/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hadoop.hive.ql.exec;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;


/**
 * After this job closed, some data (feedbacks) can be passed to descendant jobs.
 * Currently the feedbacks is a generic data structure that has a key (FeedBackType)
 * and an object as value. The type of the object depends on the key.
 **/

public class JobCloseFeedBack {

  // list of feedback types
  public static enum FeedBackType {
    DYNAMIC_PARTITIONS, // list of dynamic partitions generated by this MR job
    NONE
  };

  Map<FeedBackType, List<Object>> feedBacks; // one type corresponds to a list of values

  public JobCloseFeedBack() {
    feedBacks = new HashMap<FeedBackType, List<Object>>();
  }

  public void append(FeedBackType t, Object v) {
    List<Object> vals = feedBacks.get(t);
    if (vals == null) {
      vals = new ArrayList<Object>();
   }
    vals.add(v);
    feedBacks.put(t, vals);
  }

  public List<Object> get(FeedBackType t) {
    return feedBacks.get(t);
  }
}