/*
 * Copyright (C) 2008 The Guava Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.google.common.primitives;

import com.google.common.collect.testing.Helpers;
import com.google.common.testing.NullPointerTester;
import com.google.common.testing.SerializableTester;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.Random;
import junit.framework.TestCase;

/**
 * Unit test for {@link UnsignedBytes}.
 *
 * @author Kevin Bourrillion
 * @author Louis Wasserman
 */
public class UnsignedBytesTest extends TestCase {
  private static final byte LEAST = 0;
  private static final byte GREATEST = (byte) 255;

  // Only in this class, VALUES must be strictly ascending
  private static final byte[] VALUES =
      {LEAST, 127, (byte) 128, (byte) 129, GREATEST};

  public void testToInt() {
    assertEquals(0, UnsignedBytes.toInt((byte) 0));
    assertEquals(1, UnsignedBytes.toInt((byte) 1));
    assertEquals(127, UnsignedBytes.toInt((byte) 127));
    assertEquals(128, UnsignedBytes.toInt((byte) -128));
    assertEquals(129, UnsignedBytes.toInt((byte) -127));
    assertEquals(255, UnsignedBytes.toInt((byte) -1));
  }

  public void testCheckedCast() {
    for (byte value : VALUES) {
      assertEquals(value,
          UnsignedBytes.checkedCast(UnsignedBytes.toInt(value)));
    }
    assertCastFails(256L);
    assertCastFails(-1L);
    assertCastFails(Long.MAX_VALUE);
    assertCastFails(Long.MIN_VALUE);
  }

  public void testSaturatedCast() {
    for (byte value : VALUES) {
      assertEquals(value,
          UnsignedBytes.saturatedCast(UnsignedBytes.toInt(value)));
    }
    assertEquals(GREATEST, UnsignedBytes.saturatedCast(256L));
    assertEquals(LEAST, UnsignedBytes.saturatedCast(-1L));
    assertEquals(GREATEST, UnsignedBytes.saturatedCast(Long.MAX_VALUE));
    assertEquals(LEAST, UnsignedBytes.saturatedCast(Long.MIN_VALUE));
  }

  private static void assertCastFails(long value) {
    try {
      UnsignedBytes.checkedCast(value);
      fail("Cast to byte should have failed: " + value);
    } catch (IllegalArgumentException ex) {
      assertTrue(value + " not found in exception text: " + ex.getMessage(),
          ex.getMessage().contains(String.valueOf(value)));
    }
  }

  public void testCompare() {
    // This is the only ordering for primitives that does not have a
    // corresponding Comparable wrapper in java.lang.
    for (int i = 0; i < VALUES.length; i++) {
      for (int j = 0; j < VALUES.length; j++) {
        byte x = VALUES[i];
        byte y = VALUES[j];
        // note: spec requires only that the sign is the same
        assertEquals(x + ", " + y,
                     Math.signum(UnsignedBytes.compare(x, y)),
                     Math.signum(Ints.compare(i, j)));
      }
    }
  }

  public void testMax_noArgs() {
    try {
      UnsignedBytes.max();
      fail();
    } catch (IllegalArgumentException expected) {
    }
  }

  public void testMax() {
    assertEquals(LEAST, UnsignedBytes.max(LEAST));
    assertEquals(GREATEST, UnsignedBytes.max(GREATEST));
    assertEquals((byte) 255, UnsignedBytes.max(
        (byte) 0, (byte) -128, (byte) -1, (byte) 127, (byte) 1));
  }

  public void testMin_noArgs() {
    try {
      UnsignedBytes.min();
      fail();
    } catch (IllegalArgumentException expected) {
    }
  }

  public void testMin() {
    assertEquals(LEAST, UnsignedBytes.min(LEAST));
    assertEquals(GREATEST, UnsignedBytes.min(GREATEST));
    assertEquals((byte) 0, UnsignedBytes.min(
        (byte) 0, (byte) -128, (byte) -1, (byte) 127, (byte) 1));
    assertEquals((byte) 0, UnsignedBytes.min(
        (byte) -1, (byte) 127, (byte) 1, (byte) -128, (byte) 0));
  }

  private static void assertParseFails(String value) {
    try {
      UnsignedBytes.parseUnsignedByte(value);
      fail();
    } catch (NumberFormatException expected) {
    }
  }

  public void testParseUnsignedByte() {
    // We can easily afford to test this exhaustively.
    for (int i = 0; i <= 0xff; i++) {
      assertEquals((byte) i, UnsignedBytes.parseUnsignedByte(Integer.toString(i)));
    }
    assertParseFails("1000");
    assertParseFails("-1");
    assertParseFails("-128");
    assertParseFails("256");
  }

  public void testMaxValue() {
    assertTrue(UnsignedBytes
        .compare(UnsignedBytes.MAX_VALUE, (byte) (UnsignedBytes.MAX_VALUE + 1)) > 0);
  }

  private static void assertParseFails(String value, int radix) {
    try {
      UnsignedBytes.parseUnsignedByte(value, radix);
      fail();
    } catch (NumberFormatException expected) {
    }
  }

  public void testParseUnsignedByteWithRadix() throws NumberFormatException {
    // We can easily afford to test this exhaustively.
    for (int radix = Character.MIN_RADIX; radix <= Character.MAX_RADIX; radix++) {
      for (int i = 0; i <= 0xff; i++) {
        assertEquals((byte) i, UnsignedBytes.parseUnsignedByte(Integer.toString(i, radix), radix));
      }
      assertParseFails(Integer.toString(1000, radix), radix);
      assertParseFails(Integer.toString(-1, radix), radix);
      assertParseFails(Integer.toString(-128, radix), radix);
      assertParseFails(Integer.toString(256, radix), radix);
    }
  }

  public void testParseUnsignedByteThrowsExceptionForInvalidRadix() {
    // Valid radix values are Character.MIN_RADIX to Character.MAX_RADIX,
    // inclusive.
    try {
      UnsignedBytes.parseUnsignedByte("0", Character.MIN_RADIX - 1);
      fail();
    } catch (NumberFormatException expected) {
    }

    try {
      UnsignedBytes.parseUnsignedByte("0", Character.MAX_RADIX + 1);
      fail();
    } catch (NumberFormatException expected) {
    }

    // The radix is used as an array index, so try a negative value.
    try {
      UnsignedBytes.parseUnsignedByte("0", -1);
      fail();
    } catch (NumberFormatException expected) {
    }
  }

  public void testToString() {
    // We can easily afford to test this exhaustively.
    for (int i = 0; i <= 0xff; i++) {
      assertEquals(Integer.toString(i), UnsignedBytes.toString((byte) i));
    }
  }

  public void testToStringWithRadix() {
    // We can easily afford to test this exhaustively.
    for (int radix = Character.MIN_RADIX; radix <= Character.MAX_RADIX; radix++) {
      for (int i = 0; i <= 0xff; i++) {
        assertEquals(Integer.toString(i, radix), UnsignedBytes.toString((byte) i, radix));
      }
    }
  }

  public void testJoin() {
    assertEquals("", UnsignedBytes.join(",", new byte[] {}));
    assertEquals("1", UnsignedBytes.join(",", new byte[] {(byte) 1}));
    assertEquals("1,2", UnsignedBytes.join(",", (byte) 1, (byte) 2));
    assertEquals("123", UnsignedBytes.join("", (byte) 1, (byte) 2, (byte) 3));
    assertEquals("128,255", UnsignedBytes.join(",", (byte) 128, (byte) -1));
  }

  private static String unsafeComparatorClassName() {
    return UnsignedBytes.LexicographicalComparatorHolder.class.getName()
        + "$UnsafeComparator";
  }

  private static boolean unsafeComparatorAvailable() {
    // See Java Puzzler #44
    // Use reflection instead of catching NoClassDefFoundError
    try {
      Class.forName(unsafeComparatorClassName());
      return true;
    } catch (Error | ClassNotFoundException tolerable) {
      /*
       * We're probably running on Android.
       *
       * A note on exception types:
       *
       * Android API level 10 throws ExceptionInInitializerError the first time and
       * ClassNotFoundException thereafter.
       *
       * Android API level 26 and JVM8 both let our Error propagate directly the first time and
       * throw NoClassDefFoundError thereafter. This is the proper behavior according to the spec.
       * See https://docs.oracle.com/javase/specs/jls/se8/html/jls-12.html#jls-12.4.2 (steps #11 and
       * #5).
       *
       * Note that that "first time" might happen in a test other than
       * testLexicographicalComparatorChoice!
       */
      return false;
    }
  }

  public void testLexicographicalComparatorChoice() throws Exception {
    Comparator<byte[]> defaultComparator =
        UnsignedBytes.lexicographicalComparator();
    assertNotNull(defaultComparator);
    assertSame(defaultComparator,
        UnsignedBytes.lexicographicalComparator());
    if (unsafeComparatorAvailable()) {
      assertSame(defaultComparator.getClass(),
          Class.forName(unsafeComparatorClassName()));
    } else {
      assertSame(defaultComparator,
          UnsignedBytes.lexicographicalComparatorJavaImpl());
    }
  }

  public void testLexicographicalComparator() {
    List<byte[]> ordered = Arrays.asList(
        new byte[] {},
        new byte[] {LEAST},
        new byte[] {LEAST, LEAST},
        new byte[] {LEAST, (byte) 1},
        new byte[] {(byte) 1},
        new byte[] {(byte) 1, LEAST},
        new byte[] {GREATEST, GREATEST - (byte) 1},
        new byte[] {GREATEST, GREATEST},
        new byte[] {GREATEST, GREATEST, GREATEST});

    // The Unsafe implementation if it's available. Otherwise, the Java implementation.
    Comparator<byte[]> comparator = UnsignedBytes.lexicographicalComparator();
    Helpers.testComparator(comparator, ordered);
    assertSame(comparator, SerializableTester.reserialize(comparator));

    // The Java implementation.
    Comparator<byte[]> javaImpl = UnsignedBytes.lexicographicalComparatorJavaImpl();
    Helpers.testComparator(javaImpl, ordered);
    assertSame(javaImpl, SerializableTester.reserialize(javaImpl));
  }

  @SuppressWarnings("unchecked")
  public void testLexicographicalComparatorLongInputs() {
    Random rnd = new Random();
    for (Comparator<byte[]> comparator : Arrays.asList(
        UnsignedBytes.lexicographicalComparator(),
        UnsignedBytes.lexicographicalComparatorJavaImpl())) {
      for (int trials = 10; trials-- > 0; ) {
        byte[] left = new byte[1 + rnd.nextInt(32)];
        rnd.nextBytes(left);
        byte[] right = left.clone();
        assertTrue(comparator.compare(left, right) == 0);
        int i = rnd.nextInt(left.length);
        left[i] ^= (byte) (1 + rnd.nextInt(255));
        assertTrue(comparator.compare(left, right) != 0);
        assertEquals(
            comparator.compare(left, right) > 0,
            UnsignedBytes.compare(left[i], right[i]) > 0);
      }
    }
  }

  public void testSort() {
    testSort(new byte[] {}, new byte[] {});
    testSort(new byte[] {2}, new byte[] {2});
    testSort(new byte[] {2, 1, 0}, new byte[] {0, 1, 2});
    testSort(new byte[] {2, GREATEST, 1, LEAST}, new byte[] {LEAST, 1, 2, GREATEST});
  }

  static void testSort(byte[] input, byte[] expected) {
    input = Arrays.copyOf(input, input.length);
    UnsignedBytes.sort(input);
    assertTrue(Arrays.equals(expected, input));
  }

  public void testSortIndexed() {
    testSort(new byte[] {}, 0, 0, new byte[] {});
    testSort(new byte[] {2}, 0, 1, new byte[] {2});
    testSort(new byte[] {2, 1, 0}, 0, 2, new byte[] {1, 2, 0});
    testSort(new byte[] {2, GREATEST, 1, LEAST}, 1, 4, new byte[] {2, LEAST, 1, GREATEST});
  }

  static void testSort(byte[] input, int from, int to, byte[] expected) {
    input = Arrays.copyOf(input, input.length);
    UnsignedBytes.sort(input, from, to);
    assertTrue(Arrays.equals(expected, input));
  }

  public void testSortDescending() {
    testSortDescending(new byte[] {}, new byte[] {});
    testSortDescending(new byte[] {1}, new byte[] {1});
    testSortDescending(new byte[] {1, 2}, new byte[] {2, 1});
    testSortDescending(new byte[] {1, 3, 1}, new byte[] {3, 1, 1});
    testSortDescending(
        new byte[] {GREATEST - 1, 1, GREATEST - 2, 2},
        new byte[] {GREATEST - 1, GREATEST - 2, 2, 1});
  }

  public void testSortDescendingIndexed() {
    testSortDescending(new byte[] {}, 0, 0, new byte[] {});
    testSortDescending(new byte[] {1}, 0, 1, new byte[] {1});
    testSortDescending(new byte[] {1, 2}, 0, 2, new byte[] {2, 1});
    testSortDescending(new byte[] {1, 3, 1}, 0, 2, new byte[] {3, 1, 1});
    testSortDescending(new byte[] {1, 3, 1}, 0, 1, new byte[] {1, 3, 1});
    testSortDescending(
        new byte[] {GREATEST - 1, 1, GREATEST - 2, 2},
        1,
        3,
        new byte[] {GREATEST - 1, GREATEST - 2, 1, 2});
  }

  private static void testSortDescending(byte[] input, byte[] expectedOutput) {
    input = Arrays.copyOf(input, input.length);
    UnsignedBytes.sortDescending(input);
    assertTrue(Arrays.equals(expectedOutput, input));
  }

  private static void testSortDescending(
      byte[] input, int fromIndex, int toIndex, byte[] expectedOutput) {
    input = Arrays.copyOf(input, input.length);
    UnsignedBytes.sortDescending(input, fromIndex, toIndex);
    assertTrue(Arrays.equals(expectedOutput, input));
  }

  public void testNulls() {
    new NullPointerTester().testAllPublicStaticMethods(UnsignedBytes.class);
  }
}
