import 'package:fl_chart/fl_chart.dart';
import 'package:fl_chart/src/chart/line_chart/line_chart_painter.dart';
import 'package:fl_chart/src/chart/line_chart/line_chart_renderer.dart';
import 'package:fl_chart/src/utils/canvas_wrapper.dart';
import 'package:flutter/cupertino.dart';
import 'package:flutter_test/flutter_test.dart';
import 'package:fl_chart/src/chart/base/base_chart/base_chart_painter.dart';
import 'package:mockito/annotations.dart';
import 'package:mockito/mockito.dart';
import '../data_pool.dart';
import 'line_chart_renderer_test.mocks.dart';

@GenerateMocks([Canvas, PaintingContext, BuildContext, LineChartPainter])
void main() {
  group('LineChartRenderer', () {
    final LineChartData data = LineChartData(
        titlesData: FlTitlesData(
      leftTitles: SideTitles(reservedSize: 12, margin: 8, showTitles: true),
      rightTitles: SideTitles(reservedSize: 44, margin: 20, showTitles: true),
      topTitles: SideTitles(showTitles: false),
      bottomTitles: SideTitles(showTitles: false),
    ));

    final LineChartData targetData = LineChartData(
        titlesData: FlTitlesData(
      leftTitles: SideTitles(reservedSize: 0, margin: 8, showTitles: true),
      rightTitles: SideTitles(reservedSize: 0, margin: 20, showTitles: true),
      topTitles: SideTitles(showTitles: false),
      bottomTitles: SideTitles(showTitles: false),
    ));

    const textScale = 4.0;

    MockBuildContext _mockBuildContext = MockBuildContext();
    RenderLineChart renderLineChart = RenderLineChart(
      _mockBuildContext,
      data,
      targetData,
      textScale,
    );

    MockLineChartPainter _mockPainter = MockLineChartPainter();
    MockPaintingContext _mockPaintingContext = MockPaintingContext();
    MockCanvas _mockCanvas = MockCanvas();
    Size _mockSize = const Size(44, 44);
    when(_mockPaintingContext.canvas)
        .thenAnswer((realInvocation) => _mockCanvas);
    renderLineChart.mockTestSize = _mockSize;
    renderLineChart.painter = _mockPainter;

    test('test 1 correct data set', () {
      expect(renderLineChart.data == data, true);
      expect(renderLineChart.data == targetData, false);
      expect(renderLineChart.targetData == targetData, true);
      expect(renderLineChart.textScale == textScale, true);
      expect(renderLineChart.paintHolder.data == data, true);
      expect(renderLineChart.paintHolder.targetData == targetData, true);
      expect(renderLineChart.paintHolder.textScale == textScale, true);
    });

    test('test 2 check paint function', () {
      renderLineChart.paint(_mockPaintingContext, const Offset(10, 10));
      verify(_mockCanvas.save()).called(1);
      verify(_mockCanvas.translate(10, 10)).called(1);
      final result = verify(_mockPainter.paint(any, captureAny, captureAny));
      expect(result.callCount, 1);

      final canvasWrapper = result.captured[0] as CanvasWrapper;
      expect(canvasWrapper.size, const Size(44, 44));
      expect(canvasWrapper.canvas, _mockCanvas);

      final paintHolder = result.captured[1] as PaintHolder;
      expect(paintHolder.data, data);
      expect(paintHolder.targetData, targetData);
      expect(paintHolder.textScale, textScale);

      verify(_mockCanvas.restore()).called(1);
    });

    test('test 3 check getResponseAtLocation function', () {
      List<Map<String, dynamic>> results = [];
      when(_mockPainter.handleTouch(captureAny, captureAny, captureAny))
          .thenAnswer((inv) {
        results.add({
          'local_position': inv.positionalArguments[0] as Offset,
          'size': inv.positionalArguments[1] as Size,
          'paint_holder': (inv.positionalArguments[2] as PaintHolder),
        });
        return MockData.lineTouchResponse1.lineBarSpots;
      });
      final touchResponse =
          renderLineChart.getResponseAtLocation(MockData.offset1);
      expect(
        touchResponse.lineBarSpots,
        MockData.lineTouchResponse1.lineBarSpots,
      );
      expect(results[0]['local_position'] as Offset, MockData.offset1);
      expect(results[0]['size'] as Size, _mockSize);
      final paintHolder = results[0]['paint_holder'] as PaintHolder;
      expect(paintHolder.data, data);
      expect(paintHolder.targetData, targetData);
      expect(paintHolder.textScale, textScale);
    });

    test('test 4 check setters', () {
      renderLineChart.data = targetData;
      renderLineChart.targetData = data;
      renderLineChart.textScale = 22;

      expect(renderLineChart.data, targetData);
      expect(renderLineChart.targetData, data);
      expect(renderLineChart.textScale, 22);
    });
  });
}
