/***********************************************************************
 * Copyright (c) 2013-2025 General Atomics Integrated Intelligence, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Apache License, Version 2.0
 * which accompanies this distribution and is available at
 * https://www.apache.org/licenses/LICENSE-2.0
 ***********************************************************************/

package org.locationtech.geomesa.utils.geohash

import com.typesafe.scalalogging.LazyLogging
import org.junit.Ignore
import org.junit.runner.RunWith
import org.locationtech.geomesa.utils.geohash.GeohashUtils._
import org.locationtech.geomesa.utils.text.WKTUtils
import org.locationtech.jts.geom._
import org.specs2.mutable.Specification
import org.specs2.runner.JUnitRunner

@RunWith(classOf[JUnitRunner])
class GeohashUtilsTest extends Specification with LazyLogging {
  val NO_VALUE: Int = -1

  // Turn this on for debugging purposes
  val DEBUG_OUTPUT = false

  // collection of WKT polygons
  val testData: Map[String, (Int, Int, String, String)] = Map(
    "[POINT] CCRi" ->(NO_VALUE, 1, "dqb0tg3k", "POINT(-78.4953560 38.0752150)"),
    "[LINE] Cherry Avenue segment" ->(NO_VALUE, 11, "dqb0mdwj", "LINESTRING(-78.5000092574703 38.0272986617359,-78.5000196719491 38.0272519798381,-78.5000300864205 38.0272190279085,-78.5000370293904 38.0271853867342,-78.5000439723542 38.027151748305,-78.5000509153117 38.027118112621,-78.5000578582629 38.0270844741902,-78.5000648011924 38.0270329867966,-78.5000648011781 38.0270165108316,-78.5000682379314 38.026999348366,-78.5000752155953 38.026982185898,-78.5000786870602 38.0269657099304,-78.5000856300045 38.0269492339602,-78.5000891014656 38.0269327579921,-78.5000960444045 38.0269162820211,-78.5001064588197 38.0269004925451,-78.5001134017528 38.0268847030715,-78.50012381616 38.0268689135938,-78.5001307590877 38.0268538106175,-78.5001411734882 38.0268387076367,-78.5001550593595 38.0268236046505,-78.5001654737524 38.0268091881659,-78.5001758881429 38.0267954581791,-78.5001897740009 38.0267810416871,-78.50059593303 38.0263663951609,-78.5007972751677 38.0261625038609)"),
    "[MULTILINE] Cherry Avenue entirety" ->(NO_VALUE, 135, "dqb0mdwj", "MULTILINESTRING((-78.5000092574703 38.0272986617359,-78.5000196719491 38.0272519798381,-78.5000300864205 38.0272190279085,-78.5000370293904 38.0271853867342,-78.5000439723542 38.027151748305,-78.5000509153117 38.027118112621,-78.5000578582629 38.0270844741902,-78.5000648011924 38.0270329867966,-78.5000648011781 38.0270165108316,-78.5000682379314 38.026999348366,-78.5000752155953 38.026982185898,-78.5000786870602 38.0269657099304,-78.5000856300045 38.0269492339602,-78.5000891014656 38.0269327579921,-78.5000960444045 38.0269162820211,-78.5001064588197 38.0269004925451,-78.5001134017528 38.0268847030715,-78.50012381616 38.0268689135938,-78.5001307590877 38.0268538106175,-78.5001411734882 38.0268387076367,-78.5001550593595 38.0268236046505,-78.5001654737524 38.0268091881659,-78.5001758881429 38.0267954581791,-78.5001897740009 38.0267810416871,-78.50059593303 38.0263663951609,-78.5007972751677 38.0261625038609),(-78.5028523013144 38.0243604125841,-78.5029182561703 38.0243350105182,-78.5029599119101 38.0243205930085,-78.5029980963034 38.0243068620707,-78.5030779364612 38.0242807730671,-78.5032133180745 38.0242505635318),(-78.5007972751677 38.0261625038609,-78.5012173130024 38.025738244047,-78.5014846420529 38.02546913345),(-78.5014846420529 38.02546913345,-78.5016408173938 38.0253105501453,-78.5017588424505 38.0251959031322,-78.5018733957502 38.0250812285921,-78.5019914201064 38.0249672952982,-78.5021094441212 38.024854020922,-78.5021337430596 38.0248279335242,-78.5021615133818 38.0248025325576,-78.5021858123053 38.0247771316478,-78.5022135826113 38.0247524171677,-78.5022413528989 38.0247277026809,-78.502269123189 38.0247036746865,-78.5023003648576 38.0246803331177,-78.5023593767632 38.0246336500237,-78.5023906184182 38.024611681429,-78.5024183886838 38.0245897128963,-78.5024531016733 38.0245677442145,-78.5024843433189 38.0245471485931,-78.5025155849473 38.0245265529633,-78.5025502979257 38.0245059572511,-78.502581539567 38.0244867620621,-78.5026162525558 38.0244675091228,-78.5026891499195 38.0244311230449,-78.5027655186506 38.0243967990852,-78.5028037030413 38.0243810087123,-78.5028523013144 38.0243604125841),(-78.4969404056816 38.0284800833846,-78.4975895903523 38.0283421148715,-78.4979714625632 38.0282611151835,-78.4983949926673 38.0281677578702,-78.4987352046273 38.0280915579077),(-78.490993782089 38.0272784134596,-78.4911499870459 38.0273827729123,-78.4912957791224 38.027476833921,-78.4914450429972 38.0275715814956,-78.4915734792506 38.0276567164341),(-78.4925003144519 38.0282437343292,-78.4925523852384 38.0282670785298,-78.4925870993641 38.0282801241573,-78.492618342024 38.0282925077758,-78.4926495847667 38.0283041554598,-78.4926842990611 38.0283158280625,-78.4927190134333 38.0283268141568,-78.492750256337 38.0283371135319,-78.4930731013898 38.0284304964614,-78.4933751186572 38.028515636636,-78.4938402963052 38.0286433804125,-78.4939826273169 38.0286797446143,-78.494128429974 38.0287168227748,-78.4942742327232 38.0287545872547,-78.4944165640021 38.0287930378917,-78.494451279106 38.0288005909897,-78.494485994217 38.0288081440776,-78.4945207094357 38.0288143241593,-78.4945519530924 38.0288205040724,-78.4945866683721 38.0288259976367,-78.4946213837063 38.0288308046928,-78.4946560990941 38.0288349252408,-78.4946907798189 38.028838359279,-78.4947255299783 38.0288417933102,-78.4947602455215 38.0288438543337,-78.4947949610667 38.028845915347,-78.4948331482303 38.0288472900024,-78.4948678638734 38.0288479779984,-78.494902579564 38.0288479794862,-78.4949372952546 38.0288479809639,-78.4949720110376 38.0288466094353,-78.4950067267276 38.0288466108928,-78.4950414424176 38.0288466123402,-78.495079629767 38.0288452409245,-78.4951143455462 38.0288438693545,-78.4951490613687 38.0288418112763,-78.495170411625 38.0288401370869,-78.4951837772336 38.02883906669,-78.4952184930959 38.0288363220936,-78.4952566458955 38.0288322046275),(-78.4915734792506 38.0276567164341,-78.4917366286941 38.0277624485909,-78.4918824220269 38.0278585683647,-78.4920282157351 38.027954687957,-78.492174009675 38.0280521776182,-78.4921983083911 38.028071401161,-78.4922260787251 38.0280899384265,-78.4922538490725 38.0281084784315,-78.492281619505 38.0281263291856,-78.4923093899508 38.0281441799332,-78.4923406320174 38.0281613443987,-78.4923684026307 38.0281778221358,-78.4923996448647 38.0281936135892,-78.4924274502879 38.0282094048163,-78.4924586579016 38.0282245097537,-78.4925003144519 38.0282437343292),(-78.4902856576172 38.0268115389132,-78.4904661595601 38.0269268852315,-78.4906015358649 38.0270175137791,-78.4907403839863 38.0271081424395,-78.4908757609528 38.027198767924,-78.490993782089 38.0272784134596),(-78.4952566458955 38.0288322046275,-78.4953261125364 38.0288232828931,-78.495360793841 38.0288177922616,-78.495426789099 38.0288054378431,-78.4954615052288 38.0287978876893,-78.4960030759596 38.0286798293809,-78.4962391448211 38.0286262901994,-78.4969404056816 38.0284800833846),(-78.488966633884 38.0258873969823,-78.4891228281985 38.0260253975831,-78.4893831600733 38.0262080297014,-78.4896469647716 38.026389975042,-78.4899072993849 38.0265712329949,-78.4900496176674 38.026659116773,-78.4901884646334 38.0267483703443,-78.4902856576172 38.0268115389132),(-78.4888597973336 38.0257928149758,-78.488966633884 38.0258873969823),(-78.5101312788355 38.020930264319,-78.5102839625207 38.0205732711847,-78.5104748140153 38.0201194779227,-78.5106205652871 38.0198551624104,-78.5106344459809 38.0198277011593,-78.5106483273497 38.0198050454089,-78.5106691480632 38.0197617940248,-78.5106795580672 38.0197377655797,-78.5106830270523 38.0197226622603),(-78.5097842230447 38.0214211408033,-78.5098709932951 38.0213428699136,-78.5098883471235 38.021325705977,-78.5099057008528 38.0213078555378,-78.5099230544826 38.021289318596,-78.5099404081038 38.0212707816515,-78.5099542905008 38.02125224774,-78.509971644014 38.0212330242907,-78.509985526304 38.0212138011293,-78.509999408495 38.021193891466,-78.5100132905866 38.021173295301,-78.5100237015513 38.0211533859259,-78.5100375836286 38.0211327897578,-78.5100479943964 38.0211115073801,-78.5101312788355 38.020930264319),(-78.509326429128 38.0217944686582,-78.5097842230447 38.0214211408033),(-78.5065492485878 38.0237719370643,-78.5065943737962 38.0237479071012,-78.5066290855314 38.0237300561942,-78.5066603258586 38.0237115189713,-78.5066915661702 38.0236929817402,-78.5067228063427 38.0236730715019,-78.5067540465606 38.0236538477547,-78.5070699172241 38.0234410146365,-78.507382314983 38.0232302403788,-78.5077433065893 38.0229947481592,-78.5080140488921 38.0228189863889,-78.5080834696148 38.0227722997325,-78.5081528902506 38.0227256130349,-78.5082223106483 38.0226775505509,-78.5082882596769 38.0226294910127,-78.5084409830377 38.0225148376076,-78.5085728791014 38.0224070477514,-78.5088505540671 38.0221825393168,-78.509326429128 38.0217944686582),(-78.5043415034512 38.0241626559166,-78.5043831595454 38.024160594896,-78.5051329697906 38.0241372240259,-78.5052405813537 38.0241331003531,-78.5053516642054 38.0241282899285,-78.5054592756933 38.0241234795593,-78.5055911864062 38.0241159219599,-78.5056154856949 38.0241138613212,-78.5056397849823 38.0241118006775,-78.5056606128154 38.0241083671949,-78.5056849119952 38.0241049335434,-78.5057057397719 38.0241008135537,-78.5057265675463 38.0240966935604,-78.5059140149657 38.0240287182419,-78.506129232103 38.0239525087043,-78.5062507595161 38.0239017014502,-78.5063930447698 38.0238440279676,-78.5064277568933 38.0238296096165,-78.5064937095269 38.0237980270754,-78.5065492485878 38.0237719370643),(-78.5032133180745 38.0242505635318,-78.5032584453345 38.0242423243172,-78.5033035725845 38.0242340850855,-78.5033486998554 38.024226532336,-78.5033938271483 38.0242196660686,-78.5034285404352 38.0242141730823,-78.5034389544329 38.0242127997841,-78.5034875531266 38.0242073063797,-78.5035326804601 38.0242018130583,-78.5035778078527 38.0241976927183,-78.5036229352404 38.0241935723613,-78.5036715340093 38.0241901383797,-78.5037166614555 38.0241873909858,-78.5041506143037 38.0241709006996,-78.5043415034512 38.0241626559166),(-78.4987352046273 38.0280915579077,-78.4987629770009 38.0280853797143,-78.4988636517995 38.0280640992688,-78.4989157249487 38.0280530910668,-78.4990163997051 38.0280284027148,-78.4990650013019 38.0280153623839,-78.49916220446 38.0279879031741,-78.4992142775479 38.027973487064,-78.4992628790947 38.0279583844115,-78.4993114806282 38.0279425952407,-78.4993566106209 38.0279261195331,-78.4993843829211 38.0279151357065,-78.4994121552185 38.0279034653752,-78.4994399275125 38.027891108539,-78.4994642282732 38.0278787516807,-78.4994920005586 38.0278650218356,-78.4995163013023 38.0278526649667,-78.4995169192356 38.027852313482),(-78.4995169192356 38.027852313482,-78.499544073581 38.0278368756143,-78.4995683743188 38.0278224592396,-78.4995926750509 38.0278073563615,-78.4996135042567 38.0277915669688,-78.4996378049763 38.0277750910847,-78.4996621056852 38.0277586151955,-78.4996864063977 38.0277387068094,-78.4997107071 38.0277181119199,-78.4997315362733 38.0276975170175,-78.4997558369519 38.0276762356201,-78.4997766661031 38.0276549542112,-78.4998009667569 38.0276329835598,-78.4998217958864 38.0276103291459,-78.4998426250034 38.0275876500141,-78.4998599825995 38.02756433655,-78.4998808116931 38.0275409956268,-78.4998946977563 38.027519714189,-78.4999085838116 38.0274984327495,-78.4999224698597 38.0274764922697,-78.4999363558997 38.0274544968681,-78.4999432989171 38.0274428263983,-78.4999502419314 38.0274325289248,-78.4999606564532 38.0274098744801,-78.4999710709686 38.0273872200343,-78.4999814854776 38.0273645655874,-78.4999918999804 38.0273419111395,-78.5000023144768 38.0273185701919,-78.5000092574703 38.0272986617359),(-78.4995169192356 38.027852313482,-78.4995641309868 38.027867842256,-78.4996028531833 38.0278762156112,-78.499653086986 38.0278796098155,-78.4997536240732 38.0278750709385,-78.4999020378101 38.0278615217351,-78.499997516849 38.0278502144613))"),
    "[POLYGON] Charlottesville" ->(41, 95, "dqb0wb", "POLYGON((-78.4480347940116 38.0574871002506,-78.4478073388056 38.0570613767181,-78.447489602492 38.0556330888149,-78.4475613331476 38.0543729966714,-78.4474913359404 38.0531688593731,-78.4471033311625 38.0510124710374,-78.4468559007404 38.0503403123925,-78.4465022444764 38.049724095861,-78.4463611629831 38.0489399478115,-78.4466451302342 38.0479879914622,-78.4468935521083 38.047204016976,-78.4471774068971 38.0463920759449,-78.4473194556308 38.0458600786071,-78.4478506660003 38.0457483035168,-78.4481201134084 38.0456589845029,-78.4483408857566 38.0455080790522,-78.4489555764892 38.045281031607,-78.4495205621204 38.0452264072598,-78.4500656308001 38.0453050341156,-78.4515619871487 38.0456975562908,-78.4520278127326 38.0457369294639,-78.4525531004782 38.0457528189189,-78.4530387390881 38.0457608369821,-78.4539307629378 38.0458003696298,-78.4544066445935 38.0455810939866,-78.4546049904878 38.0454008962847,-78.4547042622303 38.0451187575263,-78.45477390994 38.0446720136184,-78.454883347649 38.0439980001295,-78.4547547718563 38.0435355014301,-78.4546195521738 38.0432666188877,-78.4544509624837 38.0428886257244,-78.4541980272186 38.0424661425413,-78.4538609688581 38.0417545844943,-78.4538050917856 38.0412654764507,-78.4538055456975 38.0405318605387,-78.4538903299325 38.0398649219979,-78.4540878941475 38.0387311822199,-78.4545383580856 38.038042167862,-78.4548198857144 38.0375532151034,-78.4552983572464 38.0369753656226,-78.4559737270544 38.0363753444128,-78.456311483729 38.0358863811039,-78.4569586439765 38.0352863437544,-78.4578027512843 38.0346864274753,-78.4584497458772 38.0345310119336,-78.4587028468452 38.0343976997343,-78.4589280721178 38.0341310048098,-78.4593782143373 38.0337087710827,-78.4597159233878 38.0333087402011,-78.4599973635754 38.0328863920337,-78.4601101325906 38.0324196117035,-78.4602229831227 38.0317971610709,-78.4604555397948 38.0307727369184,-78.4604892247959 38.0304259569015,-78.4606033941186 38.030055807579,-78.4610596435138 38.0293740452038,-78.4612745012046 38.0287701341041,-78.4611528322604 38.0282181499898,-78.4610469607118 38.0279985183977,-78.4605625005067 38.0277511374944,-78.460140841638 38.0275064392059,-78.4598315088329 38.0274173927489,-78.4594377589672 38.0274395021251,-78.4591002192831 38.0276394325831,-78.4587063573619 38.027861667526,-78.4583044761852 38.0280797244402,-78.4580512082664 38.0282998918073,-78.4578485323197 38.0286201679383,-78.4578059888932 38.0287728301014,-78.4575527254272 38.0290395134682,-78.4572433152976 38.0292616910978,-78.4569057267536 38.0294616701211,-78.4565316621609 38.0295807928969,-78.4563291050185 38.0296808378124,-78.4559240253018 38.0297607894811,-78.4555190271487 38.0297606392271,-78.455063402285 38.0297204319888,-78.4545064867333 38.0297202210031,-78.4542281271579 38.0295599134104,-78.4538992023681 38.0293195125074,-78.4535702921431 38.0290590923964,-78.4533098174528 38.028895878825,-78.4529346449186 38.0285099743953,-78.452579576467 38.0281240492591,-78.4524190060142 38.0275979353443,-78.4524326194999 38.0272628471103,-78.4525485912018 38.0265699140724,-78.4527462175347 38.0253694706732,-78.45302804476 38.0243469178326,-78.4539571672297 38.0225243215814,-78.45446402034 38.0214796357505,-78.4566010541373 38.0213470039148,-78.4569946318887 38.0215694625687,-78.4576715632512 38.0218640483203,-78.4582210821412 38.0221370295646,-78.4587572923289 38.0223164474586,-78.4593404019826 38.022495906379,-78.4603190447203 38.0226520987013,-78.4610966810329 38.0226523547397,-78.4618609491686 38.0225240063101,-78.4620689327232 38.0222123467767,-78.4625586494422 38.0214682256788,-78.4630953023495 38.0207631101049,-78.4636183940394 38.0202995810012,-78.4669173627013 38.0182586963005,-78.4683459993248 38.0162678737613,-78.4690301031858 38.015328974238,-78.4690972794554 38.0150445328578,-78.4689097864308 38.0147788080137,-78.4681435065681 38.0147112463513,-78.4695068845434 38.0121518287155,-78.4697058077266 38.0115231804633,-78.4720476723718 38.0120760020776,-78.4722822297623 38.012165686542,-78.4728652806575 38.0124736481891,-78.4732539371793 38.0126802632721,-78.4735019203215 38.0127972161935,-78.4739241493419 38.0128869389348,-78.4742793401881 38.0130272821437,-78.4744401479851 38.0132182466607,-78.4746344135635 38.0136196446557,-78.4746147418369 38.014106535084,-78.4746145815649 38.0145780793959,-78.4749267443022 38.0147128644932,-78.4751367221966 38.0147497873009,-78.4753511822194 38.0148784272864,-78.4755790828017 38.0148784741459,-78.4759879984775 38.0147499616421,-78.4765377258092 38.0145162759772,-78.4768796136275 38.0143916465332,-78.4772348718139 38.0143527489991,-78.4774812482407 38.0141969323815,-78.4780459914984 38.0140957099536,-78.4785030769381 38.0138378606924,-78.4787585686047 38.0135909030511,-78.4790140743559 38.0134113318813,-78.4793130472011 38.0133088206864,-78.4792979294484 38.0132541999618,-78.4794114855215 38.0131643976797,-78.4795533919009 38.0130970624964,-78.4797237077311 38.0129623723861,-78.4801211081832 38.0127378981307,-78.4805752656698 38.0124685070802,-78.4812504705776 38.0120778030625,-78.4815588636228 38.0118830217514,-78.48193421173 38.0118090467528,-78.4823634166166 38.0117726988772,-78.4827040194005 38.0117053891172,-78.4829594284038 38.0117278882263,-78.483158044834 38.0119524576319,-78.4831863852978 38.0121770028158,-78.4831579435691 38.0124015399732,-78.4831862993666 38.0125587255253,-78.4831578676165 38.0127383654107,-78.483186208378 38.01296291053,-78.48315777648 38.0131425229287,-78.4832428456305 38.0134568979172,-78.4831576448324 38.0137263242109,-78.4832143676419 38.0139957979267,-78.4832710511268 38.0142877064616,-78.4832993576695 38.0146694328523,-78.483294176284 38.0148681057083,-78.4835689419438 38.0151487033349,-78.4839911545882 38.0154332195007,-78.4845608651763 38.0156515196597,-78.4850635625441 38.0158308154324,-78.4856802143406 38.0160101502648,-78.486216428314 38.0160998155625,-78.4864644572302 38.0161388097697,-78.486705731891 38.0162947276079,-78.4870475493854 38.016524660737,-78.4872888351376 38.0166298861277,-78.4874832092389 38.0166416046549,-78.4878451980971 38.0166533400566,-78.4881895268302 38.0166197635353,-78.4885751882535 38.0166293297195,-78.4892139373057 38.0167628724533,-78.4895513766294 38.0168487151201,-78.4897200828032 38.016991741685,-78.4899731548683 38.0172492011166,-78.4902744340558 38.017296896683,-78.4906962792789 38.0172111176902,-78.490997615321 38.0170776575388,-78.4911904373362 38.01702047266,-78.4915038313264 38.0167535288296,-78.4919555597132 38.0166463011954,-78.492296169162 38.0163094987113,-78.4927787035232 38.0159727313467,-78.492977396659 38.0158155618519,-78.4970928967327 38.013682580216,-78.4984835788185 38.0131437019278,-78.4991931234303 38.0130538869042,-78.4994485433345 38.0128967342962,-78.4998437664113 38.0129287539271,-78.5003464802363 38.0126988302277,-78.5005877707069 38.0125936296879,-78.5008022656036 38.0124650328431,-78.5012178138321 38.0117558449659,-78.5014121704333 38.0113973512408,-78.5019743859181 38.0109207457232,-78.5022865703558 38.0106512741263,-78.5030244315817 38.0102919970931,-78.5035920201411 38.0100674400171,-78.5038474587522 38.009977610021,-78.5042163882422 38.0098878029786,-78.5069125259747 38.0106735581698,-78.5075652923892 38.0109429571282,-78.5079058876806 38.011055219413,-78.5082180759887 38.0110103010495,-78.5084167740383 38.0113021880897,-78.5084164376538 38.0113970627614,-78.5086443262663 38.0113853484664,-78.5089191301338 38.0112956984722,-78.5091134637878 38.0110618884754,-78.509327925494 38.0108202779,-78.5097221726638 38.0106060026718,-78.5100910435866 38.0101119921141,-78.510172326888 38.0098538326757,-78.5103666782886 38.009803179494,-78.510561032726 38.0097758670937,-78.5107554239778 38.0097641791796,-78.5109028585516 38.0097641657985,-78.5110637229321 38.0097524529809,-78.5113116930953 38.0096861959273,-78.5115060839921 38.0096744793298,-78.5118143923223 38.0097251131168,-78.51208919889 38.0097133875798,-78.5123974948409 38.0096860880482,-78.5128532757285 38.0097522736227,-78.5133425765796 38.0097755884693,-78.5137916582726 38.0098301006569,-78.5140195624705 38.0099313744627,-78.5142474606026 38.0099976087244,-78.5144284247841 38.0100092575271,-78.5147233296044 38.0100092215012,-78.5148841908108 38.0099819336493,-78.5151590002155 38.0099818990516,-78.5154002931798 38.0100325320629,-78.5155678583104 38.0100325103175,-78.5159500924788 38.0099007611223,-78.5161040543046 38.0100714050544,-78.5162783374999 38.0101610112144,-78.5165866805846 38.0101882366411,-78.51686818656 38.0101764991678,-78.5171564025364 38.0102154239737,-78.517338388399 38.0102882218848,-78.5175629930014 38.0103528026774,-78.5177127478026 38.0104725340821,-78.5179523654401 38.0104703289214,-78.5183213079293 38.0104927077501,-78.5186051419301 38.0106274105848,-78.5189457466686 38.0107171515413,-78.5192052153416 38.0107061260168,-78.5193582269777 38.0109250681797,-78.5193364314505 38.0111958832601,-78.5192491003715 38.011437903255,-78.5188340352807 38.011587791796,-78.518790383266 38.0117664266146,-78.5187904513964 38.0120372380663,-78.5188542773856 38.0123051012037,-78.5187978170053 38.0124809096119,-78.5187978619625 38.0126595374826,-78.5188197803747 38.0128669675239,-78.518834398527 38.0130282932169,-78.5187702286445 38.0133277280702,-78.5187421123125 38.0135945346255,-78.5188264867472 38.0138612959925,-78.518967153063 38.0142392341728,-78.5190187020363 38.0146727106298,-78.5189816270339 38.0150600682197,-78.5189321045338 38.0153640864355,-78.5189570312897 38.0157955623766,-78.5189942576665 38.016001479411,-78.5190718123973 38.0163250837603,-78.5192082876974 38.0168497135523,-78.51928587209 38.0172811807012,-78.5196392464914 38.0176341493337,-78.51974439766 38.0176838071893,-78.5197818465788 38.0178008356633,-78.5199740272022 38.0180165571445,-78.520141468358 38.0182626532725,-78.5202407303894 38.0183509753966,-78.5206129061057 38.0186256492796,-78.5206187989933 38.0187224997777,-78.5207121970512 38.0189396374862,-78.5212332036055 38.0190376605846,-78.521369581847 38.0189100844276,-78.5215555623747 38.0186647231039,-78.5216547924628 38.0186352679885,-78.5219028686934 38.0186254194828,-78.522212909395 38.0185174168012,-78.5223492903543 38.0182916973492,-78.522510492367 38.0181935523554,-78.5227214795118 38.018586026041,-78.5228456143743 38.0188411059911,-78.5229200980795 38.0190471517333,-78.5233544962418 38.0200087173655,-78.5233919406358 38.0207544415267,-78.5233920266994 38.0210292063194,-78.5235078546165 38.0213402779211,-78.5237397828542 38.022373441787,-78.5237706683738 38.0225640079623,-78.5234316582873 38.0231006980395,-78.5233106534283 38.0235606764049,-78.5233351467658 38.0244039399456,-78.523214228567 38.0251513963273,-78.5229980644635 38.0263906785026,-78.5200486370255 38.0260951647006,-78.520227651243 38.0285005977381,-78.5197027539389 38.0285063148315,-78.5197571739915 38.0285849785253,-78.5204409624135 38.0297112143958,-78.5201563565788 38.0303420169721,-78.5199075279322 38.030411065485,-78.5200174522885 38.0309744968867,-78.5202288921318 38.0330812115492,-78.5168795599589 38.0368524653907,-78.5156722277629 38.0368556506035,-78.5169224544723 38.0399869709215,-78.5162065004802 38.0404224180538,-78.5173086846758 38.0411645841191,-78.5168088996771 38.0422223517956,-78.5159817434742 38.04346924924,-78.5138958425554 38.0441253045007,-78.5145429779185 38.0448723526332,-78.507615482469 38.0562477092103,-78.5011431310711 38.0576531933158,-78.5009060110943 38.0584962880908,-78.5024471981014 38.0605196549688,-78.4949477146895 38.0652055763605,-78.4935526583464 38.0642469511724,-78.4925801543764 38.0635844877425,-78.492157628492 38.0630467200575,-78.4914220549693 38.0614749931234,-78.4912323721064 38.0616248537584,-78.4913034335911 38.0619058528612,-78.4868925118207 38.066889139645,-78.486513103012 38.0671700907363,-78.4851136285658 38.0690434872368,-78.4805851960675 38.0666634146305,-78.4802984729844 38.0657417981749,-78.4788228617472 38.0655700882117,-78.4774278973046 38.0649697592028,-78.4761327734651 38.0667658280819,-78.4745427740881 38.0676149601349,-78.4748174301878 38.0686982688173,-78.4729253470527 38.0700538600024,-78.4725295078174 38.0702446954685,-78.4717311577653 38.0705250627835,-78.4709935743572 38.0697689534562,-78.4722083419684 38.0680235864848,-78.4714503964357 38.0679766441182,-78.4712491169607 38.0681480530514,-78.4695137245745 38.0683514820538,-78.4695249339711 38.0679499444058,-78.4682882003939 38.0681111118,-78.4681945631601 38.0676091543367,-78.4668418676903 38.0678226411344,-78.4662524286672 38.0668374753678,-78.4653139480169 38.0656097925943,-78.4644758509069 38.0648692093359,-78.465335243717 38.0630575427858,-78.4663085789463 38.0613277305874,-78.4664494470514 38.061327770253,-78.4673892136686 38.0595317061067,-78.4687248297256 38.0572798254838,-78.4698455345863 38.0556396474866,-78.4711807511563 38.0540501730135,-78.4724353916266 38.0526164977684,-78.4720860496955 38.0519420067951,-78.471652148566 38.0514699274272,-78.4715798628713 38.051341206666,-78.4714714256453 38.0511266382172,-78.4713991679905 38.0509264122871,-78.4713992611406 38.0506832841434,-78.4717790748334 38.0502706003002,-78.4720943686619 38.0499394554518,-78.4723560322113 38.0496394915541,-78.4724164669572 38.0494173839294,-78.4723360778499 38.0492303372208,-78.4722499477388 38.0489815322078,-78.4718600453255 38.0489146039865,-78.4713772585177 38.0488365585299,-78.4710017199293 38.0488052736595,-78.4705680597292 38.0486664959493,-78.4703872918519 38.0484662149579,-78.4701343302694 38.0478368848681,-78.4699356469289 38.0472075403136,-78.4698289155783 38.0470086499745,-78.4695339345729 38.0468371439102,-78.4693053388149 38.0469393179596,-78.468407168336 38.0466034706965,-78.4676917660222 38.0462473734836,-78.4661192632977 38.045714987292,-78.4653292114835 38.047182785812,-78.4651074150484 38.048250382979,-78.4647453057632 38.0482502765916,-78.4642891032747 38.048663190151,-78.4639468173163 38.0492436680877,-78.4638729309242 38.0495125312664,-78.4573545185689 38.0497324748685,-78.4570999398137 38.0492609003817,-78.4571048708452 38.0499860033047,-78.4572469354638 38.0504060779264,-78.458898424969 38.0510299931243,-78.4567841773194 38.0556302795318,-78.4561273995112 38.0553332011668,-78.4557521877327 38.0549991258425,-78.4555445617933 38.0547458706718,-78.4552967746776 38.0545217622445,-78.4550136860866 38.0541576231591,-78.4547304813727 38.053989517875,-78.4542703578379 38.053569291964,-78.454093163388 38.053765231087,-78.4543764171386 38.0538493662526,-78.4544118048045 38.0539053700334,-78.4546595354771 38.0541574891887,-78.4541635943456 38.0544373334645,-78.4548714741119 38.0550816694308,-78.4492025035279 38.0589437591411,-78.4480347940116 38.0574871002506))"),
    "[POLYGON] Albemarle" ->(35, 100, "dqb32", "POLYGON((-78.61733461064 38.2620789403217,-78.5936962998423 38.253523378746,-78.574261734912 38.24762149438,-78.5544419453568 38.241171062083,-78.5438656867703 38.2379011417484,-78.5421218937646 38.2373241780986,-78.531165315332 38.233816464351,-78.5256856459506 38.2321153070391,-78.4974518000592 38.223778055769,-78.4576225187797 38.2116615603355,-78.4423761216009 38.2072833077172,-78.4381066034608 38.2057077775338,-78.431501506243 38.2037389239013,-78.4304261829744 38.2033972418705,-78.4186642595451 38.1998560726314,-78.418663082835 38.1998552690894,-78.4173597824121 38.1994632310289,-78.4168162878723 38.1992995546662,-78.4120247705715 38.1976715505003,-78.4110092128878 38.1973056075822,-78.4077434902283 38.1961313592223,-78.4030798431503 38.1944547862074,-78.4020261218293 38.1940750045315,-78.3998100241805 38.1930893657936,-78.3968014677309 38.1920876555747,-78.3963109149823 38.1920245910487,-78.3946308091241 38.1916522097602,-78.3902867144014 38.1906885941709,-78.388789637597 38.1902661766076,-78.3868818134582 38.1896512797612,-78.3861577749681 38.1894154994045,-78.3856498547559 38.1892578651137,-78.3802233412591 38.1877348098252,-78.3772682407839 38.1868648836326,-78.3771563432171 38.1868238944443,-78.3753747672691 38.186237325632,-78.3735947130076 38.185660277575,-78.3723616942592 38.1852556264042,-78.3698171252481 38.1844223335105,-78.3692869385873 38.1842476170376,-78.3688007419162 38.18408785012,-78.3647593222961 38.1830809398642,-78.3619221024757 38.1823217509485,-78.3590950837367 38.1816037371261,-78.349432004749 38.1791702853264,-78.3478366289742 38.1787799237807,-78.3468127791672 38.1784971119002,-78.3445400035707 38.1778579870474,-78.341674040901 38.1768980521589,-78.3372506785 38.1753285326734,-78.3322117486383 38.1737536686317,-78.330122803151 38.1730163719294,-78.3269030604609 38.1719350231476,-78.3223316347738 38.1703789182449,-78.3170626569758 38.168584455302,-78.3146353606766 38.1676976219279,-78.3116651441589 38.1667196110809,-78.3113554066084 38.1666169415539,-78.3073031900139 38.1653335554037,-78.2996518121669 38.1629090094161,-78.2900800797667 38.1595205361774,-78.2806944978057 38.1563010901782,-78.269260187315 38.1524158928273,-78.2518982579344 38.146120698923,-78.240816385499 38.1423640240007,-78.2339263580919 38.1399261376193,-78.2328279447061 38.1395085893557,-78.2093840879076 38.1311283231365,-78.2330809140659 38.0943202677469,-78.2539891077942 38.0628129882847,-78.2747728432618 38.0424294596005,-78.2849197292726 38.0298064257261,-78.3068660561558 38.0063643447747,-78.3290845657952 37.9808838111109,-78.335883709053 37.9731783404425,-78.3726406629157 37.9313804171004,-78.4684171744545 37.8226705194977,-78.4766591072236 37.813475956917,-78.4843477722843 37.8047119044,-78.4914549336677 37.7969628127867,-78.4918005409655 37.7969391155078,-78.4949957586579 37.7956715894926,-78.4966588324443 37.7940184012575,-78.4982693915354 37.7912308620109,-78.4987831512749 37.7892717155902,-78.4988545388821 37.786961502461,-78.4986119979158 37.7846626060694,-78.4971139406604 37.779543855235,-78.4963436094443 37.7765201615128,-78.4953879375431 37.7724432604933,-78.4953310671781 37.7693289793537,-78.4954310299408 37.7676416125141,-78.4957877718344 37.7659995492923,-78.4966009794955 37.7643688254491,-78.4977137081891 37.7631004925373,-78.4995396304141 37.7615943263442,-78.5015366656928 37.7605297993828,-78.5055163412595 37.7589555528868,-78.5087399336931 37.7579927573579,-78.510480071561 37.7574943336158,-78.51284783782 37.7572223114304,-78.5157006511108 37.7574371362717,-78.5203365856954 37.7580592893979,-78.5233890860896 37.7580474050313,-78.5255430083371 37.7582168296287,-78.5303219541321 37.7594274486062,-78.5348590001847 37.7615892177685,-78.5365711085293 37.762302158735,-78.53772662469 37.7624376960343,-78.5392672932494 37.7625504451241,-78.5413928799529 37.7627648912206,-78.5422488828754 37.7629571109646,-78.5427197066376 37.7631041613039,-78.5432619037418 37.7633304581622,-78.5436614063783 37.7634662104,-78.5440466935351 37.7636925625725,-78.5447743652413 37.7639414312497,-78.5455732442534 37.763997753087,-78.5465432212289 37.7639067771699,-78.5472563813283 37.7637592759765,-78.5479695566586 37.7636344180911,-78.548853972378 37.7636114118418,-78.5501378020461 37.7635542521494,-78.552263111307 37.7632928706825,-78.5551014908693 37.7628046286065,-78.5567132096936 37.7625207511756,-78.5578541705166 37.7622257615495,-78.5587810903134 37.7618515925718,-78.5596080380578 37.7613415737623,-78.5603066282177 37.7608655807942,-78.560948331752 37.760627437446,-78.5626740422313 37.7602754776601,-78.5650698138719 37.7595154369769,-78.5684493332082 37.7582904704184,-78.5701461303209 37.7576213234617,-78.5711586534564 37.7573942271304,-78.5722426432044 37.7573256263272,-78.5731553659339 37.7571552018128,-78.5737113538535 37.7568490920258,-78.5740820181824 37.7566563390849,-78.5752085177358 37.7563158932465,-78.5766058675283 37.7558393611144,-78.5779317995439 37.7552722644659,-78.5788299811109 37.7548753095128,-78.5794999276987 37.7544671717934,-78.5802270820423 37.7542062162507,-78.5807974356028 37.7540472879408,-78.5814107550471 37.7540468695809,-78.5821525925327 37.7541822551685,-78.5830799119084 37.7543628048077,-78.584406663642 37.7545996884049,-78.585448054977 37.7547461663297,-78.5861184475742 37.7547570086015,-78.5885146957308 37.7547552488766,-78.5893848560701 37.7548338710567,-78.5904263423579 37.7550482511062,-78.5915107938641 37.7553984861757,-78.5933943511342 37.7560198793147,-78.5948495160861 37.7562338991366,-78.5973171267227 37.756231905104,-78.600997134652 37.7562175130956,-78.6054335320189 37.7564967753083,-78.6071310318403 37.7565745311152,-78.6096128801918 37.7565496228373,-78.6120231100138 37.756320881638,-78.6139193838244 37.7557981457853,-78.6156297441011 37.7549697853732,-78.616798178304 37.7542212153323,-78.6184220632419 37.7528719730155,-78.6205298947831 37.7509446479685,-78.6223109690649 37.7498669824121,-78.6239498774376 37.7490952025738,-78.6254892544207 37.7485273513405,-78.6266578759761 37.7480051766781,-78.6279969282416 37.7470751143098,-78.6291931492806 37.7460432665413,-78.6310009290237 37.7440594642559,-78.6322965855605 37.7428576031618,-78.6338207475578 37.7418593200131,-78.635259974811 37.7412348292135,-78.6376394529931 37.7400996252257,-78.6393633426157 37.7392142803732,-78.6405309333693 37.7382729546607,-78.6416696344725 37.737150449606,-78.6427219246744 37.7355976978617,-78.6437022368611 37.7337166037874,-78.643922509596 37.7330713251369,-78.6458087884118 37.7326348143516,-78.646102484752 37.7326316359216,-78.6463778199832 37.7326801305324,-78.6465828696295 37.7327240640881,-78.6469167401908 37.732758535474,-78.6472213198765 37.7327860609888,-78.6476136470252 37.7327692989781,-78.6480293680667 37.7327362240677,-78.6484626583914 37.7327031257597,-78.6491183285698 37.732593005747,-78.6499701073947 37.7324454298119,-78.6503769463966 37.7323635189011,-78.6508715333687 37.7322349889558,-78.6511320115438 37.7321765220176,-78.6513808484979 37.7321529487045,-78.651799438593 37.7320896289036,-78.6521623735559 37.7320193990744,-78.6525777400095 37.7318142234305,-78.6528965092654 37.7316254474703,-78.653188777923 37.7313622862706,-78.6534610988292 37.7309302003488,-78.6537690216872 37.7302802634399,-78.6540975879557 37.7296412128944,-78.6542001571676 37.7293899991043,-78.6543786413935 37.729296972689,-78.6545503723249 37.729258541905,-78.6547016469308 37.7292965527188,-78.6549766058661 37.7293289437055,-78.6553065352272 37.729355809436,-78.6558563652961 37.7293769220502,-78.6562413658996 37.7294473705635,-78.6565232572719 37.729507041013,-78.6567844043258 37.7295066968278,-78.657107254993 37.7294353137381,-78.6573882551007 37.7290692369405,-78.6576615254215 37.7282992485137,-78.657975853187 37.727447335244,-78.6582278126714 37.7263498743812,-78.6582749023157 37.7258694777514,-78.6582941936152 37.7252417444778,-78.6582716994042 37.7243520645083,-78.6583113030221 37.7235823872935,-78.6584201948197 37.7230800755375,-78.6585778176117 37.7228779086445,-78.6587768422466 37.7227575586258,-78.6590447467976 37.7227135342909,-78.659326384898 37.7226640282813,-78.6596149437236 37.7226418087341,-78.6598967011172 37.7226468884378,-78.6601509757347 37.7226574643579,-78.660597648754 37.7226623172457,-78.6610168866375 37.7226890450178,-78.6612230116431 37.7226778501023,-78.661449759214 37.7226666254103,-78.6616971251735 37.7226608264585,-78.6619926547857 37.7226822634043,-78.66229515622 37.7227473477848,-78.6626940330568 37.7228941775307,-78.6632717147076 37.7231008018775,-78.6638356027046 37.7232856058489,-78.6645095108021 37.7235030111705,-78.6656855916798 37.7239544195571,-78.6666828203604 37.7243187322026,-78.6671229827337 37.724481867266,-78.667679800638 37.7245684123969,-78.6679959415511 37.7245843454902,-78.6682983456115 37.7246002901649,-78.6685595736852 37.7246435888894,-78.6688621870721 37.7247523252992,-78.6696880370077 37.7252915232722,-78.6705140229439 37.7258852982176,-78.6713677245938 37.7265718194027,-78.6718772581873 37.727007752935,-78.6723733417209 37.7275692379145,-78.6729108304292 37.728201625789,-78.6734478736416 37.7286375081771,-78.6737852500342 37.7289099356392,-78.6744048057076 37.7293620696653,-78.6747695261002 37.7295689501624,-78.6752578298173 37.7297265214838,-78.6757322313475 37.7298131559512,-78.6761240869078 37.7298671574635,-78.6765298462728 37.7299920966509,-78.6769907085094 37.7301660782978,-78.6775893341767 37.7304708505367,-78.6779199907913 37.7307978548816,-78.678099202709 37.731021379337,-78.6783204598944 37.7315832546607,-78.6784790768962 37.7318122650108,-78.6788301963791 37.7320737376523,-78.6792018000817 37.7322751342388,-78.6796213545504 37.7324109588741,-78.6798483347466 37.7324870322503,-78.6803365404036 37.7325900027061,-78.6808661665621 37.7327693219137,-78.681519886925 37.7331067409109,-78.6824008097774 37.7336075564593,-78.6831166935743 37.7340649565941,-78.6839085329431 37.7346641497298,-78.6844527975387 37.7351982180278,-78.6850523776054 37.7358632068705,-78.6855486137481 37.7364191784478,-78.6857832796963 37.7368172711591,-78.6858870464357 37.7370900230214,-78.6860873962129 37.7375045439845,-78.6861716809515 37.7382303665263,-78.6861597505845 37.7389617994007,-78.6861481063598 37.7398078528276,-78.6860879338797 37.7404902407719,-78.6859860692456 37.7409871048335,-78.6858426741372 37.7413694083506,-78.6855137228124 37.7417629224081,-78.6852810510719 37.742178112114,-78.6849048158005 37.7429101140667,-78.6845973289011 37.7436529233517,-78.6843721241368 37.7443137265291,-78.6842630703548 37.7446905188953,-78.6842638082735 37.7449907246713,-78.684340765102 37.7455364350438,-78.6844521513212 37.746109385317,-78.6845150436719 37.746524118266,-78.6845779932882 37.7469661402657,-78.6845648499004 37.7472117861477,-78.6845250178979 37.7477849735542,-78.6844169669681 37.7485711360201,-78.6843075681466 37.7488114655929,-78.6842049963287 37.749029955075,-78.684143677725 37.749253844594,-78.6841445090675 37.7495922558396,-78.684165857951 37.7498869722746,-78.6840842621757 37.7502528031182,-78.6838929214026 37.7507170534671,-78.683632930797 37.7512196182916,-78.6835101392131 37.7516073486238,-78.6834627159464 37.7518912501229,-78.6835184320243 37.7521859120356,-78.683573989904 37.7524150775851,-78.6837816154047 37.7529824161206,-78.6840167812019 37.7535660909924,-78.6842101254398 37.7539151233189,-78.6843618810261 37.7541223001082,-78.6845205571095 37.7543513051578,-78.6845829675364 37.7545695384581,-78.6847081876991 37.7551697547124,-78.6847573088951 37.7555735938877,-78.684745173608 37.7562286058007,-78.6846778481878 37.756807290063,-78.6845139424609 37.7572496657459,-78.6842879007356 37.7575884280552,-78.6841167655247 37.7578889003846,-78.6839535292635 37.7586041893189,-78.6837214567441 37.7592922923177,-78.683406276486 37.7597294385732,-78.6831044075473 37.7599918993057,-78.6826029526163 37.7601673359586,-78.681991091123 37.7601791907612,-78.6816748314852 37.7601796743776,-78.6810834751435 37.7601423677095,-78.6801894602412 37.760040023525,-78.6792336712595 37.7599814229916,-78.6785253991905 37.7599279088372,-78.6779476614489 37.7598305224813,-78.6772253056404 37.7596351036286,-78.6765303627073 37.7594014295789,-78.6760550492391 37.7590036819071,-78.6753799547159 37.7584315573237,-78.675042275703 37.7580881834962,-78.6747050196452 37.7579249334731,-78.6740375841733 37.7576802889834,-78.6732531694333 37.7573921483477,-78.6724414669397 37.7571913735167,-78.6716505705504 37.7570669763392,-78.6710249915383 37.757084250138,-78.6706058895957 37.7571994769779,-78.6703241433387 37.7572544619759,-78.6697467670942 37.7573044117277,-78.669437390993 37.757304852987,-78.6693066933247 37.757272291637,-78.6691691260611 37.7572397385109,-78.6689285813052 37.757272829425,-78.6685438045684 37.7573716250477,-78.6680972791903 37.7575250906152,-78.6677471005316 37.7577220814986,-78.6675756650566 37.7579188195654,-78.6674594611933 37.7582191903638,-78.6674394595285 37.7584975920454,-78.6674540111202 37.7588578166199,-78.6674272786152 37.7592017255419,-78.6674552428268 37.7594091005914,-78.6675863950024 37.7596436235669,-78.6677931007659 37.7598452852671,-78.6681442759429 37.7600849553055,-78.668481543121 37.7602591433927,-78.6688464098117 37.7604714972039,-78.6690944312926 37.7607003940633,-78.6694183793308 37.7610601778063,-78.66971453223 37.7612890019415,-78.669941756037 37.7614360509348,-78.6702101376125 37.7615448347198,-78.670526527271 37.7615989624272,-78.6708635721425 37.7616694389794,-78.6711732885998 37.7618109058523,-78.6720199644733 37.762251803116,-78.6729839876645 37.7628890225462,-78.6739068225706 37.7635481264216,-78.674760871552 37.7641854962525,-78.6752429753883 37.7645341173483,-78.6757183836101 37.7649591580142,-78.676138802177 37.7653897421983,-78.676517713154 37.7657057591637,-78.6768691960099 37.7660545645432,-78.6772276390017 37.7664361143559,-78.6775243247672 37.7668723277555,-78.67790373616 37.7673957562414,-78.6781936032488 37.7678483588044,-78.6785314766318 37.7682517631956,-78.6789380935548 37.7686441506439,-78.6792894073413 37.7689110718865,-78.6796614043289 37.7691998013438,-78.6800333875206 37.7694830684556,-78.6804742140561 37.7697989787463,-78.6809977037028 37.7701748022596,-78.6813560103784 37.7704853792345,-78.6817420501012 37.7708887007763,-78.6821900108176 37.7713028429162,-78.682438271905 37.7715972085674,-78.6825971191918 37.7718807942595,-78.6827700231074 37.77228989914,-78.6828393792926 37.772535410577,-78.6828400318825 37.7728028680926,-78.6828611007597 37.7729829575419,-78.6829509576209 37.773173856505,-78.68325406642 37.7733971813989,-78.6836671808664 37.7736148744992,-78.6843072521175 37.7738431313852,-78.6850158297228 37.773967567945,-78.6857861687406 37.7740373258045,-78.6865494217091 37.7740252129242,-78.6870375257307 37.7739753252509,-78.687559581225 37.7737561741422,-78.688314907163 37.7733237767541,-78.6892001550264 37.7726073384548,-78.6897899686926 37.7719950742083,-78.6900570584388 37.7715689005928,-78.6902419934812 37.7712847753606,-78.6905166638859 37.771136962515,-78.6907983936959 37.771060096503,-78.6920086000768 37.7710581487694,-78.6923658741619 37.7709484040184,-78.6927096421645 37.7709314751737,-78.6931292687406 37.7710017546795,-78.6933360270034 37.7711869962721,-78.6933917680702 37.7714707324031,-78.6933584181407 37.7718692419154,-78.6931058566649 37.7725901441545,-78.6927098209788 37.7736769803237,-78.6921627086815 37.7748459316426,-78.6917171592033 37.7754033952716,-78.691353466993 37.7757041838633,-78.6908248380965 37.7760434431817,-78.6905366135674 37.7762786133946,-78.6903377715458 37.7765081773022,-78.6900163191281 37.7771964277791,-78.6896672511565 37.7778465212079,-78.6893655656653 37.7782017867469,-78.6890773611085 37.778447867124,-78.6884043490134 37.7788146393952,-78.6878205573803 37.7791103069181,-78.6874290793482 37.7793128808187,-78.6871610027523 37.7793624267529,-78.6869064756783 37.779330079676,-78.6866314128592 37.7793359671438,-78.6862534403381 37.7794402683798,-78.6855936171009 37.7795886683507,-78.6852155814804 37.7796656731949,-78.6847892315826 37.779677254462,-78.6842872694142 37.7796998652759,-78.6840468469408 37.7798094046354,-78.6836073811142 37.7800775333576,-78.6829277534196 37.7805698281411,-78.6825709081806 37.7808814969046,-78.6824407900875 37.7811054819753,-78.6823794248905 37.781323904541,-78.6824142247406 37.7814930572671,-78.682662547094 37.7817983410401,-78.6829864080161 37.7820598398729,-78.6833651498859 37.7822612099552,-78.6837918941356 37.7824079239701,-78.6840191658507 37.7825385687629,-78.684150192163 37.7826857372121,-78.6842949889245 37.7828383453379,-78.6845222727288 37.7829744502989,-78.6847150662898 37.7830669387812,-78.6849907359211 37.783301216567,-78.6853493414997 37.7836991079178,-78.6856802768767 37.7840315453883,-78.6860245129718 37.7841783801071,-78.6862654350906 37.7842653380494,-78.6865681040118 37.7842921526294,-78.6868846850064 37.7843789878105,-78.6870706704144 37.784498776285,-78.6872019142713 37.7847278189759,-78.6872991377638 37.7851042819977,-78.6873891451198 37.7853443060593,-78.6875896150237 37.7857533561548,-78.6878036270838 37.7860750509712,-78.6880933331657 37.7864130049579,-78.6882176562568 37.7866202253816,-78.6883213481467 37.7868329310473,-78.6884392031051 37.7872039005011,-78.6884466335828 37.7874222210368,-78.6884196405269 37.7876296765921,-78.6882826366641 37.7878482229901,-78.68822129007 37.7880666509327,-78.6882218385758 37.7882849766903,-78.6882978584093 37.7884322329959,-78.6884977922148 37.7886229526516,-78.6890040697421 37.7887868725367,-78.6894081019413 37.7888360354984,-78.6899867838316 37.7911273379217,-78.6909179447047 37.7949305026462,-78.6915236246303 37.7967821889636,-78.6930223301225 37.7994318058541,-78.6930458017494 37.7994729343852,-78.6931509534254 37.7996526662319,-78.6933004286766 37.7999064506137,-78.6944301352559 37.8018389800828,-78.6952710259937 37.8032761089328,-78.6953098208257 37.8033400490186,-78.6974162097741 37.8069398026961,-78.6995447473177 37.8105768203531,-78.7008905016329 37.8128765570422,-78.7030014524879 37.816482852241,-78.7079266253154 37.8248976106098,-78.7079327686507 37.8249082760082,-78.7090181392273 37.826761955665,-78.7130395636124 37.8335170272495,-78.7130648211104 37.8335582368235,-78.716150368652 37.838506780277,-78.7175658924389 37.8406352341235,-78.7180632657808 37.8413867789138,-78.7189641247219 37.8427431115931,-78.7217712289406 37.846964959165,-78.7238283163942 37.8504329562973,-78.7258646772224 37.8538570165005,-78.7260317822426 37.8541402407893,-78.7267906824549 37.8554157629232,-78.7277689536554 37.8570636538036,-78.7302652773792 37.861261853018,-78.73083037209 37.8625459621577,-78.7309873447489 37.8629005967827,-78.732092287986 37.8651242092151,-78.7325140644018 37.8659733214932,-78.7339502972454 37.8688690769429,-78.7341074083132 37.8691393531281,-78.7378165415786 37.8754912146801,-78.738945208068 37.8774223386675,-78.7400251462735 37.8792696979002,-78.7402305629125 37.8796275771244,-78.741237227062 37.8813657239572,-78.7413351580638 37.8815346760074,-78.7420668405465 37.8827977909789,-78.7424180565993 37.8834044693533,-78.7481473869059 37.8932946170584,-78.7492072788945 37.8951235171006,-78.7492769820503 37.8952414543686,-78.7553982574472 37.9058058808363,-78.756976549622 37.908530086332,-78.758456636088 37.9110834136776,-78.7590928416299 37.912165420256,-78.7592763614775 37.9124768181554,-78.7595613293006 37.9129594494233,-78.7603842165305 37.9143561117951,-78.7617580797506 37.9166859281968,-78.7630388851853 37.918860366164,-78.7654235144014 37.9229055165737,-78.7655000360035 37.9230342024061,-78.7679136422235 37.9271287373143,-78.774318637105 37.9379880304466,-78.7744910952399 37.9382828033824,-78.7799833055781 37.9475933386339,-78.7815177682511 37.9501927105436,-78.7839668287269 37.9543446772915,-78.7868480438205 37.959226955363,-78.7923174638087 37.968072501188,-78.7944853828495 37.9715793796188,-78.7986154454043 37.9781665186261,-78.799404556567 37.9794297879503,-78.8001063912033 37.9805525326034,-78.8024006138995 37.9842250771677,-78.8030223901622 37.9852615073503,-78.8038463722896 37.9865315156264,-78.8055896212946 37.9894440541476,-78.8058517108069 37.9898971652581,-78.8070398376618 37.9919419745753,-78.8088253255936 37.9949367308266,-78.8108455125015 37.9984265073694,-78.8131636062536 38.0025703929031,-78.8140444650039 38.0040749613595,-78.8157536594619 38.006950350625,-78.8167958442362 38.0088251863943,-78.8172701707099 38.009675211855,-78.8181736409741 38.0112565574954,-78.8198065747332 38.0140778511507,-78.8199066549702 38.0142496981051,-78.8200270421741 38.0144567576739,-78.8201462705889 38.0146621588727,-78.8206631459602 38.0157539733222,-78.8213399320921 38.0172860524118,-78.8222779252492 38.0189592508209,-78.823839526755 38.0215759016315,-78.824621807651 38.0227465224677,-78.825856245085 38.0247742088648,-78.8280646196959 38.0285097607236,-78.829734503747 38.0314778743583,-78.8300745648199 38.0320981017037,-78.8301258907518 38.0322105170001,-78.8305738876986 38.0330275864999,-78.830643995282 38.0331784226537,-78.8310257894776 38.0339393803182,-78.831177610958 38.0342496974752,-78.832472208454 38.036511586899,-78.8328419542521 38.0371430340392,-78.8329581607705 38.0373606030332,-78.8336568933226 38.0385813254251,-78.8337062886995 38.0386550377655,-78.8337097985935 38.0386639529341,-78.8343989730956 38.03982221249,-78.8374884092789 38.0450364006921,-78.8388740528104 38.0473700001129,-78.8370133605374 38.0482581024147,-78.8331100199443 38.0513143575465,-78.8316293587254 38.0525823240768,-78.8301436412025 38.0535015383692,-78.8299271786087 38.0540252489615,-78.8291014380393 38.0542018959109,-78.8281076648307 38.0540735100858,-78.8265117357078 38.0544266253533,-78.8255225210022 38.0553444237514,-78.8252025784044 38.0560400293438,-78.8235689044165 38.0573110526713,-78.822220747252 38.057962058606,-78.8222936072693 38.0587499619731,-78.8216913822293 38.0592307586349,-78.8198422241673 38.059686061702,-78.8186204401966 38.0598547765252,-78.8172893142417 38.0596297239341,-78.8165366451909 38.0598603190559,-78.8162189682489 38.0602957127951,-78.8144005599877 38.0606437622394,-78.8118558454801 38.0617261686375,-78.8104968266727 38.061477770501,-78.8094844048992 38.0618922938711,-78.8082422595943 38.0621927574344,-78.8080133405726 38.0631084420453,-78.807611022603 38.0632467788062,-78.806482059074 38.0632036852884,-78.8059034569155 38.0635484159207,-78.8054439539398 38.0641447815487,-78.8050674373701 38.0646489456579,-78.8041448669299 38.064971890572,-78.8032182513384 38.0651568529811,-78.8023201472543 38.0655202265772,-78.8016171011451 38.0660238279188,-78.8003713940031 38.0663249067771,-78.7983719626817 38.0666271694034,-78.7972935461082 38.0670760804293,-78.7963086667025 38.0674691406126,-78.795251952397 38.0681410868246,-78.794761005942 38.0687189533241,-78.7927876943754 38.0695043484655,-78.7921800082168 38.0704360316296,-78.7922281497607 38.0711979193845,-78.7931299409749 38.072627725354,-78.7931768566063 38.0730738289124,-78.792786724546 38.0754548508535,-78.7928410054346 38.0760093936345,-78.793667137618 38.0767316127452,-78.7940186526446 38.0768975655859,-78.7939272631977 38.0775108206126,-78.7936470566778 38.0778080807532,-78.7926155518596 38.0778668988979,-78.7916058949299 38.0781844721634,-78.7902720695974 38.0791165380582,-78.7888395856313 38.0796952871554,-78.7881619229946 38.0805323795198,-78.7880697036732 38.0809410631134,-78.786826662607 38.0811108708416,-78.7860297824274 38.0816702078716,-78.7851609242736 38.0815796166972,-78.7835614002974 38.0811934927496,-78.7821734491983 38.0807134955637,-78.7816244179181 38.0806756631075,-78.7810515768064 38.0807731188328,-78.7808858625155 38.0810487947059,-78.7808112332915 38.0815274420512,-78.7808131244097 38.0820299535138,-78.7806479943111 38.0824614547159,-78.7801632865984 38.0829046887666,-78.7798582336156 38.0830832028789,-78.779199604865 38.0834698623639,-78.7789096979557 38.0843694704521,-78.7788992688205 38.0855541633269,-78.7789033781557 38.0861368268047,-78.7789453096161 38.087003805097,-78.7789844907324 38.087263206198,-78.7790592442258 38.0877415120846,-78.7790734861459 38.0878348376962,-78.7790841191294 38.0878897310005,-78.7790838987002 38.0878897708815,-78.7793058007012 38.0892973338877,-78.7790686196553 38.0899608674671,-78.7777547792048 38.091131999933,-78.7773184211161 38.0919538562493,-78.7772032729908 38.0932169249995,-78.7778870802257 38.0944781370797,-78.7785269061085 38.0949817345199,-78.7795690037558 38.0955791493929,-78.7806113652789 38.0965553832693,-78.7811730315827 38.0971854602294,-78.7812093845147 38.0976482589102,-78.7810469992407 38.098181271354,-78.7807033119379 38.0985977865438,-78.78070175186 38.0985978408225,-78.7807030717982 38.0989486322007,-78.7807046317672 38.0989485469921,-78.7814125796105 38.0992196910736,-78.7821194144207 38.0992050346701,-78.7827442177019 38.0992295332388,-78.7832380376808 38.0993972365117,-78.7837493914639 38.0998507244925,-78.78383323421 38.1002792186549,-78.7838188249536 38.1008118798724,-78.78367170178 38.1010330859925,-78.7829502016017 38.1015284321695,-78.7825599854836 38.1026725669869,-78.7816566934531 38.1029215307567,-78.7808633987815 38.1035455406976,-78.7801552625319 38.1042418133404,-78.779578603753 38.1050855888595,-78.7792072766699 38.1058254429901,-78.7790414260262 38.1064908875167,-78.7792112043751 38.106874756194,-78.7799794385455 38.1072276798285,-78.7809723324682 38.1076391594379,-78.781048938137 38.1081119321428,-78.780378629601 38.1089263571002,-78.7799882036415 38.1095627907388,-78.7800272649709 38.1100060783778,-78.7804032144853 38.1104929294357,-78.7805533602331 38.1110767677113,-78.780406755203 38.112071851888,-78.7806537242047 38.1127839160926,-78.7815944353669 38.1130786476105,-78.782741504707 38.1132838066117,-78.7837977673951 38.1143205753096,-78.7847941418177 38.1144221082943,-78.785602149763 38.1144053530092,-78.785790410667 38.1144939798116,-78.785774227186 38.1151769537587,-78.7858529867651 38.1161121352968,-78.7862314055667 38.1167793094219,-78.7867966970593 38.1171788286191,-78.7872853054126 38.1171776549029,-78.7875488146888 38.117280952732,-78.7872701947133 38.1181278857348,-78.7868959859423 38.1185593448862,-78.7856758676624 38.1189334327775,-78.7853114985391 38.1192418746682,-78.7852938625158 38.1197014692593,-78.7853560338336 38.1207429557405,-78.7856671649596 38.1209719891925,-78.7862876956023 38.1209858257259,-78.7867923988371 38.1211224883614,-78.7871623362296 38.121519864541,-78.7871640467548 38.1219640909817,-78.7869949232336 38.123358485823,-78.7873663449752 38.1241388310464,-78.7872733544083 38.1251653915445,-78.7880731886937 38.1264042294251,-78.7880358709215 38.1267872911355,-78.787804656555 38.1271708170654,-78.7873793906102 38.1275241423515,-78.7859397094577 38.1290308377019,-78.7855631625266 38.1292138161093,-78.7851230734722 38.1292148668464,-78.7844310963551 38.1291171973179,-78.7834887914861 38.1293180743861,-78.7821917031364 38.1299171161064,-78.7815233772181 38.1305311876565,-78.7809592512077 38.1309960230944,-78.7801437649188 38.1314945639592,-78.7797463326478 38.1316941286046,-78.7794963077823 38.1320920175333,-78.7793731736429 38.1327875654252,-78.779376274775 38.1336152562991,-78.7792322967583 38.1343439654126,-78.779464839158 38.1348731594844,-78.7794045547701 38.1355685596453,-78.7794558222397 38.1371225321322,-78.7792987030477 38.1390515265353,-78.779067151926 38.1399236680284,-78.7779418964879 38.1404084395453,-78.7772848232173 38.1405212293919,-78.7765566570613 38.140448721752,-78.7759235662189 38.1406912729861,-78.7754316550266 38.1410076634721,-78.7752466105753 38.141768425868,-78.7751547245505 38.1423249517128,-78.774689611402 38.1435499723282,-78.7741098024905 38.1455170237279,-78.7740701965707 38.1462067934725,-78.7742270360887 38.1468648961624,-78.774106901161 38.1477918802804,-78.7741453094888 38.1498403230401,-78.7735032197963 38.1515976678954,-78.7725811062194 38.1527947208903,-78.771873907186 38.1536498864893,-78.771692507473 38.1547233306981,-78.7714779183736 38.1551628027836,-78.7719436731007 38.1558445850308,-78.7719152800455 38.1565274929856,-78.7716717276582 38.1575035131556,-78.7714898201973 38.158430658951,-78.7709383752935 38.1596756459324,-78.7707536997154 38.1606020523066,-78.7703902764522 38.1610220001973,-78.7704632835422 38.1628554672177,-78.771666160862 38.1651840816649,-78.7717690371474 38.1661006632943,-78.7697852132804 38.1680173963711,-78.7691581941998 38.1689618207414,-78.768830574916 38.1701151271177,-78.7681703421283 38.1710596158634,-78.7686367790156 38.1715824499204,-78.7685743790055 38.1726827683118,-78.7683482193663 38.1734646339762,-78.7677829402018 38.1741381152406,-78.7674154052568 38.1745422854531,-78.7673327186692 38.1752370740638,-78.7669936380185 38.175663577523,-78.7660296000162 38.1759570029507,-78.7651530596839 38.1769224370685,-78.7650415950082 38.1775052701615,-78.764901028018 38.1778865135839,-78.7642771698348 38.1780671409292,-78.7634256713622 38.1780914234908,-78.7627771377907 38.1793252417365,-78.7619572434812 38.1802681196713,-78.7619309085051 38.1808507610864,-78.7620196872001 38.1818813003211,-78.7618800863364 38.1825538149089,-78.761059831152 38.1834070620569,-78.7606348925678 38.183654457055,-78.7601811384992 38.1837899091708,-78.7598426829732 38.1844180398029,-78.7589948999744 38.185540205136,-78.7574974678838 38.1867143177485,-78.7568215769284 38.187197061836,-78.7564529059334 38.1883475844525,-78.7566938051965 38.1894433379603,-78.7564939718655 38.1904330895729,-78.7556854643235 38.1917717354919,-78.7549433251577 38.1927091498296,-78.7539293728017 38.1934332403212,-78.753184652519 38.1936219727642,-78.7526107519494 38.1942381672744,-78.7523083645509 38.1949874795679,-78.7523453330888 38.1958964896704,-78.7527883247351 38.1966442307562,-78.7527897763129 38.1970720249937,-78.7521142435961 38.1976884308812,-78.7511000174685 38.1983590075548,-78.75093258352 38.1989475965714,-78.75140892918 38.1995348283611,-78.7519529682766 38.2000952014757,-78.752290199382 38.2012842814325,-78.7518591834923 38.2025831082572,-78.7508004264635 38.2042268089596,-78.7499831505861 38.2057554893454,-78.7489241741683 38.2073610011922,-78.7464707014547 38.2115651800162,-78.7451737720727 38.2144308634003,-78.7421362384731 38.2177200028188,-78.7405434256796 38.2189447687762,-78.7386580559952 38.2194829815524,-78.7376404618914 38.2192942546706,-78.7372096059463 38.2195706129815,-78.736385765386 38.2200982667629,-78.7341399921033 38.222713541754,-78.7325152018195 38.2238845768175,-78.7304979555765 38.2258349088895,-78.7298599937161 38.2270040252402,-78.7283340878751 38.2282915857203,-78.7269045611581 38.2288393146598,-78.7263147233475 38.2296189975866,-78.7268123978629 38.2309805654806,-78.7262724618029 38.2319547994272,-78.7247449825469 38.2327751771561,-78.7224260339526 38.2330130931567,-78.7212443068798 38.2339495597985,-78.7210974096541 38.2343391267609,-78.7196671375994 38.2346921239944,-78.7176829394334 38.2354553920204,-78.715574031697 38.2358674225118,-78.7112385751737 38.2386390979699,-78.7105559815687 38.2397224623294,-78.7106998502469 38.2412893257073,-78.7100059103762 38.2424452492272,-78.7085091374107 38.2432451526151,-78.7067351420239 38.244650376647,-78.7065658091379 38.2464376913505,-78.7043361363485 38.2469913848863,-78.7022770697569 38.2461975979863,-78.7008122049308 38.2459526147927,-78.697743773752 38.2457377868108,-78.6952355980601 38.2464567065167,-78.6928647146577 38.2472860906027,-78.6897867486095 38.2488376377403,-78.6893970743602 38.2498546199932,-78.6882490307654 38.2501436632636,-78.6872707246755 38.2510731526788,-78.6863469279955 38.2514060354479,-78.6853688087786 38.2524238911386,-78.6852857407184 38.2528217410838,-78.6824881598656 38.254615691921,-78.6818163433305 38.2548818365395,-78.6810318097584 38.2549051319553,-78.6803028571132 38.2547294597979,-78.6794275406394 38.2546077654733,-78.6780385773164 38.2540734337897,-78.6770301748755 38.2533883538387,-78.6762133196222 38.2531535635758,-78.6759133860927 38.2528750628033,-78.6754480156524 38.2516956783648,-78.6744109205469 38.2503454901985,-78.6723661047361 38.2485247221122,-78.6716028983715 38.2479250462097,-78.6704862844596 38.2474546188741,-78.6698058331749 38.2473268672173,-78.6698885692794 38.2478202199102,-78.6697268121 38.248464134557,-78.6694835944944 38.2491939672536,-78.6691583258287 38.2497308125829,-78.6687251285123 38.2506754937304,-78.6673872260269 38.2512452458297,-78.6661423938862 38.251601678908,-78.665065199987 38.2519717560694,-78.6621884566516 38.2534758253634,-78.661397898833 38.2544561664505,-78.6600787344458 38.2553746873655,-78.6592629795848 38.2569592445518,-78.6592378479172 38.2575634766204,-78.6589746363891 38.2580430419731,-78.6576809260624 38.2585448066928,-78.6564665409854 38.259088103832,-78.6556751873869 38.2597558543586,-78.6551493375348 38.2606318508371,-78.6562845358221 38.2605051428436,-78.6572359834927 38.2605663885211,-78.6580822846809 38.2608986299123,-78.6588233334538 38.2614393596399,-78.6591946435347 38.2620639375252,-78.6597769849316 38.2625215324201,-78.6609408854126 38.2630616822734,-78.6614171903728 38.2633527450368,-78.6621577537755 38.263643422975,-78.6621688063608 38.2638747644388,-78.6622249311902 38.2644586485252,-78.6624999607559 38.2647826856252,-78.6635703112314 38.2649975128142,-78.6643386348542 38.2651046100795,-78.6650249396043 38.2653199507279,-78.6663982040255 38.2660533852309,-78.6650276267964 38.2665310923694,-78.663739753384 38.2672033378097,-78.6630824867774 38.2677232862544,-78.6626992224576 38.2680914881065,-78.6613291483518 38.2688287011948,-78.6609191043411 38.2694997057215,-78.6606464001641 38.2702570408982,-78.6607576389954 38.270949006107,-78.6615541680612 38.2713804936486,-78.6620489120143 38.2718123606894,-78.6622151509916 38.2725474951658,-78.6626838196947 38.2735849864382,-78.6633727260961 38.2749682514539,-78.6638977602096 38.276676109249,-78.663635503078 38.2773837659488,-78.6634378517549 38.2776042329837,-78.6631445065251 38.2779330305082,-78.6249754679326 38.2647188160998,-78.61733461064 38.2620789403217))"),
    "[POLYGON] Virginia" ->(29, 87, "dnw", "POLYGON((-77.286202 38.347024,-77.249701 38.331624,-77.179339 38.341914,-77.138224 38.367916,-77.084238 38.368739,-77.041505 38.400692,-77.011464 38.374425,-77.028131 38.305317,-76.962086 38.256964,-76.94902 38.208418,-76.839368 38.163609,-76.60543 38.148933,-76.596156 38.106676,-76.536457 38.074903,-76.516609 38.026783,-76.465008 38.01322,-76.412934 37.966332,-76.333372 37.942956,-76.236458 37.886605,-76.052021 37.953578,-75.943693 37.946133,-75.952672 37.906827,-75.80124 37.912174,-75.761642 37.941366,-75.704634 37.954698,-75.6477 37.946963,-75.624449 37.994195,-75.166435 38.027834,-75.303132 37.838169,-75.360125 37.810186,-75.417073 37.82105,-75.460933 37.771356,-75.541779 37.606145,-75.529901 37.556717,-75.58911 37.477857,-75.604026 37.422024,-75.643456 37.372864,-75.644386 37.34227,-75.715515 37.275687,-75.729271 37.216901,-75.766049 37.154932,-75.889676 37.054515,-75.928307 36.938179,-75.874647 36.744305,-75.797497 36.550916,-80.122173 36.54265,-81.677393 36.588156,-81.6469 36.611918,-81.922644 36.616213,-81.934144 36.594213,-83.675395 36.600784,-83.530146 36.666049,-83.424713 36.66734,-83.309711 36.710639,-83.13512 36.742629,-83.133231 36.784585,-83.072836 36.854457,-83.006577 36.847643,-82.878683 36.889532,-82.857665 36.928669,-82.867467 36.978049,-82.722433 37.044949,-82.721413 37.12083,-82.498747 37.226958,-82.348757 37.267944,-81.968012 37.538035,-81.926243 37.513574,-81.996075 37.471526,-81.987006 37.454878,-81.935621 37.438397,-81.922956 37.411389,-81.936744 37.38073,-81.928462 37.360461,-81.860489 37.315942,-81.849949 37.285227,-81.757137 37.276147,-81.740124 37.237752,-81.723061 37.240493,-81.678221 37.20154,-81.554705 37.20799,-81.507636 37.233712,-81.499096 37.257952,-81.480144 37.251121,-81.416029 37.273626,-81.365284 37.337842,-81.225104 37.234874,-81.104147 37.280605,-80.979589 37.302279,-80.966556 37.292158,-80.89916 37.315678,-80.849451 37.346909,-80.883247 37.383933,-80.859457 37.429491,-80.784188 37.394587,-80.770082 37.372363,-80.552036 37.473563,-80.511391 37.481672,-80.49486 37.435072,-80.475601 37.422949,-80.309333 37.502883,-80.282391 37.533519,-80.330368 37.536267,-80.312201 37.54622,-80.328504 37.564315,-80.220984 37.627767,-80.267504 37.646156,-80.295916 37.692838,-80.253077 37.725899,-80.257411 37.756084,-80.216605 37.775961,-80.22769 37.797892,-80.162202 37.875122,-79.999383 37.994908,-79.926397 38.106543,-79.945258 38.132239,-79.918662 38.15479,-79.920075 38.182529,-79.788945 38.268703,-79.810154 38.306707,-79.764432 38.356514,-79.725597 38.363828,-79.729512 38.381194,-79.689667 38.431462,-79.697572 38.487223,-79.662174 38.515217,-79.669775 38.550316,-79.649175 38.591415,-79.53687 38.550917,-79.476638 38.457228,-79.312276 38.411876,-79.28294 38.4181,-79.210591 38.492913,-79.129756 38.655018,-79.092755 38.659817,-79.085455 38.724614,-79.054954 38.785713,-79.027253 38.792113,-78.999752 38.846162,-78.868971 38.763139,-78.786025 38.887187,-78.717076 38.936028,-78.719291 38.90523,-78.697437 38.914359,-78.627798 38.98179,-78.601655 38.964603,-78.550467 39.018065,-78.571901 39.031995,-78.403697 39.167451,-78.438695 39.198093,-78.399669 39.243874,-78.419422 39.257476,-78.339284 39.348605,-78.366867 39.35929,-78.343214 39.388807,-78.359918 39.409028,-78.347087 39.466012,-77.828299 39.132426,-77.730697 39.315603,-77.67681 39.324558,-77.61591 39.302548,-77.566598 39.306047,-77.543086 39.266916,-77.45771 39.224979,-77.478672 39.189671,-77.517622 39.169417,-77.520092 39.120869,-77.485401 39.10931,-77.456971 39.073399,-77.337137 39.062265,-77.248419 39.026554,-77.24503 38.982684,-77.147197 38.964439,-77.041672 38.872533,-77.043877 38.717674,-77.122632 38.685256,-77.129682 38.634704,-77.246362 38.593441,-77.310558 38.493993,-77.32291 38.432625,-77.286202 38.347024))")
  )

  val internationalDateLineSafeGeometryTestData: Map[String, (String, String, String)] = Map(
    "[LINE] no span" ->("LINESTRING(50 50, 60 60)", "POINT(55 55)", "POINT(30 30)"),
    "[LINE] span" ->("LINESTRING(160 50, -160 60)", "POINT(170 52.5)", "POINT(30 30)"),
    "[POLYGON] span" ->("POLYGON((-170 80, 170 80, 170 70, -170 70, -170 80))", "POINT(175 75)", "POINT(165 75)"),
    "[POLYGON] span transform right" ->("POLYGON((-170 80, 170 80, 170 70, -170 70, -170 80))", "POINT(-175 75)", "POINT(165 75)"),
    "[POLYGON] no span" ->("POLYGON((-170 80, 0 80, 170 80, 170 70, 0 70, -170 70, -170 80))", "POINT(165 75)", "POINT(175 75)"),
    "[MULTILINE] span" ->("MULTILINESTRING((170 40, -170 40),(40 50, 45 55))", "POINT(175 40)", "POINT(165 40)"),
    "[MULTIPOLYGON] span" ->("MULTIPOLYGON(((-170 80, 170 80, 170 70, -170 70, -170 80)),((30 30, 30 40, 40 40, 40 30, 30 30)))", "POINT(175 75)", "POINT(30 25)"),
    "[MULTIPOINT] span" ->("MULTIPOINT(160 50, 40 20)", "POINT(160 50)", "POINT(159 60)"),
    "[MULTIPOINT] no span" ->("MULTIPOINT(80 50, 40 20)", "POINT(80 50)", "POINT(159 60)"),
    "[MULTIPOINT] no span2" ->("MULTIPOINT(80 50, 40 20)", "POINT(40 20)", "POINT(159 60)"),
    "[POINT] point no span" ->("POINT(170 40)", "POINT(170 40)", "POINT(40 40)"),
    "[POLYGON] antarctica" ->("POLYGON((-179.99900834195893 -89.99982838943765, -179.99900834195893 -84.35288625138337, -164.16219722595443 -78.99626230923768, -158.26351334731075 -77.08619801855957, -158.2192008065469 -77.0736923216127, -55.37731278554597 -61.07063812275956, -45.934374960003026 -60.524831644861884, -45.7198660894195 -60.51625335736412, 135.3432939050984 -66.10066701248235, 135.4887113859158 -66.11100229913357, 143.50863529466415 -66.84460093230936, 153.878803345093 -68.27893198676848, 167.70100874912129 -70.79128509537779, 170.26488326045944 -71.29213307737697, 170.28265995306788 -71.29874766015023, 170.8138936774078 -71.69965342205771,170.88500044784155 -71.75887461315726, 170.96623579935533 -71.83452891065282, 180.0000000000001 -84.35286778378045, 180.0000000000001 -88.38070249172895, 179.99989628106425 -89.99982838943765, -179.99900834195893 -89.99982838943765))", "POINT(-179.999999 -89.99)", "POINT(40 50)"),
    "[LINE] out span" ->("LINESTRING(-200 50, -160 60)", "POINT(170 52.5)", "POINT(30 30)"),
    "[POLYGON] out span" ->("POLYGON((-170 80, -190 80, -190 70, -170 70, -170 80))", "POINT(175 75)", "POINT(165 75)"),
    "[POLYGON] out span transform right" ->("POLYGON((190 80, 170 80, 170 70, 190 70, 190 80))", "POINT(-175 75)", "POINT(165 75)"),
    "[MULTILINE] out span" ->("MULTILINESTRING((-190 40, -170 40),(40 50, 45 55))", "POINT(175 40)", "POINT(165 40)"),
    "[MULTIPOLYGON] out span" ->("MULTIPOLYGON(((-170 80, -190 80, -190 70, -170 70, -170 80)),((30 30, 30 40, 40 40, 40 30, 30 30)))", "POINT(175 75)", "POINT(30 25)"),
    "[MULTIPOINT] out span" ->("MULTIPOINT(-200 50, 40 20)", "POINT(160 50)", "POINT(159 60)"),
    "[POINT] out point no span" ->("POINT(-190 40)", "POINT(170 40)", "POINT(40 40)")
  )

  //first tuple is the point, second tuple is the BoundingBox of GH, the double is degrees of min great circle arc from point to GH
  val geodeticMinDistToGeohashTestData: Map[String, ((Double, Double), ((Int, Int), (Int, Int)), Double)] = Map(
    "within" -> ((150.0, -88.0), ((149, -89), (151, -87)), 0.0001),
    "pole wrapper" -> ((-30.0, -89.000001), ((149, -89), (151, -87)), 2.0),
    "tricksy pole wrapper" -> ((-30.0, -88.999999), ((149, -89), (151, -87)), 2.0),
    "exact pole wrapper" -> ((-30.0, -89.0), ((149, -89), (151, -87)), 2.0),
    "otherside of pole wrapper" -> ((-30.0, 89.000001), ((149, -89), (151, -87)), 178.0),
    "otherside of tricksy pole wrapper" -> ((-30.0, 88.999999), ((149, -89), (151, -87)), 178.0),
    "otherside of exact pole wrapper" -> ((-30.0, 89.0), ((149, -89), (151, -87)), 178.0),
    "northside of exact pole wrapper" -> ((150.0, 89.0), ((149, -89), (151, -87)), 176.0),
    "slightly left of northside of exact pole wrapper" -> ((148.999999, 89.0), ((149, -89), (151, -87)), 176.0),
    "IDL wrapper" -> ((179.0, 0.0), ((-180, -1), (-178, 1)), 1.0),
    "west of tl of gh in southern hemi" -> ((148.0, -87.0), ((149, -89), (151, -87)), 1.0),
    "north of tl of gh in southern hemi" -> ((149.0, -86.0), ((149, -89), (151, -87)), 1.0),
    "south of bl of gh in southern hemi" -> ((149.0, -90.0), ((149, -89), (151, -87)), 1.0),
    "west of bl of gh in southern hemi" -> ((148.0, -89.0), ((149, -89), (151, -87)), 1.0),
    "slightly WNW of gh in southern hemi" -> ((148.0, -86.999999), ((149, -89), (151, -87)), 1.0),
    "slightly NNW of gh in southern hemi" -> ((148.999999, -86.0), ((149, -89), (151, -87)), 1.0),
    "slightly WSW of gh in southern hemi" -> ((148.0, -89.000001), ((149, -89), (151, -87)), 1.0),
    "slightly SSW of gh in southern hemi" -> ((148.999999, -90.0), ((149, -89), (151, -87)), 1.0),
    "above south pole gh" -> ((99.5, -88.0), ((99, -90), (100, -89)), 1.0),
    "above south pole gh 180 around" -> ((-80.5, -89.5), ((99, -90), (100, -89)), 0.5),
    "left of south pole gh" -> ((98.0, -89.0), ((99, -90), (100, -89)), 1.0),
    "south pole to south pole gh" -> ((98.0, -90.0), ((99, -90), (100, -89)), 1e-7)
  )

  val recommendedBitsTestPolygons = Seq(
    ("POLYGON((33 -14, 82 -14, 82 24, 33 24, 33 -14))", 23, 241044),
    ("POLYGON((36 23, 36 18, 43 9, 50 10, 38 -4, 40 -14, 60 -14, 78 7, 76 8, 73 21, 71 23, 58 23, 57 19, 45 13, 40 20, 39 23, 36 23))", 23, 132496),
    ("POLYGON((0 0, 0 10, -5 25, -15 35, -25 35, -40 10, -20 0, -30 15, -20 25, -10 20, -10 10, -15 0, -25 -5, -35 -15, -40 -35, -25 -35, -15 -25, -5 -35, 0 -25, -10 -10, -25 -25, -20 -10, 0 0))", 23, 184472),
    ("POLYGON((-10 10, 0 30, 10 10, -10 10))", 25, 103564),
    ("POLYGON((-10 -10, -10 10, 10 10, 10 -10, -10 -10))", 25, 207127)
    // area bigger than ~2310 failing on Exception("Could not satisfy constraints, resolutions..." ln 361
  )

  // (reasonable) odd GeoHash resolutions
  val oddResolutions = new ResolutionRange(5, 63, 2)

  // GeoHash resolutions for decomposition
  val indexResolutions = new ResolutionRange(5, 40, 5)

  def wkt2geom(wkt: String): Geometry = WKTUtils.read(wkt)

  // testing resolution recommendation and decomposition
  testData.map({
    case (name, (resolution, numDecompositions, firstDecompositionHash, wkt)) => {
      val geom: Geometry = WKTUtils.read(wkt)

      name should {
        // all geometry types should test decomposition
        "decompose into " + numDecompositions + " GeoHashes, the first of which should be '" + firstDecompositionHash + "'" in {
          val decomposedGHs = decomposeGeometry(geom, 100, indexResolutions)
          if (DEBUG_OUTPUT)
            decomposedGHs.map(gh => logger.debug(name + "\t" + gh.hash + "\t" + WKTUtils.write(gh)))
          decomposedGHs.size must equalTo(numDecompositions)
          decomposedGHs(0).hash must equalTo(firstDecompositionHash)
        }
      }
    }
  })

  "getUniqueGeohashSubstringsInPolygon" should {
    val polygonCharlottesville = wkt2geom(testData(
      "[POLYGON] Charlottesville")._4).asInstanceOf[Polygon]

    def testGeohashSubstringsInCharlottesville(offset: Int, bits: Int): Int = {
      val ghSubstrings = getUniqueGeohashSubstringsInPolygon(
        polygonCharlottesville, offset, bits)

      ghSubstrings must beASuccessfulTry

      if (DEBUG_OUTPUT)
        ghSubstrings.get.foreach { gh => logger.debug(s"[unique Charlottesville gh($offset,$bits)] " + gh)}

      ghSubstrings.get.size
    }

    "compute (0,2) correctly for Charlottesville" in {
      testGeohashSubstringsInCharlottesville(0, 2) mustEqual 3
    }

    "compute (2,3) correctly for Charlottesville" in {
      testGeohashSubstringsInCharlottesville(2, 3) mustEqual 9
    }

    "compute (0,3) correctly for Charlottesville" in {
      testGeohashSubstringsInCharlottesville(0, 3) mustEqual 4
    }

    "compute (3,2) correctly for Charlottesville" in {
      testGeohashSubstringsInCharlottesville(3, 2) mustEqual 8
    }

    "not die when Geohash strings of more than 5 characters are requested" in {
      testGeohashSubstringsInCharlottesville(0, 6) mustEqual 83
      testGeohashSubstringsInCharlottesville(0, 7) mustEqual 1762
    }

    "not die when a point is provided" in {
      val point = WKTUtils.read("POINT(0.0 0.0)")
      forall(Seq((0, 3), (3, 2), (5, 2), (0, 7))) { case (offset, length) =>
        getUniqueGeohashSubstringsInPolygon(point, offset, length) must beASuccessfulTry
      }
    }

    "not die when a single-point collection is provided" in {
      val pointCollection = WKTUtils.read("GEOMETRYCOLLECTION( POINT(0.0 0.0) )")
      forall(Seq((0, 3), (3, 2), (5, 2), (0, 7))) { case (offset, length) =>
        getUniqueGeohashSubstringsInPolygon(pointCollection, offset, length) must beASuccessfulTry
      }
    }
  }

  internationalDateLineSafeGeometryTestData.map { case (name, (geomString, incl, excl)) =>
    "getInternationalDateLineSafeGeometry" should {
      s"work for $name" in {
        val geom = wkt2geom(geomString)
        val includedPoint = wkt2geom(incl).asInstanceOf[Point]
        val excludedPoint = wkt2geom(excl).asInstanceOf[Point]
        val decomposed = getInternationalDateLineSafeGeometry(geom).getOrElse(null)
        logger.debug("International Date Line test geom: " + geom.toText)
        decomposed match {
          case g: GeometryCollection =>
            val coords = (0 until g.getNumGeometries).map { i => g.getGeometryN(i)}
            coords.find(_.contains(includedPoint)).size must beGreaterThan(0)
            coords.find(_.contains(excludedPoint)).size must equalTo(0)
          case _ =>
            decomposed.contains(includedPoint) must equalTo(true)
            decomposed.contains(excludedPoint) must equalTo(false)
        }
      }
    }
  }
}

@Ignore
@RunWith(classOf[JUnitRunner])
class GeohashUtilsTimeTest extends Specification with LazyLogging {

  def wkt2geom(wkt: String): Geometry = WKTUtils.read(wkt)

  "performance test for getUniqueGeohashSubstringsInPolygon" should {
    val poly = wkt2geom("POLYGON((-170 -80, -170 0, -170 80, 0 80, 170 80, 170 0, 170 -80, 0 -80, -170 -80))").asInstanceOf[Polygon]
    val burnInTrials = 100
    val collectTrials = 100

    def getTime(burnInTrials: Int,
                collectTrials: Int,
                label: String,
                expectedCount: Int,
                fnx: () => Seq[String],
                useDotted: Boolean = true): Long = {

      logger.debug(s"[TIMING POLYGON $label] $poly")

      def runTest() {
        val substrings = fnx()
        val runCount = substrings.size
        if (runCount != expectedCount)
          throw new Exception("Wrong number of GeoHash sub-string results " +
            s"$label:  $runCount (expected $expectedCount)")
        if (useDotted) {
          for (i <- 1 until substrings.headOption.map(_.length).getOrElse(0)) {
            val rightDotted = "." * i
            if (substrings.find(_.takeRight(i) == rightDotted).isEmpty)
              throw new Exception(s"Missing '$rightDotted' among GeoHash sub-string results " +
                s"$label:  $runCount (expected $expectedCount)")
          }
        }
      }

      // perform a few trials to get everything warmed up
      (0 until burnInTrials).foreach(i => runTest())

      // perform the trials you want to time
      val msStart = System.currentTimeMillis()
      (0 until collectTrials).foreach(i => runTest())
      val msStop = System.currentTimeMillis()
      val msElapsed = msStop - msStart

      // output results
      logger.debug(s"\n[TIMING SUBSTRINGS $label] $collectTrials invocations over " +
        (msElapsed / 1000.0) + " seconds" +
        " -> " + (msElapsed / collectTrials) + " ms/invocation\n\n"
      )

      msElapsed / collectTrials
    }

    def timeTest(offset: Int,
                 bits: Int,
                 expectedCount: Int,
                 useDotted: Boolean = false) {

      val fnxCandidate: () => Seq[String] = () =>
        getUniqueGeohashSubstringsInPolygon(poly, offset, bits, 2 << (bits * 5), useDotted).getOrElse(Seq())

      val length =
        getTime(burnInTrials,
          collectTrials,
          s"getUniqueGeohashSubstringsInPolygon ($offset, $bits)",
          expectedCount,
          fnxCandidate,
          useDotted)
      length must not(beNull)
    }

    "collect timing data" in {
      //                                        RGHI   current   old version
      timeTest(0, 3, 27588, false)  //    50     48         83
      timeTest(0, 3, 28581, true)   //     -     78        118
      timeTest(3, 2, 1024, false)   //   777      2        545
      timeTest(3, 2, 1057, true)    //   777      5        -
      timeTest(3, 3, 32768, false)  //   N/A*    29        N/A*
      timeTest(3, 3, 33825, true)   //     -     51          -
      timeTest(5, 2, 1024, false)   //   N/A*     2        N/A*
      val last = timeTest(5, 2, 1057, true)    //     -      3          -

      last must not(beNull)
      // *"N/A" simply means that the test took more than 5 minutes without
      // finishing.
    }
  }
}
