﻿/*
 * QUANTCONNECT.COM - Democratizing Finance, Empowering Individuals.
 * Lean Algorithmic Trading Engine v2.0. Copyright 2014 QuantConnect Corporation.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
*/

using System;
using QuantConnect.Data.Market;

namespace QuantConnect.Data.Consolidators
{
    /// <summary>
    /// Represents the simplest DataConsolidator implementation, one that is defined
    /// by a straight pass through of the data. No projection or aggregation is performed.
    /// </summary>
    /// <typeparam name="T">The type of data</typeparam>
    public class IdentityDataConsolidator<T> : DataConsolidator<T>
        where T : BaseData
    {
        private static readonly bool IsTick = typeof (T) == typeof (Tick);

        private T _last;

        /// <summary>
        /// Gets the type produced by this consolidator
        /// </summary>
        public override Type OutputType
        {
            get { return typeof (T); }
        }

        /// <summary>
        /// Updates this consolidator with the specified data
        /// </summary>
        /// <param name="data">The new data for the consolidator</param>
        public override void Update(T data)
        {
            if (IsTick || _last == null || _last.EndTime != data.EndTime)
            {
                OnDataConsolidated(data);
                _last = data;
            }
        }
    }
}