﻿/*
 * QUANTCONNECT.COM - Democratizing Finance, Empowering Individuals.
 * Lean Algorithmic Trading Engine v2.0. Copyright 2014 QuantConnect Corporation.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
*/

using System;
using System.IO;
using QuantConnect.Securities;

namespace QuantConnect.Data.UniverseSelection
{
    /// <summary>
    /// Defines summary information about a single symbol for a given date
    /// </summary>
    public class CoarseFundamental : BaseData
    {
        /// <summary>
        /// Gets the market for this symbol
        /// </summary>
        public string Market { get; set; }

        /// <summary>
        /// Gets the day's dollar volume for this symbol
        /// </summary>
        public decimal DollarVolume { get; set; }

        /// <summary>
        /// Gets the day's total volume
        /// </summary>
        public long Volume { get; set; }

        /// <summary>
        /// The end time of this data.
        /// </summary>
        public override DateTime EndTime
        {
            get { return Time + QuantConnect.Time.OneDay; }
            set { Time = value - QuantConnect.Time.OneDay; }
        }

        /// <summary>
        /// Initializes a new instance of the <see cref="CoarseFundamental"/> class
        /// </summary>
        public CoarseFundamental()
        {
            DataType = MarketDataType.Auxiliary;
        }

        /// <summary>
        /// Return the URL string source of the file. This will be converted to a stream 
        /// </summary>
        /// <param name="config">Configuration object</param>
        /// <param name="date">Date of this source file</param>
        /// <param name="isLiveMode">true if we're in live mode, false for backtesting mode</param>
        /// <returns>String URL of source file.</returns>
        public override SubscriptionDataSource GetSource(SubscriptionDataConfig config, DateTime date, bool isLiveMode)
        {
            var path = Path.Combine(Constants.DataFolder, "equity", config.Market, "fundamental", "coarse", date.ToString("yyyyMMdd") + ".csv");
            return new SubscriptionDataSource(path, SubscriptionTransportMedium.LocalFile, FileFormat.Csv);
        }

        /// <summary>
        /// Reader converts each line of the data source into BaseData objects. Each data type creates its own factory method, and returns a new instance of the object 
        /// each time it is called. 
        /// </summary>
        /// <param name="config">Subscription data config setup object</param>
        /// <param name="line">Line of the source document</param>
        /// <param name="date">Date of the requested data</param>
        /// <param name="isLiveMode">true if we're in live mode, false for backtesting mode</param>
        /// <returns>Instance of the T:BaseData object generated by this line of the CSV</returns>
        public override BaseData Reader(SubscriptionDataConfig config, string line, DateTime date, bool isLiveMode)
        {
            try
            {
                var csv = line.Split(',');
                return new CoarseFundamental
                {
                    Symbol = new Symbol(SecurityIdentifier.Parse(csv[0]), csv[1]),
                    Time = date,
                    Market = config.Market,
                    Value = csv[2].ToDecimal(),
                    Volume = csv[3].ToInt64(),
                    DollarVolume = csv[4].ToDecimal()
                };
            }
            catch (Exception)
            {
                return null;
            }
        }

        /// <summary>
        /// Return a new instance clone of this object, used in fill forward
        /// </summary>
        /// <returns>A clone of the current object</returns>
        public override BaseData Clone()
        {
            return new CoarseFundamental
            {
                Symbol = Symbol,
                Time = Time,
                DollarVolume = DollarVolume,
                Market = Market,
                Value = Value,
                Volume = Volume,
                DataType = MarketDataType.Auxiliary
            };
        }

        /// <summary>
        /// Creates the symbol used for coarse fundamental data
        /// </summary>
        /// <param name="market">The market</param>
        /// <returns>A coarse universe symbol for the specified market</returns>
        public static Symbol CreateUniverseSymbol(string market)
        {
            market = market.ToLower();
            var ticker = "qc-universe-coarse-" + market;
            var sid = SecurityIdentifier.GenerateEquity(SecurityIdentifier.DefaultDate, ticker, market);
            return new Symbol(sid, ticker);
        }
    }
}
