﻿/*
 * QUANTCONNECT.COM - Democratizing Finance, Empowering Individuals.
 * Lean Algorithmic Trading Engine v2.0. Copyright 2014 QuantConnect Corporation.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
*/

/**********************************************************
* USING NAMESPACES
**********************************************************/
using System;
using System.Collections.Generic;
using Newtonsoft.Json;
using QuantConnect.Logging;

namespace QuantConnect 
{
    /******************************************************** 
    * GLOBAL CONST
    *********************************************************/
    /// <summary>
    /// Shortcut date format strings
    /// </summary>
    public static class DateFormat 
    {
        /// Year-Month-Date 6 Character Date Representation
        public static string SixCharacter = "yyMMdd";
        /// YYYY-MM-DD Eight Character Date Representation
        public static string EightCharacter = "yyyyMMdd";
        /// JSON Format Date Representation
        public static string JsonFormat = "yyyy-MM-ddThh:mm:ss";
        /// MySQL Format Date Representation
        public const string DB = "yyyy-MM-dd HH:mm:ss";
        /// QuantConnect UX Date Representation
        public const string UI = "yyyyMMdd HH:mm:ss";
        /// EXT Web Date Representation
        public const string EXT = "yyyy-MM-dd HH:mm:ss";
    }


    /******************************************************** 
    * GLOBAL STRUCT DEFINITIONS
    *********************************************************/
    /// <summary>
    /// Singular holding of assets from backend live nodes:
    /// </summary>
    [JsonObjectAttribute]
    public class Holding
    {
        /// Symbol of the Holding:
        public string Symbol = "";

        /// Average Price of our Holding
        public decimal AveragePrice = 0;

        /// Quantity of Symbol We Hold.
        public decimal Quantity = 0;

        /// Current Market Price of the Asset
        public decimal MarketPrice = 0;

        /// Create a new default holding:
        public Holding()
        { }

        /// <summary>
        /// Create a simple JSON holdings from a Security holding class.
        /// </summary>
        /// <param name="holding"></param>
        public Holding(Securities.SecurityHolding holding)
        {
            Symbol = holding.Symbol;
            Quantity = holding.Quantity;
            AveragePrice = holding.AveragePrice;
            MarketPrice = holding.Price;
        }
    }

    /******************************************************** 
    * GLOBAL ENUMS DEFINITIONS
    *********************************************************/
    /// <summary>
    /// Processing runmode of the backtest.
    /// </summary>
    /// <obsolete>The runmode enum is now obsolete and all tasks are run in series mode. This was done to ensure algorithms have memory of the day before.</obsolete>
    public enum RunMode 
    { 
        /// Automatically detect the runmode of the algorithm: series for minute data, parallel for second-tick
        Automatic,
        /// Series runmode for the algorithm
        Series,
        /// Parallel runmode for the algorithm
        Parallel
    }


    /// <summary>
    /// Multilanguage support enum: which language is this project
    /// </summary>
    public enum Language 
    { 
        /// C# Language Project
        CSharp,
        /// Java Language Project
        Java,
        /// Python Language Project
        Python
    }


    /// <summary>
    /// User subscription level
    /// </summary>
    /// <remarks>Currently there are only hobbyist plans</remarks>
    public enum UserPlan 
    {
        /// Free User 
        Free,
        /// Hobbyist User 
        Hobbyist
    }


    /// <summary>
    /// Type of tradable security / underlying asset
    /// </summary>
    public enum SecurityType 
    {
        /// Base class for all security types: 
        Base,
        /// US Equity Security
        Equity,
        /// Option Security Type
        Option,
        /// Commodity Security Type
        Commodity,
        /// FOREX Security 
        Forex,
        /// Future Security Type
        Future
    }

    /// <summary>
    /// Market data style: is the market data a summary (OHLC style) bar, or is it a time-price value.
    /// </summary>
    public enum MarketDataType 
    {
        /// Base market data type
        Base,
        /// TradeBar market data type (OHLC summary bar)
        TradeBar,
        /// Tick market data type (price-time pair)
        Tick
    }

    /// <summary>
    /// Datafeed enum options for selecting the source of the datafeed.
    /// </summary>
    public enum DataFeedEndpoint 
    { 
        /// Backtesting Datafeed Endpoint
        Backtesting,
        /// Loading files off the local system
        FileSystem,
        /// Getting datafeed from a QC-Live-Cloud
        LiveTrading,
        /// Tradier Supplied Free Data Feed 
        Tradier
    }

    /// <summary>
    /// Realtime events handler options for handling realtime events in the algorithm
    /// </summary>
    public enum RealTimeEndpoint
    { 
        /// Backtesting Faked RealTime Events
        Backtesting,
        /// Live Trading RealTime Events
        LiveTrading
    }

    /// <summary>
    /// Result events handler options for processing algorithm messages
    /// </summary>
    public enum ResultHandlerEndpoint
    {
        /// Send Results to the Backtesting Web Application
        Backtesting,
        /// Send the Results to the Local Console
        Console,
        /// Send Results to the Live Web Application
        LiveTrading,
    }

    /// <summary>
    /// Setup handler options for setting up algorithm state and the livetrading/backtest requirements.
    /// </summary>
    public enum SetupHandlerEndpoint
    {
        /// Configure algorithm+job for backtesting:
        Backtesting,
        /// Configure algorithm+job for the console:
        Console,
        /// Paper trading algorithm+job internal state configuration
        PaperTrading,
        /// Tradier Setup Handler
        Tradier
    }

    /// <summary>
    /// Transaction handler options for selecting who processes the transactions
    /// </summary>
    public enum TransactionHandlerEndpoint 
    { 
        /// Use Backtesting Models to Process Transactions
        Backtesting,
        /// Use Paper Trading Model to Process Transactions
        PaperTrading,
        /// Use Tradier to Process Transactions
        Tradier,
        /// Use Interactive Brokers to Process Transactions
        InteractiveBrokers,
        /// Use FXCM to Process Transactions
        FXCM
    }

    /// <summary>
    /// Cloud storage permission options.
    /// </summary>
    public enum StoragePermissions
    {
        /// Public Storage Permissions
        Public,

        /// Authenticated Read Storage Permissions
        Authenticated
    }

    /// <summary>
    /// Types of tick data - trades or quote ticks.
    /// </summary>
    /// <remarks>QuantConnect currently only has trade tick data but can handle quote tick data with the same data structures.</remarks>
    public enum TickType 
    {
        /// Trade type tick object.
        Trade,
        /// Quote type tick object.
        Quote
    }

    /// <summary>
    /// Resolution of data requested.
    /// </summary>
    /// <remarks>Always sort the enum from the smallest to largest resolution</remarks>
    public enum Resolution 
    {
        /// Tick Resolution (1)
        Tick,
        /// Second Resolution (2)
        Second,
        /// Minute Resolution (3)
        Minute,
        /// Hour Resolution (4)
        Hour,
        /// Daily Resolution (5)
        Daily
    }

    /// <summary>
    /// States of a live deployment.
    /// </summary>
    public enum AlgorithmStatus
    {
        /// User initiated a quit request
        Quit,           //0
        /// Error compiling algorithm at start
        DeployError,    //1
        /// Waiting for a server
        InQueue,        //2
        /// Running algorithm
        Running,        //3
        /// Stopped algorithm or exited with runtime errors
        Stopped,        //4
        /// Liquidated algorithm
        Liquidated,     //5
        /// Algorithm has been deleted
        Deleted,        //6
        /// Algorithm completed running
        Completed       //7
    }


    /// <summary>
    /// enum Period - Enum of all the analysis periods, AS integers. Reference "Period" Array to access the values
    /// </summary>
    public enum Period 
    {
        /// Period Short Codes - 10 
        TenSeconds = 10,
        /// Period Short Codes - 30 Second 
        ThirtySeconds = 30,
        /// Period Short Codes - 60 Second 
        OneMinute = 60,
        /// Period Short Codes - 120 Second 
        TwoMinutes = 120,
        /// Period Short Codes - 180 Second 
        ThreeMinutes = 180,
        /// Period Short Codes - 300 Second 
        FiveMinutes = 300,
        /// Period Short Codes - 600 Second 
        TenMinutes = 600,
        /// Period Short Codes - 900 Second 
        FifteenMinutes = 900,
        /// Period Short Codes - 1200 Second 
        TwentyMinutes = 1200,
        /// Period Short Codes - 1800 Second 
        ThirtyMinutes = 1800,
        /// Period Short Codes - 3600 Second 
        OneHour = 3600,
        /// Period Short Codes - 7200 Second 
        TwoHours = 7200,
        /// Period Short Codes - 14400 Second 
        FourHours = 14400,
        /// Period Short Codes - 21600 Second 
        SixHours = 21600
    }


    /******************************************************** 
    * GLOBAL MARKETS
    *********************************************************/
    /// <summary>
    /// Global Market Short Codes and their full versions: (used in tick objects)
    /// </summary>
    public static class MarketCodes 
    {
        /// US Market Codes
        public static Dictionary<string, string> US = new Dictionary<string, string>() 
        {
            {"A", "American Stock Exchange"},
            {"B", "Boston Stock Exchange"},
            {"C", "National Stock Exchange"},
            {"D", "FINRA ADF"},
            {"I", "International Securities Exchange"},
            {"J", "Direct Edge A"},
            {"K", "Direct Edge X"},
            {"M", "Chicago Stock Exchange"},
            {"N", "New York Stock Exchange"},
            {"P", "Nyse Arca Exchange"},
            {"Q", "NASDAQ OMX"},
            {"T", "NASDAQ OMX"},
            {"U", "OTC Bulletin Board"},
            {"u", "Over-the-Counter trade in Non-NASDAQ issue"},
            {"W", "Chicago Board Options Exchange"},
            {"X", "Philadelphia Stock Exchange"},
            {"Y", "BATS Y-Exchange, Inc"},
            {"Z", "BATS Exchange, Inc"}
        };

        /// Canada Market Short Codes:
        public static Dictionary<string, string> Canada = new Dictionary<string, string>() 
        {
            {"T", "Toronto"},
            {"V", "Venture"}
        };
    }


    /// <summary>
    /// US Public Holidays - Not Tradeable:
    /// </summary>
    public static class USHoliday 
    {
        /// <summary>
        /// Public Holidays
        /// </summary>
        public static List<DateTime> Dates = new List<DateTime>() 
        { 
            /* New Years Day*/
            new DateTime(1998, 01, 01),
            new DateTime(1999, 01, 01),
            new DateTime(2001, 01, 01),
            new DateTime(2002, 01, 01),
            new DateTime(2003, 01, 01),
            new DateTime(2004, 01, 01),
            new DateTime(2006, 01, 02),
            new DateTime(2007, 01, 01),
            new DateTime(2008, 01, 01),
            new DateTime(2009, 01, 01),
            new DateTime(2010, 01, 01),
            new DateTime(2011, 01, 01),
            new DateTime(2012, 01, 02),
            new DateTime(2013, 01, 01),
            new DateTime(2014, 01, 01),
            new DateTime(2015, 01, 01),
            new DateTime(2016, 01, 01),
            
            /* Day of Mouring */
            new DateTime(2007, 01, 02),

            /* World Trade Center */
            new DateTime(2001, 09, 11),
            new DateTime(2001, 09, 12),
            new DateTime(2001, 09, 13),
            new DateTime(2001, 09, 14),

            /* Regan Funeral */
            new DateTime(2004, 06, 11),

            /* Hurricane Sandy */
            new DateTime(2012, 10, 29),
            new DateTime(2012, 10, 30),

            /* Martin Luther King Jnr Day*/
            new DateTime(1998, 01, 19),
            new DateTime(1999, 01, 18),
            new DateTime(2000, 01, 17),
            new DateTime(2001, 01, 15),
            new DateTime(2002, 01, 21),
            new DateTime(2003, 01, 20),
            new DateTime(2004, 01, 19),
            new DateTime(2005, 01, 17),
            new DateTime(2006, 01, 16),
            new DateTime(2007, 01, 15),
            new DateTime(2008, 01, 21),
            new DateTime(2009, 01, 19),
            new DateTime(2010, 01, 18),
            new DateTime(2011, 01, 17),
            new DateTime(2012, 01, 16),
            new DateTime(2013, 01, 21),
            new DateTime(2014, 01, 20),
            new DateTime(2015, 01, 19),
            new DateTime(2016, 01, 18),

            /* Washington / Presidents Day */
            new DateTime(1998, 02, 16),
            new DateTime(1999, 02, 15),
            new DateTime(2000, 02, 21),
            new DateTime(2001, 02, 19),
            new DateTime(2002, 02, 18),
            new DateTime(2003, 02, 17),
            new DateTime(2004, 02, 16),
            new DateTime(2005, 02, 21),
            new DateTime(2006, 02, 20),
            new DateTime(2007, 02, 19),
            new DateTime(2008, 02, 18),
            new DateTime(2009, 02, 16),
            new DateTime(2010, 02, 15),
            new DateTime(2011, 02, 21),
            new DateTime(2012, 02, 20),
            new DateTime(2013, 02, 18),
            new DateTime(2014, 02, 17),
            new DateTime(2015, 02, 16),
            new DateTime(2016, 02, 15),

            /* Good Friday */
            new DateTime(1998, 04, 10),
            new DateTime(1999, 04, 02),
            new DateTime(2000, 04, 21),
            new DateTime(2001, 04, 13),
            new DateTime(2002, 03, 29),
            new DateTime(2003, 04, 18),
            new DateTime(2004, 04, 09),
            new DateTime(2005, 03, 25),
            new DateTime(2006, 04, 14),
            new DateTime(2007, 04, 06),
            new DateTime(2008, 03, 21),
            new DateTime(2009, 04, 10),
            new DateTime(2010, 04, 02),
            new DateTime(2011, 04, 22),
            new DateTime(2012, 04, 06),
            new DateTime(2013, 03, 29),
            new DateTime(2014, 04, 18),
            new DateTime(2015, 04, 03),
            new DateTime(2016, 03, 25),

            /* Memorial Day */
            new DateTime(1998, 05, 25),
            new DateTime(1999, 05, 31),
            new DateTime(2000, 05, 29),
            new DateTime(2001, 05, 28),
            new DateTime(2002, 05, 27),
            new DateTime(2003, 05, 26),
            new DateTime(2004, 05, 31),
            new DateTime(2005, 05, 30),
            new DateTime(2006, 05, 29),
            new DateTime(2007, 05, 28),
            new DateTime(2008, 05, 26),
            new DateTime(2009, 05, 25),
            new DateTime(2010, 05, 31),
            new DateTime(2011, 05, 30),
            new DateTime(2012, 05, 28),
            new DateTime(2013, 05, 27),
            new DateTime(2014, 05, 26),
            new DateTime(2015, 05, 25),
            new DateTime(2016, 05, 30),

            /* Independence Day */
            new DateTime(1998, 07, 03),
            new DateTime(1999, 07, 05),
            new DateTime(2000, 07, 04),
            new DateTime(2001, 07, 04),
            new DateTime(2002, 07, 04),
            new DateTime(2003, 07, 04),
            new DateTime(2004, 07, 05),
            new DateTime(2005, 07, 04),
            new DateTime(2006, 07, 04),
            new DateTime(2007, 07, 04),
            new DateTime(2008, 07, 04),
            new DateTime(2009, 07, 03),
            new DateTime(2010, 07, 05),
            new DateTime(2011, 07, 04),
            new DateTime(2012, 07, 04),
            new DateTime(2013, 07, 04),
            new DateTime(2014, 07, 04),
            new DateTime(2014, 07, 04),
            new DateTime(2015, 07, 03),
            new DateTime(2016, 07, 04),

            /* Labour Day */
            new DateTime(1998, 09, 07),
            new DateTime(1999, 09, 06),
            new DateTime(2000, 09, 04),
            new DateTime(2001, 09, 03),
            new DateTime(2002, 09, 02),
            new DateTime(2003, 09, 01),
            new DateTime(2004, 09, 06),
            new DateTime(2005, 09, 05),
            new DateTime(2006, 09, 04),
            new DateTime(2007, 09, 03),
            new DateTime(2008, 09, 01),
            new DateTime(2009, 09, 07),
            new DateTime(2010, 09, 06),
            new DateTime(2011, 09, 05),
            new DateTime(2012, 09, 03),
            new DateTime(2013, 09, 02),
            new DateTime(2014, 09, 01),
            new DateTime(2015, 09, 07),
            new DateTime(2016, 09, 05),

            /* Thanksgiving Day */
            new DateTime(1998, 11, 26),
            new DateTime(1999, 11, 25),
            new DateTime(2000, 11, 23),
            new DateTime(2001, 11, 22),
            new DateTime(2002, 11, 28),
            new DateTime(2003, 11, 27),
            new DateTime(2004, 11, 25),
            new DateTime(2005, 11, 24),
            new DateTime(2006, 11, 23),
            new DateTime(2007, 11, 22),
            new DateTime(2008, 11, 27),
            new DateTime(2009, 11, 26),
            new DateTime(2010, 11, 25),
            new DateTime(2011, 11, 24),
            new DateTime(2012, 11, 22),
            new DateTime(2013, 11, 28),
            new DateTime(2014, 11, 27),
            new DateTime(2015, 11, 26),
            new DateTime(2016, 11, 24),

            /* Christmas 1998-2014 */
            new DateTime(1998, 12, 25),
            new DateTime(1999, 12, 24),
            new DateTime(2000, 12, 25),
            new DateTime(2001, 12, 25),
            new DateTime(2002, 12, 25),
            new DateTime(2003, 12, 25),
            new DateTime(2004, 12, 24),
            new DateTime(2005, 12, 26),
            new DateTime(2006, 12, 25),
            new DateTime(2007, 12, 25),
            new DateTime(2008, 12, 25),
            new DateTime(2009, 12, 25),
            new DateTime(2010, 12, 24),
            new DateTime(2011, 12, 26),
            new DateTime(2012, 12, 25),
            new DateTime(2013, 12, 25),
            new DateTime(2014, 12, 25),
            new DateTime(2015, 12, 25),
            new DateTime(2016, 12, 25)
        };
    }
} // End QC Namespace:
