﻿/*
 * QUANTCONNECT.COM - Democratizing Finance, Empowering Individuals.
 * Lean Algorithmic Trading Engine v2.0. Copyright 2014 QuantConnect Corporation.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
*/

using System;
using System.Globalization;
using System.Linq;
using QuantConnect.Configuration;
using QuantConnect.Data.Market;
using QuantConnect.Logging;

namespace QuantConnect.ToolBox.DukascopyDownloader
{
    class Program
    {
        /// <summary>
        /// Primary entry point to the program
        /// </summary>
        static void Main(string[] args)
        {
            if (args.Length != 4)
            {
                Console.WriteLine("Usage: DukascopyDownloader SYMBOLS RESOLUTION FROMDATE TODATE");
                Console.WriteLine("SYMBOLS = eg EURUSD,USDJPY");
                Console.WriteLine("RESOLUTION = Tick/Second/Minute/Hour/Daily/All");
                Console.WriteLine("FROMDATE = yyyymmdd");
                Console.WriteLine("TODATE = yyyymmdd");
                Environment.Exit(1);
            }

            try
            {
                // Load settings from command line
                var symbols = args[0].Split(',');
                var allResolutions = args[1].ToLower() == "all";
                var resolution = allResolutions ? Resolution.Tick : (Resolution)Enum.Parse(typeof(Resolution), args[1]);
                var startDate = DateTime.ParseExact(args[2], "yyyyMMdd", CultureInfo.InvariantCulture);
                var endDate = DateTime.ParseExact(args[3], "yyyyMMdd", CultureInfo.InvariantCulture);

                // Load settings from config.json
                var dataDirectory = Config.Get("data-directory", "../../../Data");

                // Download the data
                const string market = Market.Dukascopy;
                var downloader = new DukascopyDataDownloader();

                foreach (var symbol in symbols)
                {
                    if (!downloader.HasSymbol(symbol))
                        throw new ArgumentException("The symbol " + symbol + " is not available.");
                }

                foreach (var symbol in symbols)
                {
                    var securityType = downloader.GetSecurityType(symbol);
                    var symbolObject = new Symbol(GetSid(symbol, securityType), symbol);
                    var data = downloader.Get(symbolObject, securityType, resolution, startDate, endDate);

                    if (allResolutions)
                    {
                        var ticks = data.Cast<Tick>().ToList();

                        // Save the data (tick resolution)
                        var writer = new LeanDataWriter(securityType, resolution, symbolObject, dataDirectory, market);
                        writer.Write(ticks);

                        // Save the data (other resolutions)
                        foreach (var res in new[] { Resolution.Second, Resolution.Minute, Resolution.Hour, Resolution.Daily })
                        {
                            var resData = DukascopyDataDownloader.AggregateTicks(symbolObject, ticks, res.ToTimeSpan());

                            writer = new LeanDataWriter(securityType, res, symbolObject, dataDirectory, market);
                            writer.Write(resData);
                        }
                    }
                    else
                    {
                        // Save the data (single resolution)
                        var writer = new LeanDataWriter(securityType, resolution, symbolObject, dataDirectory, market);
                        writer.Write(data);
                    }
                }
            }
            catch (Exception err)
            {
                Log.Error("DukascopyDownloader(): {0}", err.Message);
            }
        }

        static SecurityIdentifier GetSid(string symbol, SecurityType securityType)
        {
            if (securityType == SecurityType.Forex)
            {
                return SecurityIdentifier.GenerateForex(symbol, Market.Dukascopy);
            }
            if (securityType == SecurityType.Cfd)
            {
                return SecurityIdentifier.GenerateCfd(symbol, Market.Dukascopy);
            }

            throw new NotImplementedException("The specfied security type has not been implemented yet: " + securityType);
        }
    }
}
