﻿/*
 * QUANTCONNECT.COM - Democratizing Finance, Empowering Individuals.
 * Lean Algorithmic Trading Engine v2.0. Copyright 2014 QuantConnect Corporation.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
*/

using System;

namespace QuantConnect.Data.Market
{
    /// <summary>
    /// Split event from a security
    /// </summary>
    public class Split : BaseData
    {
        /// <summary>
        /// Initializes a new instance of the Split class
        /// </summary>
        public Split()
        {
            DataType = MarketDataType.Auxiliary;
        }

        /// <summary>
        /// Initializes a new instance of the Split class
        /// </summary>
        /// <param name="symbol">The symbol</param>
        /// <param name="date">The date</param>
        /// <param name="price">The price at the time of the split</param>
        /// <param name="splitFactor">The split factor to be applied to current holdings</param>
        public Split(string symbol, DateTime date, decimal price, decimal splitFactor)
             : this()
        {
            Symbol = symbol;
            Time = date;
            ReferencePrice = price;
            SplitFactor = splitFactor;
        }

        /// <summary>
        /// Gets the split factor
        /// </summary>
        public decimal SplitFactor
        {
            get { return Value; }
            set { Value = value; }
        }

        /// <summary>
        /// Gets the price at which the split occurred
        /// </summary>
        public decimal ReferencePrice
        {
            get; private set;
        }

        /// <summary>
        /// Reader converts each line of the data source into BaseData objects. Each data type creates its own factory method, and returns a new instance of the object 
        /// each time it is called. 
        /// </summary>
        /// <param name="config">Subscription data config setup object</param>
        /// <param name="line">Line of the source document</param>
        /// <param name="date">Date of the requested data</param>
        /// <param name="isLiveMode">true if we're in live mode, false for backtesting mode</param>
        /// <returns>Instance of the T:BaseData object generated by this line of the CSV</returns>
        public override BaseData Reader(SubscriptionDataConfig config, string line, DateTime date, bool isLiveMode)
        {
            // this is implemented in the SubscriptionDataReader.RefreshSource
            throw new NotImplementedException("This method is not supposed to be called on the Split type.");
        }

        /// <summary>
        /// Return the URL string source of the file. This will be converted to a stream 
        /// </summary>
        /// <param name="config">Configuration object</param>
        /// <param name="date">Date of this source file</param>
        /// <param name="isLiveMode">true if we're in live mode, false for backtesting mode</param>
        /// <returns>String URL of source file.</returns>
        public override SubscriptionDataSource GetSource(SubscriptionDataConfig config, DateTime date, bool isLiveMode)
        {
            // this data is derived from map files and factor files in backtesting
            throw new NotImplementedException("This method is not supposed to be called on the Split type.");
        }

        /// <summary>
        /// Returns a <see cref="System.String"/> that represents the current <see cref="QuantConnect.Data.Market.Split"/>.
        /// </summary>
        /// <returns>A <see cref="System.String"/> that represents the current <see cref="QuantConnect.Data.Market.Split"/>.</returns>
        public override string ToString()
        {
            return string.Format("{0}: {1}", Symbol, SplitFactor);
        }
    }
}
