/*
 * LensKit, an open source recommender systems toolkit.
 * Copyright 2010-2013 Regents of the University of Minnesota and contributors
 * Work on LensKit has been funded by the National Science Foundation under
 * grants IIS 05-34939, 08-08692, 08-12148, and 10-17697.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package org.grouplens.lenskit.config;

import com.google.common.base.Preconditions;
import groovy.lang.*;
import org.codehaus.groovy.control.CompilerConfiguration;
import org.codehaus.groovy.control.customizers.ImportCustomizer;
import org.codehaus.groovy.runtime.StackTraceUtils;
import org.grouplens.lenskit.core.LenskitConfiguration;
import org.grouplens.lenskit.core.RecommenderConfigurationException;

import javax.annotation.Nonnull;
import java.io.File;
import java.io.IOException;
import java.net.URL;

/**
 * Load LensKit configurations using the configuration DSL.
 *
 * @since 1.2
 * @author <a href="http://www.grouplens.org">GroupLens Research</a>
 */
public class ConfigurationLoader {
    private final ClassLoader classLoader;
    private final GroovyShell shell;
    private final Binding binding;
    private int scriptNumber;

    /**
     * Construct a new configuration loader. It uses the current thread's class loader.
     * @review Is this the classloader we should use?
     */
    public ConfigurationLoader() {
        this(Thread.currentThread().getContextClassLoader());
    }

    /**
     * Construct a new configuration loader.
     * @param loader The class loader to use.
     */
    public ConfigurationLoader(ClassLoader loader) {
        classLoader = loader;
        binding = new Binding();
        CompilerConfiguration config = new CompilerConfiguration();
        config.setScriptBaseClass(LenskitConfigScript.class.getName());
        ImportCustomizer imports = new ImportCustomizer();
        imports.addStarImports("org.grouplens.lenskit");
        config.addCompilationCustomizers(imports);
        shell = new GroovyShell(loader, binding, config);
    }

    private LenskitConfiguration load(GroovyCodeSource source) throws RecommenderConfigurationException {
        LenskitConfigScript script;
        try {
            script = (LenskitConfigScript) shell.parse(source);
        } catch (GroovyRuntimeException e) {
            throw new RecommenderConfigurationException("Error loading Groovy script", e);
        }

        try {
            script.run();
        } catch (Exception ex) {
            throw new RecommenderConfigurationException("Error running Groovy script",
                                                        StackTraceUtils.deepSanitize(ex));
        }

        return script.getConfig();
    }

    /**
     * Load a configuration from a file.
     * @param file The configuration script to load.
     * @return The resulting LensKit configuration.
     */
    public LenskitConfiguration load(@Nonnull File file) throws IOException, RecommenderConfigurationException {
        Preconditions.checkNotNull(file, "Configuration file");
        return load(new GroovyCodeSource(file));
    }

    /**
     * Load a configuration from a URL.
     * @param url The configuration script to load.
     * @return The resulting LensKit configuration.
     */
    public LenskitConfiguration load(@Nonnull URL url) throws IOException, RecommenderConfigurationException {
        Preconditions.checkNotNull(url, "Configuration URL");
        return load(new GroovyCodeSource(url));
    }

    /**
     * Load a configuration from a script source.
     * @param source The configuration script to load.
     * @return The resulting LensKit configuration.
     */
    public LenskitConfiguration load(@Nonnull String source) throws RecommenderConfigurationException {
        Preconditions.checkNotNull(source, "Configuration source text");
        return load(new GroovyCodeSource(source, "LKConfig" + (++scriptNumber),
                                         GroovyShell.DEFAULT_CODE_BASE));
    }

    /**
     * Load a configuration from a closure. The class loader is not really consulted in this case.
     * @param block The block to evaluate. This block will be evaluated with a delegate providing
     *              the LensKit DSL and the {@link Closure#DELEGATE_FIRST} resolution strategy.
     * @return The resulting LensKit configuration.
     */
    public LenskitConfiguration load(@Nonnull Closure<?> block) throws RecommenderConfigurationException {
        Preconditions.checkNotNull(block, "Configuration block");
        LenskitConfiguration config = new LenskitConfiguration();
        BindingDSL delegate = LenskitConfigDSL.forConfig(config);
        try {
            GroovyUtils.callWithDelegate(block, delegate);
        } catch (GroovyRuntimeException e) {
            // this quite possibly wraps an exception we want to throw
            if (e.getClass().equals(GroovyRuntimeException.class) && e.getCause() != null) {
                throw new RecommenderConfigurationException("Error evaluating Groovy block",
                                                            e.getCause());
            } else {
                throw new RecommenderConfigurationException("Error evaluating Groovy block", e);
            }
        } catch (RuntimeException e) {
            throw new RecommenderConfigurationException("Error evaluating Groovy block", e);
        }
        return config;
    }
}
