/*
 * LensKit, an open source recommender systems toolkit.
 * Copyright 2010-2013 Regents of the University of Minnesota and contributors
 * Work on LensKit has been funded by the National Science Foundation under
 * grants IIS 05-34939, 08-08692, 08-12148, and 10-17697.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package org.grouplens.lenskit.eval.algorithm;

import com.google.common.base.Supplier;
import org.grouplens.lenskit.RecommenderBuildException;
import org.grouplens.lenskit.data.snapshot.PreferenceSnapshot;
import org.grouplens.lenskit.eval.ExecutionInfo;
import org.grouplens.lenskit.eval.traintest.SharedPreferenceSnapshot;
import org.grouplens.lenskit.eval.data.traintest.TTDataSet;

import javax.annotation.Nonnull;
import javax.inject.Provider;
import java.util.Map;

/**
 * An algorithm instance. On its own, this doesn't do much; it exists to share some
 * metadata between {@link LenskitAlgorithmInstance} and {@link ExternalAlgorithmInstance}.
 *
 * @author <a href="http://www.grouplens.org">GroupLens Research</a>
 */
public interface AlgorithmInstance {
    /**
     * Get the name of this algorithm instance.
     * @return The instance's name.
     */
    String getName();

    /**
     * Get the attributes associated with this algorithm instance.
     * @return The algorithm instance's attributes.
     */
    @Nonnull
    Map<String, Object> getAttributes();

    /**
     * Create a testable recommender instance from this algorithm.
     *
     *
     * @param data     The data set. The test data should only be used if the recommender needs to
     *                 capture the test data (e.g. an external program that will produce predictions
     *                 en mass).
     * @param snapshot The (cached) shared preference snapshot.
     * @param info     The execution info for this execution.
     * @return A recommender instance for testing this algorithm.
     */
    RecommenderInstance makeTestableRecommender(TTDataSet data,
                                                Provider<? extends PreferenceSnapshot> snapshot,
                                                ExecutionInfo info) throws RecommenderBuildException;
}
