// Copyright 2021 gorse Project Authors
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package data

import (
	"database/sql"
	"encoding/json"
	"fmt"
	_ "github.com/go-sql-driver/mysql"
	"github.com/juju/errors"
	_ "github.com/lib/pq"
	_ "github.com/mailru/go-clickhouse"
	"github.com/scylladb/go-set/strset"
	"github.com/zhenghaoz/gorse/base"
	"go.uber.org/zap"
	"strings"
	"time"
)

type SQLDriver int

const (
	MySQL SQLDriver = iota
	Postgres
	ClickHouse
)

// SQLDatabase use MySQL as data storage.
type SQLDatabase struct {
	client *sql.DB
	driver SQLDriver
}

// Optimize is used by ClickHouse only.
func (d *SQLDatabase) Optimize() error {
	if d.driver == ClickHouse {
		for _, tableName := range []string{"users", "items", "feedback", "measurements"} {
			_, err := d.client.Exec("OPTIMIZE TABLE " + tableName)
			if err != nil {
				return errors.Trace(err)
			}
		}
	}
	return nil
}

// Init tables and indices in MySQL.
func (d *SQLDatabase) Init() error {
	switch d.driver {
	case MySQL:
		// create tables
		if _, err := d.client.Exec("CREATE TABLE IF NOT EXISTS items (" +
			"item_id varchar(256) NOT NULL," +
			"time_stamp timestamp NOT NULL," +
			"labels json NOT NULL," +
			"comment TEXT NOT NULL," +
			"PRIMARY KEY(item_id)" +
			")  ENGINE=InnoDB"); err != nil {
			return errors.Trace(err)
		}
		if _, err := d.client.Exec("CREATE TABLE IF NOT EXISTS users (" +
			"user_id varchar(256) NOT NULL," +
			"labels json NOT NULL," +
			"subscribe json NOT NULL," +
			"comment TEXT NOT NULL," +
			"PRIMARY KEY (user_id)" +
			")  ENGINE=InnoDB"); err != nil {
			return errors.Trace(err)
		}
		if _, err := d.client.Exec("CREATE TABLE IF NOT EXISTS feedback (" +
			"feedback_type varchar(256) NOT NULL," +
			"user_id varchar(256) NOT NULL," +
			"item_id varchar(256) NOT NULL," +
			"time_stamp timestamp NOT NULL," +
			"comment TEXT NOT NULL," +
			"PRIMARY KEY(feedback_type, user_id, item_id)," +
			"INDEX (user_id)" +
			")  ENGINE=InnoDB"); err != nil {
			return errors.Trace(err)
		}
		if _, err := d.client.Exec("CREATE TABLE IF NOT EXISTS measurements (" +
			"name varchar(256) NOT NULL," +
			"time_stamp timestamp NOT NULL," +
			"value double NOT NULL," +
			"comment TEXT NOT NULL," +
			"PRIMARY KEY(name, time_stamp)" +
			")  ENGINE=InnoDB"); err != nil {
			return errors.Trace(err)
		}
		// change settings
		_, err := d.client.Exec("SET SESSION sql_mode=\"" +
			"ONLY_FULL_GROUP_BY,STRICT_TRANS_TABLES,ERROR_FOR_DIVISION_BY_ZERO," +
			"NO_ENGINE_SUBSTITUTION\"")
		return errors.Trace(err)
	case Postgres:
		// create tables
		if _, err := d.client.Exec("CREATE TABLE IF NOT EXISTS items (" +
			"item_id varchar(256) NOT NULL," +
			"time_stamp timestamptz NOT NULL DEFAULT '0001-01-01'," +
			"labels json NOT NULL DEFAULT '[]'," +
			"comment TEXT NOT NULL DEFAULT ''," +
			"PRIMARY KEY(item_id)" +
			")"); err != nil {
			return errors.Trace(err)
		}
		if _, err := d.client.Exec("CREATE TABLE IF NOT EXISTS users (" +
			"user_id varchar(256) NOT NULL," +
			"labels json NOT NULL DEFAULT '[]'," +
			"subscribe json NOT NULL DEFAULT '[]'," +
			"comment TEXT NOT NULL DEFAULT ''," +
			"PRIMARY KEY (user_id)" +
			")"); err != nil {
			return errors.Trace(err)
		}
		if _, err := d.client.Exec("CREATE TABLE IF NOT EXISTS feedback (" +
			"feedback_type varchar(256) NOT NULL," +
			"user_id varchar(256) NOT NULL," +
			"item_id varchar(256) NOT NULL," +
			"time_stamp timestamptz NOT NULL DEFAULT '0001-01-01'," +
			"comment TEXT NOT NULL DEFAULT ''," +
			"PRIMARY KEY(feedback_type, user_id, item_id)" +
			")"); err != nil {
			return errors.Trace(err)
		}
		if _, err := d.client.Exec("CREATE INDEX IF NOT EXISTS user_id_index ON feedback(user_id)"); err != nil {
			return errors.Trace(err)
		}
		if _, err := d.client.Exec("CREATE TABLE IF NOT EXISTS measurements (" +
			"name varchar(256) NOT NULL," +
			"time_stamp timestamptz NOT NULL," +
			"value double precision NOT NULL," +
			"comment TEXT NOT NULL," +
			"PRIMARY KEY(name, time_stamp)" +
			")"); err != nil {
			return errors.Trace(err)
		}
	case ClickHouse:
		// create tables
		if _, err := d.client.Exec("CREATE TABLE IF NOT EXISTS items (" +
			"item_id String," +
			"time_stamp Datetime," +
			"labels String DEFAULT '[]'," +
			"comment String," +
			"version DateTime" +
			") ENGINE = ReplacingMergeTree(version) ORDER BY item_id"); err != nil {
			return errors.Trace(err)
		}
		if _, err := d.client.Exec("CREATE TABLE IF NOT EXISTS users (" +
			"user_id String," +
			"labels String DEFAULT '[]'," +
			"subscribe String DEFAULT '[]'," +
			"comment String," +
			"version DateTime" +
			") ENGINE = ReplacingMergeTree(version) ORDER BY user_id"); err != nil {
			return errors.Trace(err)
		}
		if _, err := d.client.Exec("CREATE TABLE IF NOT EXISTS feedback (" +
			"feedback_type String," +
			"user_id String," +
			"item_id String," +
			"time_stamp Datetime," +
			"comment String" +
			") ENGINE = ReplacingMergeTree() ORDER BY (feedback_type, user_id, item_id)"); err != nil {
			return errors.Trace(err)
		}
		if _, err := d.client.Exec("CREATE TABLE IF NOT EXISTS measurements (" +
			"name String," +
			"time_stamp Datetime," +
			"value Float64," +
			"comment String" +
			") ENGINE = ReplacingMergeTree() ORDER BY (name, time_stamp)"); err != nil {
			return errors.Trace(err)
		}
	}
	return nil
}

// Close MySQL connection.
func (d *SQLDatabase) Close() error {
	return d.client.Close()
}

// InsertMeasurement insert a measurement into MySQL.
func (d *SQLDatabase) InsertMeasurement(measurement Measurement) error {
	var err error
	switch d.driver {
	case MySQL:
		_, err = d.client.Exec("INSERT INTO measurements(name, time_stamp, value, `comment`) VALUES (?, ?, ?, ?) AS new "+
			"ON DUPLICATE KEY UPDATE value = new.value, `comment` = new.comment",
			measurement.Name, measurement.Timestamp, measurement.Value, measurement.Comment)
	case Postgres:
		_, err = d.client.Exec("INSERT INTO measurements(name, time_stamp, value, comment) VALUES ($1, $2, $3, $4)  "+
			"ON CONFLICT (name, time_stamp) DO UPDATE SET value = EXCLUDED.value, comment = EXCLUDED.comment",
			measurement.Name, measurement.Timestamp, measurement.Value, measurement.Comment)
	case ClickHouse:
		_, err = d.client.Exec("INSERT INTO measurements(name, time_stamp, value, `comment`) VALUES (?, ?, ?, ?)",
			measurement.Name, measurement.Timestamp, measurement.Value, measurement.Comment)
	}
	return errors.Trace(err)
}

// GetMeasurements returns recent measurements from MySQL.
func (d *SQLDatabase) GetMeasurements(name string, n int) ([]Measurement, error) {
	measurements := make([]Measurement, 0)
	var result *sql.Rows
	var err error
	switch d.driver {
	case MySQL, ClickHouse:
		result, err = d.client.Query("SELECT name, time_stamp, value, `comment` FROM measurements WHERE name = ? ORDER BY time_stamp DESC LIMIT ?", name, n)
	case Postgres:
		result, err = d.client.Query("SELECT name, time_stamp, value, comment FROM measurements WHERE name = $1 ORDER BY time_stamp DESC LIMIT $2", name, n)
	}
	if err != nil {
		return measurements, errors.Trace(err)
	}
	defer result.Close()
	for result.Next() {
		var measurement Measurement
		if err = result.Scan(&measurement.Name, &measurement.Timestamp, &measurement.Value, &measurement.Comment); err != nil {
			return measurements, errors.Trace(err)
		}
		measurements = append(measurements, measurement)
	}
	return measurements, nil
}

// InsertItem inserts a item into MySQL.
func (d *SQLDatabase) InsertItem(item Item) error {
	startTime := time.Now()
	labels, err := json.Marshal(item.Labels)
	if err != nil {
		return errors.Trace(err)
	}
	switch d.driver {
	case MySQL:
		_, err = d.client.Exec("INSERT items(item_id, time_stamp, labels, `comment`) VALUES (?, ?, ?, ?) "+
			"ON DUPLICATE KEY UPDATE time_stamp = ?, labels = ?, `comment` = ?",
			item.ItemId, item.Timestamp, labels, item.Comment, item.Timestamp, labels, item.Comment)
	case Postgres:
		_, err = d.client.Exec("INSERT INTO items(item_id, time_stamp, labels, comment) VALUES ($1, $2, $3, $4) "+
			"ON CONFLICT (item_id) "+
			"DO UPDATE SET time_stamp = EXCLUDED.time_stamp, labels = EXCLUDED.labels, comment = EXCLUDED.comment",
			item.ItemId, item.Timestamp, labels, item.Comment)
	case ClickHouse:
		_, err = d.client.Exec("INSERT INTO items(item_id, time_stamp, labels, `comment`, version) VALUES (?, ?, ?, ?, NOW())",
			item.ItemId, item.Timestamp, string(labels), item.Comment)
	}
	InsertItemLatency.Observe(time.Since(startTime).Seconds())
	return errors.Trace(err)
}

// BatchInsertItem inserts a batch of items into MySQL.
func (d *SQLDatabase) BatchInsertItem(items []Item) error {
	const batchSize = 10000
	for i := 0; i < len(items); i += batchSize {
		batchItems := items[i:base.Min(i+batchSize, len(items))]
		// build query
		builder := strings.Builder{}
		switch d.driver {
		case MySQL:
			builder.WriteString("INSERT INTO items(item_id, time_stamp, labels, `comment`) VALUES ")
		case Postgres:
			builder.WriteString("INSERT INTO items(item_id, time_stamp, labels, comment) VALUES ")
		case ClickHouse:
			builder.WriteString("INSERT INTO items(item_id, time_stamp, labels, comment, version) VALUES ")
		}
		var args []interface{}
		for i, item := range batchItems {
			labels, err := json.Marshal(item.Labels)
			if err != nil {
				return errors.Trace(err)
			}
			switch d.driver {
			case MySQL:
				builder.WriteString("(?,?,?,?)")
			case Postgres:
				builder.WriteString(fmt.Sprintf("($%d,$%d,$%d,$%d)", len(args)+1, len(args)+2, len(args)+3, len(args)+4))
			case ClickHouse:
				builder.WriteString("(?,?,?,?,NOW())")
			}
			if i+1 < len(batchItems) {
				builder.WriteString(",")
			}
			args = append(args, item.ItemId, item.Timestamp, string(labels), item.Comment)
		}
		switch d.driver {
		case MySQL:
			builder.WriteString(" AS new ON DUPLICATE KEY " +
				"UPDATE time_stamp = new.time_stamp, labels = new.labels, `comment` = new.comment")
		case Postgres:
			builder.WriteString(" ON CONFLICT (item_id) " +
				"DO UPDATE SET time_stamp = EXCLUDED.time_stamp, labels = EXCLUDED.labels, comment = EXCLUDED.comment")
		}
		_, err := d.client.Exec(builder.String(), args...)
		if err != nil {
			return errors.Trace(err)
		}
	}
	return nil
}

// DeleteItem deletes a item from MySQL.
func (d *SQLDatabase) DeleteItem(itemId string) error {
	txn, err := d.client.Begin()
	if err != nil {
		return errors.Trace(err)
	}
	switch d.driver {
	case MySQL:
		_, err = txn.Exec("DELETE FROM items WHERE item_id = ?", itemId)
	case Postgres:
		_, err = txn.Exec("DELETE FROM items WHERE item_id = $1", itemId)
	case ClickHouse:
		_, err = txn.Exec("ALTER TABLE items DELETE WHERE item_id = ?", itemId)
	}
	if err != nil {
		if err = txn.Rollback(); err != nil {
			return errors.Trace(err)
		}
		return errors.Trace(err)
	}
	switch d.driver {
	case MySQL:
		_, err = txn.Exec("DELETE FROM feedback WHERE item_id = ?", itemId)
	case Postgres:
		_, err = txn.Exec("DELETE FROM feedback WHERE item_id = $1", itemId)
	case ClickHouse:
		_, err = txn.Exec("ALTER TABLE feedback DELETE WHERE item_id = ?", itemId)
	}
	if err != nil {
		if err = txn.Rollback(); err != nil {
			return errors.Trace(err)
		}
		return errors.Trace(err)
	}
	return txn.Commit()
}

// GetItem get a item from MySQL.
func (d *SQLDatabase) GetItem(itemId string) (Item, error) {
	startTime := time.Now()
	var result *sql.Rows
	var err error
	switch d.driver {
	case MySQL, ClickHouse:
		result, err = d.client.Query("SELECT item_id, time_stamp, labels, `comment` FROM items WHERE item_id = ?", itemId)
	case Postgres:
		result, err = d.client.Query("SELECT item_id, time_stamp, labels, comment FROM items WHERE item_id = $1", itemId)
	}
	if err != nil {
		return Item{}, errors.Trace(err)
	}
	defer result.Close()
	if result.Next() {
		var item Item
		var labels string
		if err := result.Scan(&item.ItemId, &item.Timestamp, &labels, &item.Comment); err != nil {
			return Item{}, err
		}
		if err := json.Unmarshal([]byte(labels), &item.Labels); err != nil {
			return Item{}, err
		}
		GetItemLatency.Observe(time.Since(startTime).Seconds())
		return item, nil
	}
	return Item{}, ErrItemNotExist
}

// GetItems returns items from MySQL.
func (d *SQLDatabase) GetItems(cursor string, n int, timeLimit *time.Time) (string, []Item, error) {
	var result *sql.Rows
	var err error
	switch d.driver {
	case MySQL, ClickHouse:
		if timeLimit == nil {
			result, err = d.client.Query("SELECT item_id, time_stamp, labels, `comment` FROM items "+
				"WHERE item_id >= ? ORDER BY item_id LIMIT ?", cursor, n+1)
		} else {
			result, err = d.client.Query("SELECT item_id, time_stamp, labels, `comment` FROM items "+
				"WHERE item_id >= ? AND time_stamp >= ? ORDER BY item_id LIMIT ?", cursor, *timeLimit, n+1)
		}
	case Postgres:
		if timeLimit == nil {
			result, err = d.client.Query("SELECT item_id, time_stamp, labels, comment FROM items "+
				"WHERE item_id >= $1 ORDER BY item_id LIMIT $2", cursor, n+1)
		} else {
			result, err = d.client.Query("SELECT item_id, time_stamp, labels, comment FROM items "+
				"WHERE item_id >= $1 AND time_stamp >= $2 ORDER BY item_id LIMIT $3", cursor, *timeLimit, n+1)
		}
	}
	if err != nil {
		return "", nil, errors.Trace(err)
	}
	items := make([]Item, 0)
	defer result.Close()
	for result.Next() {
		var item Item
		var labels string
		if err = result.Scan(&item.ItemId, &item.Timestamp, &labels, &item.Comment); err != nil {
			return "", nil, errors.Trace(err)
		}
		if err = json.Unmarshal([]byte(labels), &item.Labels); err != nil {
			return "", nil, errors.Trace(err)
		}
		items = append(items, item)
	}
	if len(items) == n+1 {
		return items[len(items)-1].ItemId, items[:len(items)-1], nil
	}
	return "", items, nil
}

// GetItemFeedback returns feedback of a item from MySQL.
func (d *SQLDatabase) GetItemFeedback(itemId string, feedbackTypes ...string) ([]Feedback, error) {
	startTime := time.Now()
	var result *sql.Rows
	var err error
	var builder strings.Builder
	switch d.driver {
	case MySQL, ClickHouse:
		builder.WriteString("SELECT user_id, item_id, feedback_type FROM feedback WHERE item_id = ?")
	case Postgres:
		builder.WriteString("SELECT user_id, item_id, feedback_type FROM feedback WHERE item_id = $1")
	}
	args := []interface{}{itemId}
	if len(feedbackTypes) > 0 {
		builder.WriteString(" AND feedback_type IN (")
		for i, feedbackType := range feedbackTypes {
			switch d.driver {
			case MySQL, ClickHouse:
				builder.WriteString("?")
			case Postgres:
				builder.WriteString(fmt.Sprintf("$%d", len(args)+1))
			}
			if i+1 < len(feedbackTypes) {
				builder.WriteString(",")
			}
			args = append(args, feedbackType)
		}
		builder.WriteString(")")
	}
	result, err = d.client.Query(builder.String(), args...)
	if err != nil {
		return nil, errors.Trace(err)
	}
	feedbacks := make([]Feedback, 0)
	for result.Next() {
		var feedback Feedback
		if err = result.Scan(&feedback.UserId, &feedback.ItemId, &feedback.FeedbackType); err != nil {
			return nil, errors.Trace(err)
		}
		feedbacks = append(feedbacks, feedback)
	}
	GetItemFeedbackLatency.Observe(time.Since(startTime).Seconds())
	return feedbacks, nil
}

// InsertUser inserts a user into MySQL.
func (d *SQLDatabase) InsertUser(user User) error {
	labels, err := json.Marshal(user.Labels)
	if err != nil {
		return errors.Trace(err)
	}
	subscribe, err := json.Marshal(user.Subscribe)
	if err != nil {
		return errors.Trace(err)
	}
	switch d.driver {
	case MySQL:
		_, err = d.client.Exec("INSERT INTO users(user_id, labels, subscribe, `comment`) VALUES (?, ?, ?, ?) "+
			"ON DUPLICATE KEY UPDATE labels = ?, subscribe = ?, `comment` = ?",
			user.UserId, labels, subscribe, user.Comment, labels, subscribe, user.Comment)
	case Postgres:
		_, err = d.client.Exec("INSERT INTO users(user_id, labels, subscribe, comment) VALUES ($1, $2, $3, $4) "+
			"ON CONFLICT (user_id) "+
			"DO UPDATE SET labels = EXCLUDED.labels, subscribe = EXCLUDED.subscribe, comment = EXCLUDED.comment",
			user.UserId, labels, subscribe, user.Comment)
	case ClickHouse:
		_, err = d.client.Exec("INSERT INTO users(user_id, labels, subscribe, comment, version) VALUES (?, ?, ?, ?, NOW()) ",
			user.UserId, string(labels), string(subscribe), user.Comment)
	}
	return errors.Trace(err)
}

// DeleteUser deletes a user from MySQL.
func (d *SQLDatabase) DeleteUser(userId string) error {
	txn, err := d.client.Begin()
	if err != nil {
		return errors.Trace(err)
	}
	switch d.driver {
	case MySQL:
		_, err = txn.Exec("DELETE FROM users WHERE user_id = ?", userId)
	case Postgres:
		_, err = txn.Exec("DELETE FROM users WHERE user_id = $1", userId)
	case ClickHouse:
		_, err = txn.Exec("ALTER TABLE users DELETE WHERE user_id = ?", userId)
	}
	if err != nil {
		if err = txn.Rollback(); err != nil {
			return errors.Trace(err)
		}
		return errors.Trace(err)
	}
	switch d.driver {
	case MySQL:
		_, err = txn.Exec("DELETE FROM feedback WHERE user_id = ?", userId)
	case Postgres:
		_, err = txn.Exec("DELETE FROM feedback WHERE user_id = $1", userId)
	case ClickHouse:
		_, err = txn.Exec("ALTER TABLE feedback DELETE WHERE user_id = ?", userId)
	}
	if err != nil {
		if err = txn.Rollback(); err != nil {
			return errors.Trace(err)
		}
		return errors.Trace(err)
	}
	return txn.Commit()
}

// GetUser returns a user from MySQL.
func (d *SQLDatabase) GetUser(userId string) (User, error) {
	var result *sql.Rows
	var err error
	switch d.driver {
	case MySQL:
		result, err = d.client.Query("SELECT user_id, labels, subscribe, `comment` FROM users WHERE user_id = ?", userId)
	case Postgres:
		result, err = d.client.Query("SELECT user_id, labels, subscribe, comment FROM users WHERE user_id = $1", userId)
	case ClickHouse:
		result, err = d.client.Query("SELECT user_id, labels, subscribe, `comment` FROM users WHERE user_id = ?", userId)
	}
	if err != nil {
		return User{}, errors.Trace(err)
	}
	defer result.Close()
	if result.Next() {
		var user User
		var labels string
		var subscribe string
		if err = result.Scan(&user.UserId, &labels, &subscribe, &user.Comment); err != nil {
			return User{}, errors.Trace(err)
		}
		if err = json.Unmarshal([]byte(labels), &user.Labels); err != nil {
			return User{}, errors.Trace(err)
		}
		if err = json.Unmarshal([]byte(subscribe), &user.Subscribe); err != nil {
			return User{}, errors.Trace(err)
		}
		return user, nil
	}
	return User{}, ErrUserNotExist
}

// GetUsers returns users from MySQL.
func (d *SQLDatabase) GetUsers(cursor string, n int) (string, []User, error) {
	var result *sql.Rows
	var err error
	switch d.driver {
	case MySQL:
		result, err = d.client.Query("SELECT user_id, labels, subscribe, `comment` FROM users "+
			"WHERE user_id >= ? ORDER BY user_id LIMIT ?", cursor, n+1)
	case Postgres:
		result, err = d.client.Query("SELECT user_id, labels, subscribe, comment FROM users "+
			"WHERE user_id >= $1 ORDER BY user_id LIMIT $2", cursor, n+1)
	case ClickHouse:
		result, err = d.client.Query("SELECT user_id, labels, subscribe, `comment` FROM users "+
			"WHERE user_id >= ? ORDER BY user_id LIMIT ?", cursor, n+1)
	}
	if err != nil {
		return "", nil, errors.Trace(err)
	}
	users := make([]User, 0)
	defer result.Close()
	for result.Next() {
		var user User
		var labels string
		var subscribe string
		if err = result.Scan(&user.UserId, &labels, &subscribe, &user.Comment); err != nil {
			return "", nil, errors.Trace(err)
		}
		if err = json.Unmarshal([]byte(labels), &user.Labels); err != nil {
			return "", nil, errors.Trace(err)
		}
		if err = json.Unmarshal([]byte(subscribe), &user.Subscribe); err != nil {
			return "", nil, errors.Trace(err)
		}
		users = append(users, user)
	}
	if len(users) == n+1 {
		return users[len(users)-1].UserId, users[:len(users)-1], nil
	}
	return "", users, nil
}

// GetUserFeedback returns feedback of a user from MySQL.
func (d *SQLDatabase) GetUserFeedback(userId string, feedbackTypes ...string) ([]Feedback, error) {
	startTime := time.Now()
	var result *sql.Rows
	var err error
	var builder strings.Builder
	switch d.driver {
	case MySQL, ClickHouse:
		builder.WriteString("SELECT feedback_type, user_id, item_id, time_stamp, `comment` FROM feedback WHERE user_id = ?")
	case Postgres:
		builder.WriteString("SELECT feedback_type, user_id, item_id, time_stamp, comment FROM feedback WHERE user_id = $1")
	}
	args := []interface{}{userId}
	if len(feedbackTypes) > 0 {
		builder.WriteString(" AND feedback_type IN (")
		for i, feedbackType := range feedbackTypes {
			switch d.driver {
			case MySQL, ClickHouse:
				builder.WriteString("?")
			case Postgres:
				builder.WriteString(fmt.Sprintf("$%d", len(args)+1))
			}
			if i+1 < len(feedbackTypes) {
				builder.WriteString(",")
			}
			args = append(args, feedbackType)
		}
		builder.WriteString(")")
	}
	result, err = d.client.Query(builder.String(), args...)
	if err != nil {
		return nil, errors.Trace(err)
	}
	feedbacks := make([]Feedback, 0)
	for result.Next() {
		var feedback Feedback
		if err = result.Scan(&feedback.FeedbackType, &feedback.UserId, &feedback.ItemId, &feedback.Timestamp, &feedback.Comment); err != nil {
			return nil, errors.Trace(err)
		}
		feedbacks = append(feedbacks, feedback)
	}
	GetUserFeedbackLatency.Observe(time.Since(startTime).Seconds())
	return feedbacks, nil
}

// InsertFeedback insert a feedback into MySQL.
// If insertUser set, a new user will be insert to user table.
// If insertItem set, a new item will be insert to item table.
func (d *SQLDatabase) InsertFeedback(feedback Feedback, insertUser, insertItem bool) error {
	startTime := time.Now()
	var err error
	// insert users
	if insertUser {
		switch d.driver {
		case MySQL:
			_, err = d.client.Exec("INSERT IGNORE users(user_id) VALUES (?)", feedback.UserId)
		case Postgres:
			_, err = d.client.Exec("INSERT INTO users(user_id) VALUES ($1) ON CONFLICT (user_id) DO NOTHING", feedback.UserId)
		case ClickHouse:
			_, err = d.client.Exec("INSERT INTO users(user_id, version) VALUES (?,'0000-00-00 00:00:00')", feedback.UserId)
		}
		if err != nil {
			return errors.Trace(err)
		}
	} else {
		if _, err := d.GetUser(feedback.UserId); err != nil {
			if err == ErrUserNotExist {
				base.Logger().Warn("user doesn't exist", zap.String("user_id", feedback.UserId))
				return nil
			}
			return errors.Trace(err)
		}
	}
	// insert items
	if insertItem {
		switch d.driver {
		case MySQL:
			_, err = d.client.Exec("INSERT IGNORE items(item_id) VALUES (?)", feedback.ItemId)
		case Postgres:
			_, err = d.client.Exec("INSERT INTO items(item_id) VALUES ($1) ON CONFLICT (item_id) DO NOTHING", feedback.ItemId)
		case ClickHouse:
			_, err = d.client.Exec("INSERT INTO items(item_id, version) VALUES (?,'0000-00-00 00:00:00')", feedback.ItemId)
		}
		if err != nil {
			return errors.Trace(err)
		}
	} else {
		if _, err = d.GetItem(feedback.ItemId); err != nil {
			if err == ErrItemNotExist {
				base.Logger().Warn("item doesn't exist", zap.String("item_id", feedback.ItemId))
				return nil
			}
			return errors.Trace(err)
		}
	}
	// insert feedback
	switch d.driver {
	case MySQL:
		_, err = d.client.Exec("INSERT feedback(feedback_type, user_id, item_id, time_stamp, `comment`) VALUES (?,?,?,?,?) "+
			"ON DUPLICATE KEY UPDATE time_stamp = ?, `comment` = ?",
			feedback.FeedbackType, feedback.UserId, feedback.ItemId, feedback.Timestamp, feedback.Comment, feedback.Timestamp, feedback.Comment)
	case Postgres:
		_, err = d.client.Exec("INSERT INTO feedback(feedback_type, user_id, item_id, time_stamp, comment) VALUES ($1,$2,$3,$4,$5) "+
			"ON CONFLICT (feedback_type, user_id, item_id) DO UPDATE SET time_stamp = EXCLUDED.time_stamp, comment = EXCLUDED.comment",
			feedback.FeedbackType, feedback.UserId, feedback.ItemId, feedback.Timestamp, feedback.Comment)
	case ClickHouse:
		_, err = d.client.Exec("INSERT INTO feedback(feedback_type, user_id, item_id, time_stamp, `comment`) VALUES (?,?,?,?,?) ",
			feedback.FeedbackType, feedback.UserId, feedback.ItemId, feedback.Timestamp, feedback.Comment)
	}
	InsertFeedbackLatency.Observe(time.Since(startTime).Seconds())
	return errors.Trace(err)
}

// BatchInsertFeedback insert a batch feedback into MySQL.
// If insertUser set, new users will be insert to user table.
// If insertItem set, new items will be insert to item table.
func (d *SQLDatabase) BatchInsertFeedback(feedback []Feedback, insertUser, insertItem bool) error {
	const batchSize = 10000
	// collect users and items
	users := strset.New()
	items := strset.New()
	for _, v := range feedback {
		users.Add(v.UserId)
		items.Add(v.ItemId)
	}
	// insert users
	if insertUser {
		userList := users.List()
		for i := 0; i < len(userList); i += batchSize {
			batchUsers := userList[i:base.Min(i+batchSize, len(userList))]
			builder := strings.Builder{}
			switch d.driver {
			case MySQL:
				builder.WriteString("INSERT IGNORE users(user_id) VALUES ")
			case Postgres:
				builder.WriteString("INSERT INTO users(user_id) VALUES ")
			case ClickHouse:
				builder.WriteString("INSERT INTO users(user_id, version) VALUES ")
			}
			var args []interface{}
			for i, user := range batchUsers {
				switch d.driver {
				case MySQL:
					builder.WriteString("(?)")
				case Postgres:
					builder.WriteString(fmt.Sprintf("($%d)", i+1))
				case ClickHouse:
					builder.WriteString("(?,'0000-00-00 00:00:00')")
				}
				if i+1 < len(batchUsers) {
					builder.WriteString(",")
				}
				args = append(args, user)
			}
			if d.driver == Postgres {
				builder.WriteString(" ON CONFLICT (user_id) DO NOTHING")
			}
			if _, err := d.client.Exec(builder.String(), args...); err != nil {
				return errors.Trace(err)
			}
		}
	} else {
		for _, user := range users.List() {
			var rs *sql.Rows
			var err error
			switch d.driver {
			case MySQL:
				rs, err = d.client.Query("SELECT user_id FROM users WHERE user_id = ?", user)
			case Postgres:
				rs, err = d.client.Query("SELECT user_id FROM users WHERE user_id = $1", user)
			}
			if err != nil {
				return errors.Trace(err)
			} else if !rs.Next() {
				users.Remove(user)
			}
			rs.Close()
		}
	}
	// insert items
	if insertItem {
		itemList := items.List()
		for i := 0; i < len(itemList); i += batchSize {
			batchItems := itemList[i:base.Min(i+batchSize, len(itemList))]
			builder := strings.Builder{}
			switch d.driver {
			case MySQL:
				builder.WriteString("INSERT IGNORE items(item_id) VALUES ")
			case Postgres:
				builder.WriteString("INSERT INTO items(item_id) VALUES ")
			case ClickHouse:
				builder.WriteString("INSERT INTO items(item_id, version) VALUES ")
			}
			var args []interface{}
			for i, item := range batchItems {
				switch d.driver {
				case MySQL:
					builder.WriteString("(?)")
				case Postgres:
					builder.WriteString(fmt.Sprintf("($%d)", i+1))
				case ClickHouse:
					builder.WriteString("(?,'0000-00-00 00:00:00')")
				}
				if i+1 < len(batchItems) {
					builder.WriteString(",")
				}
				args = append(args, item)
			}
			if d.driver == Postgres {
				builder.WriteString(" ON CONFLICT (item_id) DO NOTHING")
			}
			if _, err := d.client.Exec(builder.String(), args...); err != nil {
				return errors.Trace(err)
			}
		}
	} else {
		for _, item := range items.List() {
			var rs *sql.Rows
			var err error
			switch d.driver {
			case MySQL:
				rs, err = d.client.Query("SELECT item_id FROM items WHERE item_id = ?", item)
			case Postgres:
				rs, err = d.client.Query("SELECT item_id FROM items WHERE item_id = $1", item)
			}
			if err != nil {
				return errors.Trace(err)
			} else if !rs.Next() {
				users.Remove(item)
			}
			rs.Close()
		}
	}
	// insert feedback
	for i := 0; i < len(feedback); i += batchSize {
		batchFeedback := feedback[i:base.Min(i+batchSize, len(feedback))]
		builder := strings.Builder{}
		switch d.driver {
		case MySQL, ClickHouse:
			builder.WriteString("INSERT INTO feedback(feedback_type, user_id, item_id, time_stamp, `comment`) VALUES ")
		case Postgres:
			builder.WriteString("INSERT INTO feedback(feedback_type, user_id, item_id, time_stamp, comment) VALUES ")
		}
		var args []interface{}
		for i, f := range batchFeedback {
			if users.Has(f.UserId) && items.Has(f.ItemId) {
				switch d.driver {
				case MySQL, ClickHouse:
					builder.WriteString("(?,?,?,?,?)")
				case Postgres:
					builder.WriteString(fmt.Sprintf("($%d,$%d,$%d,$%d,$%d)",
						len(args)+1, len(args)+2, len(args)+3, len(args)+4, len(args)+5))
				}
				if i+1 < len(batchFeedback) {
					builder.WriteString(",")
				}
				args = append(args, f.FeedbackType, f.UserId, f.ItemId, f.Timestamp, f.Comment)
			}
		}
		switch d.driver {
		case MySQL:
			builder.WriteString(" AS new ON DUPLICATE KEY UPDATE time_stamp = new.time_stamp, `comment` = new.`comment`")
		case Postgres:
			builder.WriteString(" ON CONFLICT (feedback_type, user_id, item_id) DO UPDATE SET time_stamp = EXCLUDED.time_stamp, comment = EXCLUDED.comment")
		}
		_, err := d.client.Exec(builder.String(), args...)
		if err != nil {
			return errors.Trace(err)
		}
	}
	return nil
}

// GetFeedback returns feedback from MySQL.
func (d *SQLDatabase) GetFeedback(cursor string, n int, timeLimit *time.Time, feedbackTypes ...string) (string, []Feedback, error) {
	var cursorKey FeedbackKey
	if cursor != "" {
		if err := json.Unmarshal([]byte(cursor), &cursorKey); err != nil {
			return "", nil, err
		}
	}
	var result *sql.Rows
	var err error
	var builder strings.Builder
	switch d.driver {
	case MySQL, ClickHouse:
		builder.WriteString("SELECT feedback_type, user_id, item_id, time_stamp, `comment` FROM feedback WHERE feedback_type >= ? AND user_id >= ? AND item_id >= ?")
	case Postgres:
		builder.WriteString("SELECT feedback_type, user_id, item_id, time_stamp, comment FROM feedback WHERE feedback_type >= $1 AND user_id >= $2 AND item_id >= $3")
	}
	args := []interface{}{cursorKey.FeedbackType, cursorKey.UserId, cursorKey.ItemId}
	if len(feedbackTypes) > 0 {
		builder.WriteString(" AND feedback_type IN (")
		for i, feedbackType := range feedbackTypes {
			switch d.driver {
			case MySQL, ClickHouse:
				builder.WriteString("?")
			case Postgres:
				builder.WriteString(fmt.Sprintf("$%d", len(args)+1))
			}
			if i+1 < len(feedbackTypes) {
				builder.WriteString(",")
			}
			args = append(args, feedbackType)
		}
		builder.WriteString(")")
	}
	if timeLimit != nil {
		switch d.driver {
		case MySQL, ClickHouse:
			builder.WriteString(" AND time_stamp >= ?")
		case Postgres:
			builder.WriteString(fmt.Sprintf(" AND time_stamp >= $%d", len(args)+1))
		}
		args = append(args, *timeLimit)
	}
	switch d.driver {
	case MySQL, ClickHouse:
		builder.WriteString(" ORDER BY feedback_type, user_id, item_id LIMIT ?")
	case Postgres:
		builder.WriteString(fmt.Sprintf(" ORDER BY feedback_type, user_id, item_id LIMIT $%d", len(args)+1))
	}
	args = append(args, n+1)
	result, err = d.client.Query(builder.String(), args...)
	if err != nil {
		return "", nil, errors.Trace(err)
	}
	feedbacks := make([]Feedback, 0)
	for result.Next() {
		var feedback Feedback
		if err = result.Scan(&feedback.FeedbackType, &feedback.UserId, &feedback.ItemId, &feedback.Timestamp, &feedback.Comment); err != nil {
			return "", nil, errors.Trace(err)
		}
		feedbacks = append(feedbacks, feedback)
	}
	if len(feedbacks) == n+1 {
		nextCursorKey := feedbacks[len(feedbacks)-1].FeedbackKey
		nextCursor, err := json.Marshal(nextCursorKey)
		if err != nil {
			return "", nil, errors.Trace(err)
		}
		return string(nextCursor), feedbacks[:len(feedbacks)-1], nil
	}
	return "", feedbacks, nil
}

// GetUserItemFeedback gets a feedback by user id and item id from MySQL.
func (d *SQLDatabase) GetUserItemFeedback(userId, itemId string, feedbackTypes ...string) ([]Feedback, error) {
	startTime := time.Now()
	var result *sql.Rows
	var err error
	var builder strings.Builder
	switch d.driver {
	case MySQL, ClickHouse:
		builder.WriteString("SELECT feedback_type, user_id, item_id, time_stamp, `comment` FROM feedback WHERE user_id = ? AND item_id = ?")
	case Postgres:
		builder.WriteString("SELECT feedback_type, user_id, item_id, time_stamp, comment FROM feedback WHERE user_id = $1 AND item_id = $2")
	}
	args := []interface{}{userId, itemId}
	if len(feedbackTypes) > 0 {
		builder.WriteString(" AND feedback_type IN (")
		for i, feedbackType := range feedbackTypes {
			switch d.driver {
			case MySQL, ClickHouse:
				builder.WriteString("?")
			case Postgres:
				builder.WriteString(fmt.Sprintf("$%d", i+3))
			}
			if i+1 < len(feedbackTypes) {
				builder.WriteString(",")
			}
			args = append(args, feedbackType)
		}
		builder.WriteString(")")
	}
	result, err = d.client.Query(builder.String(), args...)
	if err != nil {
		return nil, errors.Trace(err)
	}
	feedbacks := make([]Feedback, 0)
	for result.Next() {
		var feedback Feedback
		if err = result.Scan(&feedback.FeedbackType, &feedback.UserId, &feedback.ItemId, &feedback.Timestamp, &feedback.Comment); err != nil {
			return nil, errors.Trace(err)
		}
		feedbacks = append(feedbacks, feedback)
	}
	GetUserItemFeedbackLatency.Observe(time.Since(startTime).Seconds())
	return feedbacks, nil
}

// DeleteUserItemFeedback deletes a feedback by user id and item id from MySQL.
func (d *SQLDatabase) DeleteUserItemFeedback(userId, itemId string, feedbackTypes ...string) (int, error) {
	var rs sql.Result
	var err error
	var builder strings.Builder
	switch d.driver {
	case MySQL:
		builder.WriteString("DELETE FROM feedback WHERE user_id = ? AND item_id = ?")
	case Postgres:
		builder.WriteString("DELETE FROM feedback WHERE user_id = $1 AND item_id = $2")
	case ClickHouse:
		builder.WriteString("ALTER TABLE feedback DELETE WHERE user_id = ? AND item_id = ?")
	}
	args := []interface{}{userId, itemId}
	if len(feedbackTypes) > 0 {
		builder.WriteString(" AND feedback_type IN (")
		for i, feedbackType := range feedbackTypes {
			switch d.driver {
			case MySQL, ClickHouse:
				builder.WriteString("?")
			case Postgres:
				builder.WriteString(fmt.Sprintf("$%d", len(args)+1))
			}
			if i+1 < len(feedbackTypes) {
				builder.WriteString(",")
			}
			args = append(args, feedbackType)
		}
		builder.WriteString(")")
	}
	rs, err = d.client.Exec(builder.String(), args...)
	if err != nil {
		return 0, errors.Trace(err)
	}
	deleteCount, err := rs.RowsAffected()
	if err != nil && d.driver != ClickHouse {
		return 0, errors.Trace(err)
	}
	return int(deleteCount), nil
}

// GetClickThroughRate computes the click-through-rate of a specified date.
func (d *SQLDatabase) GetClickThroughRate(date time.Time, positiveTypes []string, readType string) (float64, error) {
	builder := strings.Builder{}
	var args []interface{}
	switch d.driver {
	case MySQL, ClickHouse:
		builder.WriteString("SELECT positive_count.positive_count / read_count.read_count FROM (")
	case Postgres:
		builder.WriteString("SELECT positive_count.positive_count :: DOUBLE PRECISION / read_count.read_count :: DOUBLE PRECISION FROM (")
	}
	builder.WriteString("SELECT user_id, COUNT(*) AS positive_count FROM (")
	switch d.driver {
	case MySQL, ClickHouse:
		builder.WriteString("SELECT DISTINCT user_id, item_id FROM feedback WHERE DATE(time_stamp) = DATE(?) AND feedback_type IN (")
	case Postgres:
		builder.WriteString(fmt.Sprintf("SELECT DISTINCT user_id, item_id FROM feedback "+
			"WHERE DATE(time_stamp) = DATE($%d) AND feedback_type IN (", len(args)+1))
	}
	args = append(args, date)
	for i, positiveType := range positiveTypes {
		if i > 0 {
			builder.WriteString(",")
		}
		switch d.driver {
		case MySQL, ClickHouse:
			builder.WriteString("?")
		case Postgres:
			builder.WriteString(fmt.Sprintf("$%d", len(args)+1))
		}
		args = append(args, positiveType)
	}
	builder.WriteString(")) AS positive_feedback GROUP BY user_id) AS positive_count ")
	builder.WriteString("JOIN (")
	builder.WriteString("SELECT user_id, COUNT(*) AS read_count FROM (")
	switch d.driver {
	case MySQL, ClickHouse:
		builder.WriteString("SELECT DISTINCT user_id, item_id FROM feedback WHERE DATE(time_stamp) = DATE(?) AND feedback_type IN (?")
	case Postgres:
		builder.WriteString(fmt.Sprintf("SELECT DISTINCT user_id, item_id FROM feedback "+
			"WHERE DATE(time_stamp) = DATE($%d) AND feedback_type IN ($%d", len(args)+1, len(args)+2))
	}
	args = append(args, date, readType)
	for _, positiveType := range positiveTypes {
		switch d.driver {
		case MySQL, ClickHouse:
			builder.WriteString(",?")
		case Postgres:
			builder.WriteString(fmt.Sprintf(",$%d", len(args)+1))
		}
		args = append(args, positiveType)
	}
	builder.WriteString(")) AS read_feedback GROUP BY user_id) AS read_count ")
	builder.WriteString("ON positive_count.user_id = read_count.user_id")
	base.Logger().Info("get click through rate from MySQL", zap.String("query", builder.String()))
	rs, err := d.client.Query(builder.String(), args...)
	if err != nil {
		return 0, errors.Trace(err)
	}
	var sum, count float64
	for rs.Next() {
		var temp float64
		if err = rs.Scan(&temp); err != nil {
			return 0, errors.Trace(err)
		}
		sum += temp
		count++
	}
	if count > 0 {
		sum /= count
	}
	return sum, errors.Trace(err)
}

// CountActiveUsers returns the number active users starting from a specified date.
func (d *SQLDatabase) CountActiveUsers(date time.Time) (int, error) {
	var rs *sql.Rows
	var err error
	switch d.driver {
	case MySQL, ClickHouse:
		rs, err = d.client.Query("SELECT COUNT(DISTINCT user_id) FROM feedback WHERE DATE(time_stamp) >= DATE(?)", date)
	case Postgres:
		rs, err = d.client.Query("SELECT COUNT(DISTINCT user_id) FROM feedback WHERE DATE(time_stamp) >= DATE($1)", date)
	}
	if err != nil {
		return 0, err
	}
	var count int
	if rs.Next() {
		if err = rs.Scan(&count); err != nil {
			return 0, err
		}
	}
	return count, nil
}
