(ns frontend.components.theme
  (:require [frontend.extensions.pdf.highlights :as pdf]
            [frontend.config :as config]
            [frontend.handler.plugin :refer [lsp-enabled?] :as plugin-handler]
            [frontend.handler.route :as route-handler]
            [frontend.handler.ui :as ui-handler]
            [frontend.ui :as ui]
            [frontend.util :as util]
            [frontend.state :as state]
            [frontend.components.settings :as settings]
            [frontend.rum :refer [use-mounted]]
            [rum.core :as rum]))

(rum/defc container
  [{:keys [route theme on-click current-repo nfs-granted? db-restoring?
           settings-open? sidebar-open? system-theme? sidebar-blocks-len]} child]
  (let [mounted-fn (use-mounted)
        [restored-sidebar? set-restored-sidebar?] (rum/use-state false)]

    (rum/use-effect!
     #(let [doc js/document.documentElement
            cls (.-classList doc)]
        (.setAttribute doc "data-theme" theme)
        (if (= theme "dark") ;; for tailwind dark mode
          (.add cls "dark")
          (.remove cls "dark"))
        (ui/apply-custom-theme-effect! theme)
        (plugin-handler/hook-plugin-app :theme-mode-changed {:mode theme} nil))
     [theme])

    (rum/use-effect!
     #(when (and restored-sidebar?
                 (mounted-fn))
        (plugin-handler/hook-plugin-app :sidebar-visible-changed {:visible sidebar-open?})
        (ui-handler/persist-right-sidebar-state!))
     [sidebar-open? restored-sidebar? sidebar-blocks-len])

    (rum/use-effect!
     #(when lsp-enabled?
        (plugin-handler/setup-install-listener!)
        (plugin-handler/load-plugin-preferences))
     [])

    (rum/use-effect!
     (fn []
       (ui-handler/add-style-if-exists!)
       (pdf/reset-current-pdf!)
       (plugin-handler/hook-plugin-app :current-graph-changed {}))
     [current-repo])

    (rum/use-effect!
     #(let [db-restored? (false? db-restoring?)]
        (if db-restoring?
          (util/set-title! "Loading")
          (when (or nfs-granted? db-restored?)
            (route-handler/update-page-title! route))))
     [nfs-granted? db-restoring? route])

    (rum/use-effect!
     (fn []
       (when-not db-restoring?
         (let [repos (state/get-repos)]
           (if-not (or
                    ;; demo graph only
                    (and (= 1 (count repos)) (:example? (first repos))
                         (not (util/mobile?)))
                    ;; not in publising mode
                    config/publishing?
                    ;; other graphs exists
                    (seq repos))
            (route-handler/redirect! {:to :repo-add})
            (do
              (ui-handler/restore-right-sidebar-state!)
              (set-restored-sidebar? true))))))
     [db-restoring?])

    (rum/use-effect!
     #(when system-theme?
        (ui/setup-system-theme-effect!))
     [system-theme?])

    (rum/use-effect!
     #(state/set-modal!
       (when settings-open?
         (fn [] [:div.settings-modal (settings/settings)])))
     [settings-open?])

    [:div
     {:class    (util/classnames
                 [(str theme "-theme")
                  {:white-theme (= "light" theme)}]) ; The white-theme is for backward compatibility. See: https://github.com/logseq/logseq/pull/4652.
      :on-click on-click}
     child

     (pdf/playground)]))
