/* Copyright 2025 Dnote Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package config

import (
	"net/url"
	"os"
	"path/filepath"

	"github.com/dnote/dnote/pkg/dirs"
	"github.com/dnote/dnote/pkg/server/assets"
	"github.com/pkg/errors"
)

const (
	// DefaultDBDir is the default directory name for Dnote data
	DefaultDBDir = "dnote"
	// DefaultDBFilename is the default database filename
	DefaultDBFilename = "server.db"
)

var (
	// DefaultDBPath is the default path to the database file
	DefaultDBPath = filepath.Join(dirs.DataHome, DefaultDBDir, DefaultDBFilename)
)

var (
	// ErrDBMissingPath is an error for an incomplete configuration missing the database path
	ErrDBMissingPath = errors.New("DB Path is empty")
	// ErrBaseURLInvalid is an error for an incomplete configuration with invalid base url
	ErrBaseURLInvalid = errors.New("Invalid BaseURL")
	// ErrPortInvalid is an error for an incomplete configuration with invalid port
	ErrPortInvalid = errors.New("Invalid Port")
)

func readBoolEnv(name string) bool {
	return os.Getenv(name) == "true"
}

// getOrEnv returns value if non-empty, otherwise env var, otherwise default
func getOrEnv(value, envKey, defaultVal string) string {
	if value != "" {
		return value
	}
	if env := os.Getenv(envKey); env != "" {
		return env
	}
	return defaultVal
}

// Config is an application configuration
type Config struct {
	BaseURL             string
	DisableRegistration bool
	Port                string
	DBPath              string
	AssetBaseURL        string
	HTTP500Page         []byte
	LogLevel            string
}

// Params are the configuration parameters for creating a new Config
type Params struct {
	Port                string
	BaseURL             string
	DBPath              string
	DisableRegistration bool
	LogLevel            string
}

// New constructs and returns a new validated config.
// Empty string params will fall back to environment variables and defaults.
func New(p Params) (Config, error) {
	c := Config{
		Port:                getOrEnv(p.Port, "PORT", "3001"),
		BaseURL:             getOrEnv(p.BaseURL, "BaseURL", "http://localhost:3001"),
		DBPath:              getOrEnv(p.DBPath, "DBPath", DefaultDBPath),
		DisableRegistration: p.DisableRegistration || readBoolEnv("DisableRegistration"),
		LogLevel:            getOrEnv(p.LogLevel, "LOG_LEVEL", "info"),
		AssetBaseURL:        "/static",
		HTTP500Page:         assets.MustGetHTTP500ErrorPage(),
	}

	if err := validate(c); err != nil {
		return Config{}, err
	}

	return c, nil
}

func validate(c Config) error {
	if _, err := url.ParseRequestURI(c.BaseURL); err != nil {
		return errors.Wrapf(ErrBaseURLInvalid, "'%s'", c.BaseURL)
	}
	if c.Port == "" {
		return ErrPortInvalid
	}

	if c.DBPath == "" {
		return ErrDBMissingPath
	}

	return nil
}
