use crate::{
    data::models::{
        ClusterAnalyticsFilter, ClusterTopicsClickhouse, DatasetAnalytics, EventDataClickhouse,
        Granularity, HeadQueries, Pool, PopularFilters, PopularFiltersClickhouse,
        RAGAnalyticsFilter, RAGSortBy, RAGUsageGraphResponse, RAGUsageResponse, RagQueryEvent,
        RagQueryEventClickhouse, RecommendationAnalyticsFilter, RecommendationCTRMetrics,
        RecommendationEvent, RecommendationEventClickhouse, RecommendationsWithClicksCTRResponse,
        RecommendationsWithClicksCTRResponseClickhouse, RecommendationsWithoutClicksCTRResponse,
        RecommendationsWithoutClicksCTRResponseClickhouse, SearchAnalyticsFilter, SearchCTRMetrics,
        SearchCTRMetricsClickhouse, SearchClusterTopics, SearchLatencyGraph,
        SearchLatencyGraphClickhouse, SearchQueriesWithClicksCTRResponse,
        SearchQueriesWithClicksCTRResponseClickhouse, SearchQueriesWithoutClicksCTRResponse,
        SearchQueriesWithoutClicksCTRResponseClickhouse, SearchQueryEvent,
        SearchQueryEventClickhouse, SearchSortBy, SearchTypeCount, SortOrder, TopDatasetsResponse,
        TopDatasetsResponseClickhouse, UsageGraphPoint, UsageGraphPointClickhouse,
    },
    errors::ServiceError,
    handlers::analytics_handler::{GetTopDatasetsRequestBody, RateQueryRequest},
};
use actix_web::web;
use diesel::prelude::*;
use diesel_async::RunQueryDsl;
use futures::future::join_all;
use itertools::Itertools;
use serde::{Deserialize, Serialize};
use ureq::json;
use utoipa::ToSchema;

#[derive(Debug, Serialize, Deserialize, ToSchema)]
pub struct SearchClusterResponse {
    pub clusters: Vec<SearchClusterTopics>,
}

pub async fn get_clusters_query(
    dataset_id: uuid::Uuid,
    filters: Option<ClusterAnalyticsFilter>,
    clickhouse_client: &clickhouse::Client,
) -> Result<SearchClusterResponse, ServiceError> {
    let mut query_string = String::from("SELECT ?fields FROM cluster_topics WHERE dataset_id = ?");

    if let Some(filters) = filters {
        query_string = filters.add_to_query(query_string);
    }

    query_string.push_str(" ORDER BY density DESC LIMIT 10");

    let clickhouse_topics = clickhouse_client
        .query(query_string.as_str())
        .bind(dataset_id)
        .fetch_all::<ClusterTopicsClickhouse>()
        .await
        .map_err(|e| {
            log::error!("Error fetching topics: {:?}", e);
            ServiceError::InternalServerError("Error fetching topics".to_string())
        })?;

    let topics: Vec<SearchClusterTopics> = clickhouse_topics
        .into_iter()
        .map(|t| t.into())
        .collect::<Vec<_>>();

    Ok(SearchClusterResponse { clusters: topics })
}

#[derive(Debug, Serialize, Deserialize, ToSchema)]
pub struct SearchQueryResponse {
    pub queries: Vec<SearchQueryEvent>,
}

pub async fn get_queries_for_cluster_query(
    dataset_id: uuid::Uuid,
    cluster_id: uuid::Uuid,
    page: Option<u32>,
    clickhouse_client: &clickhouse::Client,
) -> Result<SearchQueryResponse, ServiceError> {
    let query_string = String::from(
        "
        SELECT DISTINCT ON (search_queries.request_params, search_queries.query) ?fields 
        FROM search_queries 
        JOIN search_cluster_memberships ON search_queries.id = search_cluster_memberships.search_id 
        WHERE search_cluster_memberships.cluster_id = ? 
            AND search_queries.dataset_id = ? AND search_queries.is_duplicate = 0
        ORDER BY
            search_cluster_memberships.distance_to_centroid DESC
        LIMIT 15 
        OFFSET ?
    ",
    );

    let clickhouse_queries = clickhouse_client
        .query(query_string.as_str())
        .bind(cluster_id)
        .bind(dataset_id)
        .bind((page.unwrap_or(1) - 1) * 15)
        .fetch_all::<SearchQueryEventClickhouse>()
        .await
        .map_err(|e| {
            log::error!("Error fetching queries: {:?}", e);
            ServiceError::InternalServerError("Error fetching queries".to_string())
        })?;

    let queries: Vec<SearchQueryEvent> = clickhouse_queries
        .into_iter()
        .map(|q| q.into())
        .collect_vec();

    Ok(SearchQueryResponse { queries })
}

pub async fn get_search_query(
    dataset_id: uuid::Uuid,
    search_id: uuid::Uuid,
    clickhouse_client: &clickhouse::Client,
) -> Result<SearchQueryEvent, ServiceError> {
    let clickhouse_query = clickhouse_client
        .query("SELECT ?fields FROM search_queries WHERE id = ? AND dataset_id = ? AND search_queries.is_duplicate = 0")
        .bind(search_id)
        .bind(dataset_id)
        .fetch_one::<SearchQueryEventClickhouse>()
        .await
        .map_err(|e| {
            log::error!("Error fetching query: {:?}", e);
            ServiceError::InternalServerError("Error fetching query".to_string())
        })?;

    let query: SearchQueryEvent = clickhouse_query.into();

    Ok(query)
}

pub async fn get_search_metrics_query(
    dataset_id: uuid::Uuid,
    filter: Option<SearchAnalyticsFilter>,
    clickhouse_client: &clickhouse::Client,
) -> Result<DatasetAnalytics, ServiceError> {
    let mut query_string = String::from(
        "SELECT 
            total_queries,
            total_queries / dateDiff('second', min_created_at, max_created_at) AS search_rps,
            avg_latency,
            p99,
            p95,
            p50
        FROM (
            SELECT 
                count(*) as total_queries,
                min(created_at) as min_created_at,
                max(created_at) as max_created_at,
                avg(latency) as avg_latency,
                quantile(0.99)(latency) as p99,
                quantile(0.95)(latency) as p95,
                quantile(0.5)(latency) as p50
            FROM default.search_queries
            WHERE dataset_id = ?
        ) subquery",
    );

    if let Some(filter) = filter {
        query_string = filter.add_to_query(query_string);
    }

    let clickhouse_query = clickhouse_client
        .query(query_string.as_str())
        .bind(dataset_id)
        .fetch_one::<DatasetAnalytics>()
        .await
        .map_err(|e| {
            log::error!("Error fetching query: {:?}", e);
            ServiceError::InternalServerError("Error fetching query".to_string())
        })?;

    Ok(clickhouse_query)
}

#[derive(Debug, Serialize, Deserialize, ToSchema)]
pub struct HeadQueryResponse {
    pub queries: Vec<HeadQueries>,
}

pub async fn get_head_queries_query(
    dataset_id: uuid::Uuid,
    filter: Option<SearchAnalyticsFilter>,
    page: Option<u32>,
    clickhouse_client: &clickhouse::Client,
) -> Result<HeadQueryResponse, ServiceError> {
    let mut query_string = String::from(
        "SELECT 
            query, 
            count(*) AS count
        FROM 
            default.search_queries
        WHERE dataset_id = ? AND search_queries.is_duplicate = 0",
    );

    if let Some(filter) = filter {
        query_string = filter.add_to_query(query_string);
    }

    query_string.push_str(
        " GROUP BY 
            query
        ORDER BY 
            count DESC
        LIMIT 10
        OFFSET ?",
    );

    let head_queries = clickhouse_client
        .query(query_string.as_str())
        .bind(dataset_id)
        .bind((page.unwrap_or(1) - 1) * 10)
        .fetch_all::<HeadQueries>()
        .await
        .map_err(|e| {
            log::error!("Error fetching query: {:?}", e);
            ServiceError::InternalServerError("Error fetching query".to_string())
        })?;

    Ok(HeadQueryResponse {
        queries: head_queries,
    })
}

pub async fn get_low_confidence_queries_query(
    dataset_id: uuid::Uuid,
    filter: Option<SearchAnalyticsFilter>,
    threshold: Option<f32>,
    page: Option<u32>,
    clickhouse_client: &clickhouse::Client,
) -> Result<SearchQueryResponse, ServiceError> {
    let mut query_string = String::from(
        "SELECT 
            ?fields
        FROM 
            default.search_queries
        WHERE dataset_id = ? AND search_queries.is_duplicate = 0",
    );

    if let Some(filter) = filter {
        query_string = filter.add_to_query(query_string);
    }

    if let Some(threshold) = threshold {
        query_string.push_str(
            format!(
                " 
                AND top_score < {}
                ",
                threshold
            )
            .as_str(),
        );
    }

    query_string.push_str(
        "
        ORDER BY 
            top_score ASC
        LIMIT 10
        OFFSET ?",
    );

    let clickhouse_query = clickhouse_client
        .query(query_string.as_str())
        .bind(dataset_id)
        .bind((page.unwrap_or(1) - 1) * 10)
        .fetch_all::<SearchQueryEventClickhouse>()
        .await
        .map_err(|e| {
            log::error!("Error fetching query: {:?}", e);
            ServiceError::InternalServerError("Error fetching query".to_string())
        })?;

    let queries: Vec<SearchQueryEvent> =
        clickhouse_query.into_iter().map(|q| q.into()).collect_vec();

    Ok(SearchQueryResponse { queries })
}

pub async fn get_no_result_queries_query(
    dataset_id: uuid::Uuid,
    filter: Option<SearchAnalyticsFilter>,
    page: Option<u32>,
    clickhouse_client: &clickhouse::Client,
) -> Result<SearchQueryResponse, ServiceError> {
    let mut query_string = String::from(
        "SELECT 
            ?fields
        FROM 
            default.search_queries
        WHERE dataset_id = ?
        AND top_score = 0 AND search_queries.is_duplicate = 0",
    );

    if let Some(filter) = filter {
        query_string = filter.add_to_query(query_string);
    }

    query_string.push_str(
        "
        ORDER BY 
            created_at DESC
        LIMIT 10
        OFFSET ?",
    );

    let clickhouse_query = clickhouse_client
        .query(query_string.as_str())
        .bind(dataset_id)
        .bind((page.unwrap_or(1) - 1) * 10)
        .fetch_all::<SearchQueryEventClickhouse>()
        .await
        .map_err(|e| {
            log::error!("Error fetching query: {:?}", e);
            ServiceError::InternalServerError("Error fetching query".to_string())
        })?;

    let queries: Vec<SearchQueryEvent> =
        clickhouse_query.into_iter().map(|q| q.into()).collect_vec();

    Ok(SearchQueryResponse { queries })
}

pub async fn get_all_queries_query(
    dataset_id: uuid::Uuid,
    filter: Option<SearchAnalyticsFilter>,
    sort_by: Option<SearchSortBy>,
    sort_order: Option<SortOrder>,
    page: Option<u32>,
    clickhouse_client: &clickhouse::Client,
) -> Result<SearchQueryResponse, ServiceError> {
    let mut query_string = String::from(
        "SELECT 
            ?fields
        FROM 
            default.search_queries
        WHERE dataset_id = ? AND search_queries.is_duplicate = 0",
    );

    if let Some(filter) = filter {
        query_string = filter.add_to_query(query_string);
    }

    query_string.push_str(&format!(
        "
        ORDER BY 
        {} {}
        LIMIT 10
        OFFSET ?",
        sort_by.clone().unwrap_or(SearchSortBy::CreatedAt),
        sort_order.clone().unwrap_or(SortOrder::Desc)
    ));

    let clickhouse_query = clickhouse_client
        .query(query_string.as_str())
        .bind(dataset_id)
        .bind((page.unwrap_or(1) - 1) * 10)
        .fetch_all::<SearchQueryEventClickhouse>()
        .await
        .map_err(|e| {
            log::error!("Error fetching query: {:?}", e);
            ServiceError::InternalServerError("Error fetching query".to_string())
        })?;

    let queries: Vec<SearchQueryEvent> =
        clickhouse_query.into_iter().map(|q| q.into()).collect_vec();

    Ok(SearchQueryResponse { queries })
}

#[derive(Debug, Serialize, Deserialize, ToSchema)]
pub struct QueryCountResponse {
    pub total_queries: Vec<SearchTypeCount>,
}

pub async fn get_query_counts_query(
    dataset_id: uuid::Uuid,
    filter: Option<SearchAnalyticsFilter>,
    clickhouse_client: &clickhouse::Client,
) -> Result<QueryCountResponse, ServiceError> {
    let mut query_string = String::from(
        "SELECT 
            search_type,
            JSONExtractString(request_params, 'search_type') as search_method,
            COUNT(*) as search_count
        FROM 
            search_queries
        WHERE dataset_id = ?",
    );

    if let Some(filter) = filter {
        query_string = filter.add_to_query(query_string);
    }

    query_string.push_str(
        "
        GROUP BY 
            search_type, search_method
        ORDER BY 
            search_count DESC",
    );

    let result_counts = clickhouse_client
        .query(query_string.as_str())
        .bind(dataset_id)
        .fetch_all::<SearchTypeCount>()
        .await
        .map_err(|e| {
            log::error!("Error fetching query: {:?}", e);
            ServiceError::InternalServerError("Error fetching query".to_string())
        })?;

    Ok(QueryCountResponse {
        total_queries: result_counts,
    })
}

#[derive(Debug, Serialize, Deserialize, ToSchema)]
pub struct PopularFiltersResponse {
    pub popular_filters: Vec<PopularFilters>,
}

pub async fn get_popular_filter_values_query(
    dataset_id: uuid::Uuid,
    filter: Option<SearchAnalyticsFilter>,
    clickhouse_client: &clickhouse::Client,
) -> Result<PopularFiltersResponse, ServiceError> {
    let mut filter_string = String::new();
    if let Some(filter) = filter {
        filter_string = filter.add_to_query(String::from(""));
    }

    let query_string = format!(
                "WITH filter_data AS (
            SELECT
                'must' AS clause,
                JSONExtractArrayRaw(JSONExtractString(request_params, 'filters'), 'must') AS conditions
            FROM search_queries
            WHERE JSONExtractString(request_params, 'filters', 'must') != '[]' AND dataset_id = '{dataset_id}'::UUID {filter_string}

            UNION ALL

            SELECT
                'should' AS clause,
                JSONExtractArrayRaw(JSONExtractString(request_params, 'filters'), 'should') AS conditions
            FROM search_queries
            WHERE JSONExtractString(request_params, 'filters', 'should') != '[]' AND dataset_id = '{dataset_id}'::UUID {filter_string}

            UNION ALL

            SELECT
                'must_not' AS clause,
                JSONExtractArrayRaw(JSONExtractString(request_params, 'filters'), 'must_not') AS conditions
            FROM search_queries
            WHERE JSONExtractString(request_params, 'filters', 'must_not') != '[]' AND dataset_id = '{dataset_id}'::UUID {filter_string}
        ),
        parsed_conditions AS (
            SELECT
                clause,
                JSONExtractString(condition, 'field') AS field,
                multiIf(
                    JSONExtractString(condition, 'match_any') != '', 'match_any',
                    JSONExtractString(condition, 'match_all') != '', 'match_all',
                    JSONExtractString(condition, 'range') != '', 'range',
                    JSONExtractString(condition, 'date_range') != '', 'date_range',
                    JSONExtractString(condition, 'geo_bounding_box') != '', 'geo_bounding_box',
                    JSONExtractString(condition, 'geo_radius') != '', 'geo_radius',
                    JSONExtractString(condition, 'geo_polygon') != '', 'geo_polygon',
                    'unknown'
                ) AS filter_type,
                JSONExtractString(condition, 'match_any') AS match_any_value,
                JSONExtractString(condition, 'match_all') AS match_all_value,
                JSONExtractKeysAndValues(condition, 'range', 'Float32') AS range_value,
                JSONExtractKeysAndValues(condition, 'date_range', 'String') AS date_range_value
            FROM filter_data
            ARRAY JOIN conditions AS condition
        ),
        aggregated_conditions AS (
            SELECT
                clause,
                field,
                filter_type,
                match_any_value,
                match_all_value,
                range_value,
                date_range_value,
                count() OVER (PARTITION BY clause, field, filter_type) AS total_count,
                count() OVER (PARTITION BY clause, field, filter_type, match_any_value) AS match_any_count,
                count() OVER (PARTITION BY clause, field, filter_type, match_all_value) AS match_all_count,
                count() OVER (PARTITION BY clause, field, filter_type, range_value) AS range_count,
                count() OVER (PARTITION BY clause, field, filter_type, date_range_value) AS date_range_count
            FROM parsed_conditions
        ),
        final_aggregation AS (
            SELECT
                clause,
                field,
                filter_type,
                any(total_count) AS count,
                arraySort(groupArray((match_any_value, match_any_count))) AS match_any_agg,
                arraySort(groupArray((match_all_value, match_all_count))) AS match_all_agg,
                arraySort(groupArray((range_value, range_count))) AS range_agg,
                arraySort(groupArray((date_range_value, date_range_count))) AS date_range_agg
            FROM aggregated_conditions
            GROUP BY clause, field, filter_type
        )
        SELECT
            clause,
            field,
            filter_type,
            count,
            CASE
                WHEN filter_type = 'match_any' THEN
                    arrayStringConcat(
                        arrayMap(x -> concat(x.1, ': ', toString(x.2)),
                            match_any_agg),
                        ', '
                    )
                WHEN filter_type = 'match_all' THEN
                    arrayStringConcat(
                        arrayMap(x -> concat(x.1, ': ', toString(x.2)),
                            match_all_agg),
                        ', '
                    )
                WHEN filter_type = 'range' THEN
                    arrayStringConcat(
                        arrayMap(x -> concat(x.1, ': ', toString(x.2)),
                            range_agg),
                        ', '
                    )
                WHEN filter_type = 'date_range' THEN
                    arrayStringConcat(
                        arrayMap(x -> concat(x.1, ': ', toString(x.2)),
                            date_range_agg),
                        ', '
                    )
                ELSE 'N/A'
            END AS common_values
        FROM final_aggregation
        ORDER BY count DESC
        LIMIT 10", dataset_id = dataset_id, filter_string = filter_string);

    let popular_filters = clickhouse_client
        .query(query_string.as_str())
        .fetch_all::<PopularFiltersClickhouse>()
        .await
        .map_err(|e| {
            log::error!("Error fetching query: {:?}", e);
            ServiceError::InternalServerError("Error fetching query".to_string())
        })?;

    let popular_filters: Vec<PopularFilters> =
        popular_filters.into_iter().map(|f| f.into()).collect_vec();

    Ok(PopularFiltersResponse { popular_filters })
}

#[derive(Debug, Serialize, Deserialize, ToSchema)]
pub struct SearchUsageGraphResponse {
    pub usage_points: Vec<UsageGraphPoint>,
}

pub async fn get_search_usage_graph_query(
    dataset_id: uuid::Uuid,
    filter: Option<SearchAnalyticsFilter>,
    granularity: Option<Granularity>,
    clickhouse_client: &clickhouse::Client,
) -> Result<SearchUsageGraphResponse, ServiceError> {
    let granularity = granularity.unwrap_or(Granularity::Hour);
    let interval = match granularity {
        Granularity::Second => "1 SECOND",
        Granularity::Minute => "1 MINUTE",
        Granularity::Hour => "1 HOUR",
        Granularity::Day => "1 DAY",
        // Add other granularities as needed
    };

    let mut query_string = format!(
        "SELECT 
            toStartOfInterval(created_at, INTERVAL {}) AS time_stamp,
            count(*) AS requests
        FROM 
            default.search_queries
        WHERE 
            dataset_id = ?
        ",
        interval
    );

    if let Some(filter) = filter {
        query_string = filter.add_to_query(query_string);
    }

    query_string.push_str(
        "
        GROUP BY 
            time_stamp
        ORDER BY 
            time_stamp
        LIMIT
            1000",
    );

    let clickhouse_query = clickhouse_client
        .query(query_string.as_str())
        .bind(dataset_id)
        .fetch_all::<UsageGraphPointClickhouse>()
        .await
        .map_err(|e| {
            log::error!("Error fetching query: {:?}", e);
            ServiceError::InternalServerError("Error fetching query".to_string())
        })?;

    let rps_graph: Vec<UsageGraphPoint> = clickhouse_query
        .into_iter()
        .map(|q| q.into())
        .collect::<Vec<_>>();

    Ok(SearchUsageGraphResponse {
        usage_points: rps_graph,
    })
}

#[derive(Debug, Serialize, Deserialize, ToSchema)]
pub struct LatencyGraphResponse {
    pub latency_points: Vec<SearchLatencyGraph>,
}

pub async fn get_latency_graph_query(
    dataset_id: uuid::Uuid,
    filter: Option<SearchAnalyticsFilter>,
    granularity: Option<Granularity>,
    clickhouse_client: &clickhouse::Client,
) -> Result<LatencyGraphResponse, ServiceError> {
    let mut query_string = String::from(
        "WITH per_second_latency AS (
            SELECT 
                toDateTime(toUnixTimestamp(created_at) - (toUnixTimestamp(created_at) % 1)) AS second,
                avg(latency) AS latency_per_second
            FROM 
                default.search_queries
            WHERE 
                dataset_id = ?
        ",
    );

    if let Some(filter) = filter {
        query_string = filter.add_to_query(query_string);
    }

    query_string.push_str(&format!(
        "
            GROUP BY 
                second
        ),
        per_interval_latency AS (
            SELECT 
                toStartOfInterval(second, INTERVAL '1 {}') AS time_stamp,
                avg(latency_per_second) AS average_latency
            FROM 
                per_second_latency
            GROUP BY 
                time_stamp
        )
        SELECT 
            time_stamp,
            average_latency
        FROM 
            per_interval_latency
        ORDER BY 
            time_stamp
        LIMIT
            1000",
        granularity.clone().unwrap_or(Granularity::Hour)
    ));

    let clickhouse_query = clickhouse_client
        .query(query_string.as_str())
        .bind(dataset_id)
        .fetch_all::<SearchLatencyGraphClickhouse>()
        .await
        .map_err(|e| {
            log::error!("Error fetching query: {:?}", e);
            ServiceError::InternalServerError("Error fetching query".to_string())
        })?;

    let latency_query: Vec<SearchLatencyGraph> = clickhouse_query
        .into_iter()
        .map(|q| q.into())
        .collect::<Vec<_>>();

    Ok(LatencyGraphResponse {
        latency_points: latency_query,
    })
}

#[derive(Debug, Serialize, Deserialize, ToSchema)]
pub struct RagQueryResponse {
    pub queries: Vec<RagQueryEvent>,
}

pub async fn get_rag_queries_query(
    dataset_id: uuid::Uuid,
    filter: Option<RAGAnalyticsFilter>,
    sort_by: Option<RAGSortBy>,
    sort_order: Option<SortOrder>,
    page: Option<u32>,
    pool: web::Data<Pool>,
    clickhouse_client: &clickhouse::Client,
) -> Result<RagQueryResponse, ServiceError> {
    let mut query_string = String::from(
        "SELECT 
            ?fields
        FROM 
            default.rag_queries
        WHERE dataset_id = ?",
    );

    if let Some(filter) = filter {
        query_string = filter.add_to_query(query_string);
    }

    query_string.push_str(&format!(
        "
        ORDER BY 
        {} {}
        LIMIT 10
        OFFSET ?",
        sort_by.clone().unwrap_or(RAGSortBy::CreatedAt),
        sort_order.clone().unwrap_or(SortOrder::Desc)
    ));

    let clickhouse_query = clickhouse_client
        .query(query_string.as_str())
        .bind(dataset_id)
        .bind((page.unwrap_or(1) - 1) * 10)
        .fetch_all::<RagQueryEventClickhouse>()
        .await
        .map_err(|e| {
            log::error!("Error fetching query: {:?}", e);
            ServiceError::InternalServerError("Error fetching query".to_string())
        })?;

    let queries: Vec<RagQueryEvent> = join_all(
        clickhouse_query
            .into_iter()
            .map(|q| q.from_clickhouse(pool.clone())),
    )
    .await;

    Ok(RagQueryResponse { queries })
}

pub async fn get_rag_usage_query(
    dataset_id: uuid::Uuid,
    filter: Option<RAGAnalyticsFilter>,
    clickhouse_client: &clickhouse::Client,
) -> Result<RAGUsageResponse, ServiceError> {
    let mut query_string = String::from(
        "SELECT 
            count(*) as total_queries
        FROM 
            default.rag_queries
        WHERE dataset_id = ?",
    );

    if let Some(filter) = filter {
        query_string = filter.add_to_query(query_string);
    }

    let clickhouse_query = clickhouse_client
        .query(query_string.as_str())
        .bind(dataset_id)
        .fetch_one::<RAGUsageResponse>()
        .await
        .map_err(|e| {
            log::error!("Error fetching query: {:?}", e);
            ServiceError::InternalServerError("Error fetching query".to_string())
        })?;

    Ok(clickhouse_query)
}

pub async fn get_rag_usage_graph_query(
    dataset_id: uuid::Uuid,
    filter: Option<RAGAnalyticsFilter>,
    granularity: Option<Granularity>,
    clickhouse_client: &clickhouse::Client,
) -> Result<RAGUsageGraphResponse, ServiceError> {
    let granularity = granularity.unwrap_or(Granularity::Hour);
    let interval = match granularity {
        Granularity::Second => "1 SECOND",
        Granularity::Minute => "1 MINUTE",
        Granularity::Hour => "1 HOUR",
        Granularity::Day => "1 DAY",
    };

    let mut query_string = format!(
        "SELECT 
            toStartOfInterval(created_at, INTERVAL {}) AS time_stamp,
            count(*) AS requests
        FROM 
            default.rag_queries
        WHERE 
            dataset_id = ?
        ",
        interval
    );

    if let Some(filter) = filter {
        query_string = filter.add_to_query(query_string);
    }

    query_string.push_str(
        "
        GROUP BY 
            time_stamp
        ORDER BY 
            time_stamp
        LIMIT
            1000",
    );

    let clickhouse_query = clickhouse_client
        .query(query_string.as_str())
        .bind(dataset_id)
        .fetch_all::<UsageGraphPointClickhouse>()
        .await
        .map_err(|e| {
            log::error!("Error fetching query: {:?}", e);
            ServiceError::InternalServerError("Error fetching query".to_string())
        })?;

    let rps_graph: Vec<UsageGraphPoint> = clickhouse_query
        .into_iter()
        .map(|q| q.into())
        .collect::<Vec<_>>();

    Ok(RAGUsageGraphResponse {
        usage_points: rps_graph,
    })
}

#[derive(Debug, Serialize, Deserialize, ToSchema)]
pub struct RecommendationsEventResponse {
    pub queries: Vec<RecommendationEvent>,
}

pub async fn get_low_confidence_recommendations_query(
    dataset_id: uuid::Uuid,
    filter: Option<RecommendationAnalyticsFilter>,
    threshold: Option<f32>,
    page: Option<u32>,
    clickhouse_client: &clickhouse::Client,
) -> Result<RecommendationsEventResponse, ServiceError> {
    let mut query_string = String::from(
        "SELECT 
            ?fields
        FROM 
            default.recommendations
        WHERE dataset_id = ?",
    );

    if let Some(filter) = filter {
        query_string = filter.add_to_query(query_string);
    }

    if let Some(threshold) = threshold {
        query_string.push_str(
            format!(
                " 
                AND top_score < {}
                ",
                threshold
            )
            .as_str(),
        );
    }

    query_string.push_str(
        "
        ORDER BY 
            top_score ASC
        LIMIT 10
        OFFSET ?",
    );

    let clickhouse_query = clickhouse_client
        .query(query_string.as_str())
        .bind(dataset_id)
        .bind((page.unwrap_or(1) - 1) * 10)
        .fetch_all::<RecommendationEventClickhouse>()
        .await
        .map_err(|e| {
            log::error!("Error fetching query: {:?}", e);
            ServiceError::InternalServerError("Error fetching query".to_string())
        })?;

    let queries: Vec<RecommendationEvent> =
        clickhouse_query.into_iter().map(|q| q.into()).collect_vec();

    Ok(RecommendationsEventResponse { queries })
}

pub async fn get_recommendation_queries_query(
    dataset_id: uuid::Uuid,
    filter: Option<RecommendationAnalyticsFilter>,
    sort_by: Option<SearchSortBy>,
    sort_order: Option<SortOrder>,
    page: Option<u32>,
    clickhouse_client: &clickhouse::Client,
) -> Result<RecommendationsEventResponse, ServiceError> {
    let mut query_string = String::from(
        "SELECT 
            ?fields
        FROM 
            default.recommendations
        WHERE dataset_id = ?",
    );

    if let Some(filter) = filter {
        query_string = filter.add_to_query(query_string);
    }

    query_string.push_str(&format!(
        "
        ORDER BY 
        {} {}
        LIMIT 10
        OFFSET ?",
        sort_by.clone().unwrap_or(SearchSortBy::CreatedAt),
        sort_order.clone().unwrap_or(SortOrder::Desc)
    ));

    let clickhouse_query = clickhouse_client
        .query(query_string.as_str())
        .bind(dataset_id)
        .bind((page.unwrap_or(1) - 1) * 10)
        .fetch_all::<RecommendationEventClickhouse>()
        .await
        .map_err(|e| {
            log::error!("Error fetching query: {:?}", e);
            ServiceError::InternalServerError("Error fetching query".to_string())
        })?;

    let queries: Vec<RecommendationEvent> =
        clickhouse_query.into_iter().map(|q| q.into()).collect_vec();

    Ok(RecommendationsEventResponse { queries })
}

pub async fn send_event_data_query(
    data: EventDataClickhouse,
    clickhouse_client: &clickhouse::Client,
) -> Result<(), ServiceError> {
    let query_string = format!(
        "INSERT INTO default.events (id, event_type, event_name, items, metadata, user_id, is_conversion, request_id, dataset_id, created_at, updated_at) VALUES ('{}', '{}', '{}', ['{:?}'], '{}', '{}', '{}', '{}', '{}', now(), now())",
        data.id, data.event_type, data.event_name.replace('\'', "''").replace('?', "|q").replace('\n', ""), data.items.join(",").replace('\'', "''").replace('?', "|q").replace('\n', ""), data.metadata.replace('\'', "''").replace('?', "|q").replace('\n', ""), data.user_id, data.is_conversion, data.request_id, data.dataset_id
    );

    clickhouse_client
        .query(&query_string)
        .execute()
        .await
        .map_err(|e| {
            log::error!("Error sending event data: {:?}", e);
            ServiceError::InternalServerError("Error sending event data".to_string())
        })?;

    Ok(())
}

#[derive(Debug, Serialize, Deserialize, ToSchema)]
pub struct CTRSearchQueryWithClicksResponse {
    pub queries: Vec<SearchQueriesWithClicksCTRResponse>,
}

#[derive(Debug, Serialize, Deserialize, ToSchema)]
pub struct CTRSearchQueryWithoutClicksResponse {
    pub queries: Vec<SearchQueriesWithoutClicksCTRResponse>,
}

pub async fn get_search_ctr_metrics_query(
    dataset_id: uuid::Uuid,
    filter: Option<SearchAnalyticsFilter>,
    clickhouse_client: &clickhouse::Client,
) -> Result<SearchCTRMetrics, ServiceError> {
    let mut query_string = String::from(
        "WITH total_searches AS (
            SELECT COUNT(*) AS total
            FROM default.search_queries
            WHERE dataset_id = ? AND is_duplicate = 0
        ),
        metadata_values AS (
            SELECT arrayJoin(JSONExtractKeys(metadata)) AS key,
                JSONExtractFloat(metadata, key) AS value
            FROM default.events
            JOIN default.search_queries ON toUUID(events.request_id) = search_queries.id
            WHERE search_queries.dataset_id = ? AND events.event_type = 'click'
        ",
    );

    if let Some(filter) = filter {
        query_string = filter.add_to_query(query_string);
    }

    query_string.push_str(
        "
            )
        SELECT 
            searches_with_clicks,
            (searches_with_clicks * 100.0 / total) AS percent_searches_with_click,
            ((total - searches_with_clicks) * 100.0 / total) AS percent_searches_without_click,
            avg_metadata_value
        FROM (
            SELECT 
                COUNT(*) AS searches_with_clicks,
                AVG(value) AS avg_metadata_value
            FROM metadata_values
        ) AS subquery
        CROSS JOIN total_searches
        ",
    );

    let clickhouse_query = clickhouse_client
        .query(query_string.as_str())
        .bind(dataset_id)
        .bind(dataset_id)
        .fetch_one::<SearchCTRMetricsClickhouse>()
        .await
        .map_err(|e| {
            log::error!("Error fetching query: {:?}", e);
            ServiceError::InternalServerError("Error fetching query".to_string())
        })?;

    Ok(clickhouse_query.into())
}

pub async fn get_searches_with_clicks_query(
    dataset_id: uuid::Uuid,
    page: Option<u32>,
    filter: Option<SearchAnalyticsFilter>,
    pool: web::Data<Pool>,
    clickhouse_client: &clickhouse::Client,
) -> Result<CTRSearchQueryWithClicksResponse, ServiceError> {
    let mut query_string = String::from(
        "SELECT 
            search_queries.query,
            events.dataset_id,
            events.metadata,
            events.created_at
        FROM default.events 
        JOIN default.search_queries ON toUUID(events.request_id) = search_queries.id 
        WHERE search_queries.dataset_id = ? AND search_queries.is_duplicate = 0 AND events.event_type = 'click'",
    );

    if let Some(filter) = filter {
        query_string = filter.add_to_query(query_string);
    }

    query_string.push_str(
        "
        ORDER BY 
            events.created_at DESC
        LIMIT 10
        OFFSET ?",
    );

    let clickhouse_query = clickhouse_client
        .query(query_string.as_str())
        .bind(dataset_id)
        .bind((page.unwrap_or(1) - 1) * 10)
        .fetch_all::<SearchQueriesWithClicksCTRResponseClickhouse>()
        .await
        .map_err(|e| {
            log::error!("Error fetching query: {:?}", e);
            ServiceError::InternalServerError("Error fetching query".to_string())
        })?;

    let queries: Vec<SearchQueriesWithClicksCTRResponse> = join_all(
        clickhouse_query
            .into_iter()
            .map(|q| q.from_clickhouse(pool.clone())),
    )
    .await;

    Ok(CTRSearchQueryWithClicksResponse { queries })
}

pub async fn get_searches_without_clicks_query(
    dataset_id: uuid::Uuid,
    page: Option<u32>,
    filter: Option<SearchAnalyticsFilter>,
    clickhouse_client: &clickhouse::Client,
) -> Result<CTRSearchQueryWithoutClicksResponse, ServiceError> {
    let mut query_string = String::from(
        "SELECT search_queries.query, search_queries.created_at
        FROM default.search_queries sq
        LEFT JOIN default.events cd ON sq.id = toUUID(cd.request_id) AND 
            events.event_type = 'click'
        WHERE cd.request_id = '' AND search_queries.dataset_id = ? AND search_queries.is_duplicate = 0",
    );

    if let Some(filter) = filter {
        query_string = filter.add_to_query(query_string);
    }

    query_string.push_str(
        "
        ORDER BY 
            search_queries.created_at DESC
        LIMIT 10
        OFFSET ?",
    );

    let queries = clickhouse_client
        .query(query_string.as_str())
        .bind(dataset_id)
        .bind((page.unwrap_or(1) - 1) * 10)
        .fetch_all::<SearchQueriesWithoutClicksCTRResponseClickhouse>()
        .await
        .map_err(|e| {
            log::error!("Error fetching query: {:?}", e);
            ServiceError::InternalServerError("Error fetching query".to_string())
        })?;

    let queries: Vec<SearchQueriesWithoutClicksCTRResponse> =
        queries.into_iter().map(|q| q.into()).collect::<Vec<_>>();

    Ok(CTRSearchQueryWithoutClicksResponse { queries })
}

pub async fn get_recommendation_ctr_metrics_query(
    dataset_id: uuid::Uuid,
    filter: Option<RecommendationAnalyticsFilter>,
    clickhouse_client: &clickhouse::Client,
) -> Result<RecommendationCTRMetrics, ServiceError> {
    let mut query_string = String::from(
        "WITH total_recommendations AS (
            SELECT COUNT(*) AS total
            FROM default.recommendations
            WHERE dataset_id = ? 
        ),
        metadata_values AS (
            SELECT arrayJoin(JSONExtractKeys(metadata)) AS key,
                JSONExtractFloat(metadata, key) AS value
            FROM default.events
            JOIN default.recommendations ON toUUID(events.request_id) = recommendations.id
            WHERE recommendations.dataset_id = ? AND events.event_type = 'click'
        ",
    );

    if let Some(filter) = filter {
        query_string = filter.add_to_query(query_string);
    }

    query_string.push_str(
        "
         )
        SELECT 
            recommendations_with_clicks,
            (recommendations_with_clicks * 100.0 / total) AS percent_recommendations_with_click,
            ((total - recommendations_with_clicks) * 100.0 / total) AS percent_recommendations_without_click,
            avg_metadata_value
        FROM (
            SELECT 
                COUNT(*) AS recommendations_with_clicks,
                AVG(value) AS avg_metadata_value
            FROM metadata_values
        ) AS subquery
        CROSS JOIN total_recommendations
        ",
    );

    let clickhouse_query = clickhouse_client
        .query(query_string.as_str())
        .bind(dataset_id)
        .bind(dataset_id)
        .fetch_one::<RecommendationCTRMetrics>()
        .await
        .map_err(|e| {
            log::error!("Error fetching query: {:?}", e);
            ServiceError::InternalServerError("Error fetching query".to_string())
        })?;

    Ok(clickhouse_query)
}

#[derive(Debug, Serialize, Deserialize, ToSchema)]
pub struct CTRRecommendationsWithClicksResponse {
    pub recommendations: Vec<RecommendationsWithClicksCTRResponse>,
}

#[derive(Debug, Serialize, Deserialize, ToSchema)]
pub struct CTRRecommendationsWithoutClicksResponse {
    pub recommendations: Vec<RecommendationsWithoutClicksCTRResponse>,
}

pub async fn get_recommendations_with_clicks_query(
    dataset_id: uuid::Uuid,
    page: Option<u32>,
    filter: Option<RecommendationAnalyticsFilter>,
    pool: web::Data<Pool>,
    clickhouse_client: &clickhouse::Client,
) -> Result<CTRRecommendationsWithClicksResponse, ServiceError> {
    let mut query_string = String::from(
        "SELECT 
            recommendations.positive_ids,
            recommendations.negative_ids,
            recommendations.positive_tracking_ids,
            recommendations.negative_tracking_ids,
            events.dataset_id,
            events.metadata,
            events.created_at
        FROM default.events 
        JOIN default.recommendations ON toUUID(events.request_id) = recommendations.id 
        WHERE recommendations.dataset_id = ? AND events.event_type = 'click'",
    );

    if let Some(filter) = filter {
        query_string = filter.add_to_query(query_string);
    }

    query_string.push_str(
        "
        ORDER BY 
            events.created_at DESC
        LIMIT 10
        OFFSET ?",
    );

    let clickhouse_query = clickhouse_client
        .query(query_string.as_str())
        .bind(dataset_id)
        .bind((page.unwrap_or(1) - 1) * 10)
        .fetch_all::<RecommendationsWithClicksCTRResponseClickhouse>()
        .await
        .map_err(|e| {
            log::error!("Error fetching query: {:?}", e);
            ServiceError::InternalServerError("Error fetching query".to_string())
        })?;

    let recommendations: Vec<RecommendationsWithClicksCTRResponse> = join_all(
        clickhouse_query
            .into_iter()
            .map(|q| q.from_clickhouse(pool.clone())),
    )
    .await;

    Ok(CTRRecommendationsWithClicksResponse { recommendations })
}

pub async fn get_recommendations_without_clicks_query(
    dataset_id: uuid::Uuid,
    page: Option<u32>,
    filter: Option<RecommendationAnalyticsFilter>,
    clickhouse_client: &clickhouse::Client,
) -> Result<CTRRecommendationsWithoutClicksResponse, ServiceError> {
    let mut query_string = String::from(
        "SELECT   
            recommendations.positive_ids,
            recommendations.negative_ids,
            recommendations.positive_tracking_ids,
            recommendations.negative_tracking_ids,
            recommendations.created_at
        FROM default.recommendations r
        LEFT JOIN default.events cd ON r.id = toUUID(cd.request_id) AND 
            events.event_type = 'click'
        WHERE cd.request_id = '' AND recommendations.dataset_id = ?",
    );

    if let Some(filter) = filter {
        query_string = filter.add_to_query(query_string);
    }

    query_string.push_str(
        "
        ORDER BY 
            recommendations.created_at DESC
        LIMIT 10
        OFFSET ?",
    );

    let clickhouse_query = clickhouse_client
        .query(query_string.as_str())
        .bind(dataset_id)
        .bind((page.unwrap_or(1) - 1) * 10)
        .fetch_all::<RecommendationsWithoutClicksCTRResponseClickhouse>()
        .await
        .map_err(|e| {
            log::error!("Error fetching query: {:?}", e);
            ServiceError::InternalServerError("Error fetching query".to_string())
        })?;

    let recommendations: Vec<RecommendationsWithoutClicksCTRResponse> =
        clickhouse_query.into_iter().map(|q| q.into()).collect();

    Ok(CTRRecommendationsWithoutClicksResponse { recommendations })
}

pub async fn set_query_rating_query(
    data: RateQueryRequest,
    dataset_id: uuid::Uuid,
    clickhouse_client: &clickhouse::Client,
) -> Result<(), ServiceError> {
    let json_data = json!({
        "rating": data.rating,
        "comment": data.note
    });

    let stringified_data = serde_json::to_string(&json_data).unwrap_or_default();

    clickhouse_client
        .query(
            "ALTER TABLE default.search_queries
        UPDATE query_rating = ?
        WHERE id = ? AND dataset_id = ?",
        )
        .bind(stringified_data)
        .bind(data.query_id)
        .bind(dataset_id)
        .execute()
        .await
        .map_err(|err| {
            log::error!(
                "Error altering to ClickHouse default.search_queries: {:?}",
                err
            );
            sentry::capture_message(
                &format!(
                    "Error altering to ClickHouse default.search_queries: {:?}",
                    err
                ),
                sentry::Level::Error,
            );
            ServiceError::InternalServerError(
                "Error altering to ClickHouse default.search_queries".to_string(),
            )
        })?;

    Ok(())
}

pub async fn get_top_datasets_query(
    data: GetTopDatasetsRequestBody,
    clickhouse_client: &clickhouse::Client,
    pool: web::Data<Pool>,
) -> Result<Vec<TopDatasetsResponse>, ServiceError> {
    use crate::data::schema::datasets::dsl as datasets_columns;

    let mut query_string = format!(
        "SELECT 
            dataset_id,
            COUNT(*) as total_queries
        FROM 
            default.{}",
        data.r#type
    );

    if let Some(date_range) = data.date_range {
        if let Some(gt) = &date_range.gt {
            query_string.push_str(&format!(" AND created_at > '{}'", gt));
        }
        if let Some(lt) = &date_range.lt {
            query_string.push_str(&format!(" AND created_at < '{}'", lt));
        }
        if let Some(gte) = &date_range.gte {
            query_string.push_str(&format!(" AND created_at >= '{}'", gte));
        }
        if let Some(lte) = &date_range.lte {
            query_string.push_str(&format!(" AND created_at <= '{}'", lte));
        }
    }

    query_string.push_str(
        "
        GROUP BY 
            dataset_id
        ORDER BY 
            total_queries DESC
        LIMIT 10",
    );

    let clickhouse_resp_data = clickhouse_client
        .query(query_string.as_str())
        .fetch_all::<TopDatasetsResponseClickhouse>()
        .await
        .map_err(|e| {
            log::error!("Error fetching query: {:?}", e);
            ServiceError::InternalServerError("Error fetching query".to_string())
        })?;

    let dataset_ids = clickhouse_resp_data
        .iter()
        .map(|x| x.dataset_id)
        .collect::<Vec<_>>();
    let mut conn = pool
        .get()
        .await
        .map_err(|_| ServiceError::BadRequest("Could not get database connection".to_string()))?;
    let dataset_id_and_tracking_ids = datasets_columns::datasets
        .select((datasets_columns::id, datasets_columns::tracking_id))
        .filter(datasets_columns::id.eq_any(dataset_ids))
        .load::<(uuid::Uuid, Option<String>)>(&mut conn)
        .await
        .map_err(|e| {
            log::error!("Error fetching dataset ids: {:?}", e);
            ServiceError::InternalServerError("Error fetching dataset ids".to_string())
        })?;

    let response = clickhouse_resp_data
        .into_iter()
        .map(|x| {
            let mut top_dataset_resps = TopDatasetsResponse::from(x.clone());
            top_dataset_resps.dataset_tracking_id = dataset_id_and_tracking_ids
                .iter()
                .find(|(id, _)| id == &x.dataset_id)
                .and_then(|(_, tracking_id)| tracking_id.clone());
            top_dataset_resps
        })
        .collect::<Vec<_>>();

    Ok(response)
}
