use super::auth_handler::{AdminOnly, LoggedUser};
use crate::data::models::{
    ChatMessageProxy, ChunkMetadata, ChunkMetadataStringTagSet, ChunkMetadataWithScore,
    ConditionType, CountSearchMethod, DatasetAndOrgWithSubAndPlan, DatasetConfiguration, GeoInfo,
    HighlightOptions, IngestSpecificChunkMetadata, Pool, QueryTypes, RagQueryEventClickhouse,
    RecommendType, RecommendationEventClickhouse, RecommendationStrategy, RedisPool, ScoreChunk,
    ScoreChunkDTO, SearchMethod, SearchQueryEventClickhouse, SlimChunkMetadataWithScore,
    SortByField, SortOptions, TypoOptions, UnifiedId, UpdateSpecificChunkMetadata,
};
use crate::errors::ServiceError;
use crate::get_env;
use crate::middleware::api_version::APIVersion;
use crate::operators::chunk_operator::get_metadata_from_id_query;
use crate::operators::chunk_operator::*;
use crate::operators::clickhouse_operator::{get_latency_from_header, ClickHouseEvent, EventQueue};
use crate::operators::dataset_operator::get_dataset_usage_query;
use crate::operators::parse_operator::convert_html_to_text;
use crate::operators::qdrant_operator::{
    point_ids_exists_in_qdrant, recommend_qdrant_query, scroll_dataset_points,
};
use crate::operators::search_operator::{
    assemble_qdrant_filter, autocomplete_chunks_query, count_chunks_query, search_chunks_query,
    search_hybrid_chunks,
};
use actix::Arbiter;
use actix_web::web::Bytes;
use actix_web::{web, HttpResponse};
use chrono::NaiveDateTime;
use crossbeam_channel::unbounded;
use dateparser::DateTimeUtc;
use itertools::Itertools;
use openai_dive::v1::api::Client;
use openai_dive::v1::resources::chat::{
    ChatCompletionParameters, ChatMessage, ChatMessageContent, Role,
};
use openai_dive::v1::resources::shared::StopToken;
use regex::Regex;
use serde::{Deserialize, Serialize};
use serde_json::json;
use simple_server_timing_header::Timer;
use tokio_stream::StreamExt;
use utoipa::ToSchema;

/// Boost phrase is useful for when you want to boost certain phrases in the fulltext (SPLADE) and BM25 search results. I.e. making sure that the listing for AirBNB itself ranks higher than companies who make software for AirBNB hosts by boosting the in-document-frequency of the AirBNB token (AKA word) for its official listing. Conceptually it multiples the in-document-importance second value in the tuples of the SPLADE or BM25 sparse vector of the chunk_html innerText for all tokens present in the boost phrase by the boost factor like so: (token, in-document-importance) -> (token, in-document-importance*boost_factor).
#[derive(Serialize, Deserialize, Debug, Clone, ToSchema)]
pub struct FullTextBoost {
    /// The phrase to boost in the fulltext document frequency index
    pub phrase: String,
    /// Amount to multiplicatevly increase the frequency of the tokens in the phrase by
    pub boost_factor: f64,
}

/// Distance phrase is useful for moving the embedding vector of the chunk in the direction of the distance phrase. I.e. you can push a chunk with a chunk_html of "iphone" 25% closer to the term "flagship" by using the distance phrase "flagship" and a distance factor of 0.25. Conceptually it's drawing a line (euclidean/L2 distance) between the vector for the innerText of the chunk_html and distance_phrase then moving the vector of the chunk_html distance_factor*L2Distance closer to or away from the distance_phrase point along the line between the two points.
#[derive(Serialize, Deserialize, Debug, Clone, ToSchema)]
pub struct SemanticBoost {
    /// Terms to embed in order to create the vector which is weighted summed with the chunk_html embedding vector
    pub phrase: String,
    /// Arbitrary float (positive or negative) specifying the multiplicate factor to apply before summing the phrase vector with the chunk_html embedding vector
    pub distance_factor: f32,
}

/// Scoring options provides ways to modify the sparse or dense vector created for the query in order to change how potential matches are scored. If not specified, this defaults to no modifications.
#[derive(Serialize, Deserialize, Debug, Clone, ToSchema)]
pub struct ScoringOptions {
    ///  Full text boost is useful for when you want to boost certain phrases in the fulltext (SPLADE) and BM25 search results. I.e. making sure that the listing for AirBNB itself ranks higher than companies who make software for AirBNB hosts by boosting the in-document-frequency of the AirBNB token (AKA word) for its official listing. Conceptually it multiples the in-document-importance second value in the tuples of the SPLADE or BM25 sparse vector of the chunk_html innerText for all tokens present in the boost phrase by the boost factor like so: (token, in-document-importance) -> (token, in-document-importance*boost_factor).
    pub fulltext_boost: Option<FullTextBoost>,
    /// Semantic boost is useful for moving the embedding vector of the chunk in the direction of the distance phrase. I.e. you can push a chunk with a chunk_html of "iphone" 25% closer to the term "flagship" by using the distance phrase "flagship" and a distance factor of 0.25. Conceptually it's drawing a line (euclidean/L2 distance) between the vector for the innerText of the chunk_html and distance_phrase then moving the vector of the chunk_html distance_factor*L2Distance closer to or away from the distance_phrase point along the line between the two points.
    pub semantic_boost: Option<SemanticBoost>,
}

#[derive(Serialize, Deserialize, Debug, ToSchema, Clone, Default)]
#[schema(example = json!({
    "chunk_html": "<p>Some HTML content</p>",
    "link": "https://example.com",
    "tag_set": ["tag1", "tag2"],
    "metadata": {"key1": "value1", "key2": "value2"},
    "tracking_id": "tracking_id",
    "group_ids": ["d290f1ee-6c54-4b01-90e6-d701748f0851"],
    "group_tracking_ids": ["group_tracking_id"],
    "time_stamp": "2021-01-01 00:00:00.000",
    "location": {
        "lat": -34,
        "lon": 151
    },
    "image_urls": ["https://example.com/red", "https://example.com/blue"],
    "fulltext_boost": {"phrase": "foo", "boost_factor": 5.0},
    "semantic_boost": {"phrase": "flagship", "distance_factor": 0.5}
}))]
pub struct ChunkReqPayload {
    /// HTML content of the chunk. This can also be plaintext. The innerText of the HTML will be used to create the embedding vector. The point of using HTML is for convienience, as some users have applications where users submit HTML content.
    pub chunk_html: Option<String>,
    /// If semantic_content is present, it will be used for creating semantic embeddings instead of the innerText `chunk_html`. `chunk_html` will still be the only thing stored and always used for fulltext functionality.
    pub semantic_content: Option<String>,
    /// Link to the chunk. This can also be any string. Frequently, this is a link to the source of the chunk. The link value will not affect the embedding creation.
    pub link: Option<String>,
    /// Tag set is a list of tags. This can be used to filter chunks by tag. Unlike with metadata filtering, HNSW indices will exist for each tag such that there is not a performance hit for filtering on them.
    pub tag_set: Option<Vec<String>>,
    /// Num value is an arbitrary numerical value that can be used to filter chunks. This is useful for when you want to filter chunks by numerical value. There is no performance hit for filtering on num_value.
    pub num_value: Option<f64>,
    /// Metadata is a JSON object which can be used to filter chunks. This is useful for when you want to filter chunks by arbitrary metadata. Unlike with tag filtering, there is a performance hit for filtering on metadata.
    pub metadata: Option<serde_json::Value>,
    /// Tracking_id is a string which can be used to identify a chunk. This is useful for when you are coordinating with an external system and want to use the tracking_id to identify the chunk.
    pub tracking_id: Option<String>,
    /// Upsert when a chunk with the same tracking_id exists. By default this is false, and the request will fail if a chunk with the same tracking_id exists. If this is true, the chunk will be updated if a chunk with the same tracking_id exists.
    pub upsert_by_tracking_id: Option<bool>,
    /// Group ids are the Trieve generated ids of the groups that the chunk should be placed into. This is useful for when you want to create a chunk and add it to a group or multiple groups in one request. Groups with these Trieve generated ids must be created first, it cannot be arbitrarily created through this route.
    pub group_ids: Option<Vec<uuid::Uuid>>,
    /// Group tracking_ids are the user-assigned tracking_ids of the groups that the chunk should be placed into. This is useful for when you want to create a chunk and add it to a group or multiple groups in one request. If a group with the tracking_id does not exist, it will be created.
    pub group_tracking_ids: Option<Vec<String>>,
    /// Time_stamp should be an ISO 8601 combined date and time without timezone. It is used for time window filtering and recency-biasing search results.
    pub time_stamp: Option<String>,
    /// Location is a GeoInfo object which lets you specify a latitude and longitude which can be used later to filter results.
    pub location: Option<GeoInfo>,
    /// Image urls are a list of urls to images that are associated with the chunk. This is useful for when you want to associate images with a chunk.
    pub image_urls: Option<Vec<String>>,
    /// Weight is a float which can be used to bias search results. This is useful for when you want to bias search results for a chunk. The magnitude only matters relative to other chunks in the chunk's dataset dataset.
    pub weight: Option<f64>,
    /// Split avg is a boolean which tells the server to split the text in the chunk_html into smaller chunks and average their resulting vectors. This is useful for when you want to create a chunk from a large piece of text and want to split it into smaller chunks to create a more fuzzy average dense vector. The sparse vector will be generated normally with no averaging. By default this is false.
    pub split_avg: Option<bool>,
    /// Convert HTML to raw text before processing to avoid adding noise to the vector embeddings. By default this is true. If you are using HTML content that you want to be included in the vector embeddings, set this to false.
    pub convert_html_to_text: Option<bool>,
    ///  Full text boost is useful for when you want to boost certain phrases in the fulltext (SPLADE) and BM25 search results. I.e. making sure that the listing for AirBNB itself ranks higher than companies who make software for AirBNB hosts by boosting the in-document-frequency of the AirBNB token (AKA word) for its official listing. Conceptually it multiples the in-document-importance second value in the tuples of the SPLADE or BM25 sparse vector of the chunk_html innerText for all tokens present in the boost phrase by the boost factor like so: (token, in-document-importance) -> (token, in-document-importance*boost_factor).
    #[serde(alias = "boost_phrase")]
    pub fulltext_boost: Option<FullTextBoost>,
    /// Semantic boost is useful for moving the embedding vector of the chunk in the direction of the distance phrase. I.e. you can push a chunk with a chunk_html of "iphone" 25% closer to the term "flagship" by using the distance phrase "flagship" and a distance factor of 0.25. Conceptually it's drawing a line (euclidean/L2 distance) between the vector for the innerText of the chunk_html and distance_phrase then moving the vector of the chunk_html distance_factor*L2Distance closer to or away from the distance_phrase point along the line between the two points.
    #[serde(alias = "distance_phrase")]
    pub semantic_boost: Option<SemanticBoost>,
}

#[derive(Serialize, Deserialize, Clone, ToSchema)]
pub struct FailedChunk {
    index: usize,
    message: String,
}

#[derive(Serialize, Deserialize, Clone, ToSchema)]
#[serde(untagged)]
pub enum ReturnQueuedChunk {
    /// All Chunks that have been queue'd with any errors filtered out
    Single(SingleQueuedChunkResponse),
    Batch(BatchQueuedChunkResponse),
}

#[derive(Serialize, Deserialize, Clone, ToSchema)]
#[schema(example = json!({
    "chunk_metadata": [{
        "content": "Some content",
        "link": "https://example.com",
        "tag_set": ["tag1", "tag2"],
        "metadata": {"key1": "value1", "key2": "value2"},
        "tracking_id": "tracking_id",
        "time_stamp": "2021-01-01 00:00:00.000",
        "weight": 0.5
    }],
    "pos_in_queue": 1
}))]
pub struct SingleQueuedChunkResponse {
    /// The chunk that got queue'd
    pub chunk_metadata: ChunkMetadata,
    /// The current position the last access item is in the queue
    pub pos_in_queue: i32,
}

#[derive(Serialize, Deserialize, Clone, ToSchema)]
#[schema(example = json!({
    "chunk_metadata": [{
        "content": "Some content",
        "link": "https://example.com",
        "tag_set": ["tag1", "tag2"],
        "file_id": "d290f1ee-6c54-4b01-90e6-d701748f0851",
        "metadata": {"key1": "value1", "key2": "value2"},
        "tracking_id": "tracking_id",
        "time_stamp": "2021-01-01 00:00:00.000",
        "weight": 0.5
    },
    {
        "content": "Some content",
        "link": "https://example.com",
        "tag_set": ["tag1", "tag2"],
        "file_id": "d290f1ee-6c54-4b01-90e6-d701748f0851",
        "metadata": {"key1": "value1", "key2": "value2"},
        "tracking_id": "tracking_id",
        "time_stamp": "2021-01-01 00:00:00.000",
        "weight": 0.5
    }],
    "pos_in_queue": 2
}))]
pub struct BatchQueuedChunkResponse {
    // All the chunks that got queue'd
    pub chunk_metadata: Vec<ChunkMetadata>,
    /// The current position the last access item is in the queue
    pub pos_in_queue: i32,
}

#[derive(Serialize, Deserialize, Debug, Clone)]
pub struct UploadIngestionMessage {
    pub ingest_specific_chunk_metadata: IngestSpecificChunkMetadata,
    pub chunk: ChunkReqPayload,
    pub dataset_id: uuid::Uuid,
    pub upsert_by_tracking_id: bool,
}

#[derive(Serialize, Deserialize, Debug, Clone)]
pub struct BulkUploadIngestionMessage {
    pub attempt_number: usize,
    pub dataset_id: uuid::Uuid,
    pub ingestion_messages: Vec<UploadIngestionMessage>,
}

#[derive(Serialize, Deserialize, Debug, Clone, ToSchema)]
#[schema(example = json!({
    "chunk_html": "<p>Some HTML content</p>",
    "link": "https://example.com",
    "tag_set": ["tag1", "tag2"],
    "metadata": {"key1": "value1", "key2": "value2"},
    "tracking_id": "tracking_id",
    "upsert_by_tracking_id": true,
    "group_ids": ["d290f1ee-6c54-4b01-90e6-d701748f0851"],
    "group_tracking_ids": ["group_tracking_id"],
    "time_stamp": "2021-01-01 00:00:00.000",
    "location": {
        "lat": -34,
        "lon": 151
    },
    "weight": 0.5,
    "split_avg": false,
    "convert_html_to_text": false,
}))]
pub struct CreateSingleChunkReqPayload(pub ChunkReqPayload);

#[derive(Serialize, Deserialize, Debug, Clone, ToSchema)]
#[schema(example = json!([{
    "chunk_html": "<p>Some HTML content</p>",
    "link": "https://example.com",
    "tag_set": ["tag1", "tag2"],
    "metadata": {"key1": "value1", "key2": "value2"},
    "tracking_id": "tracking_id",
    "upsert_by_tracking_id": true,
    "group_ids": ["d290f1ee-6c54-4b01-90e6-d701748f0851"],
    "group_tracking_ids": ["group_tracking_id"],
    "time_stamp": "2021-01-01 00:00:00.000",
    "image_urls": ["https://example.com/red", "https://example.com/blue"],
    "location": {
        "lat": -34,
        "lon": 151
    },
}, {
    "chunk_html": "<p>Some more HTML content</p>",
    "link": "https://explain.com",
    "tag_set": ["tag3", "tag4"],
    "metadata": {"key1": "value1", "key2": "value2"},
    "tracking_id": "tracking_id",
    "upsert_by_tracking_id": true,
    "group_ids": ["d290f1ee-6c54-4b01-90e6-d701748f0851"],
    "group_tracking_ids": ["group_tracking_id"],
    "time_stamp": "2021-01-01 00:00:00.000",
    "image_urls": [],
    "location": {
        "lat": -34,
        "lon": 151
    },
    "weight": 0.5,
}]
))]
pub struct CreateBatchChunkReqPayload(pub Vec<ChunkReqPayload>);

#[derive(Serialize, Deserialize, Debug, Clone, ToSchema)]
#[serde(untagged)]
pub enum CreateChunkReqPayloadEnum {
    Single(CreateSingleChunkReqPayload),
    Batch(CreateBatchChunkReqPayload),
}

/// Create or Upsert Chunk or Chunks
///
/// Create new chunk(s). If the chunk has the same tracking_id as an existing chunk, the request will fail. Once a chunk is created, it can be searched for using the search endpoint.
/// If uploading in bulk, the maximum amount of chunks that can be uploaded at once is 120 chunks. Auth'ed user or api key must have an admin or owner role for the specified dataset's organization.
#[utoipa::path(
    post,
    path = "/chunk",
    context_path = "/api",
    tag = "Chunk",
    request_body(content = CreateChunkReqPayloadEnum, description = "JSON request payload to create a new chunk (chunk)", content_type = "application/json"),
    responses(
        (status = 200, description = "JSON response payload containing the created chunk", body = ReturnQueuedChunk),
        (status = 426, description = "Error when upgrade is needed to process more chunks", body = ErrorResponseBody),
        (status = 413, description = "Error when more than 120 chunks are provided in bulk", body = ErrorResponseBody),
        (status = 400, description = "Error typically due to deserialization issues", body = ErrorResponseBody),
    ),
    params(
        ("TR-Dataset" = String, Header, description = "The dataset id or tracking_id to use for the request. We assume you intend to use an id if the value is a valid uuid."),
    ),
    security(
        ("ApiKey" = ["admin"]),
    )
)]
#[tracing::instrument(skip(redis_pool, pool))]
pub async fn create_chunk(
    create_chunk_data: web::Json<CreateChunkReqPayloadEnum>,
    pool: web::Data<Pool>,
    _user: AdminOnly,
    dataset_org_plan_sub: DatasetAndOrgWithSubAndPlan,
    redis_pool: web::Data<RedisPool>,
) -> Result<HttpResponse, actix_web::Error> {
    let chunks = match create_chunk_data.clone() {
        CreateChunkReqPayloadEnum::Single(chunk) => vec![chunk.0],
        CreateChunkReqPayloadEnum::Batch(chunks) => chunks.0,
    };

    let mut timer = Timer::new();

    let unlimited = std::env::var("UNLIMITED").unwrap_or("false".to_string());
    if unlimited == "false" {
        let chunk_count = get_row_count_for_organization_id_query(
            dataset_org_plan_sub.organization.organization.id,
            pool.clone(),
        )
        .await?;

        if chunk_count + chunks.len()
            > dataset_org_plan_sub
                .organization
                .plan
                .unwrap_or_default()
                .chunk_count as usize
        {
            return Ok(HttpResponse::UpgradeRequired()
                .json(json!({"message": "Must upgrade your plan to add more chunks"})));
        }

        timer.add("get dataset count");
    }

    if chunks.len() > 120 {
        return Err(ServiceError::PayloadTooLarge(
            "Too many chunks provided in bulk. The limit is 120 chunks per bulk upload".to_string(),
        )
        .into());
    }

    let dataset_config =
        DatasetConfiguration::from_json(dataset_org_plan_sub.dataset.server_configuration.clone());

    let chunks = chunks.into_iter().map(|chunk| {
        let non_empty_tracking_id = chunk
            .tracking_id
            .clone()
            .filter(|tracking_id| !tracking_id.is_empty());
        ChunkReqPayload {
            tracking_id: non_empty_tracking_id,
            ..chunk.clone()
        }
    });

    let (upsert_chunks, non_upsert_chunks): (Vec<ChunkReqPayload>, Vec<ChunkReqPayload>) =
        chunks.partition(|chunk| chunk.upsert_by_tracking_id.unwrap_or(false));

    let (non_upsert_chunk_ingestion_message, non_upsert_chunk_metadatas) = create_chunk_metadata(
        non_upsert_chunks,
        dataset_org_plan_sub.dataset.id,
        dataset_config.clone(),
        pool.clone(),
    )
    .await?;

    let (upsert_chunk_ingestion_message, upsert_chunk_metadatas) = create_chunk_metadata(
        upsert_chunks,
        dataset_org_plan_sub.dataset.id,
        dataset_config.clone(),
        pool.clone(),
    )
    .await?;

    let chunk_metadatas = non_upsert_chunk_metadatas
        .clone()
        .into_iter()
        .chain(upsert_chunk_metadatas.clone().into_iter())
        .collect::<Vec<ChunkMetadata>>();

    let mut redis_conn = redis_pool
        .get()
        .await
        .map_err(|err| ServiceError::BadRequest(err.to_string()))?;

    timer.add("got redis connection");

    let mut pos_in_queue = 0;
    if !non_upsert_chunk_metadatas.is_empty() {
        let serialized_message: String = serde_json::to_string(&non_upsert_chunk_ingestion_message)
            .map_err(|_| {
                ServiceError::BadRequest("Failed to Serialize BulkUploadMessage".to_string())
            })?;

        pos_in_queue = redis::cmd("lpush")
            .arg("ingestion")
            .arg(&serialized_message)
            .query_async(&mut *redis_conn)
            .await
            .map_err(|err| ServiceError::BadRequest(err.to_string()))?;
    }
    if !upsert_chunk_metadatas.is_empty() {
        let serialized_message: String = serde_json::to_string(&upsert_chunk_ingestion_message)
            .map_err(|_| {
                ServiceError::BadRequest("Failed to Serialize BulkUploadMessage".to_string())
            })?;

        pos_in_queue = redis::cmd("lpush")
            .arg("ingestion")
            .arg(&serialized_message)
            .query_async(&mut *redis_conn)
            .await
            .map_err(|err| ServiceError::BadRequest(err.to_string()))?;
    }

    let response = match create_chunk_data.into_inner() {
        CreateChunkReqPayloadEnum::Single(_) => ReturnQueuedChunk::Single(SingleQueuedChunkResponse {
            chunk_metadata: chunk_metadatas
                .get(0)
                .ok_or(ServiceError::BadRequest(
                    "Failed to queue a single chunk due to deriving 0 ingestion_messages from the request data".to_string(),
                ))?
                .clone(),
            pos_in_queue,
        }),
        CreateChunkReqPayloadEnum::Batch(_) => ReturnQueuedChunk::Batch(BatchQueuedChunkResponse {
            chunk_metadata: chunk_metadatas,
            pos_in_queue,
        }),
    };

    Ok(HttpResponse::Ok()
        .insert_header((Timer::header_key(), timer.header_value()))
        .json(response))
}

/// Delete Chunk
///
/// Delete a chunk by its id. If deleting a root chunk which has a collision, the most recently created collision will become a new root chunk. Auth'ed user or api key must have an admin or owner role for the specified dataset's organization.
#[utoipa::path(
    delete,
    path = "/chunk/{chunk_id}",
    context_path = "/api",
    tag = "Chunk",
    responses(
        (status = 204, description = "Confirmation that the chunk with the id specified was deleted"),
        (status = 400, description = "Service error relating to finding a chunk by tracking_id", body = ErrorResponseBody),
    ),
    params(
        ("TR-Dataset" = String, Header, description = "The dataset id or tracking_id to use for the request. We assume you intend to use an id if the value is a valid uuid."),
        ("chunk_id" = Option<uuid::Uuid>, Path, description = "Id of the chunk you want to fetch."),
    ),
    security(
        ("ApiKey" = ["admin"]),
    )
)]
#[tracing::instrument(skip(pool))]
pub async fn delete_chunk(
    chunk_id: web::Path<uuid::Uuid>,
    pool: web::Data<Pool>,
    _user: AdminOnly,
    dataset_org_plan_sub: DatasetAndOrgWithSubAndPlan,
) -> Result<HttpResponse, actix_web::Error> {
    let dataset_config =
        DatasetConfiguration::from_json(dataset_org_plan_sub.dataset.server_configuration.clone());

    let chunk_id = chunk_id.into_inner();

    let deleted_at = chrono::Utc::now().naive_utc();

    delete_chunk_metadata_query(
        vec![chunk_id],
        deleted_at,
        dataset_org_plan_sub.dataset,
        pool,
        dataset_config,
    )
    .await?;

    Ok(HttpResponse::NoContent().finish())
}

/// Delete Chunk By Tracking Id
///
/// Delete a chunk by tracking_id. This is useful for when you are coordinating with an external system and want to use the tracking_id to identify the chunk. If deleting a root chunk which has a collision, the most recently created collision will become a new root chunk. Auth'ed user or api key must have an admin or owner role for the specified dataset's organization.
#[utoipa::path(
    delete,
    path = "/chunk/tracking_id/{tracking_id}",
    context_path = "/api",
    tag = "Chunk",
    responses(
        (status = 204, description = "Confirmation that the chunk with the tracking_id specified was deleted"),
        (status = 400, description = "Service error relating to finding a chunk by tracking_id", body = ErrorResponseBody),
    ),
    params(
        ("TR-Dataset" = String, Header, description = "The dataset id or tracking_id to use for the request. We assume you intend to use an id if the value is a valid uuid."),
        ("tracking_id" = Option<String>, Path, description = "tracking_id of the chunk you want to delete"),
    ),
    security(
        ("ApiKey" = ["admin"]),
    )
)]
#[tracing::instrument(skip(pool))]
pub async fn delete_chunk_by_tracking_id(
    tracking_id: web::Path<String>,
    pool: web::Data<Pool>,
    _user: AdminOnly,
    dataset_org_plan_sub: DatasetAndOrgWithSubAndPlan,
) -> Result<HttpResponse, actix_web::Error> {
    let tracking_id_inner = tracking_id.into_inner();
    let dataset_id = dataset_org_plan_sub.dataset.id;
    let dataset_config =
        DatasetConfiguration::from_json(dataset_org_plan_sub.dataset.server_configuration.clone());

    let chunk_metadata =
        get_metadata_from_tracking_id_query(tracking_id_inner, dataset_id, pool.clone()).await?;

    let deleted_at = chrono::Utc::now().naive_utc();

    delete_chunk_metadata_query(
        vec![chunk_metadata.id],
        deleted_at,
        dataset_org_plan_sub.dataset,
        pool,
        dataset_config,
    )
    .await?;

    Ok(HttpResponse::NoContent().finish())
}

#[derive(Serialize, Deserialize, Debug, Clone, ToSchema)]
#[schema(example = json!({
    "chunk_id": "d290f1ee-6c54-4b01-90e6-d701748f0851",
    "link": "https://example.com",
    "chunk_html": "<p>Some HTML content</p>",
    "metadata": {"key1": "value1", "key2": "value2"},
    "time_stamp": "2021-01-01 00:00:00.000",
    "weight": 0.5,
    "group_ids": ["d290f1ee-6c54-4b01-90e6-d701748f0851"],
}))]
pub struct UpdateChunkReqPayload {
    /// Id of the chunk you want to update. You can provide either the chunk_id or the tracking_id. If both are provided, the chunk_id will be used.
    chunk_id: Option<uuid::Uuid>,
    /// Tracking_id of the chunk you want to update. This is required to match an existing chunk.
    tracking_id: Option<String>,
    /// Tag set is a list of tags. This can be used to filter chunks by tag. Unlike with metadata filtering, HNSW indices will exist for each tag such that there is not a performance hit for filtering on them. If no tag_set is provided, the existing tag_set will be used.
    tag_set: Option<Vec<String>>,
    /// Link of the chunk you want to update. This can also be any string. Frequently, this is a link to the source of the chunk. The link value will not affect the embedding creation. If no link is provided, the existing link will be used.
    link: Option<String>,
    ///Num value is an arbitrary numerical value that can be used to filter chunks. This is useful for when you want to filter chunks by numerical value. If no num_value is provided, the existing num_value will be used.
    num_value: Option<f64>,
    /// HTML content of the chunk you want to update. This can also be plaintext. The innerText of the HTML will be used to create the embedding vector. The point of using HTML is for convienience, as some users have applications where users submit HTML content. If no chunk_html is provided, the existing chunk_html will be used.
    chunk_html: Option<String>,
    /// The metadata is a JSON object which can be used to filter chunks. This is useful for when you want to filter chunks by arbitrary metadata. Unlike with tag filtering, there is a performance hit for filtering on metadata. If no metadata is provided, the existing metadata will be used.
    metadata: Option<serde_json::Value>,
    /// Time_stamp should be an ISO 8601 combined date and time without timezone. It is used for time window filtering and recency-biasing search results. If no time_stamp is provided, the existing time_stamp will be used.
    time_stamp: Option<String>,
    /// Weight is a float which can be used to bias search results. This is useful for when you want to bias search results for a chunk. The magnitude only matters relative to other chunks in the chunk's dataset dataset. If no weight is provided, the existing weight will be used.
    weight: Option<f64>,
    /// Group ids are the ids of the groups that the chunk should be placed into. This is useful for when you want to update a chunk and add it to a group or multiple groups in one request.
    group_ids: Option<Vec<uuid::Uuid>>,
    /// Group tracking_ids are the tracking_ids of the groups that the chunk should be placed into. This is useful for when you want to update a chunk and add it to a group or multiple groups in one request.
    group_tracking_ids: Option<Vec<String>>,
    /// Location is a GeoInfo object which lets you specify a latitude and longitude which can be used later to filter results.
    location: Option<GeoInfo>,
    /// Image urls are a list of urls to images that are associated with the chunk. This is useful for when you want to associate images with a chunk. If no image_urls are provided, the existing image_urls will be used.
    image_urls: Option<Vec<String>>,
    /// Convert HTML to raw text before processing to avoid adding noise to the vector embeddings. By default this is true. If you are using HTML content that you want to be included in the vector embeddings, set this to false.
    convert_html_to_text: Option<bool>,
    ///  Full text boost is useful for when you want to boost certain phrases in the fulltext (SPLADE) and BM25 search results. I.e. making sure that the listing for AirBNB itself ranks higher than companies who make software for AirBNB hosts by boosting the in-document-frequency of the AirBNB token (AKA word) for its official listing. Conceptually it multiples the in-document-importance second value in the tuples of the SPLADE or BM25 sparse vector of the chunk_html innerText for all tokens present in the boost phrase by the boost factor like so: (token, in-document-importance) -> (token, in-document-importance*boost_factor).
    #[serde(alias = "boost_phrase")]
    pub fulltext_boost: Option<FullTextBoost>,
    /// Semantic boost is useful for moving the embedding vector of the chunk in the direction of the distance phrase. I.e. you can push a chunk with a chunk_html of "iphone" 25% closer to the term "flagship" by using the distance phrase "flagship" and a distance factor of 0.25. Conceptually it's drawing a line (euclidean/L2 distance) between the vector for the innerText of the chunk_html and distance_phrase then moving the vector of the chunk_html distance_factor*L2Distance closer to or away from the distance_phrase point along the line between the two points.
    #[serde(alias = "distance_phrase")]
    pub semantic_boost: Option<SemanticBoost>,
}

#[derive(Serialize, Deserialize, Debug, Clone)]
pub struct UpdateIngestionMessage {
    pub chunk_metadata: UpdateSpecificChunkMetadata,
    pub dataset_id: uuid::Uuid,
    pub group_ids: Option<Vec<UnifiedId>>,
    pub convert_html_to_text: Option<bool>,
    pub fulltext_boost: Option<FullTextBoost>,
    pub semantic_boost: Option<SemanticBoost>,
}

/// Update Chunk
///
/// Update a chunk. If you try to change the tracking_id of the chunk to have the same tracking_id as an existing chunk, the request will fail. Auth'ed user or api key must have an admin or owner role for the specified dataset's organization.
#[utoipa::path(
    put,
    path = "/chunk",
    context_path = "/api",
    tag = "Chunk",
    request_body(content = UpdateChunkReqPayload, description = "JSON request payload to update a chunk (chunk)", content_type = "application/json"),
    responses(
        (status = 204, description = "No content Ok response indicating the chunk was updated as requested",),
        (status = 400, description = "Service error relating to to updating chunk, likely due to conflicting tracking_id", body = ErrorResponseBody),
    ),
    params(
        ("TR-Dataset" = String, Header, description = "The dataset id or tracking_id to use for the request. We assume you intend to use an id if the value is a valid uuid."),
    ),
    security(
        ("ApiKey" = ["admin"]),
    )
)]
#[tracing::instrument(skip(pool, redis_pool))]
pub async fn update_chunk(
    update_chunk_data: web::Json<UpdateChunkReqPayload>,
    pool: web::Data<Pool>,
    redis_pool: web::Data<RedisPool>,
    _user: AdminOnly,
    dataset_org_plan_sub: DatasetAndOrgWithSubAndPlan,
) -> Result<HttpResponse, actix_web::Error> {
    let dataset_id = dataset_org_plan_sub.dataset.id;
    let chunk_id = update_chunk_data.chunk_id;

    let chunk_metadata = if let Some(chunk_id) = chunk_id {
        get_metadata_from_id_query(chunk_id, dataset_id, pool).await?
    } else if let Some(tracking_id) = update_chunk_data.tracking_id.clone() {
        get_metadata_from_tracking_id_query(tracking_id.clone(), dataset_id, pool).await?
    } else {
        return Err(ServiceError::BadRequest(
            "Either chunk_id or tracking_id must be provided to update a chunk".into(),
        )
        .into());
    };

    let link = match update_chunk_data.link.clone() {
        Some(link) => Some(link),
        None => chunk_metadata.link,
    };

    let chunk_tracking_id = match update_chunk_data.tracking_id.clone() {
        Some(tracking_id) => {
            if tracking_id.is_empty() {
                None
            } else {
                Some(tracking_id)
            }
        }
        None => chunk_metadata.tracking_id,
    };

    let chunk_tag_set = update_chunk_data.tag_set.clone().map(|tag_set| {
        tag_set
            .into_iter()
            .map(Some)
            .collect::<Vec<Option<String>>>()
    });

    let chunk_html = match update_chunk_data.chunk_html.clone() {
        Some(chunk_html) => Some(chunk_html),
        None => chunk_metadata.chunk_html,
    };

    let chunk_metadata = ChunkMetadata::from_details_with_id(
        chunk_metadata.id,
        chunk_html,
        &link,
        &chunk_tag_set.or(chunk_metadata.tag_set),
        chunk_metadata.qdrant_point_id,
        <std::option::Option<serde_json::Value> as Clone>::clone(&update_chunk_data.metadata)
            .or(chunk_metadata.metadata),
        chunk_tracking_id,
        update_chunk_data
            .time_stamp
            .clone()
            .map(|ts| -> Result<NaiveDateTime, ServiceError> {
                //TODO: change all ts parsing to this crate
                Ok(ts
                    .parse::<DateTimeUtc>()
                    .map_err(|_| ServiceError::BadRequest("Invalid timestamp format".to_string()))?
                    .0
                    .with_timezone(&chrono::Local)
                    .naive_local())
            })
            .transpose()?
            .or(chunk_metadata.time_stamp),
        update_chunk_data.location.or(chunk_metadata.location),
        update_chunk_data.image_urls.clone().or(chunk_metadata
            .image_urls
            .map(|x| x.into_iter().map(|x| x.unwrap()).collect())),
        dataset_id,
        update_chunk_data.weight.unwrap_or(chunk_metadata.weight),
        update_chunk_data.num_value.or(chunk_metadata.num_value),
    );

    let group_ids = if let Some(group_ids) = update_chunk_data.group_ids.clone() {
        Some(
            group_ids
                .into_iter()
                .map(UnifiedId::from)
                .collect::<Vec<UnifiedId>>(),
        )
    } else {
        update_chunk_data
            .group_tracking_ids
            .clone()
            .map(|group_tracking_ids| {
                group_tracking_ids
                    .into_iter()
                    .map(UnifiedId::from)
                    .collect::<Vec<UnifiedId>>()
            })
    };

    let message = UpdateIngestionMessage {
        chunk_metadata: chunk_metadata.clone().into(),
        dataset_id,
        group_ids,
        convert_html_to_text: update_chunk_data.convert_html_to_text,
        fulltext_boost: update_chunk_data.fulltext_boost.clone(),
        semantic_boost: update_chunk_data.semantic_boost.clone(),
    };

    let mut redis_conn = redis_pool
        .get()
        .await
        .map_err(|err| ServiceError::BadRequest(err.to_string()))?;

    redis::cmd("lpush")
        .arg("ingestion")
        .arg(serde_json::to_string(&message)?)
        .query_async::<_, ()>(&mut *redis_conn)
        .await
        .map_err(|err| ServiceError::BadRequest(err.to_string()))?;

    Ok(HttpResponse::NoContent().finish())
}

#[derive(Serialize, Deserialize, Debug, Clone, ToSchema)]
pub struct UpdateChunkByTrackingIdData {
    /// Tracking_id of the chunk you want to update. This is required to match an existing chunk.
    tracking_id: String,
    /// Link of the chunk you want to update. This can also be any string. Frequently, this is a link to the source of the chunk. The link value will not affect the embedding creation. If no link is provided, the existing link will be used.
    link: Option<String>,
    /// HTML content of the chunk you want to update. This can also be plaintext. The innerText of the HTML will be used to create the embedding vector. The point of using HTML is for convienience, as some users have applications where users submit HTML content. If no chunk_html is provided, the existing chunk_html will be used.
    chunk_html: Option<String>,
    /// The metadata is a JSON object which can be used to filter chunks. This is useful for when you want to filter chunks by arbitrary metadata. Unlike with tag filtering, there is a performance hit for filtering on metadata. If no metadata is provided, the existing metadata will be used.
    metadata: Option<serde_json::Value>,
    /// Time_stamp should be an ISO 8601 combined date and time without timezone. It is used for time window filtering and recency-biasing search results. If no time_stamp is provided, the existing time_stamp will be used.
    time_stamp: Option<String>,
    /// Weight is a float which can be used to bias search results. This is useful for when you want to bias search results for a chunk. The magnitude only matters relative to other chunks in the chunk's dataset dataset. If no weight is provided, the existing weight will be used.
    weight: Option<f64>,
    /// Group ids are the ids of the groups that the chunk should be placed into. This is useful for when you want to update a chunk and add it to a group or multiple groups in one request.
    group_ids: Option<Vec<uuid::Uuid>>,
    /// Group tracking_ids are the tracking_ids of the groups that the chunk should be placed into. This is useful for when you want to update a chunk and add it to a group or multiple groups in one request.
    group_tracking_ids: Option<Vec<String>>,
    /// Convert HTML to raw text before processing to avoid adding noise to the vector embeddings. By default this is true. If you are using HTML content that you want to be included in the vector embeddings, set this to false.
    pub convert_html_to_text: Option<bool>,
}

/// Update Chunk By Tracking Id
///
/// Update a chunk by tracking_id. This is useful for when you are coordinating with an external system and want to use the tracking_id to identify the chunk. Auth'ed user or api key must have an admin or owner role for the specified dataset's organization.
#[utoipa::path(
    put,
    path = "/chunk/tracking_id/update",
    context_path = "/api",
    tag = "Chunk",
    request_body(content = UpdateChunkByTrackingIdData, description = "JSON request payload to update a chunk by tracking_id (chunks)", content_type = "application/json"),
    responses(
        (status = 204, description = "Confirmation that the chunk has been updated as per your request",),
        (status = 400, description = "Service error relating to to updating chunk", body = ErrorResponseBody),
    ),
    params(
        ("TR-Dataset" = String, Header, description = "The dataset id or tracking_id to use for the request. We assume you intend to use an id if the value is a valid uuid."),
    ),
    security(
        ("ApiKey" = ["admin"]),
    )
)]
#[deprecated]
#[tracing::instrument(skip(pool, redis_pool))]
pub async fn update_chunk_by_tracking_id(
    update_chunk_data: web::Json<UpdateChunkByTrackingIdData>,
    pool: web::Data<Pool>,
    redis_pool: web::Data<RedisPool>,
    _user: AdminOnly,
    dataset_org_plan_sub: DatasetAndOrgWithSubAndPlan,
) -> Result<HttpResponse, actix_web::Error> {
    if update_chunk_data.tracking_id.is_empty() {
        return Err(ServiceError::BadRequest(
            "Tracking id must be provided to update by tracking_id".into(),
        )
        .into());
    }
    let tracking_id = update_chunk_data.tracking_id.clone();

    let dataset_id = dataset_org_plan_sub.dataset.id;

    let chunk_metadata =
        get_metadata_from_tracking_id_query(tracking_id, dataset_id, pool.clone()).await?;

    let link = update_chunk_data
        .link
        .clone()
        .unwrap_or_else(|| chunk_metadata.link.clone().unwrap_or_default());

    let chunk_html = match update_chunk_data.chunk_html.clone() {
        Some(chunk_html) => Some(chunk_html),
        None => chunk_metadata.chunk_html,
    };

    let metadata = ChunkMetadata::from_details_with_id(
        chunk_metadata.id,
        chunk_html,
        &Some(link),
        &chunk_metadata.tag_set,
        chunk_metadata.qdrant_point_id,
        <std::option::Option<serde_json::Value> as Clone>::clone(&update_chunk_data.metadata)
            .or(chunk_metadata.metadata),
        Some(update_chunk_data.tracking_id.clone()),
        update_chunk_data
            .time_stamp
            .clone()
            .map(|ts| -> Result<NaiveDateTime, ServiceError> {
                //TODO: change all ts parsing to this crate
                Ok(ts
                    .parse::<DateTimeUtc>()
                    .map_err(|_| ServiceError::BadRequest("Invalid timestamp format".to_string()))?
                    .0
                    .with_timezone(&chrono::Local)
                    .naive_local())
            })
            .transpose()?
            .or(chunk_metadata.time_stamp),
        None,
        None,
        dataset_org_plan_sub.dataset.id,
        update_chunk_data.weight.unwrap_or(1.0),
        None,
    );
    let group_ids = if let Some(group_ids) = update_chunk_data.group_ids.clone() {
        Some(
            group_ids
                .into_iter()
                .map(UnifiedId::from)
                .collect::<Vec<UnifiedId>>(),
        )
    } else {
        update_chunk_data
            .group_tracking_ids
            .clone()
            .map(|group_tracking_ids| {
                group_tracking_ids
                    .into_iter()
                    .map(UnifiedId::from)
                    .collect::<Vec<UnifiedId>>()
            })
    };

    let message = UpdateIngestionMessage {
        chunk_metadata: metadata.clone().into(),
        dataset_id,
        group_ids,
        convert_html_to_text: update_chunk_data.convert_html_to_text,
        fulltext_boost: None,
        semantic_boost: None,
    };

    let mut redis_conn = redis_pool
        .get()
        .await
        .map_err(|err| ServiceError::BadRequest(err.to_string()))?;

    redis::cmd("lpush")
        .arg("ingestion")
        .arg(serde_json::to_string(&message)?)
        .query_async::<_, ()>(&mut *redis_conn)
        .await
        .map_err(|err| ServiceError::BadRequest(err.to_string()))?;

    Ok(HttpResponse::NoContent().finish())
}

#[derive(Serialize, Deserialize, Debug, Clone, ToSchema)]
#[schema(example = json!({
    "must": [
        {
            "field": "tag_set",
            "match_all": ["A", "B"],
        },
        {
            "field": "num_value",
            "range": {
                "gte": 10,
                "lte": 25,
            }
        }
    ]
}))]
/// Filters is a JSON object which can be used to filter chunks. This is useful for when you want to filter chunks by arbitrary metadata. Unlike with tag filtering, there is a performance hit for filtering on metadata.
pub struct ChunkFilter {
    /// Only one of these field conditions has to match for the chunk to be included in the result set.
    pub should: Option<Vec<ConditionType>>,
    /// All of these field conditions have to match for the chunk to be included in the result set.
    pub must: Option<Vec<ConditionType>>,
    /// None of these field conditions can match for the chunk to be included in the result set.
    pub must_not: Option<Vec<ConditionType>>,
    /// JOSNB prefilter tells the server to perform a full scan over the metadata JSONB column instead of using the filtered HNSW. Datasets on the enterprise plan with custom metadata indices will perform better with the filtered HNSW instead. When false, the server will use the filtered HNSW index to filter chunks. When true, the server will perform a full scan over the metadata JSONB column to filter chunks. Default is true.
    pub jsonb_prefilter: Option<bool>,
}

#[derive(Serialize, Clone, Debug, ToSchema)]
#[schema(example = json!({
    "search_type": "semantic",
    "query": "Some search query",
    "page": 1,
    "page_size": 10,
    "filters": {
        "should": [
            {
                "field": "metadata.key1",
                "match": ["value1", "value2"],
            }
        ],
        "must": [
            {
                "field": "num_value",
                "range": {
                    "gte": 0.0,
                    "lte": 1.0,
                    "gt": 0.0,
                    "lt": 1.0
                }
            }
        ],
        "must_not": [
            {
                "field": "metadata.key3",
                "match": ["value5", "value6"],
            }
        ]
    },
    "use_weights": true,
    "highlight_results": true,
    "highlight_delimiters": ["?", ",", ".", "!"],
    "score_threshold": 0.5
}))]
pub struct SearchChunksReqPayload {
    /// Can be either "semantic", "fulltext", "hybrid, or "bm25". If specified as "hybrid", it will pull in one page of both semantic and full-text results then re-rank them using scores from a cross encoder model. "semantic" will pull in one page of the nearest cosine distant vectors. "fulltext" will pull in one page of full-text results based on SPLADE. "bm25" will get one page of results scored using BM25 with the terms OR'd together.
    pub search_type: SearchMethod,
    /// Query is the search query. This can be any string. The query will be used to create an embedding vector and/or SPLADE vector which will be used to find the result set.  You can either provide one query, or multiple with weights. Multi-query only works with Semantic Search.
    pub query: QueryTypes,
    /// Page of chunks to fetch. Page is 1-indexed.
    pub page: Option<u64>,
    /// Page size is the number of chunks to fetch. This can be used to fetch more than 10 chunks at a time.
    pub page_size: Option<u64>,
    /// Get total page count for the query accounting for the applied filters. Defaults to false, but can be set to true when the latency penalty is acceptable (typically 50-200ms).
    pub get_total_pages: Option<bool>,
    /// Filters is a JSON object which can be used to filter chunks. This is useful for when you want to filter chunks by arbitrary metadata. Unlike with tag filtering, there is a performance hit for filtering on metadata.
    pub filters: Option<ChunkFilter>,
    /// Sort Options lets you specify different methods to rerank the chunks in the result set. If not specified, this defaults to the score of the chunks.
    pub sort_options: Option<SortOptions>,
    /// Scoring options provides ways to modify the sparse or dense vector created for the query in order to change how potential matches are scored. If not specified, this defaults to no modifications.
    pub scoring_options: Option<ScoringOptions>,
    /// Highlight Options lets you specify different methods to highlight the chunks in the result set. If not specified, this defaults to the score of the chunks.
    pub highlight_options: Option<HighlightOptions>,
    /// Set score_threshold to a float to filter out chunks with a score below the threshold for cosine distance metric. For Manhattan Distance, Euclidean Distance, and Dot Product, it will filter out scores above the threshold distance. This threshold applies before weight and bias modifications. If not specified, this defaults to no threshold. A threshold of 0 will default to no threshold.
    pub score_threshold: Option<f32>,
    /// Set slim_chunks to true to avoid returning the content and chunk_html of the chunks. This is useful for when you want to reduce amount of data over the wire for latency improvement (typically 10-50ms). Default is false.
    pub slim_chunks: Option<bool>,
    /// Set content_only to true to only returning the chunk_html of the chunks. This is useful for when you want to reduce amount of data over the wire for latency improvement (typically 10-50ms). Default is false.
    pub content_only: Option<bool>,
    /// If true, quoted and - prefixed words will be parsed from the queries and used as required and negated words respectively. Default is false.
    pub use_quote_negated_terms: Option<bool>,
    /// If true, stop words (specified in server/src/stop-words.txt in the git repo) will be removed. Queries that are entirely stop words will be preserved.
    pub remove_stop_words: Option<bool>,
    /// User ID is the id of the user who is making the request. This is used to track user interactions with the search results.
    pub user_id: Option<String>,
    /// Typo options lets you specify different methods to handle typos in the search query. If not specified, this defaults to no typo handling.
    pub typo_options: Option<TypoOptions>,
}

impl Default for SearchChunksReqPayload {
    fn default() -> Self {
        SearchChunksReqPayload {
            search_type: SearchMethod::Hybrid,
            query: QueryTypes::Single("".to_string()),
            page: Some(1),
            get_total_pages: None,
            page_size: Some(10),
            filters: None,
            sort_options: None,
            scoring_options: None,
            highlight_options: None,
            score_threshold: None,
            slim_chunks: None,
            content_only: None,
            use_quote_negated_terms: None,
            remove_stop_words: None,
            user_id: None,
            typo_options: None,
        }
    }
}

#[derive(Serialize, Deserialize, ToSchema, Debug, Clone)]
#[schema(title = "V1")]
pub struct SearchChunkQueryResponseBody {
    pub score_chunks: Vec<ScoreChunkDTO>,
    pub corrected_query: Option<String>,
    pub total_chunk_pages: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, ToSchema)]
#[schema(title = "V2")]
pub struct SearchResponseBody {
    pub id: uuid::Uuid,
    pub chunks: Vec<ScoreChunk>,
    pub corrected_query: Option<String>,
    pub total_pages: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, ToSchema)]
#[serde(untagged)]
pub enum SearchResponseTypes {
    #[schema(title = "V2")]
    V2(SearchResponseBody),
    #[schema(title = "V1")]
    V1(SearchChunkQueryResponseBody),
}

impl SearchChunkQueryResponseBody {
    fn into_v2(self, search_id: uuid::Uuid) -> SearchResponseBody {
        SearchResponseBody {
            id: search_id,
            chunks: self
                .score_chunks
                .into_iter()
                .map(|chunk| chunk.into())
                .collect(),
            corrected_query: self.corrected_query,
            total_pages: self.total_chunk_pages,
        }
    }
}

#[derive(Clone, Debug)]
pub struct ParsedQuery {
    pub query: String,
    pub quote_words: Option<Vec<String>>,
    pub negated_words: Option<Vec<String>>,
}

#[derive(Clone, Debug)]
pub enum ParsedQueryTypes {
    Single(ParsedQuery),
    Multi(Vec<(ParsedQuery, f32)>),
}

impl ParsedQueryTypes {
    pub fn to_parsed_query(&self) -> Result<ParsedQuery, ServiceError> {
        match self {
            ParsedQueryTypes::Single(query) => Ok(query.clone()),
            ParsedQueryTypes::Multi(_) => Err(ServiceError::BadRequest(
                "Cannot use Multi Query with cross encoder or highlights".to_string(),
            )),
        }
    }
}

pub fn parse_query(
    query: String,
    use_quote_negated_terms: Option<bool>,
    remove_stop_words: Option<bool>,
) -> ParsedQuery {
    let stop_words = get_stop_words();
    let query = match remove_stop_words {
        Some(true) => {
            let mut query_parts_split_by_stop_words: Vec<String> = Vec::new();
            let mut current_chunk: Vec<String> = Vec::new();
            for word in query.split(' ') {
                if !stop_words.contains(&word.to_lowercase()) {
                    current_chunk.push(word.to_string());
                } else if !current_chunk.is_empty() {
                    query_parts_split_by_stop_words.push(current_chunk.join(" "));
                    current_chunk.clear();
                }
            }
            if !current_chunk.is_empty() {
                query_parts_split_by_stop_words.push(current_chunk.join(" "));
            }
            let new_query = query_parts_split_by_stop_words.join(" ");
            match new_query.is_empty() {
                true => query,
                false => new_query,
            }
        }
        _ => query,
    };

    match use_quote_negated_terms {
        Some(true) => {
            let re = Regex::new(r#""(?:[^"\\]|\\.)*""#).expect("Regex pattern is always valid");
            let quote_words: Vec<String> = re
                .captures_iter(&query)
                .filter_map(|capture| capture.get(0).map(|capture| capture.as_str().to_string()))
                .filter(|word| !word.is_empty())
                .collect::<Vec<String>>();

            let quote_words = if quote_words.is_empty() {
                None
            } else {
                Some(quote_words)
            };

            let negated_words: Vec<String> = query
                .split_whitespace()
                .filter(|word| word.starts_with('-'))
                .map(|word| word.strip_prefix('-').unwrap().to_string())
                .collect::<Vec<String>>();

            let negated_words = if negated_words.is_empty() {
                None
            } else {
                Some(negated_words)
            };

            ParsedQuery {
                query,
                quote_words,
                negated_words,
            }
        }
        _ => ParsedQuery {
            query,
            quote_words: None,
            negated_words: None,
        },
    }
}

/// Search
///
/// This route provides the primary search functionality for the API. It can be used to search for chunks by semantic similarity, full-text similarity, or a combination of both. Results' `chunk_html` values will be modified with `<b><mark>` tags for sub-sentence highlighting.
#[utoipa::path(
    post,
    path = "/chunk/search",
    context_path = "/api",
    tag = "Chunk",
    request_body(content = SearchChunksReqPayload, description = "JSON request payload to semantically search for chunks (chunks)", content_type = "application/json"),
    responses(
        (status = 200, description = "Chunks with embedding vectors which are similar to those in the request body", body = SearchResponseTypes),
        (status = 400, description = "Service error relating to searching", body = ErrorResponseBody),
    ),
    params(
        ("TR-Dataset" = String, Header, description = "The dataset id or tracking_id to use for the request. We assume you intend to use an id if the value is a valid uuid."),
        ("X-API-Version" = Option<APIVersion>, Header, description = "The API version to use for this request. Defaults to V2 for orgs created after July 12, 2024 and V1 otherwise.")
    ),
    security(
        ("ApiKey" = ["readonly"]),
    )
)]
#[tracing::instrument(skip(pool, event_queue, redis_pool))]
pub async fn search_chunks(
    data: web::Json<SearchChunksReqPayload>,
    _user: LoggedUser,
    pool: web::Data<Pool>,
    event_queue: web::Data<EventQueue>,
    redis_pool: web::Data<RedisPool>,
    api_version: APIVersion,
    dataset_org_plan_sub: DatasetAndOrgWithSubAndPlan,
) -> Result<HttpResponse, actix_web::Error> {
    let dataset_config =
        DatasetConfiguration::from_json(dataset_org_plan_sub.dataset.server_configuration.clone());

    let mut data = data.into_inner();

    let parsed_query = match data.query.clone() {
        QueryTypes::Single(query) => ParsedQueryTypes::Single(parse_query(
            query.clone(),
            data.use_quote_negated_terms,
            data.remove_stop_words,
        )),
        QueryTypes::Multi(query) => ParsedQueryTypes::Multi(
            query
                .into_iter()
                .map(|multi_query| {
                    let parsed_query = parse_query(
                        multi_query.query.clone(),
                        data.use_quote_negated_terms,
                        data.remove_stop_words,
                    );
                    (parsed_query, multi_query.weight)
                })
                .collect::<Vec<(ParsedQuery, f32)>>(),
        ),
    };

    data.score_threshold = data.score_threshold.filter(|threshold| *threshold != 0.0);

    let tx_ctx = sentry::TransactionContext::new("search", "search_chunks");
    let transaction = sentry::start_transaction(tx_ctx);
    sentry::configure_scope(|scope| scope.set_span(Some(transaction.clone().into())));
    let mut timer = Timer::new();

    let result_chunks = match data.search_type {
        SearchMethod::Hybrid => {
            search_hybrid_chunks(
                data.clone(),
                parsed_query.to_parsed_query()?,
                pool,
                redis_pool,
                dataset_org_plan_sub.dataset.clone(),
                &dataset_config,
                &mut timer,
            )
            .await?
        }
        _ => {
            search_chunks_query(
                data.clone(),
                parsed_query,
                pool,
                redis_pool,
                dataset_org_plan_sub.dataset.clone(),
                &dataset_config,
                &mut timer,
            )
            .await?
        }
    };
    timer.add("search_chunks");

    let search_id = uuid::Uuid::new_v4();

    let query = match &data.query {
        QueryTypes::Single(query) => query.clone(),
        QueryTypes::Multi(query) => serde_json::to_string(&query).unwrap_or_default(),
    };

    let clickhouse_event = SearchQueryEventClickhouse {
        id: search_id,
        search_type: String::from("search"),
        query,
        request_params: serde_json::to_string(&data.clone()).unwrap_or_default(),
        latency: get_latency_from_header(timer.header_value()),
        top_score: result_chunks
            .score_chunks
            .first()
            .map(|x| x.score as f32)
            .unwrap_or(0.0),
        results: result_chunks
            .score_chunks
            .clone()
            .into_iter()
            .map(|x| serde_json::to_string(&x).unwrap_or_default())
            .collect(),
        dataset_id: dataset_org_plan_sub.dataset.id,
        created_at: time::OffsetDateTime::now_utc(),
        query_rating: String::from(""),
        user_id: data.user_id.clone().unwrap_or_default(),
    };

    event_queue
        .send(ClickHouseEvent::SearchQueryEvent(clickhouse_event))
        .await;

    timer.add("send_to_clickhouse");

    transaction.finish();

    if api_version == APIVersion::V2 {
        return Ok(HttpResponse::Ok()
            .insert_header((Timer::header_key(), timer.header_value()))
            .json(SearchResponseTypes::V2(result_chunks.into_v2(search_id))));
    }

    Ok(HttpResponse::Ok()
        .insert_header((Timer::header_key(), timer.header_value()))
        .json(result_chunks))
}

#[derive(Serialize, Clone, Debug, ToSchema)]
#[schema(example = json!({
    "search_type": "semantic",
    "query": "Some search query",
    "page": 1,
    "page_size": 10,
    "filters": {
        "should": [
            {
                "field": "metadata.key1",
                "match": ["value1", "value2"],
                "range": {
                    "gte": 0.0,
                    "lte": 1.0,
                    "gt": 0.0,
                    "lt": 1.0
                }
            }
        ],
        "must": [
            {
                "field": "metadata.key2",
                "match": ["value3", "value4"],
                "range": {
                    "gte": 0.0,
                    "lte": 1.0,
                    "gt": 0.0,
                    "lt": 1.0
                }
            }
        ],
        "must_not": [
            {
                "field": "metadata.key3",
                "match": ["value5", "value6"],
                "range": {
                    "gte": 0.0,
                    "lte": 1.0,
                    "gt": 0.0,
                    "lt": 1.0
                }
            }
        ]
    },
    "recency_bias": 1.0,
    "use_weights": true,
    "highlight_results": true,
    "highlight_delimiters": ["?", ",", ".", "!"],
    "score_threshold": 0.5
}))]
pub struct AutocompleteReqPayload {
    /// Can be either "semantic", or "fulltext". "semantic" will pull in one page_size of the nearest cosine distant vectors. "fulltext" will pull in one page_size of full-text results based on SPLADE. "bm25" will pull in one page_size of results based on the BM25 algorithim
    pub search_type: SearchMethod,
    /// If specified to true, this will extend the search results to include non-exact prefix matches of the same search_type such that a full page_size of results are returned. Default is false.
    pub extend_results: Option<bool>,
    /// Query is the search query. This can be any string. The query will be used to create an embedding vector and/or SPLADE vector which will be used to find the result set.
    pub query: String,
    /// Page size is the number of chunks to fetch. This can be used to fetch more than 10 chunks at a time.
    pub page_size: Option<u64>,
    /// Filters is a JSON object which can be used to filter chunks. This is useful for when you want to filter chunks by arbitrary metadata. Unlike with tag filtering, there is a performance hit for filtering on metadata.
    pub filters: Option<ChunkFilter>,
    /// Sort Options lets you specify different methods to rerank the chunks in the result set. If not specified, this defaults to the score of the chunks.
    pub sort_options: Option<SortOptions>,
    /// Scoring options provides ways to modify the sparse or dense vector created for the query in order to change how potential matches are scored. If not specified, this defaults to no modifications.
    pub scoring_options: Option<ScoringOptions>,
    /// Highlight Options lets you specify different methods to highlight the chunks in the result set. If not specified, this defaults to the score of the chunks.
    pub highlight_options: Option<HighlightOptions>,
    /// Set score_threshold to a float to filter out chunks with a score below the threshold. This threshold applies before weight and bias modifications. If not specified, this defaults to 0.0.
    pub score_threshold: Option<f32>,
    /// Set slim_chunks to true to avoid returning the content and chunk_html of the chunks. This is useful for when you want to reduce amount of data over the wire for latency improvement (typically 10-50ms). Default is false.
    pub slim_chunks: Option<bool>,
    /// Set content_only to true to only returning the chunk_html of the chunks. This is useful for when you want to reduce amount of data over the wire for latency improvement (typically 10-50ms). Default is false.
    pub content_only: Option<bool>,
    /// If true, quoted and - prefixed words will be parsed from the queries and used as required and negated words respectively. Default is false.
    pub use_quote_negated_terms: Option<bool>,
    /// If true, stop words (specified in server/src/stop-words.txt in the git repo) will be removed. Queries that are entirely stop words will be preserved.
    pub remove_stop_words: Option<bool>,
    /// User ID is the id of the user who is making the request. This is used to track user interactions with the search results.
    pub user_id: Option<String>,
    pub typo_options: Option<TypoOptions>,
}

impl From<AutocompleteReqPayload> for SearchChunksReqPayload {
    fn from(autocomplete_data: AutocompleteReqPayload) -> Self {
        SearchChunksReqPayload {
            search_type: autocomplete_data.search_type,
            query: QueryTypes::Single(autocomplete_data.query),
            page: Some(1),
            get_total_pages: None,
            page_size: autocomplete_data.page_size,
            filters: autocomplete_data.filters,
            sort_options: autocomplete_data.sort_options,
            scoring_options: autocomplete_data.scoring_options,
            highlight_options: autocomplete_data.highlight_options,
            score_threshold: autocomplete_data.score_threshold,
            slim_chunks: autocomplete_data.slim_chunks,
            content_only: autocomplete_data.content_only,
            use_quote_negated_terms: autocomplete_data.use_quote_negated_terms,
            remove_stop_words: autocomplete_data.remove_stop_words,
            user_id: autocomplete_data.user_id,
            typo_options: autocomplete_data.typo_options,
        }
    }
}

/// Autocomplete
///
/// This route provides the primary autocomplete functionality for the API. This prioritize prefix matching with semantic or full-text search.
#[utoipa::path(
    post,
    path = "/chunk/autocomplete",
    context_path = "/api",
    tag = "Chunk",
    request_body(content = AutocompleteReqPayload, description = "JSON request payload to semantically search for chunks (chunks)", content_type = "application/json"),
    responses(
        (status = 200, description = "Chunks with embedding vectors which are similar to those in the request body", body = SearchResponseTypes),
        (status = 400, description = "Service error relating to searching", body = ErrorResponseBody),
    ),
    params(
        ("TR-Dataset" = String, Header, description = "The dataset id or tracking_id to use for the request. We assume you intend to use an id if the value is a valid uuid."),
        ("X-API-Version" = Option<APIVersion>, Header, description = "The API version to use for this request. Defaults to V2 for orgs created after July 12, 2024 and V1 otherwise.")
    ),
    security(
        ("ApiKey" = ["readonly"]),
    )
)]
#[tracing::instrument(skip(pool, event_queue, redis_pool))]
pub async fn autocomplete(
    data: web::Json<AutocompleteReqPayload>,
    _user: LoggedUser,
    pool: web::Data<Pool>,
    event_queue: web::Data<EventQueue>,
    redis_pool: web::Data<RedisPool>,
    api_version: APIVersion,
    dataset_org_plan_sub: DatasetAndOrgWithSubAndPlan,
) -> Result<HttpResponse, actix_web::Error> {
    let dataset_config =
        DatasetConfiguration::from_json(dataset_org_plan_sub.dataset.server_configuration.clone());

    let parsed_query = parse_query(
        data.query.clone(),
        data.use_quote_negated_terms,
        data.remove_stop_words,
    );

    let tx_ctx = sentry::TransactionContext::new("search", "search_chunks");
    let transaction = sentry::start_transaction(tx_ctx);
    sentry::configure_scope(|scope| scope.set_span(Some(transaction.clone().into())));

    let mut timer = Timer::new();

    let result_chunks = autocomplete_chunks_query(
        data.clone(),
        parsed_query,
        pool,
        redis_pool,
        dataset_org_plan_sub.dataset.clone(),
        &dataset_config,
        &mut timer,
    )
    .await?;

    timer.add("autocomplete_chunks");

    let search_id = uuid::Uuid::new_v4();

    let clickhouse_event = SearchQueryEventClickhouse {
        id: search_id,
        search_type: String::from("autocomplete"),
        query: data.query.clone(),
        request_params: serde_json::to_string(&data.clone()).unwrap_or_default(),
        latency: get_latency_from_header(timer.header_value()),
        top_score: result_chunks
            .score_chunks
            .first()
            .map(|x| x.score as f32)
            .unwrap_or(0.0),
        results: result_chunks
            .score_chunks
            .clone()
            .into_iter()
            .map(|x| serde_json::to_string(&x).unwrap_or_default())
            .collect(),
        dataset_id: dataset_org_plan_sub.dataset.id,
        created_at: time::OffsetDateTime::now_utc(),
        query_rating: String::from(""),
        user_id: data.user_id.clone().unwrap_or_default(),
    };

    event_queue
        .send(ClickHouseEvent::SearchQueryEvent(clickhouse_event))
        .await;

    timer.add("send_to_clickhouse");

    transaction.finish();

    if api_version == APIVersion::V2 {
        return Ok(HttpResponse::Ok()
            .insert_header((Timer::header_key(), timer.header_value()))
            .json(SearchResponseTypes::V2(result_chunks.into_v2(search_id))));
    }

    Ok(HttpResponse::Ok()
        .insert_header((Timer::header_key(), timer.header_value()))
        .json(result_chunks))
}

#[derive(Serialize, Deserialize, Clone, Debug, ToSchema)]
#[serde(untagged)]
pub enum ChunkReturnTypes {
    V2(ChunkMetadata),
    V1(ChunkMetadataStringTagSet),
}

#[derive(Serialize, Deserialize, Debug, Clone, ToSchema)]
pub struct ScrollChunksResponseBody {
    pub chunks: Vec<ChunkMetadata>,
}

#[derive(Serialize, Deserialize, Debug, Clone, ToSchema)]
pub struct ScrollChunksReqPayload {
    /// Page size is the number of chunks to fetch. This can be used to fetch more than 10 chunks at a time.
    pub page_size: Option<u64>,
    /// Offset chunk id is the id of the chunk to start the page from. If not specified, this defaults to the first chunk in the dataset sorted by id ascending.
    pub offset_chunk_id: Option<uuid::Uuid>,
    /// Get total page count for the query accounting for the applied filters. Defaults to false, but can be set to true when the latency penalty is acceptable (typically 50-200ms).
    pub filters: Option<ChunkFilter>,
    /// Sort by lets you specify a key to sort the results by. If not specified, this defaults to the id's of the chunks. If specified, the field can be num_value, time_stamp, or any key in the chunk metadata. This key must be a numeric value within the payload.
    pub sort_by: Option<SortByField>,
}

/// Scroll Chunks
///
/// Get paginated chunks from your dataset with filters and custom sorting. If sort by is not specified, the results will sort by the id's of the chunks in ascending order. Sort by and offset_chunk_id cannot be used together; if you want to scroll with a sort by then you need to use a must_not filter with the ids you have already seen. There is a limit of 1000 id's in a must_not filter at a time.
#[utoipa::path(
    post,
    path = "/chunks/scroll",
    context_path = "/api",
    tag = "Chunk",
    request_body(content = ScrollChunksReqPayload, description = "JSON request payload to scroll through chunks (chunks)", content_type = "application/json"),
    responses(
        (status = 200, description = "Number of chunks equivalent to page_size starting from offset_chunk_id", body = ScrollChunksResponseBody),
        (status = 400, description = "Service error relating to scrolling chunks", body = ErrorResponseBody)
    ),
    params(
        ("TR-Dataset" = String, Header, description = "The dataset id or tracking_id to use for the request. We assume you intend to use an id if the value is a valid uuid."),
    ),
    security(
        ("ApiKey" = ["readonly"]),
    )
)]
#[tracing::instrument(skip(pool))]
pub async fn scroll_dataset_chunks(
    data: web::Json<ScrollChunksReqPayload>,
    _user: LoggedUser,
    dataset_org_plan_sub: DatasetAndOrgWithSubAndPlan,
    pool: web::Data<Pool>,
) -> Result<HttpResponse, actix_web::Error> {
    let dataset_config =
        DatasetConfiguration::from_json(dataset_org_plan_sub.dataset.server_configuration.clone());
    let dataset_id = dataset_org_plan_sub.dataset.id;
    let filters = data.filters.clone();

    let filter = assemble_qdrant_filter(filters, None, None, dataset_id, pool.clone()).await?;

    let qdrant_point_id_of_offset_chunk = match data.offset_chunk_id {
        Some(offset_chunk_id) => {
            let chunk =
                get_metadata_from_id_query(offset_chunk_id, dataset_id, pool.clone()).await?;
            Some(chunk.qdrant_point_id)
        }
        None => None,
    };

    let qdrant_point_ids = scroll_dataset_points(
        data.page_size.unwrap_or(10),
        qdrant_point_id_of_offset_chunk,
        data.sort_by.clone(),
        dataset_config,
        filter,
    )
    .await?;

    let chunks: Vec<ChunkMetadata> =
        get_chunk_metadatas_from_point_ids(qdrant_point_ids.clone(), pool)
            .await?
            .into_iter()
            .map(ChunkMetadata::from)
            .collect();

    let ordered_chunks = qdrant_point_ids
        .into_iter()
        .filter_map(|point_id| {
            chunks
                .iter()
                .find(|chunk| chunk.qdrant_point_id == point_id)
                .cloned()
        })
        .collect();

    let resp = ScrollChunksResponseBody {
        chunks: ordered_chunks,
    };

    Ok(HttpResponse::Ok().json(resp))
}

/// Get Chunk By Id
///
/// Get a singular chunk by id.
#[utoipa::path(
    get,
    path = "/chunk/{chunk_id}",
    context_path = "/api",
    tag = "Chunk",
    responses(
        (status = 200, description = "chunk with the id that you were searching for", body = ChunkReturnTypes),
        (status = 400, description = "Service error relating to fidning a chunk by tracking_id", body = ErrorResponseBody),
        (status = 404, description = "Chunk not found", body = ErrorResponseBody)
    ),
    params(
        ("TR-Dataset" = String, Header, description = "The dataset id or tracking_id to use for the request. We assume you intend to use an id if the value is a valid uuid."),
        ("X-API-Version" = Option<APIVersion>, Header, description = "The API version to use for this request. Defaults to V2 for orgs created after July 12, 2024 and V1 otherwise."),
        ("chunk_id" = Option<uuid::Uuid>, Path, description = "Id of the chunk you want to fetch."),
    ),
    security(
        ("ApiKey" = ["readonly"]),
    )
)]
#[tracing::instrument(skip(pool))]
pub async fn get_chunk_by_id(
    chunk_id: web::Path<uuid::Uuid>,
    _user: LoggedUser,
    pool: web::Data<Pool>,
    api_version: APIVersion,
    dataset_org_plan_sub: DatasetAndOrgWithSubAndPlan,
) -> Result<HttpResponse, ServiceError> {
    let chunk_id = chunk_id.into_inner();

    let dataset_configuration =
        DatasetConfiguration::from_json(dataset_org_plan_sub.dataset.server_configuration.clone());

    let chunk = get_metadata_from_id_query(chunk_id, dataset_org_plan_sub.dataset.id, pool).await?;
    let chunk_string_tag_set = ChunkMetadataStringTagSet::from(chunk);

    let point_id = chunk_string_tag_set.qdrant_point_id;
    let pointid_exists = point_ids_exists_in_qdrant(vec![point_id], dataset_configuration).await?;

    if pointid_exists {
        if api_version == APIVersion::V2 {
            return Ok(HttpResponse::Ok().json(ChunkMetadata::from(chunk_string_tag_set)));
        }
        Ok(HttpResponse::Ok().json(chunk_string_tag_set))
    } else {
        Err(ServiceError::NotFound("Chunk not found".to_string()))
    }
}

#[derive(Serialize, Deserialize, Clone, Debug, ToSchema)]
#[schema(example = json!({
    "search_type": "semantic",
    "query": "Some search query",
    "score_threshold": 0.5
}))]
pub struct CountChunksReqPayload {
    /// Can be either "semantic", "fulltext", or "bm25". "hybrid" is not supported due to latency limitations with using the reranker. These search types are applied without the reranker cross-encoder model, so if you are using it be aware that the count may not directly correlate with an actual search query.
    pub search_type: CountSearchMethod,
    /// Query is the search query. This can be any string. The query will be used to create an embedding vector and/or SPLADE vector which will be used to find the result set.
    pub query: QueryTypes,
    /// Filters is a JSON object which can be used to filter chunks. This is useful for when you want to filter chunks by arbitrary metadata. Unlike with tag filtering, there is a performance hit for filtering on metadata.
    pub filters: Option<ChunkFilter>,
    /// Set score_threshold to a float to filter out chunks with a score below the threshold. This threshold applies before weight and bias modifications. If not specified, this defaults to 0.0.
    pub score_threshold: Option<f32>,
    /// Set limit to restrict the maximum number of chunks to count. This is useful for when you want to reduce the latency of the count operation. By default the limit will be the number of chunks in the dataset.
    pub limit: Option<u64>,
    /// If true, quoted and - prefixed words will be parsed from the queries and used as required and negated words respectively. Default is false.
    pub use_quote_negated_terms: Option<bool>,
}

impl From<CountChunksReqPayload> for SearchChunksReqPayload {
    fn from(count_data: CountChunksReqPayload) -> Self {
        SearchChunksReqPayload {
            search_type: count_data.search_type.into(),
            query: count_data.query,
            page: Some(1),
            get_total_pages: None,
            page_size: count_data.limit,
            filters: count_data.filters,
            sort_options: None,
            scoring_options: None,
            highlight_options: None,
            score_threshold: count_data.score_threshold,
            slim_chunks: None,
            content_only: None,
            use_quote_negated_terms: count_data.use_quote_negated_terms,
            remove_stop_words: None,
            user_id: None,
            typo_options: None,
        }
    }
}

#[derive(Serialize, Deserialize, Clone, Debug, ToSchema)]
pub struct CountChunkQueryResponseBody {
    pub count: u32,
}

/// Count chunks above threshold
///
/// This route can be used to determine the number of chunk results that match a search query including score threshold and filters. It may be high latency for large limits. There is a dataset configuration imposed restriction on the maximum limit value (default 10,000) which is used to prevent DDOS attacks. Auth'ed user or api key must have an admin or owner role for the specified dataset's organization.
#[utoipa::path(
    post,
    path = "/chunk/count",
    context_path = "/api",
    tag = "Chunk",
    request_body(content = CountChunksReqPayload, description = "JSON request payload to count chunks for a search query", content_type = "application/json"),
    responses(
        (status = 200, description = "Number of chunks satisfying the query", body = CountChunkQueryResponseBody),
        (status = 404, description = "Failed to count chunks", body = ErrorResponseBody)
    ),
    params(
        ("TR-Dataset" = String, Header, description = "The dataset id or tracking_id to use for the request. We assume you intend to use an id if the value is a valid uuid."),
    ),
    security(
        ("ApiKey" = ["readonly"]),
    )
)]
#[tracing::instrument(skip(pool))]
pub async fn count_chunks(
    data: web::Json<CountChunksReqPayload>,
    _user: LoggedUser,
    pool: web::Data<Pool>,
    dataset_org_plan_sub: DatasetAndOrgWithSubAndPlan,
) -> Result<HttpResponse, actix_web::Error> {
    let dataset_config =
        DatasetConfiguration::from_json(dataset_org_plan_sub.dataset.server_configuration.clone());

    let parsed_query = match data.query.clone() {
        QueryTypes::Single(query) => ParsedQueryTypes::Single(parse_query(
            query.clone(),
            data.use_quote_negated_terms,
            None,
        )),
        QueryTypes::Multi(query) => ParsedQueryTypes::Multi(
            query
                .into_iter()
                .map(|multi_query| {
                    let parsed_query = parse_query(
                        multi_query.query.clone(),
                        data.use_quote_negated_terms,
                        None,
                    );
                    (parsed_query, multi_query.weight)
                })
                .collect::<Vec<(ParsedQuery, f32)>>(),
        ),
    };

    let limit = match data.limit {
        Some(limit) => limit,
        None => {
            let dataset_usage =
                get_dataset_usage_query(dataset_org_plan_sub.dataset.id, pool.clone()).await?;
            dataset_usage.chunk_count as u64
        }
    };

    let search_req_data = CountChunksReqPayload {
        limit: Some(limit),
        ..data.clone()
    };

    if limit > dataset_config.MAX_LIMIT {
        return Err(ServiceError::BadRequest(format!(
            "Limit of {} is greater than the maximum limit of {}. Please reduce the limit.",
            limit, dataset_config.MAX_LIMIT
        ))
        .into());
    }

    let result_chunks = count_chunks_query(
        search_req_data.clone(),
        parsed_query,
        pool,
        dataset_org_plan_sub.dataset.clone(),
        &dataset_config,
    )
    .await?;

    Ok(HttpResponse::Ok().json(result_chunks))
}

/// Get Chunk By Tracking Id
///
/// Get a singular chunk by tracking_id. This is useful for when you are coordinating with an external system and want to use your own id as the primary reference for a chunk.
#[utoipa::path(
    get,
    path = "/chunk/tracking_id/{tracking_id}",
    context_path = "/api",
    tag = "Chunk",
    responses(
        (status = 200, description = "chunk with the tracking_id that you were searching for", body = ChunkReturnTypes),
        (status = 400, description = "Service error relating to fidning a chunk by tracking_id", body = ErrorResponseBody),
        (status = 404, description = "Chunk not found", body = ErrorResponseBody)
    ),
    params(
        ("TR-Dataset" = String, Header, description = "The dataset id or tracking_id to use for the request. We assume you intend to use an id if the value is a valid uuid."),
        ("X-API-Version" = Option<APIVersion>, Header, description = "The API version to use for this request. Defaults to V2 for orgs created after July 12, 2024 and V1 otherwise."),
        ("tracking_id" = Option<String>, Path, description = "tracking_id of the chunk you want to fetch"),
    ),
    security(
        ("ApiKey" = ["readonly"]),
    )
)]
#[tracing::instrument(skip(pool))]
pub async fn get_chunk_by_tracking_id(
    tracking_id: web::Path<String>,
    _user: LoggedUser,
    pool: web::Data<Pool>,
    api_version: APIVersion,
    dataset_org_plan_sub: DatasetAndOrgWithSubAndPlan,
) -> Result<HttpResponse, ServiceError> {
    let dataset_configuration =
        DatasetConfiguration::from_json(dataset_org_plan_sub.dataset.server_configuration.clone());

    let chunk = get_metadata_from_tracking_id_query(
        tracking_id.into_inner(),
        dataset_org_plan_sub.dataset.id,
        pool,
    )
    .await?;
    let chunk_tag_set_string = ChunkMetadataStringTagSet::from(chunk);

    let point_id = chunk_tag_set_string.qdrant_point_id;

    let pointid_exists = point_ids_exists_in_qdrant(vec![point_id], dataset_configuration).await?;

    if pointid_exists {
        if api_version == APIVersion::V2 {
            return Ok(HttpResponse::Ok().json(ChunkMetadata::from(chunk_tag_set_string)));
        }
        Ok(HttpResponse::Ok().json(chunk_tag_set_string))
    } else {
        Err(ServiceError::NotFound("Chunk not found".to_string()))
    }
}

#[derive(Serialize, Deserialize, Debug, ToSchema)]
pub struct GetChunksData {
    pub ids: Vec<uuid::Uuid>,
}

/// Get Chunks By Ids
///
/// Get multiple chunks by multiple ids.
#[utoipa::path(
    post,
    path = "/chunks",
    context_path = "/api",
    tag = "Chunk",
    request_body(content = GetChunksData, description = "JSON request payload to get the chunks in the request", content_type = "application/json"),
    responses(
        (status = 200, description = "chunks with the id that you were searching for", body = Vec<ChunkReturnTypes>),
        (status = 400, description = "Service error relating to fidning a chunk by tracking_id", body = ErrorResponseBody),
        (status = 404, description = "Any one of the specified chunks not found", body = ErrorResponseBody)
    ),
    params(
        ("TR-Dataset" = String, Header, description = "The dataset id or tracking_id to use for the request. We assume you intend to use an id if the value is a valid uuid."),
        ("X-API-Version" = Option<APIVersion>, Header, description = "The API version to use for this request. Defaults to V2 for orgs created after July 12, 2024 and V1 otherwise.")
    ),
    security(
        ("ApiKey" = ["readonly"]),
    )
)]
#[tracing::instrument(skip(pool))]
pub async fn get_chunks_by_ids(
    chunk_payload: web::Json<GetChunksData>,
    _user: LoggedUser,
    pool: web::Data<Pool>,
    api_version: APIVersion,
    dataset_org_plan_sub: DatasetAndOrgWithSubAndPlan,
) -> Result<HttpResponse, ServiceError> {
    let dataset_configuration =
        DatasetConfiguration::from_json(dataset_org_plan_sub.dataset.server_configuration.clone());

    let chunks = get_metadata_from_ids_query(
        chunk_payload.ids.clone(),
        dataset_org_plan_sub.dataset.id,
        pool,
    )
    .await?;
    let chunk_string_tag_sets = chunks
        .into_iter()
        .map(ChunkMetadataStringTagSet::from)
        .collect::<Vec<ChunkMetadataStringTagSet>>();

    let point_ids = chunk_string_tag_sets
        .iter()
        .map(|x| x.qdrant_point_id)
        .collect();

    let pointids_exists = point_ids_exists_in_qdrant(point_ids, dataset_configuration).await?;

    if pointids_exists {
        if api_version == APIVersion::V2 {
            return Ok(HttpResponse::Ok().json(
                chunk_string_tag_sets
                    .into_iter()
                    .map(ChunkMetadata::from)
                    .collect::<Vec<ChunkMetadata>>(),
            ));
        }
        Ok(HttpResponse::Ok().json(chunk_string_tag_sets))
    } else {
        Err(ServiceError::NotFound(
            "Any one of the specified chunks not found".to_string(),
        ))
    }
}

#[derive(Serialize, Deserialize, Debug, ToSchema)]
pub struct GetTrackingChunksData {
    pub tracking_ids: Vec<String>,
}

/// Get Chunks By Tracking Ids
///
/// Get multiple chunks by ids.
#[utoipa::path(
    post,
    path = "/chunks/tracking",
    context_path = "/api",
    tag = "Chunk",
    request_body(content = GetTrackingChunksData, description = "JSON request payload to get the chunks in the request", content_type = "application/json"),
    responses(
        (status = 200, description = "Chunks with one the ids which were specified", body = Vec<ChunkReturnTypes>),
        (status = 400, description = "Service error relating to finding a chunk by tracking_id", body = ErrorResponseBody),
    ),
    params(
        ("TR-Dataset" = String, Header, description = "The dataset id or tracking_id to use for the request. We assume you intend to use an id if the value is a valid uuid."),
        ("X-API-Version" = Option<APIVersion>, Header, description = "The API version to use for this request. Defaults to V2 for orgs created after July 12, 2024 and V1 otherwise.")
    ),
    security(
        ("ApiKey" = ["readonly"]),
    )
)]
#[tracing::instrument(skip(pool))]
pub async fn get_chunks_by_tracking_ids(
    chunk_payload: web::Json<GetTrackingChunksData>,
    _user: LoggedUser,
    pool: web::Data<Pool>,
    api_version: APIVersion,
    dataset_org_plan_sub: DatasetAndOrgWithSubAndPlan,
) -> Result<HttpResponse, ServiceError> {
    let dataset_configuration =
        DatasetConfiguration::from_json(dataset_org_plan_sub.dataset.server_configuration.clone());

    let chunks = get_metadata_from_tracking_ids_query(
        chunk_payload.tracking_ids.clone(),
        dataset_org_plan_sub.dataset.id,
        pool,
    )
    .await?;
    let chunk_string_tag_sets = chunks
        .into_iter()
        .map(ChunkMetadataStringTagSet::from)
        .collect::<Vec<ChunkMetadataStringTagSet>>();

    let point_ids = chunk_string_tag_sets
        .iter()
        .map(|x| x.qdrant_point_id)
        .collect();

    let pointids_exists = point_ids_exists_in_qdrant(point_ids, dataset_configuration).await?;

    if pointids_exists {
        if api_version == APIVersion::V2 {
            return Ok(HttpResponse::Ok().json(
                chunk_string_tag_sets
                    .into_iter()
                    .map(ChunkMetadata::from)
                    .collect::<Vec<ChunkMetadata>>(),
            ));
        }
        Ok(HttpResponse::Ok().json(chunk_string_tag_sets))
    } else {
        Err(ServiceError::NotFound(
            "Any one of the specified chunks not found".to_string(),
        ))
    }
}

#[derive(Serialize, Deserialize, Debug, ToSchema, Clone)]
pub struct RecommendChunksRequest {
    /// The ids of the chunks to be used as positive examples for the recommendation. The chunks in this array will be used to find similar chunks.
    pub positive_chunk_ids: Option<Vec<uuid::Uuid>>,
    /// The ids of the chunks to be used as negative examples for the recommendation. The chunks in this array will be used to filter out similar chunks.
    pub negative_chunk_ids: Option<Vec<uuid::Uuid>>,
    /// The tracking_ids of the chunks to be used as positive examples for the recommendation. The chunks in this array will be used to find similar chunks.
    pub positive_tracking_ids: Option<Vec<String>>,
    /// The tracking_ids of the chunks to be used as negative examples for the recommendation. The chunks in this array will be used to filter out similar chunks.
    pub negative_tracking_ids: Option<Vec<String>>,
    /// Strategy to use for recommendations, either "average_vector" or "best_score". The default is "average_vector". The "average_vector" strategy will construct a single average vector from the positive and negative samples then use it to perform a pseudo-search (You must provide at least one of either positive_chunk_ids or positive_tracking_ids for `average_vector` strategy). The "best_score" strategy is more advanced and navigates the HNSW with a heuristic of picking edges where the point is closer to the positive samples than it is the negatives for best score you can provide a list of only negatives or only positives or both.
    pub strategy: Option<RecommendationStrategy>,
    /// The type of recommendation to make. This lets you choose whether to recommend based off of `semantic` or `fulltext` similarity. The default is `semantic`.
    pub recommend_type: Option<RecommendType>,
    /// Filters to apply to the chunks to be recommended. This is a JSON object which contains the filters to apply to the chunks to be recommended. The default is None.
    pub filters: Option<ChunkFilter>,
    /// The number of chunks to return. This is the number of chunks which will be returned in the response. The default is 10.
    pub limit: Option<u64>,
    /// Set slim_chunks to true to avoid returning the content and chunk_html of the chunks. This is useful for when you want to reduce amount of data over the wire for latency improvement (typicall 10-50ms). Default is false.
    pub slim_chunks: Option<bool>,
    /// User ID is the id of the user who is making the request. This is used to track user interactions with the recommendation results.
    pub user_id: Option<String>,
}

#[derive(Serialize, Deserialize, Debug, Clone, ToSchema)]
#[schema(title = "V2")]
pub struct RecommendChunksResponseBody {
    pub id: uuid::Uuid,
    pub chunks: Vec<ScoreChunk>,
}

#[derive(Serialize, Deserialize, Debug, ToSchema)]
#[serde(untagged)]
pub enum RecommendResponseTypes {
    #[schema(title = "V2")]
    V2(RecommendChunksResponseBody),
    #[schema(title = "V1")]
    V1(Vec<ChunkMetadataWithScore>),
}

/// Get Recommended Chunks
///
/// Get recommendations of chunks similar to the positive samples in the request and dissimilar to the negative.
#[utoipa::path(
    post,
    path = "/chunk/recommend",
    context_path = "/api",
    tag = "Chunk",
    request_body(content = RecommendChunksRequest, description = "JSON request payload to get recommendations of chunks similar to the chunks in the request", content_type = "application/json"),
    responses(
        (status = 200, description = "Chunks with embedding vectors which are similar to positives and dissimilar to negatives", body = RecommendResponseTypes),
        (status = 400, description = "Service error relating to to getting similar chunks", body = ErrorResponseBody),
    ),
    params(
        ("TR-Dataset" = String, Header, description = "The dataset id or tracking_id to use for the request. We assume you intend to use an id if the value is a valid uuid."),
        ("X-API-Version" = Option<APIVersion>, Header, description = "The API version to use for this request. Defaults to V2 for orgs created after July 12, 2024 and V1 otherwise.")
    ),
    security(
        ("ApiKey" = ["readonly"]),
    )
)]
#[tracing::instrument(skip(pool, event_queue))]
pub async fn get_recommended_chunks(
    data: web::Json<RecommendChunksRequest>,
    pool: web::Data<Pool>,
    _user: LoggedUser,
    event_queue: web::Data<EventQueue>,
    api_version: APIVersion,
    dataset_org_plan_sub: DatasetAndOrgWithSubAndPlan,
) -> Result<HttpResponse, actix_web::Error> {
    let positive_chunk_ids = data.positive_chunk_ids.clone();
    let negative_chunk_ids = data.negative_chunk_ids.clone();
    let positive_tracking_ids = data.positive_tracking_ids.clone();
    let negative_tracking_ids = data.negative_tracking_ids.clone();
    let limit = data.limit.unwrap_or(10);
    let dataset_config =
        DatasetConfiguration::from_json(dataset_org_plan_sub.dataset.server_configuration);

    if positive_chunk_ids.is_none()
        && positive_tracking_ids.is_none()
        && data
            .strategy
            .clone()
            .unwrap_or(RecommendationStrategy::AverageVector)
            == RecommendationStrategy::AverageVector
    {
        return Err(ServiceError::BadRequest(
            "Either positive_chunk_ids or positive_tracking_ids must be provided for average_vector strategy".to_string(),
        )
        .into());
    }

    let dataset_id = dataset_org_plan_sub.dataset.id;

    let mut positive_qdrant_ids = vec![];

    let mut timer = Timer::new();

    timer.add("start extending tracking_ids and chunk_ids to qdrant_point_ids");

    if let Some(positive_chunk_ids) = positive_chunk_ids.clone() {
        positive_qdrant_ids.extend(
            get_point_ids_from_unified_chunk_ids(
                positive_chunk_ids
                    .into_iter()
                    .map(UnifiedId::TrieveUuid)
                    .collect(),
                dataset_id,
                pool.clone(),
            )
            .await
            .map_err(|err| {
                ServiceError::BadRequest(format!(
                    "Could not get positive qdrant_point_ids from positive_chunk_ids: {}",
                    err
                ))
            })?,
        )
    }

    if let Some(positive_chunk_tracking_ids) = positive_tracking_ids.clone() {
        positive_qdrant_ids.extend(
            get_point_ids_from_unified_chunk_ids(
                positive_chunk_tracking_ids
                    .into_iter()
                    .map(UnifiedId::TrackingId)
                    .collect(),
                dataset_id,
                pool.clone(),
            )
            .await
            .map_err(|err| {
                ServiceError::BadRequest(format!(
                    "Could not get positive qdrant_point_ids from positive_tracking_ids: {}",
                    err
                ))
            })?,
        )
    }

    let mut negative_qdrant_ids = vec![];

    if let Some(negative_chunk_ids) = negative_chunk_ids.clone() {
        negative_qdrant_ids.extend(
            get_point_ids_from_unified_chunk_ids(
                negative_chunk_ids
                    .into_iter()
                    .map(UnifiedId::TrieveUuid)
                    .collect(),
                dataset_id,
                pool.clone(),
            )
            .await
            .map_err(|err| {
                ServiceError::BadRequest(format!(
                    "Could not get negative qdrant_point_ids from negative_chunk_ids: {}",
                    err
                ))
            })?,
        )
    }

    if let Some(negative_chunk_tracking_ids) = negative_tracking_ids.clone() {
        negative_qdrant_ids.extend(
            get_point_ids_from_unified_chunk_ids(
                negative_chunk_tracking_ids
                    .into_iter()
                    .map(UnifiedId::TrackingId)
                    .collect(),
                dataset_id,
                pool.clone(),
            )
            .await
            .map_err(|err| {
                ServiceError::BadRequest(format!(
                    "Could not get negative qdrant_point_ids from negative_tracking_ids: {}",
                    err
                ))
            })?,
        )
    }

    if positive_qdrant_ids.is_empty() && negative_qdrant_ids.is_empty() {
        return Err(
            ServiceError::BadRequest("Positive chunk ids could not be found".to_string()).into(),
        );
    }

    timer.add("fetched ids from postgres");

    let recommended_qdrant_results = recommend_qdrant_query(
        positive_qdrant_ids,
        negative_qdrant_ids,
        data.strategy.clone(),
        data.recommend_type.clone(),
        data.filters.clone(),
        limit,
        dataset_org_plan_sub.dataset.id,
        dataset_config,
        pool.clone(),
    )
    .await
    .map_err(|err| {
        ServiceError::BadRequest(format!("Could not get recommended chunks: {}", err))
    })?;

    timer.add("recommend_qdrant_query");

    let recommended_chunk_metadatas = match data.slim_chunks {
        Some(true) => get_slim_chunks_from_point_ids_query(
            recommended_qdrant_results
                .clone()
                .into_iter()
                .map(|recommend_qdrant_result| recommend_qdrant_result.point_id)
                .collect(),
            pool,
        )
        .await
        .map_err(|err| {
            ServiceError::BadRequest(format!(
                "Could not get recommended slim chunk_metadatas from qdrant_point_ids: {}",
                err
            ))
        })?,
        _ => get_chunk_metadatas_from_point_ids(
            recommended_qdrant_results
                .clone()
                .into_iter()
                .map(|recommend_qdrant_result| recommend_qdrant_result.point_id)
                .collect(),
            pool,
        )
        .await
        .map_err(|err| {
            ServiceError::BadRequest(format!(
                "Could not get recommended chunk_metadatas from qdrant_point_ids: {}",
                err
            ))
        })?,
    };

    let recommended_chunk_metadatas_with_score = recommended_chunk_metadatas
        .into_iter()
        .map(|chunk_metadata| {
            let score = recommended_qdrant_results
                .iter()
                .find(|recommend_qdrant_result| {
                    recommend_qdrant_result.point_id == chunk_metadata.metadata().qdrant_point_id
                })
                .map(|recommend_qdrant_result| recommend_qdrant_result.score)
                .unwrap_or(0.0);

            ChunkMetadataWithScore::from((chunk_metadata.metadata(), score))
        })
        .collect::<Vec<ChunkMetadataWithScore>>();

    let recommended_chunk_metadatas_with_score = recommended_chunk_metadatas_with_score
        .into_iter()
        .sorted_by(|a, b| {
            b.score
                .partial_cmp(&a.score)
                .unwrap_or(std::cmp::Ordering::Equal)
        })
        .collect::<Vec<ChunkMetadataWithScore>>();

    timer.add("fetched metadata from point_ids");

    let recommendation_id = uuid::Uuid::new_v4();

    let clickhouse_event = RecommendationEventClickhouse {
        id: recommendation_id,
        recommendation_type: String::from("chunk"),
        positive_ids: positive_chunk_ids
            .unwrap_or_default()
            .into_iter()
            .map(|x| x.to_string())
            .collect(),
        negative_ids: negative_chunk_ids
            .unwrap_or_default()
            .into_iter()
            .map(|x| x.to_string())
            .collect(),
        positive_tracking_ids: positive_tracking_ids.unwrap_or_default(),
        negative_tracking_ids: negative_tracking_ids.unwrap_or_default(),
        request_params: serde_json::to_string(&data.clone()).unwrap_or_default(),
        top_score: recommended_chunk_metadatas_with_score
            .first()
            .map(|x| x.score)
            .unwrap_or(0.0),
        results: recommended_chunk_metadatas_with_score
            .iter()
            .map(|x| serde_json::to_string(x).unwrap_or_default())
            .collect(),
        dataset_id: dataset_org_plan_sub.dataset.id,
        created_at: time::OffsetDateTime::now_utc(),
        user_id: data.user_id.clone().unwrap_or_default(),
    };

    event_queue
        .send(ClickHouseEvent::RecommendationEvent(clickhouse_event))
        .await;

    timer.add("send_to_clickhouse");

    if data.slim_chunks.unwrap_or(false) {
        let res = recommended_chunk_metadatas_with_score
            .into_iter()
            .map(|chunk| chunk.into())
            .collect::<Vec<SlimChunkMetadataWithScore>>();

        return Ok(HttpResponse::PartialContent().json(res));
    }

    if api_version == APIVersion::V2 {
        let new_payload = RecommendChunksResponseBody {
            id: recommendation_id,
            chunks: recommended_chunk_metadatas_with_score
                .into_iter()
                .map(|chunk| chunk.into())
                .collect::<Vec<ScoreChunk>>(),
        };

        return Ok(HttpResponse::Ok()
            .insert_header((Timer::header_key(), timer.header_value()))
            .json(new_payload));
    }

    Ok(HttpResponse::Ok()
        .insert_header((Timer::header_key(), timer.header_value()))
        .json(recommended_chunk_metadatas_with_score))
}

#[derive(Debug, Serialize, Deserialize, ToSchema)]
#[schema(example = json!({
    "prev_messages": [
        {
            "role": "user",
            "content": "How do I setup RAG with Trieve?",
        }
    ],
    "chunk_ids": ["d290f1ee-6c54-4b01-90e6-d701748f0851"],
    "prompt": "Respond to the instruction and include the doc numbers that you used in square brackets at the end of the sentences that you used the docs for:",
    "stream_response": true
}))]
pub struct GenerateOffChunksReqPayload {
    /// The previous messages to be placed into the chat history. There must be at least one previous message.
    pub prev_messages: Vec<ChatMessageProxy>,
    /// The ids of the chunks to be retrieved and injected into the context window for RAG.
    pub chunk_ids: Vec<uuid::Uuid>,
    /// Prompt will be used to tell the model what to generate in the next message in the chat. The default is 'Respond to the previous instruction and include the doc numbers that you used in square brackets at the end of the sentences that you used the docs for:'. You can also specify an empty string to leave the final message alone such that your user's final message can be used as the prompt. See docs.trieve.ai or contact us for more information.
    pub prompt: Option<String>,
    /// Whether or not to stream the response. If this is set to true or not included, the response will be a stream. If this is set to false, the response will be a normal JSON response. Default is true.
    pub stream_response: Option<bool>,
    /// Set highlight_results to false for a slight latency improvement (1-10ms). If not specified, this defaults to true. This will add `<b><mark>` tags to the chunk_html of the chunks to highlight matching splits.
    pub highlight_results: Option<bool>,
    /// What sampling temperature to use, between 0 and 2. Higher values like 0.8 will make the output more random, while lower values like 0.2 will make it more focused and deterministic. Default is 0.5.
    pub temperature: Option<f32>,
    /// Frequency penalty is a number between -2.0 and 2.0. Positive values penalize new tokens based on their existing frequency in the text so far, decreasing the model's likelihood to repeat the same line verbatim. Default is 0.7.
    pub frequency_penalty: Option<f32>,
    /// Presence penalty is a number between -2.0 and 2.0. Positive values penalize new tokens based on whether they appear in the text so far, increasing the model's likelihood to talk about new topics. Default is 0.7.
    pub presence_penalty: Option<f32>,
    /// The maximum number of tokens to generate in the chat completion. Default is None.
    pub max_tokens: Option<u32>,
    /// Stop tokens are up to 4 sequences where the API will stop generating further tokens. Default is None.
    pub stop_tokens: Option<Vec<String>>,
    /// User ID is the id of the user who is making the request. This is used to track user interactions with the RAG results.
    pub user_id: Option<String>,
}

/// RAG on Specified Chunks
///
/// This endpoint exists as an alternative to the topic+message resource pattern where our Trieve handles chat memory. With this endpoint, the user is responsible for providing the context window and the prompt and the conversation is ephemeral.
#[utoipa::path(
    post,
    path = "/chunk/generate",
    context_path = "/api",
    tag = "Chunk",
    request_body(content = GenerateOffChunksReqPayload, description = "JSON request payload to perform RAG on some chunks (chunks)", content_type = "application/json"),
    responses(
        (status = 200, description = "This will be a HTTP stream of a string, check the chat or search UI for an example how to process this. Response if streaming.",),
        (status = 200, description = "This will be a JSON response of a string containing the LLM's generated inference. Response if not streaming.", body = String),
        (status = 400, description = "Service error relating to to updating chunk, likely due to conflicting tracking_id", body = ErrorResponseBody),
    ),
    params(
        ("TR-Dataset" = String, Header, description = "The dataset id or tracking_id to use for the request. We assume you intend to use an id if the value is a valid uuid."),
    ),
    security(
        ("ApiKey" = ["readonly"]),
    )
)]
#[tracing::instrument(skip(pool, event_queue))]
pub async fn generate_off_chunks(
    data: web::Json<GenerateOffChunksReqPayload>,
    pool: web::Data<Pool>,
    event_queue: web::Data<EventQueue>,
    _user: LoggedUser,
    dataset_org_plan_sub: DatasetAndOrgWithSubAndPlan,
) -> Result<HttpResponse, actix_web::Error> {
    let prev_messages = data.prev_messages.clone();

    if prev_messages.iter().len() < 1 {
        return Err(
            ServiceError::BadRequest("There needs to be at least 1 prior message".into()).into(),
        );
    };

    let chunk_ids = data.chunk_ids.clone();

    let prompt = data.prompt.clone();

    let stream_response = data.stream_response;

    let mut chunks =
        get_metadata_from_ids_query(chunk_ids, dataset_org_plan_sub.dataset.id, pool).await?;

    let dataset_config =
        DatasetConfiguration::from_json(dataset_org_plan_sub.dataset.server_configuration);

    let base_url = dataset_config.LLM_BASE_URL;

    let default_model = dataset_config.LLM_DEFAULT_MODEL;

    let base_url = if base_url.is_empty() {
        "https://openrouter.ai/api/v1".into()
    } else {
        base_url
    };

    let llm_api_key = if !dataset_config.LLM_API_KEY.is_empty() {
        dataset_config.LLM_API_KEY.clone()
    } else if base_url.contains("openai.com") {
        get_env!("OPENAI_API_KEY", "OPENAI_API_KEY for openai should be set").into()
    } else {
        get_env!(
            "LLM_API_KEY",
            "LLM_API_KEY for openrouter or self-hosted should be set"
        )
        .into()
    };

    let client = Client {
        project: None,
        api_key: llm_api_key,
        http_client: reqwest::Client::new(),
        base_url,
        organization: None,
    };

    let mut messages: Vec<ChatMessage> = vec![];

    check_completion_param_validity(
        data.temperature,
        data.frequency_penalty,
        data.presence_penalty,
        data.stop_tokens.clone(),
    )?;

    messages.truncate(prev_messages.len() - 1);

    messages.push(ChatMessage {
        role: Role::User,
        content: ChatMessageContent::Text("I am going to provide several pieces of information (documents) for you to use in response to a request or question.".to_string()),
        tool_calls: None,
        name: None,
        tool_call_id: None,
    });

    messages.push(ChatMessage {
        role: Role::Assistant,
        content: ChatMessageContent::Text(
            "Understood, I will use the provided documents as information to respond to any future questions or instructions."
                .to_string(),
        ),
        tool_calls: None,
        name: None,
        tool_call_id: None,
    });

    chunks.sort_by(|a, b| {
        data.chunk_ids
            .iter()
            .position(|&id| id == a.id)
            .unwrap()
            .cmp(&data.chunk_ids.iter().position(|&id| id == b.id).unwrap())
    });

    chunks.iter().enumerate().for_each(|(idx, bookmark)| {
        let content = convert_html_to_text(&(bookmark.chunk_html.clone().unwrap_or_default()));
        let first_2000_words = content
            .split_whitespace()
            .take(2000)
            .collect::<Vec<_>>()
            .join(" ");

        messages.push(ChatMessage {
            role: Role::User,
            content: ChatMessageContent::Text(format!("Doc {}: {}", idx + 1, first_2000_words)),
            tool_calls: None,
            name: None,
            tool_call_id: None,
        });

        messages.push(ChatMessage {
            role: Role::Assistant,
            content: ChatMessageContent::Text("".to_string()),
            tool_calls: None,
            name: None,
            tool_call_id: None,
        });
    });

    let last_prev_message = prev_messages
        .last()
        .expect("There needs to be at least 1 prior message");

    let mut prev_messages = prev_messages.clone();

    prev_messages.truncate(prev_messages.len() - 1);

    prev_messages
        .iter()
        .for_each(|message| messages.push(ChatMessage::from(message.clone())));

    let prompt = prompt.unwrap_or("Respond to the question or instruction using the docs and include the doc numbers that you used in square brackets at the end of the sentences that you used the docs for:\n\n".to_string());

    messages.push(ChatMessage {
        role: Role::User,
        content: ChatMessageContent::Text(format!(
            "{} {}",
            prompt,
            last_prev_message.content.clone()
        )),
        tool_calls: None,
        name: None,
        tool_call_id: None,
    });

    let parameters = ChatCompletionParameters {
        model: default_model,
        stream: stream_response,
        messages,
        top_p: None,
        n: None,
        temperature: Some(data.temperature.unwrap_or(0.5)),
        frequency_penalty: Some(data.frequency_penalty.unwrap_or(0.7)),
        presence_penalty: Some(data.presence_penalty.unwrap_or(0.7)),
        stop: data
            .stop_tokens
            .as_ref()
            .map(|stop_tokens| StopToken::Array(stop_tokens.clone())),
        max_tokens: data.max_tokens,
        logit_bias: None,
        user: None,
        response_format: None,
        tools: None,
        tool_choice: None,
        logprobs: None,
        top_logprobs: None,
        seed: None,
    };

    if !stream_response.unwrap_or(true) {
        let assistant_completion =
            client
                .chat()
                .create(parameters.clone())
                .await
                .map_err(|err| {
                    ServiceError::BadRequest(format!(
                        "Bad response from LLM server provider: {}",
                        err
                    ))
                })?;

        let chat_content = match assistant_completion.choices.get(0) {
            Some(choice) => choice.message.content.clone(),
            None => {
                return Err(ServiceError::InternalServerError(
                    "Failed to get response completion".into(),
                )
                .into())
            }
        };

        let completion_content = match chat_content.clone() {
            ChatMessageContent::Text(text) => text.clone(),
            _ => "Failed to get response completion".to_string(),
        };

        let clickhouse_rag_event = RagQueryEventClickhouse {
            id: uuid::Uuid::new_v4(),
            created_at: time::OffsetDateTime::now_utc(),
            dataset_id: dataset_org_plan_sub.dataset.id,
            search_id: uuid::Uuid::nil(),
            results: data
                .chunk_ids
                .clone()
                .into_iter()
                .map(|s| s.to_string())
                .collect(),
            user_message: prompt,
            rag_type: "chosen_chunks".to_string(),
            llm_response: completion_content.clone(),
            user_id: data.user_id.clone().unwrap_or_default(),
        };

        event_queue
            .send(ClickHouseEvent::RagQueryEvent(clickhouse_rag_event))
            .await;

        return Ok(HttpResponse::Ok().json(chat_content));
    }

    let (s, r) = unbounded::<String>();
    let stream = client
        .chat()
        .create_stream(parameters.clone())
        .await
        .unwrap();

    Arbiter::new().spawn(async move {
        let chunk_v: Vec<String> = r.iter().collect();
        let completion = chunk_v.join("");

        let clickhouse_rag_event = RagQueryEventClickhouse {
            id: uuid::Uuid::new_v4(),
            created_at: time::OffsetDateTime::now_utc(),
            dataset_id: dataset_org_plan_sub.dataset.id,
            search_id: uuid::Uuid::nil(),
            results: data
                .chunk_ids
                .clone()
                .into_iter()
                .map(|s| s.to_string())
                .collect(),
            user_message: prompt,
            rag_type: "chosen_chunks".to_string(),
            llm_response: completion,
            user_id: data.user_id.clone().unwrap_or_default(),
        };

        event_queue
            .send(ClickHouseEvent::RagQueryEvent(clickhouse_rag_event))
            .await;
    });

    let completion_stream = stream.map(move |response| -> Result<Bytes, actix_web::Error> {
        if let Ok(response) = response {
            let chat_content = match response.choices.get(0) {
                Some(choice) => choice.delta.content.clone(),
                None => Some("failed to get response completion".to_string()),
            };
            if let Some(message) = chat_content.clone() {
                s.send(message).unwrap();
            }
            return Ok(Bytes::from(chat_content.unwrap_or("".to_string())));
        }
        Err(ServiceError::InternalServerError(
            "Model Response Error. Please try again later.".into(),
        )
        .into())
    });

    Ok(HttpResponse::Ok().streaming(completion_stream))
}

pub fn check_completion_param_validity(
    temperature: Option<f32>,
    frequency_penalty: Option<f32>,
    presence_penalty: Option<f32>,
    stop_tokens: Option<Vec<String>>,
) -> Result<(), ServiceError> {
    if let Some(temperature) = temperature {
        if !(0.0..=2.0).contains(&temperature) {
            return Err(ServiceError::BadRequest(
                "Temperature must be between 0 and 2".to_string(),
            ));
        }
    }

    if let Some(frequency_penalty) = frequency_penalty {
        if !(-2.0..=2.0).contains(&frequency_penalty) {
            return Err(ServiceError::BadRequest(
                "Frequency penalty must be between -2.0 and 2.0".to_string(),
            ));
        }
    }

    if let Some(presence_penalty) = presence_penalty {
        if !(-2.0..=2.0).contains(&presence_penalty) {
            return Err(ServiceError::BadRequest(
                "Presence penalty must be between -2.0 and 2.0".to_string(),
            ));
        }
    }

    if let Some(stop_tokens) = stop_tokens {
        if stop_tokens.len() > 4 {
            return Err(ServiceError::BadRequest(
                "Stop tokens must be less than or equal to 4".to_string(),
            ));
        }
    }

    Ok(())
}

#[derive(Serialize, Deserialize, Debug, ToSchema, Clone)]
#[serde(rename_all = "snake_case")]
pub enum CrawlInterval {
    Daily,
    Weekly,
    Monthly,
}
