export class ConcurrentProcessor<T, R> {
  constructor(private maxConcurrent: number = 3) {}

  async *processQueue(
    items: T[],
    processor: (item: T) => Promise<R>
  ): AsyncGenerator<R> {
    const queue = [...items];
    const inFlight = new Map<Promise<R>, T>();
    
    while (queue.length > 0 || inFlight.size > 0) {
      // Start new work up to limit
      while (inFlight.size < this.maxConcurrent && queue.length > 0) {
        const item = queue.shift()!;
        const promise = processor(item);
        inFlight.set(promise, item);
        
        promise.finally(() => inFlight.delete(promise));
      }
      
      // Wait for next completion
      if (inFlight.size > 0) {
        const result = await Promise.race(inFlight.keys());
        yield result;
      }
    }
  }

  async processAll(
    items: T[],
    processor: (item: T) => Promise<R>
  ): Promise<R[]> {
    const results: R[] = [];
    for await (const result of this.processQueue(items, processor)) {
      results.push(result);
    }
    return results;
  }
}

export class StreamMerger<T> {
  async *merge(
    streams: Map<string, AsyncIterator<T>>
  ): AsyncGenerator<{ source: string; value: T }> {
    const active = new Map(streams);
    
    while (active.size > 0) {
      const promises = Array.from(active.entries()).map(async ([source, iter]) => {
        const { done, value } = await iter.next();
        return { source, done, value };
      });
      
      const result = await Promise.race(promises);
      
      if (result.done) {
        active.delete(result.source);
      } else {
        yield { source: result.source, value: result.value };
      }
    }
  }
}


class Channel<T> {
  private queue: T[] = [];
  private waiters: ((value: T) => void)[] = [];
  private closed = false;

  async send(value: T): Promise<void> {
    if (this.closed) throw new Error('Channel closed');
    
    if (this.waiters.length > 0) {
      const waiter = this.waiters.shift()!;
      waiter(value);
    } else {
      this.queue.push(value);
    }
  }

  async receive(): Promise<T | undefined> {
    if (this.queue.length > 0) {
      return this.queue.shift()!;
    }
    
    if (this.closed) {
      return undefined;
    }
    
    return new Promise(resolve => {
      this.waiters.push(resolve);
    });
  }

  close() {
    this.closed = true;
    this.waiters.forEach(w => w(undefined as any));
  }

  async *[Symbol.asyncIterator]() {
    while (true) {
      const value = await this.receive();
      if (value === undefined) break;
      yield value;
    }
  }
}
