require "grit"

class Project < ActiveRecord::Base
  has_many :users_projects, :dependent => :destroy
  has_many :users, :through => :users_projects

  validates_presence_of :name
  validates_presence_of :path

  before_save :valid_repo?

  scope :public_only, where(:private_flag => false)

  def self.allowed(user, project)
    return [] unless project.instance_of?(Project)
    
    rules = []
    rules << :read_project if project.public? ||
      project.readers.include?(user)

    rules << :write_project if project.writers.include?(user)
    rules << :admin_project if project.admins.include?(user)
    rules
  end

  def add_access(user, *access)
    opts = { :project => self, :user => user }
    access.each { |name| opts.merge!(name => true) }
    UsersProject.create(opts)
  end

  def writers
    users_projects.includes(:user).where(:write => true).map(&:user)
  end

  def readers
    users_projects.includes(:user).where(:read => true).map(&:user)
  end

  def admins
    users_projects.includes(:user).where(:admin => true).map(&:user)
  end

  def public?
    !private_flag
  end

  def private?
    private_flag
  end

  def valid_repo?
    repo
  rescue
    errors.add(:path, "Invalid repository path")
    false
  end

  def repo 
    @repo ||= Grit::Repo.new(path)
  end

  def commit(commit_id = nil)
    if commit_id
      repo.commits(commit_id).first
    else 
      repo.commits.first
    end
  end

  def tree(fcommit, path = nil)
    fcommit = commit if fcommit == :head
    tree = fcommit.tree
    path ? (tree / path) : tree
  end
end
