require 'spec_helper'

describe Project do
  describe "Associations" do
    it { should have_many(:users) }
    it { should have_many(:users_projects) }
  end

  describe "Validation" do
    it { should validate_presence_of(:name) }
    it { should validate_presence_of(:path) }
  end

  describe :valid_repo? do 
    it "should be valid repo" do 
      project = Factory :project
      project.valid_repo?.should be_true 
    end

    it "should be invalid repo" do
      project = Project.new(:name => "ok_name", :path => "/INVALID_PATH/")
      project.valid_repo?.should be_false
    end
  end

  describe "Git methods" do 
    let(:project) { Factory :project }

    describe :repo do 
      it "should return valid repo" do 
        project.repo.should be_kind_of(Grit::Repo)
      end

      it "should return nil" do 
        lambda { Project.new(:path => "invalid").repo }.should raise_error(Grit::NoSuchPathError)
      end

      it "should return nil" do 
        lambda { Project.new.repo }.should raise_error(TypeError)
      end
    end

    describe :commit do 
      it "should return first head commit if without params" do 
        project.commit.id.should == project.repo.commits.first.id
      end

      it "should return valid commit" do 
        project.commit(ValidCommit::ID).should be_valid_commit
      end

      it "should return nil" do 
        project.commit("+123_4532530XYZ").should be_nil
      end
    end

    describe :tree do 
      before do 
        @commit = project.commit(ValidCommit::ID)
      end

      it "should raise error w/o arguments" do 
        lambda { project.tree }.should raise_error
      end

      it "should return root tree for commit" do
        tree = project.tree(@commit)
        tree.contents.size.should == ValidCommit::FILES_COUNT
        tree.contents.map(&:name).should == ValidCommit::FILES
      end

      it "should return root tree for commit with correct path" do
        tree = project.tree(@commit, ValidCommit::C_FILE_PATH)
        tree.contents.map(&:name).should == ValidCommit::C_FILES
      end

      it "should return root tree for commit with incorrect path" do
        project.tree(@commit, "invalid_path").should be_nil
      end
    end
  end

  describe :authorization do 
    before do 
      @p1 = Factory :project
      @u1 = Factory :user
      @u2 = Factory :user
      @abilities = Six.new
      @abilities << Project
    end

    describe :read do 
      before do 
        UsersProject.create(:project => @p1, :user => @u1, :read => false) 
        UsersProject.create(:project => @p1, :user => @u2, :read => true) 
      end

      it { @abilities.allowed?(@u1, :read_project, @p1).should be_false }
      it { @abilities.allowed?(@u2, :read_project, @p1).should be_true }
    end

    describe :write do 
      before do 
        UsersProject.create(:project => @p1, :user => @u1, :write => false) 
        UsersProject.create(:project => @p1, :user => @u2, :write => true) 
      end

      it { @abilities.allowed?(@u1, :write_project, @p1).should be_false }
      it { @abilities.allowed?(@u2, :write_project, @p1).should be_true }
    end

    describe :admin do 
      before do 
        UsersProject.create(:project => @p1, :user => @u1, :admin => false) 
        UsersProject.create(:project => @p1, :user => @u2, :admin => true) 
      end

      it { @abilities.allowed?(@u1, :admin_project, @p1).should be_false }
      it { @abilities.allowed?(@u2, :admin_project, @p1).should be_true }
    end
  end
end
