class ProjectsController < ApplicationController
  before_filter :authenticate_user!, :except => [:index, :show, :tree, :blob] 
  before_filter :project, :except => [:index, :new, :create] 

  # Authorize
  before_filter :add_abilities
  before_filter :authorize_read!, :except => [:index, :new, :create] 
  before_filter :authorize_admin!, :only => [:edit, :update, :destroy] 

  # GET /projects
  # GET /projects.json
  def index
    @projects = if current_user
                  current_user.projects.all
                else 
                  Project.public_only.all
                end

    respond_to do |format|
      format.html # index.html.erb
      format.json { render json: @projects }
    end
  end

  # GET /projects/1
  # GET /projects/1.json
  def show
    @repo = project.repo
    @commit = @repo.commits.first
    @tree = @commit.tree
    @tree = @tree / params[:path] if params[:path]

    respond_to do |format|
      format.html # show.html.erb
      format.json { render json: project }
    end
  end

  def tree
    @repo = project.repo
    if params[:commit_id]
      @commit = @repo.commits(params[:commit_id]).first
    else 
      @commit = @repo.commits.first
    end
    @tree = @commit.tree
    @tree = @tree / params[:path] if params[:path]

    respond_to do |format|
      format.html # show.html.erb
      format.js
      format.json { render json: project }
    end
  end

  def blob
    @repo = project.repo
    @commit = project.commit(params[:commit_id])
    @tree = project.tree(@commit, params[:path])

    if @tree.is_a?(Grit::Blob)
      send_data(@tree.data, :type => @tree.mime_type, :disposition => 'inline')
    end
  end

  # GET /projects/new
  # GET /projects/new.json
  def new
    @project = Project.new

    respond_to do |format|
      format.html # new.html.erb
      format.json { render json: @project }
    end
  end

  # GET /projects/1/edit
  def edit
  end

  # POST /projects
  # POST /projects.json
  def create
    @project = Project.new(params[:project])

    Project.transaction do 
      @project.save!

      UsersProject.create!(:project => @project, :admin => true, :read => true,
                           :write => true, :user => current_user)
    end

    respond_to do |format|
      if @project.valid?
        format.html { redirect_to @project, notice: 'Project was successfully created.' }
        format.json { render json: @project, status: :created, location: @project }
      else
        format.html { render action: "new" }
        format.json { render json: @project.errors, status: :unprocessable_entity }
      end
    end
  rescue StandardError => ex
    @project.errors.add(:base, "Cant save project. Please try again later")
    respond_to do |format|
      format.html { render action: "new" }
      format.json { render json: @project.errors, status: :unprocessable_entity }
    end
  end

  # PUT /projects/1
  # PUT /projects/1.json
  def update
    respond_to do |format|
      if project.update_attributes(params[:project])
        format.html { redirect_to project, notice: 'Project was successfully updated.' }
        format.json { head :ok }
      else
        format.html { render action: "edit" }
        format.json { render json: project.errors, status: :unprocessable_entity }
      end
    end
  end

  # DELETE /projects/1
  # DELETE /projects/1.json
  def destroy
    project.destroy

    respond_to do |format|
      format.html { redirect_to projects_url }
      format.json { head :ok }
    end
  end

  protected 

  def add_abilities
    abilities << Project
  end

  def project 
    @project ||= Project.find(params[:id])
  end

  def authorize_read!
    head(404) unless can?(current_user, :read_project, project)
  end

  def authorize_admin!
    head(404) unless can?(current_user, :admin_project, project)
  end
end
