/**
 * Copyright 2014 Google Inc. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import * as assert from 'assert';
import * as util from '../src/util';
import * as proxyquire from 'proxyquire';
import * as pfy from '@google-cloud/promisify';
import * as sinon from 'sinon';

let promisified = false;
const fakePromisify = Object.assign({}, pfy, {
  promisifyAll: function(Class) {
    if (Class.name === 'Snapshot') {
      promisified = true;
    }
  },
});

describe('Snapshot', function() {
  let Snapshot;
  let snapshot;

  const SNAPSHOT_NAME = 'a';
  const PROJECT_ID = 'grape-spaceship-123';

  const PUBSUB = {
    projectId: PROJECT_ID,
  };

  const SUBSCRIPTION: any = {
    Promise: {},
    projectId: PROJECT_ID,
    pubsub: PUBSUB,
    api: {},
    createSnapshot: function() {},
    seek: function() {},
  };

  before(function() {
    Snapshot = proxyquire('../src/snapshot', {
      '@google-cloud/promisify': fakePromisify,
    }).Snapshot;
  });

  const sandbox = sinon.createSandbox();
  beforeEach(function() {
    snapshot = new Snapshot(SUBSCRIPTION, SNAPSHOT_NAME);
  });
  afterEach(() => sandbox.restore());

  describe('initialization', function() {
    const FULL_SNAPSHOT_NAME = 'a/b/c/d';
    let formatName_;

    before(function() {
      formatName_ = Snapshot.formatName_;
      Snapshot.formatName_ = function() {
        return FULL_SNAPSHOT_NAME;
      };
    });

    after(function() {
      Snapshot.formatName_ = formatName_;
    });

    it('should promisify all the things', function() {
      assert(promisified);
    });

    it('should localize parent.Promise', function() {
      assert.strictEqual(snapshot.Promise, SUBSCRIPTION.Promise);
    });

    it('should localize the parent', function() {
      assert.strictEqual(snapshot.parent, SUBSCRIPTION);
    });

    describe('name', function() {
      it('should create and cache the full name', function() {
        Snapshot.formatName_ = function(projectId, name) {
          assert.strictEqual(projectId, PROJECT_ID);
          assert.strictEqual(name, SNAPSHOT_NAME);
          return FULL_SNAPSHOT_NAME;
        };

        const snapshot = new Snapshot(SUBSCRIPTION, SNAPSHOT_NAME);
        assert.strictEqual(snapshot.name, FULL_SNAPSHOT_NAME);
      });

      it('should pull the projectId from parent object', function() {
        Snapshot.formatName_ = function(projectId, name) {
          assert.strictEqual(projectId, PROJECT_ID);
          assert.strictEqual(name, SNAPSHOT_NAME);
          return FULL_SNAPSHOT_NAME;
        };

        const snapshot = new Snapshot(SUBSCRIPTION, SNAPSHOT_NAME);
        assert.strictEqual(snapshot.name, FULL_SNAPSHOT_NAME);
      });
    });

    describe('with Subscription parent', function() {
      it('should include the create method', function(done) {
        SUBSCRIPTION.createSnapshot = function(name, callback) {
          assert.strictEqual(name, SNAPSHOT_NAME);
          callback(); // The done function
        };

        const snapshot = new Snapshot(SUBSCRIPTION, SNAPSHOT_NAME);
        snapshot.create(done);
      });

      it('should create a seek method', function(done) {
        SUBSCRIPTION.seek = function(name, callback) {
          assert.strictEqual(name, SNAPSHOT_NAME);
          callback(); // The done function
        };

        const snapshot = new Snapshot(SUBSCRIPTION, SNAPSHOT_NAME);
        snapshot.seek(done);
      });
    });

    describe('with PubSub parent', function() {
      let snapshot;

      beforeEach(function() {
        snapshot = new Snapshot(PUBSUB, SNAPSHOT_NAME);
      });

      it('should not include the create method', function() {
        assert.strictEqual(snapshot.create, undefined);
      });

      it('should not include a seek method', function() {
        assert.strictEqual(snapshot.seek, undefined);
      });
    });
  });

  describe('formatName_', function() {
    const EXPECTED = 'projects/' + PROJECT_ID + '/snapshots/' + SNAPSHOT_NAME;

    it('should format the name', function() {
      const name = Snapshot.formatName_(PROJECT_ID, SNAPSHOT_NAME);
      assert.strictEqual(name, EXPECTED);
    });

    it('should not re-format the name', function() {
      const name = Snapshot.formatName_(PROJECT_ID, EXPECTED);
      assert.strictEqual(name, EXPECTED);
    });
  });

  describe('delete', function() {
    it('should make the correct request', function(done) {
      snapshot.parent.request = function(config, callback) {
        assert.strictEqual(config.client, 'SubscriberClient');
        assert.strictEqual(config.method, 'deleteSnapshot');
        assert.deepStrictEqual(config.reqOpts, {snapshot: snapshot.name});
        callback(); // the done fn
      };

      snapshot.delete(done);
    });

    it('should optionally accept a callback', function(done) {
      sandbox.stub(util, 'noop').callsFake(done);
      snapshot.parent.request = function(config, callback) {
        callback(); // the done fn
      };
      snapshot.delete();
    });
  });
});
