package collectors

import (
	"time"

	"bosun.org/metadata"
	"bosun.org/opentsdb"
)

func init() {
	c_dns := &IntervalCollector{
		F:        c_dns_windows,
		Interval: time.Second * 30, //Depending on how busy your DNS servers are, 30 might encompass a lot of change, or barely any at all. Seems like a good compromise.
	}
	c_dns.init = wmiInitNamespace(c_dns, func() interface{} { return &[]MicrosoftDNS_Statistic{} }, "", &dnsQuery, rootDNS)
	collectors = append(collectors, c_dns)
}

var (
	dnsQuery string //This will be created by wmiInitNamespace() by using reflection over the struct that's been passed
	rootDNS  = "root\\MicrosoftDNS"
)

func c_dns_windows() (opentsdb.MultiDataPoint, error) {
	var dst []MicrosoftDNS_Statistic                    //Data returned from WMI will be stored here
	var md opentsdb.MultiDataPoint                      //Data to push out will be here
	var dupes = make(map[string]MicrosoftDNS_Statistic) //We are going to push duplicate values that need to be summed here (for more details see comment block above getTagsFromDesc())
	err := queryWmiNamespace(dnsQuery, &dst, rootDNS)   //Get the data. dnsQuery is populated by wmiInitNamespace() in the init function
	if err != nil {
		return nil, err
	}
	for _, v := range dst { //ForEach record in WMI dataset

		if p, ok := DNSStatPropertyMap[v.Name]; ok {
			if p.DoubleUp { //This is a stat that has multiple entries we need to collate it for later
				if _, ok := dupes[p.Metric]; ok { //Key already exists, so we need to add to it
					*dupes[p.Metric].Value += *v.Value
				} else { //Key does not exist, so we need to create it
					dupes[p.Metric] = v
				}
			} else { //Unique stat, we can put it straight away
				if "" == p.Grouping {
					Add(&md, "dns."+p.Category+"."+p.Metric, v.Value, opentsdb.TagSet{}, p.RateType, metadata.Count, p.Description)
				} else {
					Add(&md, "dns."+p.Category+"."+p.Metric, v.Value, opentsdb.TagSet{"type": p.Grouping}, p.RateType, metadata.Count, p.Description)
				}
			}
		}
	}
	for _, v := range dupes { //ForEach record in our added up duplicates
		if p, ok := DNSStatPropertyMap[v.Name]; ok {
			if "" == p.Grouping {
				Add(&md, "dns."+p.Category+"."+p.Metric, v.Value, opentsdb.TagSet{}, p.RateType, metadata.Count, p.Description)
			} else {
				Add(&md, "dns."+p.Category+"."+p.Metric, v.Value, opentsdb.TagSet{"type": p.Grouping}, p.RateType, metadata.Count, p.Description)
			}
		}

	}
	return md, nil
}

type DNSStatProperties struct {
	Category    string
	Metric      string
	DoubleUp    bool
	Grouping    string
	RateType    metadata.RateType
	Description string
}

type MicrosoftDNS_Statistic struct {
	CollectionId   uint32
	CollectionName string
	DnsServerName  string
	Name           string
	//StringValue *string
	Value *uint32 //The namespace documentation does not mention that this is a nullable field, but apparently it is.
}

/*
	So what's the story here? WMI only returns a descriptive text for the counter, which is not particularly useful for Bosun.
	The good news is that there is a mapping between the WMI description and the counters actual internal name (exposed
	in the Get-DNSServerStatus powershell cmdlet. So an awfully long time was spent mapping the description to the actual
	metric name and category.

	There is also a problem where a bunch of the metrics have the same description. This is likely a bug in the namespace
	where there is one counter for UDP and one counter for TCP. For these doubleups, "true" will be returned in the
	third parameter, and these numbers will be summed together and submitted as a single counter.

	The sheet I worked all this out on is here: https://docs.google.com/spreadsheets/d/17Smri2o6Y9EiCrdELRy8TtwTpU5xh_yEoo728XtOgX4/edit?usp=sharing
*/
var DNSStatPropertyMap = map[string]DNSStatProperties{
	"A queries":                            {"query", "queries", false, "typea", metadata.Counter, "The number of queries received by the DNS server. The 'Type' tag indicates the type of request that was received (A, MX, PTR, etc)."},
	"Active record nodes checked":          {"timeout", "activerecord", false, "", metadata.Counter, "The number of times while performing checks the server encountered a cache node that had records present while checking nodes for deletion."},
	"Additional questions recursed":        {"recursion", "additionalrecursed", false, "", metadata.Counter, "The number of recursions performed to return additional data or CNAME."},
	"ALL queries":                          {"query", "queries", false, "typeall", metadata.Counter, "The number of queries received by the DNS server. The 'Type' tag indicates the type of request that was received (A, MX, PTR, etc)."},
	"Allocation count":                     {"memory", "alloc", false, "", metadata.Gauge, "The cumulative number of times memory allocations have been performed for a given purpose."},
	"Answer responses":                     {"recursion", "responseanswer", false, "", metadata.Counter, "The number of responses received from other servers for recursive queries."},
	"Authoritative responses":              {"recursion", "responseauthoritative", false, "", metadata.Counter, "The number of responses received from the server authoritative for the zone."},
	"AXFR in IXFR":                         {"query", "ixfraxfr", false, "", metadata.Counter, "The number of incremental zone transfer requests received by the server which required a full zone transfer."},
	"AXFR in IXFR requests":                {"query", "requests", false, "ixfraxfrreq", metadata.Counter, "The number of requests that the DNS server has made to other DNS servers. For example, these could be recursive queries or zone transfers. The 'Type' tag indicates the sort of request that the DNS server has made."},
	"AXFR invalid responses":               {"query", "axfrinvalid", false, "", metadata.Counter, "The number of full zone transfer invalid responses received by the server."},
	"AXFR queries":                         {"query", "queries", false, "typeaxfr", metadata.Counter, "The number of queries received by the DNS server. The 'Type' tag indicates the type of request that was received (A, MX, PTR, etc)."},
	"AXFR refused responses":               {"query", "axfrrefused", false, "", metadata.Counter, "The number of full zone transfer rejection responses received by the server."},
	"AXFR requests":                        {"query", "requests", true, "axfrrequest", metadata.Counter, "The number of requests that the DNS server has made to other DNS servers. For example, these could be recursive queries or zone transfers. The 'Type' tag indicates the sort of request that the DNS server has made."},
	"AXFR responses":                       {"query", "axfrresponse", false, "", metadata.Counter, "The number of full zone transfer responses received by the server."},
	"AXFR success responses":               {"query", "afxrsuccessresponse", false, "", metadata.Counter, "The number of incremental zone transfer requests received by the server which required a full zone transfer."},
	"AXFR successes":                       {"query", "axfrsuccess", false, "", metadata.Counter, "The number of full zone transfers successfully completed by the server."},
	"AxfrLimit refused":                    {"query", "axfrlimit", false, "", metadata.Counter, "The number of full zone transfer requests rejected due to time restrictions between successive full zone transfers."},
	"Bad packet responses":                 {"recursion", "responsebadpacket", false, "", metadata.Counter, "The number of bad response packets received."},
	"Bad time TKEYs":                       {"security", "securitytkeybadtime", false, "", metadata.Counter, "The number of secure DNS update messages that had TKEY with a skewed time stamp."},
	"BadKey":                               {"error", "errors", false, "badkey", metadata.Counter, "The number of errors that the DNS server has processed. The 'Type' metric is the type of error that the DNS server has processed."},
	"BadKey TSIGs":                         {"security", "securitytsigbadkey", false, "", metadata.Counter, "The number of TSIG records received for which the cached security context could not be found."},
	"BadSig":                               {"error", "errors", false, "badsig", metadata.Counter, "The number of errors that the DNS server has processed. The 'Type' metric is the type of error that the DNS server has processed."},
	"BadTime":                              {"error", "errors", false, "badtime", metadata.Counter, "The number of errors that the DNS server has processed. The 'Type' metric is the type of error that the DNS server has processed."},
	"Cache update failures":                {"recursion", "cacheupdatefailure", false, "", metadata.Counter, "The number of times the server received failure for self-generated cache update recursion queries to remote servers."},
	"Cache update queries allocated":       {"recursion", "cacheupdatealloc", false, "", metadata.Counter, "The number of times the server allocated a query to be sent to update a cache entry."},
	"Cache update queries for root NS":     {"recursion", "rootnsquery", false, "", metadata.Counter, "The number of times the server sent a query for a root name server."},
	"Cache update queries freed":           {"recursion", "cacheupdatefree", false, "", metadata.Counter, "The number of times the server released a query request or response packet sent to update a cache entry."},
	"Cache update queries resumed":         {"recursion", "resumesuspendedquery", false, "", metadata.Counter, "The number of times the server resumed a suspended query that was needed to update cache entry information."},
	"Cache update queries suspended":       {"recursion", "suspendedquery", false, "", metadata.Counter, "The number of times the server suspended sending a query needed to update cache entry information."},
	"Cache update query responses":         {"recursion", "cacheupdateresponse", false, "", metadata.Counter, "The number of times the server received responses for a query sent to update a cache entry."},
	"Cache update query retries":           {"recursion", "cacheupdateretry", false, "", metadata.Counter, "The number of times the server reattempted a query to update cache entry information."},
	"Cache update responses for root NS":   {"recursion", "rootnsresponse", false, "", metadata.Counter, "The number of times the server processed a response from one of its root servers."},
	"Cached records timed out":             {"record", "cachetimeouts", false, "", metadata.Counter, "The cumulative number of resource records that have been freed from the DNS server's cache."},
	"Can not delete nodes checked":         {"timeout", "cannotdelete", false, "", metadata.Counter, "The number of times the server encountered a cache node that was marked for deletion that could not be deleted because it had been recently accessed or because it had active records or child nodes."},
	"Checks where cache exceeded limit":    {"cache", "exceededlimit", false, "", metadata.Counter, "Not used. Senders MUST set this value to zero and receivers MUST ignore it."},
	"Continue current lookup responses":    {"recursion", "continuecurrentlookup", false, "", metadata.Counter, "The number of times the server received a response from a remote DNS server while processing a client query and restarted recursion."},
	"Continue current recursion responses": {"recursion", "continuecurrentrecursion", false, "", metadata.Counter, "The number of additional remote queries generated by the DNS server during normal query processing."},
	"Continue next lookup responses":       {"recursion", "continuenextlookup", false, "", metadata.Counter, "The number of times the server started a lookup with the next query."},
	"Database nodes in use":                {"database", "nodeinuse", false, "", metadata.Gauge, "The number of nodes currently allocated for use in the record database."},
	"Database nodes memory":                {"database", "nodememory", false, "", metadata.Gauge, "The total size in bytes of server memory currently used for nodes."},
	"Database nodes returned":              {"database", "nodereturn", false, "", metadata.Gauge, "The cumulative number of nodes freed from the record database."},
	"Database nodes used":                  {"database", "nodeused", false, "", metadata.Gauge, "The cumulative number of nodes allocated for use in the record database."},
	"Delayed frees executed":               {"timeout", "delayedfreesexecuted", false, "", metadata.Counter, "The number of times the server released a block of memory that had previously been entered into an internal list of memory blocks that can be freed in the future."},
	"Delayed frees executed with function": {"timeout", "delayedfreesexecutedwithfunction", false, "", metadata.Counter, "The number of times the server released a block of memory that had previously been entered into an internal list of memory blocks that can be freed in the future where a function other than the standard memory free function was used for release."},
	"Delayed frees queued":                 {"timeout", "delayedfreesqueued", false, "", metadata.Counter, "The number of times the server entered a block of memory into an internal list of memory blocks that can be freed in the future."},
	"Delayed frees queued with function":   {"timeout", "delayedfreesqueuedwithfunction", false, "", metadata.Counter, "The number of times the server entered a block of memory into an internal list of memory blocks that can be freed in the future where the block is freed using a function other than the standard memory free function."},
	"Delegation responses":                 {"recursion", "responsedelegation", false, "", metadata.Counter, "The number of delegation responses received by the server."},
	"Deleted nodes checked":                {"timeout", "deleted", false, "", metadata.Counter, "The number of times the server successfully deleted a cache node that was marked as eligible for deletion."},
	"DnsLib blocks allocated":              {"memory", "dnslib", false, "dnslibblocksallocated", metadata.Gauge, "Memory statistics for DnsLib."},
	"DnsLib blocks freed":                  {"memory", "dnslib", false, "dnslibblocksfreed", metadata.Gauge, "Memory statistics for DnsLib."},
	"DnsLib blocks in use":                 {"memory", "dnslib", false, "dnslibblocksinuse", metadata.Gauge, "Memory statistics for DnsLib."},
	"DnsLib memory":                        {"memory", "dnslib", false, "dnslibmemory", metadata.Gauge, "Memory statistics for DnsLib."},
	"DS DN blocks allocated":               {"domainservices", "dnblocksallocated", false, "", metadata.Counter, "The number of full zone transfers successfully completed by the server."},
	"DS DN blocks freed":                   {"domainservices", "dnblocksfreed", false, "", metadata.Gauge, "The number of blocks freed from Directory Services Distinguished Names."},
	"DS DN blocks in use":                  {"domainservices", "dnblocksinuse", false, "", metadata.Gauge, "The number of blocks in use for Directory Services Distinguished Names."},
	"DS DN memory":                         {"domainservices", "dnmemory", false, "", metadata.Gauge, "The amount of memory allocated to Directory Services Distinguished Names."},
	"DS Mod blocks allocated":              {"domainservices", "modblocksallocated", false, "", metadata.Gauge, "The number of blocks in use for Directory Services Module."},
	"DS Mod blocks freed":                  {"domainservices", "modblocksfreed", false, "", metadata.Gauge, "The number of blocks freed from Directory Services Mod Module."},
	"DS Mod blocks in use":                 {"domainservices", "modblocksinuse", false, "", metadata.Gauge, "The number of blocks in use for Directory Services Mod Module."},
	"DS Mod memory":                        {"domainservices", "modmemory", false, "", metadata.Gauge, "The amount of memory allocated to Directory Services Mod Module."},
	"DS Other blocks allocated":            {"domainservices", "otherblocksallocated", false, "", metadata.Gauge, "The number of blocks in use for Other directory server related operations."},
	"DS Other blocks freed":                {"domainservices", "otherblocksfreed", false, "", metadata.Gauge, "The number of blocks freed from Other directory server related operations."},
	"DS Other blocks in use":               {"domainservices", "otherblocksinuse", false, "", metadata.Gauge, "The number of blocks in use for Other directory server related operations."},
	"DS Other memory":                      {"domainservices", "othermemory", false, "", metadata.Gauge, "The amount of memory allocated to Directory Services Other."},
	"DS Record blocks allocated":           {"domainservices", "recordblocksallocated", false, "", metadata.Gauge, "The number of blocks in use for Directory Services Records."},
	"DS Record blocks freed":               {"domainservices", "recordblocksfreed", false, "", metadata.Gauge, "The number of blocks freed from Directory Services Records."},
	"DS Record blocks in use":              {"domainservices", "recordblocksinuse", false, "", metadata.Gauge, "The number of blocks in use for Directory Services Records."},
	"DS Record memory":                     {"domainservices", "recordmemory", false, "", metadata.Gauge, "The amount of memory allocated to Directory Services Records."},
	"Echo TSIGs":                           {"security", "securitytsigecho", false, "", metadata.Counter, "The number of echo TSIG records received by the server indicating that the remote server is not security aware. [RFC2845]"},
	"Empty responses":                      {"recursion", "responseempty", false, "", metadata.Counter, "The number of empty responses received from other servers."},
	"EventControl blocks allocated":        {"memory", "eventcontrol", false, "eventcontrolblocksallocated", metadata.Gauge, "Memory statistics for Event Control."},
	"EventControl blocks freed":            {"memory", "eventcontrol", false, "eventcontrolblocksfreed", metadata.Gauge, "Memory statistics for Event Control."},
	"EventControl blocks in use":           {"memory", "eventcontrol", false, "eventcontrolblocksinuse", metadata.Gauge, "Memory statistics for Event Control."},
	"EventControl memory":                  {"memory", "eventcontrol", false, "eventcontrolmemory", metadata.Gauge, "Memory statistics for Event Control."},
	"Failed adds":                          {"domainservices", "failedldapadd", false, "", metadata.Counter, "The number of times the server failed to add entries to the directory server."},
	"Failed cache enforcement passes":      {"cache", "failedfreepasses", false, "", metadata.Counter, "The number of times since the server last started that the server cache was found to exceed the cache size limit which is 90 percent of the MaxCacheSize (section 3.1.1.1.1) and that an attempt to free nodes was unsuccessful in meeting the cache size limit. After reaching 0xFFFFFFFF the value increments to 0x00000000."},
	"Failed deletions":                     {"domainservices", "faileddeletedsentries", false, "", metadata.Counter, "The number of times the server failed to delete entries from the directory server."},
	"Failed modifies":                      {"domainservices", "failedldapmodify", false, "", metadata.Counter, "The number of times the server failed to modify records in the directory server."},
	"Failed reads":                         {"domainservices", "failedreadrecords", false, "", metadata.Counter, "The number of times the server failed to read records from the directory server."},
	"File Buffer blocks allocated":         {"memory", "filebuffer", false, "filebufferblocksallocated", metadata.Gauge, "Memory statistics for File Buffers."},
	"File Buffer blocks freed":             {"memory", "filebuffer", false, "filebufferblocksfreed", metadata.Gauge, "Memory statistics for File Buffers."},
	"File Buffer blocks in use":            {"memory", "filebuffer", false, "filebufferblocksinuse", metadata.Gauge, "Memory statistics for File Buffers."},
	"File Buffer memory":                   {"memory", "filebuffer", false, "filebuffermemory", metadata.Gauge, "Memory statistics for File Buffers."},
	"Final queued timeouts":                {"recursion", "finaltimeoutqueued", false, "", metadata.Counter, "The number of recursive queries enlisted to wait for final time-out before they expire."},
	"Final timeouts expired":               {"recursion", "finaltimeoutexpired", false, "", metadata.Counter, "The number of recursive queries expired without the server receiving any response."},
	"FormErr TSIGs":                        {"security", "securitytsigformerror", false, "", metadata.Counter, "The number of TSIG records from which signature extraction failed."},
	"FormError":                            {"error", "errors", false, "formerror", metadata.Counter, "The number of errors that the DNS server has processed. The 'Type' metric is the type of error that the DNS server has processed."},
	"FormErrors":                           {"query", "formerror", false, "", metadata.Counter, "The number of invalid format error responses returned by the server."},
	"Forwarded responses":                  {"query", "forwardedresponses", false, "", metadata.Counter, "The number of response packets received for the update requests that were forwarded to other servers."},
	"Forwarded updates in queue":           {"update", "forwardinqueue", true, "", metadata.Counter, "The number of update packets forwarded to other servers and which are waiting for a response."},
	"Free count":                           {"memory", "free", false, "", metadata.Gauge, "The cumulative number of times memory has been released for a given purpose."},
	"Into authority failures":              {"error", "errors", false, "intoauthorityfailures", metadata.Counter, "The number of errors that the DNS server has processed. The 'Type' metric is the type of error that the DNS server has processed."},
	"Invalid TKEYs":                        {"security", "securitytkeyinvalid", false, "", metadata.Counter, "The number of secure DNS update messages from which TKEY was successfully retrieved."},
	"IXFR queries":                         {"query", "queries", false, "typeixfr", metadata.Counter, "The number of queries received by the DNS server. The 'Type' tag indicates the type of request that was received (A, MX, PTR, etc)."},
	"IXFR requests":                        {"query", "requests", false, "ixfrrequest", metadata.Counter, "The number of requests that the DNS server has made to other DNS servers. For example, these could be recursive queries or zone transfers. The 'Type' tag indicates the sort of request that the DNS server has made."},
	"IXFR successes":                       {"query", "ixfrsuccesses", false, "", metadata.Counter, "The number of secure DNS update messages from which TKEY was successfully retrieved."},
	"IXFR TCP AXFR responses":              {"query", "ixfrtcpaxfr", false, "", metadata.Counter, "The number of incremental zone transfer responses received by the server over TCP indicating that full zone transfer is needed."},
	"IXFR TCP force AXFR":                  {"query", "ixfrtcpforceaxfr", false, "", metadata.Counter, "The number of IXFR TCP requests that have forced AFXR."},
	"IXFR TCP FormError responses":         {"query", "ixfrtcpformerror", false, "", metadata.Counter, "The number of incremental zone transfer responses received by the server over TCP where either the primary DNS server does not support incremental zone transfer or the primary DNS server indicated that the zone transfer request was malformed."},
	"IXFR TCP invalid responses":           {"query", "ixfrtcpinvalid", false, "", metadata.Counter, "The number of incremental zone transfer invalid responses received by the server over TCP."},
	"IXFR TCP refused responses":           {"query", "ixfrtcprefused", false, "", metadata.Counter, "The number of incremental zone transfer rejection responses received by the server over TCP."},
	"IXFR TCP requests":                    {"query", "requests", true, "ixfrtcprequest", metadata.Counter, "The number of requests that the DNS server has made to other DNS servers. For example, these could be recursive queries or zone transfers. The 'Type' tag indicates the sort of request that the DNS server has made."},
	"IXFR TCP responses":                   {"query", "ixfrtcpresponse", false, "", metadata.Counter, "The number of incremental zone transfer success responses received by the server over TCP."},
	"IXFR TCP success responses":           {"query", "ixfrtcpsuccessresponses", false, "", metadata.Counter, "The number of success responses for incremental zone transfer sent by the server."},
	"IXFR TCP successes":                   {"query", "ixfrtcpsuccess", false, "", metadata.Counter, "The number of success responses for incremental zone transfers sent by the server over TCP."},
	"IXFR UDP force AXFR":                  {"query", "ixfrudpforceaxfr", false, "", metadata.Counter, "The number of incremental zone transfer requests received by the server over UDP for which the server responded with a full zone transfer."},
	"IXFR UDP force TCP":                   {"query", "ixfrudpforcetcp", false, "", metadata.Counter, "The number of incremental zone transfer requests received by the server over UDP for which the server responded using TCP."},
	"IXFR UDP FormError responses":         {"query", "ixfrudpformerror", false, "", metadata.Counter, "The number of incremental zone transfer responses received by the server over UDP where either the master does not support incremental zone transfer or the master indicated that the zone transfer request was malformed."},
	"IXFR UDP invalid responses":           {"query", "ixfrudpinvalid", false, "", metadata.Counter, "The number of incremental zone transfer invalid responses received by the server over UDP."},
	"IXFR UDP new primary responses":       {"query", "ixfrudpnewprimary", false, "", metadata.Counter, "The number of incremental zone transfer responses received by the server over UDP where the SOA indicates a new primary server name."},
	"IXFR UDP refused responses":           {"query", "ixfrudprefused", false, "", metadata.Counter, "The number of incremental zone transfer rejection responses received by the server over UDP."},
	"IXFR UDP requests":                    {"query", "requests", true, "ixfrudprequest", metadata.Counter, "The number of requests that the DNS server has made to other DNS servers. For example, these could be recursive queries or zone transfers. The 'Type' tag indicates the sort of request that the DNS server has made."},
	"IXFR UDP responses":                   {"query", "ixfrudpresponse", false, "", metadata.Counter, "The number of incremental zone transfer success responses received by the server over UDP."},
	"IXFR UDP success responses":           {"query", "ixfrudpsuccessresponses", false, "", metadata.Counter, "The number of incremental zone transfer success responses received by the server over UDP."},
	"IXFR UDP successes":                   {"query", "ixfrudpsuccess", false, "", metadata.Counter, "The number of success responses for incremental zone transfers sent by the server over UDP."},
	"IXFR UDP UseAxfr responses":           {"query", "ixfrudpuseaxfr", false, "", metadata.Counter, "The number of incremental zone transfer responses received by the server over UDP indicating that full zone transfer is needed."},
	"IXFR UDP UseTcp responses":            {"query", "ixfrudpusetcp", false, "", metadata.Counter, "The number of incremental zone transfer responses received by the server over UDP indicating that TCP is needed."},
	"IXFR UDP wrong server responses":      {"query", "ixfrudpwrongserver", false, "", metadata.Counter, "The number of incremental zone transfer responses received by the server over UDP where the remote sender is not among the masters for this zone."},
	"LDAP average write time":              {"domainservices", "ldapwriteaverage", false, "", metadata.Counter, "The average time in milliseconds for all server performed timed LDAP write operations since the server was last restarted."},
	"LDAP reconnects":                      {"domainservices", "ldapreconnects", false, "", metadata.Counter, "The number of times the server attempted to reconnect to the directory server. This field MUST be present if and only if PollingPassesWithDsErrors is present. A client can tell whether the field is present based on the size of the buffer holding this structure."},
	"LDAP timed writes":                    {"domainservices", "ldaptimedwrites", false, "", metadata.Counter, "The number of times the server performed a timed LDAP write operation."},
	"LDAP total write time":                {"domainservices", "ldapwritetimetotal", false, "", metadata.Counter, "The cumulative time in milliseconds consumed by server-performed timed LDAP write operations."},
	"LDAP writes < 10 ms":                  {"domainservices", "writespeed", false, "ldapwritesunder10ms", metadata.Counter, "The speed at which the DNS server was able to write to the LDAP Directory Server. The 'type' tag indicates the number of writes that were undertaken in each time bracket (the time is in milliseconds)."},
	"LDAP writes < 100 ms":                 {"domainservices", "writespeed", false, "ldapwritesunder100ms", metadata.Counter, "The speed at which the DNS server was able to write to the LDAP Directory Server. The 'type' tag indicates the number of writes that were undertaken in each time bracket (the time is in milliseconds)."},
	"LDAP writes < 1 s":                    {"domainservices", "writespeed", false, "ldapwritesunder1s", metadata.Counter, "The speed at which the DNS server was able to write to the LDAP Directory Server. The 'type' tag indicates the number of writes that were undertaken in each time bracket (the time is in milliseconds)."},
	"LDAP writes < 10 s":                   {"domainservices", "writespeed", false, "ldapwritesunder10s", metadata.Counter, "The speed at which the DNS server was able to write to the LDAP Directory Server. The 'type' tag indicates the number of writes that were undertaken in each time bracket (the time is in milliseconds)."},
	"LDAP writes < 100 s":                  {"domainservices", "writespeed", false, "ldapwritesunder100s", metadata.Counter, "The speed at which the DNS server was able to write to the LDAP Directory Server. The 'type' tag indicates the number of writes that were undertaken in each time bracket (the time is in milliseconds)."},
	"LDAP writes > 100 s":                  {"domainservices", "writespeed", false, "ldapwritesover100s", metadata.Counter, "The speed at which the DNS server was able to write to the LDAP Directory Server. The 'type' tag indicates the number of writes that were undertaken in each time bracket (the time is in milliseconds)."},
	"LDAP writes for A type":               {"domainservices", "writetypes", false, "writesforatype", metadata.Counter, "The number of LDAP Directory Services updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"LDAP writes for A6 type":              {"domainservices", "writetypes", false, "writesfora6type", metadata.Counter, "The number of LDAP Directory Services updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"LDAP writes for AAAA type":            {"domainservices", "writetypes", false, "writesforaaaatype", metadata.Counter, "The number of LDAP Directory Services updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"LDAP writes for AFSDB type":           {"domainservices", "writetypes", false, "writesforafsdbtype", metadata.Counter, "The number of LDAP Directory Services updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"LDAP writes for ATMA type":            {"domainservices", "writetypes", false, "writesforatmatype", metadata.Counter, "The number of LDAP Directory Services updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"LDAP writes for CERT type":            {"domainservices", "writetypes", false, "writesforcerttype", metadata.Counter, "The number of LDAP Directory Services updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"LDAP writes for CNAME type":           {"domainservices", "writetypes", false, "writesforcnametype", metadata.Counter, "The number of LDAP Directory Services updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"LDAP writes for EID type":             {"domainservices", "writetypes", false, "writesforeidtype", metadata.Counter, "The number of LDAP Directory Services updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"LDAP writes for GPOS type":            {"domainservices", "writetypes", false, "writesforgpostype", metadata.Counter, "The number of LDAP Directory Services updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"LDAP writes for HINFO type":           {"domainservices", "writetypes", false, "writesforhinfotype", metadata.Counter, "The number of LDAP Directory Services updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"LDAP writes for ISDN type":            {"domainservices", "writetypes", false, "writesforisdntype", metadata.Counter, "The number of LDAP Directory Services updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"LDAP writes for KEY type":             {"domainservices", "writetypes", false, "writesforkeytype", metadata.Counter, "The number of LDAP Directory Services updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"LDAP writes for KX type":              {"domainservices", "writetypes", false, "writesforkxtype", metadata.Counter, "The number of LDAP Directory Services updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"LDAP writes for LOC type":             {"domainservices", "writetypes", false, "writesforloctype", metadata.Counter, "The number of LDAP Directory Services updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"LDAP writes for MB type":              {"domainservices", "writetypes", false, "writesformbtype", metadata.Counter, "The number of LDAP Directory Services updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"LDAP writes for MG type":              {"domainservices", "writetypes", false, "writesformgtype", metadata.Counter, "The number of LDAP Directory Services updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"LDAP writes for MINFO type":           {"domainservices", "writetypes", false, "writesforminfotype", metadata.Counter, "The number of LDAP Directory Services updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"LDAP writes for mixed type":           {"domainservices", "writetypes", false, "writesformixedtype", metadata.Counter, "The number of LDAP Directory Services updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"LDAP writes for MR type":              {"domainservices", "writetypes", false, "writesformrtype", metadata.Counter, "The number of LDAP Directory Services updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"LDAP writes for MX type":              {"domainservices", "writetypes", false, "writesformxtype", metadata.Counter, "The number of LDAP Directory Services updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"LDAP writes for NAPTR type":           {"domainservices", "writetypes", false, "writesfornaptrtype", metadata.Counter, "The number of LDAP Directory Services updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"LDAP writes for NIMLOC type":          {"domainservices", "writetypes", false, "writesfornimloctype", metadata.Counter, "The number of LDAP Directory Services updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"LDAP writes for NS type":              {"domainservices", "writetypes", false, "writesfornstype", metadata.Counter, "The number of LDAP Directory Services updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"LDAP writes for NSAP type":            {"domainservices", "writetypes", false, "writesfornsaptype", metadata.Counter, "The number of LDAP Directory Services updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"LDAP writes for NSAPPTR type":         {"domainservices", "writetypes", false, "writesfornsapptrtype", metadata.Counter, "The number of LDAP Directory Services updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"LDAP writes for NULL type":            {"domainservices", "writetypes", false, "writesfornulltype", metadata.Counter, "The number of LDAP Directory Services updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"LDAP writes for NXT type":             {"domainservices", "writetypes", false, "writesfornxttype", metadata.Counter, "The number of LDAP Directory Services updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"LDAP writes for PTR type":             {"domainservices", "writetypes", false, "writesforptrtype", metadata.Counter, "The number of LDAP Directory Services updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"LDAP writes for PX type":              {"domainservices", "writetypes", false, "writesforpxtype", metadata.Counter, "The number of LDAP Directory Services updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"LDAP writes for RP type":              {"domainservices", "writetypes", false, "writesforrptype", metadata.Counter, "The number of LDAP Directory Services updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"LDAP writes for RT type":              {"domainservices", "writetypes", false, "writesforrttype", metadata.Counter, "The number of LDAP Directory Services updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"LDAP writes for SIG type":             {"domainservices", "writetypes", false, "writesforsigtype", metadata.Counter, "The number of LDAP Directory Services updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"LDAP writes for SOA type":             {"domainservices", "writetypes", false, "writesforsoatype", metadata.Counter, "The number of LDAP Directory Services updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"LDAP writes for SRV type":             {"domainservices", "writetypes", false, "writesforsrvtype", metadata.Counter, "The number of LDAP Directory Services updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"LDAP writes for TXT type":             {"domainservices", "writetypes", false, "writesfortxttype", metadata.Counter, "The number of LDAP Directory Services updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"LDAP writes for unknown type":         {"domainservices", "writetypes", false, "writesforunknowntype", metadata.Counter, "The number of LDAP Directory Services updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"LDAP writes for WKS type":             {"domainservices", "writetypes", false, "writesforwkstype", metadata.Counter, "The number of LDAP Directory Services updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"LDAP writes for X25 type":             {"domainservices", "writetypes", false, "writesforx25type", metadata.Counter, "The number of LDAP Directory Services updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"LDAP writes for ZERO type":            {"domainservices", "writetypes", false, "writesforzerotype", metadata.Counter, "The number of LDAP Directory Services updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"LDAP writes max timeout":              {"domainservices", "writetypes", false, "writesmaxtimeout", metadata.Counter, "The number of LDAP Directory Services updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"Max":                                           {"error", "errors", false, "max", metadata.Counter, "The number of errors that the DNS server has processed. The 'Type' metric is the type of error that the DNS server has processed."},
	"MX queries":                                    {"query", "queries", false, "typemx", metadata.Counter, "The number of queries received by the DNS server. The 'Type' tag indicates the type of request that was received (A, MX, PTR, etc)."},
	"Name blocks allocated":                         {"memory", "name", false, "nameblocksallocated", metadata.Gauge, "Memory statistics for Name-related operations."},
	"Name blocks freed":                             {"memory", "name", false, "nameblocksfreed", metadata.Gauge, "Memory statistics for Name-related operations."},
	"Name blocks in use":                            {"memory", "name", false, "nameblocksinuse", metadata.Gauge, "Memory statistics for Name-related operations."},
	"Name error responses":                          {"recursion", "responsenameerror", false, "", metadata.Counter, "The number of name errors received by the server."},
	"Name memory":                                   {"memory", "name", false, "namememory", metadata.Gauge, "Memory statistics for Name-related operations."},
	"NameErrors":                                    {"query", "nameerror", false, "", metadata.Counter, "The number of name error responses returned by the server."},
	"Nbstat blocks allocated":                       {"netbios", "nbstatallocated", false, "", metadata.Counter, "The number of NetBIOS blocks that are currently allocated by the server."},
	"Nbstat blocks freed":                           {"netbios", "nbstatfree", false, "", metadata.Counter, "The cumulative number of NetBIOS packet buffers returned by the server to system memory."},
	"Nbstat blocks in use":                          {"netbios", "nbstatused", false, "", metadata.Counter, "The cumulative number of NetBIOS buffers currently in use by the server either servicing queries or in a free list."},
	"Nbstat buffers in use":                         {"netbios", "nbstatinuse", false, "", metadata.Counter, "The number of NetBIOS buffers currently being used by the server to service queries or being held in a free list."},
	"Nbstat buffers on free list":                   {"netbios", "nbstatinfreelist", false, "", metadata.Counter, "The number of NetBIOS buffers currently in a free list."},
	"Nbstat buffers returned":                       {"netbios", "nbstatreturn", false, "", metadata.Counter, "The cumulative number of NetBIOS buffers freed or returned by the server to a free list."},
	"Nbstat memory":                                 {"netbios", "nbstatmemory", false, "", metadata.Counter, "The total memory used by the NetBIOS packet buffers currently allocated by the server."},
	"Nbstat memory highwater mark":                  {"netbios", "nbstatnetallocs", false, "", metadata.Counter, "The number of NetBIOS packet buffers currently allocated by the server."},
	"Nbstat net buffers allocated":                  {"memory", "nbstat", true, "nbstatnetbuffersallocated", metadata.Gauge, "Memory statistics for NetBIOS."},
	"Nbstat net bytes allocated":                    {"memory", "nbstat", true, "nbstatnetbytesallocated", metadata.Gauge, "Memory statistics for NetBIOS."},
	"Nbstat total buffers allocated":                {"memory", "nbstat", true, "nbstattotalbuffersallocated", metadata.Gauge, "Memory statistics for NetBIOS."},
	"Nbstat total buffers freed":                    {"memory", "nbstat", true, "nbstattotalbuffersfreed", metadata.Gauge, "Memory statistics for NetBIOS."},
	"Node Admin blocks allocated":                   {"memory", "node.admin", false, "nodeadminblocksallocated", metadata.Gauge, "Memory statistics for Node operations for administration."},
	"Node Admin blocks freed":                       {"memory", "node.admin", false, "nodeadminblocksfreed", metadata.Gauge, "Memory statistics for Node operations for administration."},
	"Node Admin blocks in use":                      {"memory", "node.admin", false, "nodeadminblocksinuse", metadata.Gauge, "Memory statistics for Node operations for administration."},
	"Node Admin memory":                             {"memory", "node.admin", false, "nodeadminmemory", metadata.Gauge, "Memory statistics for Node operations for administration."},
	"Node Auto blocks allocated":                    {"memory", "node.auto", false, "nodeautoblocksallocated", metadata.Gauge, "Memory statistics for Node operations for autoconfig."},
	"Node Auto blocks freed":                        {"memory", "node.auto", false, "nodeautoblocksfreed", metadata.Gauge, "Memory statistics for Node operations for autoconfig."},
	"Node Auto blocks in use":                       {"memory", "node.auto", false, "nodeautoblocksinuse", metadata.Gauge, "Memory statistics for Node operations for autoconfig."},
	"Node Auto memory":                              {"memory", "node.auto", false, "nodeautomemory", metadata.Gauge, "Memory statistics for Node operations for autoconfig."},
	"Node AXFR blocks allocated":                    {"memory", "node.axfr", false, "nodeaxfrblocksallocated", metadata.Gauge, "Memory statistics for Node operations for complete zone transfers."},
	"Node AXFR blocks freed":                        {"memory", "node.axfr", false, "nodeaxfrblocksfreed", metadata.Gauge, "Memory statistics for Node operations for complete zone transfers."},
	"Node AXFR blocks in use":                       {"memory", "node.axfr", false, "nodeaxfrblocksinuse", metadata.Gauge, "Memory statistics for Node operations for complete zone transfers."},
	"Node AXFR memory":                              {"memory", "node.axfr", false, "nodeaxfrmemory", metadata.Gauge, "Memory statistics for Node operations for complete zone transfers."},
	"Node blocks allocated":                         {"memory", "node", false, "nodeblocksallocated", metadata.Gauge, "Memory statistics for Node operations for database."},
	"Node blocks freed":                             {"memory", "node", false, "nodeblocksfreed", metadata.Gauge, "Memory statistics for Node operations for database."},
	"Node blocks in use":                            {"memory", "node", false, "nodeblocksinuse", metadata.Gauge, "Memory statistics for Node operations for database."},
	"Node Cache blocks allocated":                   {"memory", "node.cache", false, "nodecacheblocksallocated", metadata.Gauge, "Memory statistics for Node operations for cache."},
	"Node Cache blocks freed":                       {"memory", "node.cache", false, "nodecacheblocksfreed", metadata.Gauge, "Memory statistics for Node operations for cache."},
	"Node Cache blocks in use":                      {"memory", "node.cache", false, "nodecacheblocksinuse", metadata.Gauge, "Memory statistics for Node operations for cache."},
	"Node Cache memory":                             {"memory", "node.cache", false, "nodecachememory", metadata.Gauge, "Memory statistics for Node operations for cache."},
	"Node Copy blocks allocated":                    {"memory", "node.copy", false, "nodecopyblocksallocated", metadata.Gauge, "Memory statistics for Node operations for copy."},
	"Node Copy blocks freed":                        {"memory", "node.copy", false, "nodecopyblocksfreed", metadata.Gauge, "Memory statistics for Node operations for copy."},
	"Node Copy blocks in use":                       {"memory", "node.copy", false, "nodecopyblocksinuse", metadata.Gauge, "Memory statistics for Node operations for copy."},
	"Node Copy memory":                              {"memory", "node.copy", false, "nodecopymemory", metadata.Gauge, "Memory statistics for Node operations for copy."},
	"Node DS blocks allocated":                      {"memory", "node.ds", false, "nodedsblocksallocated", metadata.Gauge, "Memory statistics for Node operations for directory server."},
	"Node DS blocks freed":                          {"memory", "node.ds", false, "nodedsblocksfreed", metadata.Gauge, "Memory statistics for Node operations for directory server."},
	"Node DS blocks in use":                         {"memory", "node.ds", false, "nodedsblocksinuse", metadata.Gauge, "Memory statistics for Node operations for directory server."},
	"Node DS memory":                                {"memory", "node.ds", false, "nodedsmemory", metadata.Gauge, "Memory statistics for Node operations for directory server."},
	"Node File blocks allocated":                    {"memory", "node.file", false, "nodefileblocksallocated", metadata.Gauge, "Memory statistics for Node operations for file."},
	"Node File blocks freed":                        {"memory", "node.file", false, "nodefileblocksfreed", metadata.Gauge, "Memory statistics for Node operations for file."},
	"Node File blocks in use":                       {"memory", "node.file", false, "nodefileblocksinuse", metadata.Gauge, "Memory statistics for Node operations for file."},
	"Node File memory":                              {"memory", "node.file", false, "nodefilememory", metadata.Gauge, "Memory statistics for Node operations for file."},
	"Node Hash blocks allocated":                    {"memory", "node.hash", false, "nodehashblocksallocated", metadata.Gauge, "Memory statistics for Node hash"},
	"Node Hash blocks freed":                        {"memory", "node.hash", false, "nodehashblocksfreed", metadata.Gauge, "Memory statistics for Node hash"},
	"Node Hash blocks in use":                       {"memory", "node.hash", false, "nodehashblocksinuse", metadata.Gauge, "Memory statistics for Node hash"},
	"Node Hash memory":                              {"memory", "node.hash", false, "nodehashmemory", metadata.Gauge, "Memory statistics for Node hash"},
	"Node IXFR blocks allocated":                    {"memory", "node.ixfr", false, "nodeixfrblocksallocated", metadata.Gauge, "Memory statistics for Single Record transfer operations."},
	"Node IXFR blocks freed":                        {"memory", "node.ixfr", false, "nodeixfrblocksfreed", metadata.Gauge, "Memory statistics for Single Record transfer operations."},
	"Node IXFR blocks in use":                       {"memory", "node.ixfr", false, "nodeixfrblocksinuse", metadata.Gauge, "Memory statistics for Single Record transfer operations."},
	"Node IXFR memory":                              {"memory", "node.ixfr", false, "nodeixfrmemory", metadata.Gauge, "Memory statistics for Single Record transfer operations."},
	"Node memory":                                   {"memory", "nodememory", false, "", metadata.Gauge, "The total size in bytes of server memory currently used for nodes."},
	"Node NoExist blocks allocated":                 {"memory", "node.noexist", false, "nodenoexistblocksallocated", metadata.Gauge, "Memory statistics for Node operations for non-existent records."},
	"Node NoExist blocks freed":                     {"memory", "node.noexist", false, "nodenoexistblocksfreed", metadata.Gauge, "Memory statistics for Node operations for non-existent records."},
	"Node NoExist blocks in use":                    {"memory", "node.noexist", false, "nodenoexistblocksinuse", metadata.Gauge, "Memory statistics for Node operations for non-existent records."},
	"Node NoExist memory":                           {"memory", "node.noexist", false, "nodenoexistmemory", metadata.Gauge, "Memory statistics for Node operations for non-existent records."},
	"Node Update blocks allocated":                  {"memory", "node.update", false, "nodeupdateblocksallocated", metadata.Gauge, "Memory statistics for Node operations for dynamic update."},
	"Node Update blocks freed":                      {"memory", "node.update", false, "nodeupdateblocksfreed", metadata.Gauge, "Memory statistics for Node operations for dynamic update."},
	"Node Update blocks in use":                     {"memory", "node.update", false, "nodeupdateblocksinuse", metadata.Gauge, "Memory statistics for Node operations for dynamic update."},
	"Node Update memory":                            {"memory", "node.update", false, "nodeupdatememory", metadata.Gauge, "Memory statistics for Node operations for dynamic update."},
	"Node WINS blocks allocated":                    {"memory", "node.wins", false, "nodewinsblocksallocated", metadata.Gauge, "Memory statistics for Node operations for WINS."},
	"Node WINS blocks freed":                        {"memory", "node.wins", false, "nodewinsblocksfreed", metadata.Gauge, "Memory statistics for Node operations for WINS."},
	"Node WINS blocks in use":                       {"memory", "node.wins", false, "nodewinsblocksinuse", metadata.Gauge, "Memory statistics for Node operations for WINS."},
	"Node WINS memory":                              {"memory", "node.wins", false, "nodewinsmemory", metadata.Gauge, "Memory statistics for Node operations for WINS."},
	"Node WINS-PTR blocks allocated":                {"memory", "node.winsptr", false, "nodewins-ptrblocksallocated", metadata.Gauge, "Memory statistics for Node operations for WINS-PTR."},
	"Node WINS-PTR blocks freed":                    {"memory", "node.winsptr", false, "nodewins-ptrblocksfreed", metadata.Gauge, "Memory statistics for Node operations for WINS-PTR."},
	"Node WINS-PTR blocks in use":                   {"memory", "node.winsptr", false, "nodewins-ptrblocksinuse", metadata.Gauge, "Memory statistics for Node operations for WINS-PTR."},
	"Node WINS-PTR memory":                          {"memory", "node.winsptr", false, "nodewins-ptrmemory", metadata.Gauge, "Memory statistics for Node operations for WINS-PTR."},
	"Nodes added":                                   {"domainservices", "dsnodesadded", false, "", metadata.Counter, "The number of new nodes added to the directory server."},
	"Nodes checked":                                 {"timeout", "checks", false, "", metadata.Counter, "The number of times the server performed any node timeout marking operation."},
	"Nodes deleted":                                 {"domainservices", "dsnodesdeleted", false, "", metadata.Counter, "The number of nodes deleted from the directory server."},
	"Nodes directed queued":                         {"timeout", "setdirect", false, "", metadata.Counter, "The number of times the server marked a node as being eligible for deletion when it is no longer in use by the cache by directly referencing the node."},
	"Nodes duplicate (already queued)":              {"timeout", "alreadyinsystem", false, "", metadata.Counter, "The number of times the server marked a node as being eligible for deletion when it is no longer in use by the cache when the node was already so marked."},
	"Nodes loaded":                                  {"domainservices", "dsnodesloaded", false, "", metadata.Counter, "The number of valid DNS nodes found in the directory server and loaded in memory by the server."},
	"Nodes modified":                                {"domainservices", "dsnodesmodified", false, "", metadata.Counter, "The number of nodes modified in the directory server."},
	"Nodes queued":                                  {"memory", "nodesqueued", false, "", metadata.Gauge, "The cumulative number of nodes that have been queued."},
	"Nodes queued from child delete":                {"timeout", "setfromchilddelete", false, "", metadata.Counter, "The number of times the server marked a node as being eligible for deletion when it is no longer in use by the cache because the node's last child was deleted."},
	"Nodes queued from reference":                   {"timeout", "setfromdereference", false, "", metadata.Counter, "The number of times the server marked a node as being eligible for deletion when it is no longer in use by the cache because the last reference was deleted."},
	"Nodes read":                                    {"domainservices", "dstotalnodesread", false, "", metadata.Counter, "The total number of DNS nodes read from the directory server."},
	"Nodes tombstoned":                              {"domainservices", "dsnodestombstoned", false, "", metadata.Counter, "The number of nodes tombstoned in the directory server."},
	"Nodes write suppressed":                        {"domainservices", "dswritesuppressed", false, "", metadata.Counter, "The number of records added to the directory server."},
	"NoError":                                       {"error", "errors", false, "noerror", metadata.Counter, "The number of errors that the DNS server has processed. The 'Type' metric is the type of error that the DNS server has processed."},
	"Non-zone data responses":                       {"recursion", "responsenonzonedata", false, "", metadata.Counter, "The number of error responses when a name is not found in the zone."},
	"None blocks allocated":                         {"memory", "none", false, "noneblocksallocated", metadata.Gauge, "Memory statistics for Not related to a particular operation."},
	"None blocks freed":                             {"memory", "none", false, "noneblocksfreed", metadata.Gauge, "Memory statistics for Not related to a particular operation."},
	"None blocks in use":                            {"memory", "none", false, "noneblocksinuse", metadata.Gauge, "Memory statistics for Not related to a particular operation."},
	"None memory":                                   {"memory", "none", false, "nonememory", metadata.Gauge, "Memory statistics for Not related to a particular operation."},
	"NotAuth":                                       {"error", "errors", false, "notauthoritative", metadata.Counter, "The number of errors that the DNS server has processed. The 'Type' metric is the type of error that the DNS server has processed."},
	"NotAuthoritative responses":                    {"recursion", "responsenotauthoritative", false, "", metadata.Counter, "The number of responses received from a server not authoritative for the zone."},
	"Notifies current version":                      {"query", "notifycurrentversion", false, "", metadata.Counter, "The number of zone notifications received by the server where the received SOA has same version number as that of the SOA already present on the server."},
	"Notifies invalid":                              {"query", "notifyinvalid", false, "", metadata.Counter, "The number of invalid zone notifications received by the server."},
	"Notifies master unknown":                       {"query", "notifymasterunknown", false, "", metadata.Counter, "The number of zone notifications received by the server where notifications are received from a server that is not present in the list of masters for the zone."},
	"Notifies new version":                          {"query", "notifynewversion", false, "", metadata.Counter, "The number of zone notifications received by the server where the received SOA has a newer version number than that of the SOA already present on the server."},
	"Notifies no version":                           {"query", "notifynoversion", false, "", metadata.Counter, "The number of zone notifications received by the server for which the server has no SOA."},
	"Notifies old version":                          {"query", "notifyoldversion", false, "", metadata.Counter, "The number of zone notifications received by the server where the received SOA has an older version number than the SOA already present on the server."},
	"Notifies primary":                              {"query", "notifyprimary", false, "", metadata.Counter, "The number of zone notifications for primary zones received by the server."},
	"Notifies received":                             {"query", "notifyreceived", false, "", metadata.Counter, "The number of zone notifications received by the server."},
	"Notifies sent":                                 {"query", "notifysent", false, "", metadata.Counter, "The number of update notifications sent to secondaries by the server."},
	"Notify queries":                                {"query", "queries", false, "notify", metadata.Counter, "The number of queries received by the DNS server. The 'Type' tag indicates the type of request that was received (A, MX, PTR, etc)."},
	"NotImpl":                                       {"error", "errors", false, "notimpl", metadata.Counter, "The number of errors that the DNS server has processed. The 'Type' metric is the type of error that the DNS server has processed."},
	"NotZone":                                       {"error", "errors", false, "notzone", metadata.Counter, "The number of errors that the DNS server has processed. The 'Type' metric is the type of error that the DNS server has processed."},
	"NS queries":                                    {"query", "queries", false, "typens", metadata.Counter, "The number of queries received by the DNS server. The 'Type' tag indicates the type of request that was received (A, MX, PTR, etc)."},
	"NxDomain":                                      {"error", "errors", false, "nxdomain", metadata.Counter, "The number of errors that the DNS server has processed. The 'Type' metric is the type of error that the DNS server has processed."},
	"NxRRSet":                                       {"error", "errors", false, "nxrrset", metadata.Counter, "The number of errors that the DNS server has processed. The 'Type' metric is the type of error that the DNS server has processed."},
	"Original questions recursed":                   {"recursion", "originalquestionrecursed", true, "", metadata.Counter, "The number of new recursive queries initiated."},
	"Other queries":                                 {"query", "queries", false, "typeother", metadata.Counter, "The number of queries received by the DNS server. The 'Type' tag indicates the type of request that was received (A, MX, PTR, etc)."},
	"Passes requiring aggressive free":              {"cache", "passesrequiringaggressivefree", false, "", metadata.Counter, "The number of times since the server last started that the server cache was found to exceed the cache size limit which is 90 percent of the MaxCacheSize (section 3.1.1.1.1) and that the server scanned the cache aggressively attempting to free even nodes that contain unexpired records. An aggressive scan frees in order nodes containing records that are to expire within the next hour records that are to expire within the next day and all records if needed and halts the freeing process immediately once the cache size limit is reached. After reaching 0xFFFFFFFF the value increments to 0x00000000."},
	"Passes where nothing was freed":                {"cache", "passeswithnofrees", false, "", metadata.Counter, "The number of times since the server last started that the server cache was found to exceed the cache size limit which is 90 percent of the MaxCacheSize (section 3.1.1.1.1) but when the server scanned the cache looking for nodes containing  no records or only expired DNS records to free it found no nodes that could be freed. After reaching 0xFFFFFFFF the value increments to 0x00000000."},
	"Polling passes with DS errors":                 {"domainservices", "pollingpasseswithdserrors", false, "", metadata.Counter, "The number of times the server hit failure while polling zones in the directory server. This field SHOULD<73> be present. A client can tell whether the field is present based on the size of the buffer holding this structure."},
	"Previous zone failures":                        {"recursion", "failurereachpreviousresponse", false, "", metadata.Counter, "The number of times the server received failure while performing recursive lookup on queries because the query recursed back to the domain from which a name server had already responded."},
	"PTR queries":                                   {"query", "queries", false, "typeptr", metadata.Counter, "The number of queries received by the DNS server. The 'Type' tag indicates the type of request that was received (A, MX, PTR, etc)."},
	"Queries received":                              {"query", "queriesreceived", false, "", metadata.Counter, "The cumulative number of queries received."},
	"Queries recursed":                              {"recursion", "queriesrecursed", false, "", metadata.Counter, "The number of queries received that required recursive lookups."},
	"Rcode responses":                               {"recursion", "responsercode", false, "", metadata.Counter, "The number of errors other than name errors received by the server."},
	"Recent access nodes checked":                   {"timeout", "recentaccess", false, "", metadata.Counter, "The number of times the server encountered a cache node that it could not delete because the node had recently been accessed."},
	"Record blocks allocated":                       {"memory", "record", false, "recordblocksallocated", metadata.Gauge, "Memory statistics for Record operations."},
	"Record blocks freed":                           {"memory", "record", false, "recordblocksfreed", metadata.Gauge, "Memory statistics for Record operations."},
	"Record blocks in use":                          {"memory", "record", false, "recordblocksinuse", metadata.Gauge, "Memory statistics for Record operations."},
	"Record memory":                                 {"memory", "record", false, "recordmemory", metadata.Gauge, "Memory statistics for Record operations."},
	"Records currently cached":                      {"record", "cachecurrent", false, "", metadata.Counter, "The number of resource records currently cached by the server."},
	"Records in use":                                {"record", "inuse", false, "", metadata.Counter, "The number of resource records currently allocated by the server."},
	"Records loaded":                                {"domainservices", "dsrecordsloaded", false, "", metadata.Counter, "The number of resource records loaded in memory by the server."},
	"Records memory":                                {"domainservices", "memory", false, "", metadata.Counter, "The amount of memory in bytes currently allocated for resource records by the server."},
	"Records queued for slow free":                  {"record", "slowfreequeued", false, "", metadata.Counter, "Some cached record types such as NS and SOA are not immediately freed to the pool of allocated records instead they are placed in a timeout queue and returned after this timeout expires. This is the cumulative count of such slow-free records that have been entered into the timeout queue."},
	"Records read":                                  {"domainservices", "dstotalrecordsread", false, "", metadata.Counter, "The total number of resource records read from the directory server."},
	"Records returned":                              {"record", "return", false, "", metadata.Counter, "The cumulative number of resource records freed by the server."},
	"Records slow freed":                            {"record", "slowfreefinished", false, "", metadata.Counter, "The number of slow frees (see SlowFreeQueued above) that have been completed."},
	"Records used":                                  {"record", "used", false, "", metadata.Counter, "The cumulative number of resource records allocated by the server."},
	"Recurse failures":                              {"recursion", "recursionfailure", false, "", metadata.Counter, "The number of times the server received failures for recursion queries to remote servers."},
	"Recursion messages returned":                   {"packet", "recursepacketreturn", false, "", metadata.Counter, "The cumulative number of packets that were used for recursive queries and then returned by the server to the pool of packets."},
	"Recursion messages used":                       {"packet", "recursepacketused", false, "", metadata.Counter, "The cumulative number of packets used by the server for recursion queries."},
	"Refused":                                       {"error", "errors", true, "refused", metadata.Counter, "The number of errors that the DNS server has processed. The 'Type' metric is the type of error that the DNS server has processed."},
	"Registry blocks allocated":                     {"memory", "registry", false, "registryblocksallocated", metadata.Gauge, "Memory statistics for Registry operations."},
	"Registry blocks freed":                         {"memory", "registry", false, "registryblocksfreed", metadata.Gauge, "Memory statistics for Registry operations."},
	"Registry blocks in use":                        {"memory", "registry", false, "registryblocksinuse", metadata.Gauge, "Memory statistics for Registry operations."},
	"Registry memory":                               {"memory", "registry", false, "registrymemory", metadata.Gauge, "Memory statistics for Registry operations."},
	"Remote IP blocks allocated":                    {"memory", "remoteip", false, "remoteipblocksallocated", metadata.Gauge, "Memory statistics for Remote IP address operations."},
	"Remote IP blocks freed":                        {"memory", "remoteip", false, "remoteipblocksfreed", metadata.Gauge, "Memory statistics for Remote IP address operations."},
	"Remote IP blocks in use":                       {"memory", "remoteip", false, "remoteipblocksinuse", metadata.Gauge, "Memory statistics for Remote IP address operations."},
	"Remote IP memory":                              {"memory", "remoteip", false, "remoteipmemory", metadata.Gauge, "Memory statistics for Remote IP address operations."},
	"Requests":                                      {"query", "request", false, "", metadata.Counter, ""},
	"Responses for forwarded updates":               {"query", "responsesforforwardedupdates", true, "", metadata.Counter, ""},
	"Responses from forwarders":                     {"recursion", "responsefromforwarder", false, "", metadata.Counter, ""},
	"Responses mismatched":                          {"recursion", "responsesmismatched", false, "", metadata.Counter, ""},
	"Responses sent":                                {"query", "responsessent", false, "", metadata.Counter, ""},
	"Responses unmatched":                           {"recursion", "responsesunmatched", false, "", metadata.Counter, ""},
	"Retries":                                       {"recursion", "retries", false, "", metadata.Counter, ""},
	"Retry count failures":                          {"query", "retrycountfailures", false, "", metadata.Counter, ""},
	"RPC blocks allocated":                          {"memory", "rpc", false, "rpcblocksallocated", metadata.Gauge, "Memory statistics for RPC operations."},
	"RPC blocks freed":                              {"memory", "rpc", false, "rpcblocksfreed", metadata.Gauge, "Memory statistics for RPC operations."},
	"RPC blocks in use":                             {"memory", "rpc", false, "rpcblocksinuse", metadata.Gauge, "Memory statistics for RPC operations."},
	"RPC memory":                                    {"memory", "rpc", false, "rpcmemory", metadata.Gauge, "Memory statistics for RPC operations."},
	"RR Admin blocks allocated":                     {"memory", "rr.admin", false, "rradminblocksallocated", metadata.Gauge, "Memory statistics for RR operations for administration."},
	"RR Admin blocks freed":                         {"memory", "rr.admin", false, "rradminblocksfreed", metadata.Gauge, "Memory statistics for RR operations for administration."},
	"RR Admin blocks in use":                        {"memory", "rr.admin", false, "rradminblocksinuse", metadata.Gauge, "Memory statistics for RR operations for administration."},
	"RR Admin memory":                               {"memory", "rr.admin", false, "rradminmemory", metadata.Gauge, "Memory statistics for RR operations for administration."},
	"RR Auto blocks allocated":                      {"memory", "rr.auto", false, "rrautoblocksallocated", metadata.Gauge, "Memory statistics for RR operations for autoconfig."},
	"RR Auto blocks freed":                          {"memory", "rr.auto", false, "rrautoblocksfreed", metadata.Gauge, "Memory statistics for RR operations for autoconfig."},
	"RR Auto blocks in use":                         {"memory", "rr.auto", false, "rrautoblocksinuse", metadata.Gauge, "Memory statistics for RR operations for autoconfig."},
	"RR Auto memory":                                {"memory", "rr.auto", false, "rrautomemory", metadata.Gauge, "Memory statistics for RR operations for autoconfig."},
	"RR AXFR blocks allocated":                      {"memory", "rr.axfr", false, "rraxfrblocksallocated", metadata.Gauge, "Memory statistics for Complete zone transfer operations."},
	"RR AXFR blocks freed":                          {"memory", "rr.axfr", false, "rraxfrblocksfreed", metadata.Gauge, "Memory statistics for Complete zone transfer operations."},
	"RR AXFR blocks in use":                         {"memory", "rr.axfr", false, "rraxfrblocksinuse", metadata.Gauge, "Memory statistics for Complete zone transfer operations."},
	"RR AXFR memory":                                {"memory", "rr.axfr", false, "rraxfrmemory", metadata.Gauge, "Memory statistics for Complete zone transfer operations."},
	"RR Cache blocks allocated":                     {"memory", "rr.cache", false, "rrcacheblocksallocated", metadata.Gauge, "Memory statistics for RR operations for cache."},
	"RR Cache blocks freed":                         {"memory", "rr.cache", false, "rrcacheblocksfreed", metadata.Gauge, "Memory statistics for RR operations for cache."},
	"RR Cache blocks in use":                        {"memory", "rr.cache", false, "rrcacheblocksinuse", metadata.Gauge, "Memory statistics for RR operations for cache."},
	"RR Cache memory":                               {"memory", "rr.cache", false, "rrcachememory", metadata.Gauge, "Memory statistics for RR operations for cache."},
	"RR Copy blocks allocated":                      {"memory", "rr.copy", false, "rrcopyblocksallocated", metadata.Gauge, "Memory statistics for RR copy operations."},
	"RR Copy blocks freed":                          {"memory", "rr.copy", false, "rrcopyblocksfreed", metadata.Gauge, "Memory statistics for RR copy operations."},
	"RR Copy blocks in use":                         {"memory", "rr.copy", false, "rrcopyblocksinuse", metadata.Gauge, "Memory statistics for RR copy operations."},
	"RR Copy memory":                                {"memory", "rr.copy", false, "rrcopymemory", metadata.Gauge, "Memory statistics for RR copy operations."},
	"RR DS blocks allocated":                        {"memory", "rr.ds", false, "rrdsblocksallocated", metadata.Gauge, "Memory statistics for Directory server-based RR operations."},
	"RR DS blocks freed":                            {"memory", "rr.ds", false, "rrdsblocksfreed", metadata.Gauge, "Memory statistics for Directory server-based RR operations."},
	"RR DS blocks in use":                           {"memory", "rr.ds", false, "rrdsblocksinuse", metadata.Gauge, "Memory statistics for Directory server-based RR operations."},
	"RR DS memory":                                  {"memory", "rr.ds", false, "rrdsmemory", metadata.Gauge, "Memory statistics for Directory server-based RR operations."},
	"RR File blocks allocated":                      {"memory", "rr.file", false, "rrfileblocksallocated", metadata.Gauge, "Memory statistics for File-based operations."},
	"RR File blocks freed":                          {"memory", "rr.file", false, "rrfileblocksfreed", metadata.Gauge, "Memory statistics for File-based operations."},
	"RR File blocks in use":                         {"memory", "rr.file", false, "rrfileblocksinuse", metadata.Gauge, "Memory statistics for File-based operations."},
	"RR File memory":                                {"memory", "rr.file", false, "rrfilememory", metadata.Gauge, "Memory statistics for File-based operations."},
	"RR IXFR blocks allocated":                      {"memory", "rr.ixfr", false, "rrixfrblocksallocated", metadata.Gauge, "Memory statistics for Single Record transfer operations."},
	"RR IXFR blocks freed":                          {"memory", "rr.ixfr", false, "rrixfrblocksfreed", metadata.Gauge, "Memory statistics for Single Record transfer operations."},
	"RR IXFR blocks in use":                         {"memory", "rr.ixfr", false, "rrixfrblocksinuse", metadata.Gauge, "Memory statistics for Single Record transfer operations."},
	"RR IXFR memory":                                {"memory", "rr.ixfr", false, "rrixfrmemory", metadata.Gauge, "Memory statistics for Single Record transfer operations."},
	"RR NoExist blocks allocated":                   {"memory", "rr.noexist", false, "rrnoexistblocksallocated", metadata.Gauge, "Memory statistics for RR operations for non-existent records."},
	"RR NoExist blocks freed":                       {"memory", "rr.noexist", false, "rrnoexistblocksfreed", metadata.Gauge, "Memory statistics for RR operations for non-existent records."},
	"RR NoExist blocks in use":                      {"memory", "rr.noexist", false, "rrnoexistblocksinuse", metadata.Gauge, "Memory statistics for RR operations for non-existent records."},
	"RR NoExist memory":                             {"memory", "rr.noexist", false, "rrnoexistmemory", metadata.Gauge, "Memory statistics for RR operations for non-existent records."},
	"RR sets added":                                 {"memory", "rrsetsadded", false, "", metadata.Gauge, ""},
	"RR sets replaced":                              {"domainservices", "dsrecordsreplaced", false, "", metadata.Counter, ""},
	"RR Update blocks allocated":                    {"memory", "rr.update", false, "rrupdateblocksallocated", metadata.Gauge, "Memory statistics for RR operations for dynamic update."},
	"RR Update blocks freed":                        {"memory", "rr.update", false, "rrupdateblocksfreed", metadata.Gauge, "Memory statistics for RR operations for dynamic update."},
	"RR Update blocks in use":                       {"memory", "rr.update", false, "rrupdateblocksinuse", metadata.Gauge, "Memory statistics for RR operations for dynamic update."},
	"RR Update memory":                              {"memory", "rr.update", false, "rrupdatememory", metadata.Gauge, "Memory statistics for RR operations for dynamic update."},
	"RR WINS blocks allocated":                      {"memory", "rr.wins", false, "rrwinsblocksallocated", metadata.Gauge, "Memory statistics for RR operations for WINS."},
	"RR WINS blocks freed":                          {"memory", "rr.wins", false, "rrwinsblocksfreed", metadata.Gauge, "Memory statistics for RR operations for WINS."},
	"RR WINS blocks in use":                         {"memory", "rr.wins", false, "rrwinsblocksinuse", metadata.Gauge, "Memory statistics for RR operations for WINS."},
	"RR WINS memory":                                {"memory", "rr.wins", false, "rrwinsmemory", metadata.Gauge, "Memory statistics for RR operations for WINS."},
	"RR WINS-PTR blocks allocated":                  {"memory", "rr.winsptr", false, "rrwins-ptrblocksallocated", metadata.Gauge, "Memory statistics for RR operations for WINS-PTR."},
	"RR WINS-PTR blocks freed":                      {"memory", "rr.winsptr", false, "rrwins-ptrblocksfreed", metadata.Gauge, "Memory statistics for RR operations for WINS-PTR."},
	"RR WINS-PTR blocks in use":                     {"memory", "rr.winsptr", false, "rrwins-ptrblocksinuse", metadata.Gauge, "Memory statistics for RR operations for WINS-PTR."},
	"RR WINS-PTR memory":                            {"memory", "rr.winsptr", false, "rrwins-ptrmemory", metadata.Gauge, "Memory statistics for RR operations for WINS-PTR."},
	"Safe blocks allocated":                         {"memory", "safe", false, "safeblocksallocated", metadata.Gauge, "Memory statistics for Miscellaneous queuing operations."},
	"Safe blocks freed":                             {"memory", "safe", false, "safeblocksfreed", metadata.Gauge, "Memory statistics for Miscellaneous queuing operations."},
	"Safe blocks in use":                            {"memory", "safe", false, "safeblocksinuse", metadata.Gauge, "Memory statistics for Miscellaneous queuing operations."},
	"Safe memory":                                   {"memory", "safe", false, "safememory", metadata.Gauge, "Memory statistics for Miscellaneous queuing operations."},
	"Seconds since start":                           {"time", "timeelapsedsinceserverstart", false, "", metadata.Counter, ""},
	"Seconds since statistics cleared":              {"time", "timeelapsedsincelastclearedstatistics", false, "", metadata.Counter, ""},
	"Secure update continues":                       {"update", "secureupdatecontinues", true, "", metadata.Counter, ""},
	"Secure update DS write failures":               {"update", "securedswritefailure", true, "", metadata.Counter, ""},
	"Secure update failures":                        {"update", "securefailure", true, "", metadata.Counter, ""},
	"Secure update successes":                       {"update", "securesuccess", true, "", metadata.Counter, ""},
	"Security contexts created":                     {"security", "securitycontextcreate", false, "", metadata.Counter, ""},
	"Security contexts dequeued":                    {"security", "securitycontextdequeue", false, "", metadata.Counter, ""},
	"Security contexts freed":                       {"security", "securitycontextfree", false, "", metadata.Counter, ""},
	"Security contexts queue length":                {"security", "securitycontextqueuelength", false, "", metadata.Gauge, ""},
	"Security contexts queued":                      {"security", "securitycontextqueue", false, "", metadata.Counter, ""},
	"Security contexts queued in negotiation":       {"security", "securitycontextqueueinnego", false, "", metadata.Counter, ""},
	"Security contexts queued negotiation complete": {"security", "securitycontextqueueinnegocomplete", false, "", metadata.Counter, ""},
	"Security contexts timed out":                   {"security", "securitycontexttimeout", false, "", metadata.Counter, ""},
	"Security packet contexts allocated":            {"memory", "securitypacketcontextsallocated", false, "", metadata.Gauge, ""},
	"Security packet contexts freed":                {"security", "securitypackfree", false, "", metadata.Counter, ""},
	"Security refused":                              {"query", "refusesecurity", false, "", metadata.Counter, ""},
	"Send timeouts":                                 {"recursion", "timedoutqueries", false, "", metadata.Counter, ""},
	"Sends":                                         {"recursion", "sends", false, "", metadata.Counter, ""},
	"Serial number writes":                          {"domainservices", "dsserialwrites", false, "", metadata.Counter, ""},
	"Server failure responses":                      {"recursion", "serverfailure", false, "", metadata.Counter, ""},
	"ServFail":                                      {"error", "errors", false, "servfail", metadata.Counter, "The number of errors that the DNS server has processed. The 'Type' metric is the type of error that the DNS server has processed."},
	"ServFail refused":                              {"query", "refuseserverfailure", false, "", metadata.Counter, ""},
	"Shutdown refused":                              {"query", "refuseshutdown", false, "", metadata.Counter, ""},
	"SOA invalid responses":                         {"query", "soaresponseinvalid", false, "", metadata.Counter, ""},
	"SOA NameError responses":                       {"query", "soanameerrorresponses", false, "", metadata.Counter, ""},
	"SOA queries":                                   {"query", "queries", false, "typesoa", metadata.Counter, "The number of queries received by the DNS server. The 'Type' tag indicates the type of request that was received (A, MX, PTR, etc)."},
	"SOA requests":                                  {"query", "requests", false, "soarequest", metadata.Counter, "The number of requests that the DNS server has made to other DNS servers. For example, these could be recursive queries or zone transfers. The 'Type' tag indicates the sort of request that the DNS server has made."},
	"SOA responses":                                 {"query", "soaresponse", false, "", metadata.Counter, ""},
	"Socket blocks allocated":                       {"memory", "socket", false, "socketblocksallocated", metadata.Gauge, "Memory statistics for Socket operations."},
	"Socket blocks freed":                           {"memory", "socket", false, "socketblocksfreed", metadata.Gauge, "Memory statistics for Socket operations."},
	"Socket blocks in use":                          {"memory", "socket", false, "socketblocksinuse", metadata.Gauge, "Memory statistics for Socket operations."},
	"Socket memory":                                 {"memory", "socket", false, "socketmemory", metadata.Gauge, "Memory statistics for Socket operations."},
	"SRV queries":                                   {"query", "queries", false, "typesrv", metadata.Counter, "The number of queries received by the DNS server. The 'Type' tag indicates the type of request that was received (A, MX, PTR, etc)."},
	"Standard allocs in use":                        {"memory", "standard", false, "stdinuse", metadata.Gauge, "Memory statistics for common-size blocks"},
	"Standard allocs memory":                        {"memory", "standard", false, "stdmemory", metadata.Gauge, "Memory statistics for common-size blocks"},
	"Standard allocs returned":                      {"memory", "standard", false, "stdreturn", metadata.Gauge, "Memory statistics for common-size blocks"},
	"Standard allocs used":                          {"memory", "standard", false, "stdused", metadata.Gauge, "Memory statistics for common-size blocks"},
	"Standard block memory in free list":            {"memory", "standard", false, "stdblockfreelistmemory", metadata.Gauge, "Memory statistics for common-size blocks"},
	"Standard block total memory":                   {"memory", "standard", false, "stdblockmemory", metadata.Gauge, "Memory statistics for common-size blocks"},
	"Standard blocks allocated":                     {"memory", "standard", false, "stdblockalloc", metadata.Gauge, "Memory statistics for common-size blocks"},
	"Standard blocks in free list":                  {"memory", "standard", false, "stdblockfreelist", metadata.Gauge, "Memory statistics for common-size blocks"},
	"Standard blocks in use":                        {"memory", "standard", false, "stdblockinuse", metadata.Gauge, "Memory statistics for common-size blocks"},
	"Standard blocks returned":                      {"memory", "standard", false, "stdblockreturn", metadata.Gauge, "Memory statistics for common-size blocks"},
	"Standard blocks used":                          {"memory", "standard", false, "stdblockused", metadata.Gauge, "Memory statistics for common-size blocks"},
	"Standard queries":                              {"query", "standard", false, "", metadata.Counter, ""},
	"Standard to heap allocs in use":                {"memory", "standard.toheap", false, "stdtoheapinuse", metadata.Gauge, "The number of arbitrary-size blocks of memory currently in use."},
	"Standard to heap allocs memory":                {"memory", "standard.toheap", false, "stdtoheapmemory", metadata.Gauge, "The number of arbitrary-size blocks of memory currently in use."},
	"Standard to heap allocs returned":              {"memory", "standard.toheap", false, "stdtoheapfree", metadata.Gauge, "The number of arbitrary-size blocks of memory currently in use."},
	"Standard to heap allocs used":                  {"memory", "standard.toheap", false, "stdtoheapalloc", metadata.Gauge, "The number of arbitrary-size blocks of memory currently in use."},
	"Stub zone AXFR invalid responses":              {"query", "stubaxfrinvalid", false, "", metadata.Counter, ""},
	"Stub zone AXFR refused responses":              {"query", "stubaxfrrefused", false, "", metadata.Counter, ""},
	"Stub zone AXFR requests":                       {"query", "requests", false, "stubaxfrrequest", metadata.Counter, "The number of requests that the DNS server has made to other DNS servers. For example, these could be recursive queries or zone transfers. The 'Type' tag indicates the sort of request that the DNS server has made."},
	"Stub zone AXFR responses":                      {"query", "stubaxfrresponse", false, "", metadata.Counter, ""},
	"Stub zone AXFR success responses":              {"query", "stubaxfrsuccess", false, "", metadata.Counter, ""},
	"Stuff blocks allocated":                        {"memory", "stuff", false, "stuffblocksallocated", metadata.Gauge, "Memory statistics for Miscellaneous operations."},
	"Stuff blocks freed":                            {"memory", "stuff", false, "stuffblocksfreed", metadata.Gauge, "Memory statistics for Miscellaneous operations."},
	"Stuff blocks in use":                           {"memory", "stuff", false, "stuffblocksinuse", metadata.Gauge, "Memory statistics for Miscellaneous operations."},
	"Stuff memory":                                  {"memory", "stuff", false, "stuffmemory", metadata.Gauge, "Memory statistics for Miscellaneous operations."},
	"Successful cache enforcement passes":           {"cache", "successfulfreepasses", false, "", metadata.Counter, ""},
	"Table blocks allocated":                        {"memory", "table", false, "tableblocksallocated", metadata.Gauge, "Memory statistics for Record table operations."},
	"Table blocks freed":                            {"memory", "table", false, "tableblocksfreed", metadata.Gauge, "Memory statistics for Record table operations."},
	"Table blocks in use":                           {"memory", "table", false, "tableblocksinuse", metadata.Gauge, "Memory statistics for Record table operations."},
	"Table memory":                                  {"memory", "table", false, "tablememory", metadata.Gauge, "Memory statistics for Record table operations."},
	"TCP client connections":                        {"query", "tcpclientconnections", false, "", metadata.Counter, ""},
	"TCP Connection blocks allocated":               {"memory", "tcp", false, "tcpconnectionblocksallocated", metadata.Gauge, "Memory statistics for TCP Packets."},
	"TCP Connection blocks freed":                   {"memory", "tcp", false, "tcpconnectionblocksfreed", metadata.Gauge, "Memory statistics for TCP Packets."},
	"TCP Connection blocks in use":                  {"memory", "tcp", false, "tcpconnectionblocksinuse", metadata.Gauge, "Memory statistics for TCP Packets."},
	"TCP Connection memory":                         {"memory", "tcp", false, "tcpconnectionmemory", metadata.Gauge, "Memory statistics for TCP Packets."},
	"TCP Packet blocks allocated":                   {"packet", "tcpalloc", false, "", metadata.Counter, ""},
	"TCP Packet blocks freed":                       {"packet", "tcpfree", false, "", metadata.Counter, ""},
	"TCP Packet blocks in use":                      {"packet", "tcpnetallocs", false, "", metadata.Counter, ""},
	"TCP Packet memory":                             {"packet", "tcpmemory", false, "", metadata.Counter, ""},
	"TCP queries received":                          {"query", "queriesreceived", false, "tcp", metadata.Counter, "The number of queries that the DNS server has received. The 'Type' tag contains whether or not the queries were TCP or UDP."},
	"TCP queries sent":                              {"query", "queriessent", false, "tcp", metadata.Counter, "The number of queries that the DNS server has sent to other DNS servers (for example, recursive requests or zone transfers). The 'Type' tag contains whether or not the queries were TCP or UDP."},
	"TCP recursion disconnects":                     {"recursion", "tcpdisconnect", false, "", metadata.Counter, ""},
	"TCP recursion queries":                         {"recursion", "tcpquery", false, "", metadata.Counter, ""},
	"TCP recursion responses":                       {"recursion", "tcpresponse", false, "", metadata.Counter, ""},
	"TCP recursions tried":                          {"recursion", "tcptry", false, "", metadata.Counter, ""},
	"TCP responses received":                        {"query", "responsesreceived", false, "tcp", metadata.Counter, ""},
	"TCP responses sent":                            {"query", "respondessent", false, "tcp", metadata.Counter, ""},
	"Thread blocks allocated":                       {"memory", "thread", false, "threadblocksallocated", metadata.Gauge, "Memory statistics for Thread management."},
	"Thread blocks freed":                           {"memory", "thread", false, "threadblocksfreed", metadata.Gauge, "Memory statistics for Thread management."},
	"Thread blocks in use":                          {"memory", "thread", false, "threadblocksinuse", metadata.Gauge, "Memory statistics for Thread management."},
	"Thread memory":                                 {"memory", "thread", false, "threadmemory", metadata.Gauge, "Memory statistics for Thread management."},
	"Timeout blocks allocated":                      {"memory", "timeout", false, "timeoutblocksallocated", metadata.Gauge, "Memory statistics for Timeout."},
	"Timeout blocks created":                        {"timeout", "arrayblockscreated", false, "", metadata.Counter, ""},
	"Timeout blocks deleted":                        {"timeout", "arrayblocksdeleted", false, "", metadata.Counter, ""},
	"Timeout blocks freed":                          {"memory", "timeout", false, "timeoutblocksfreed", metadata.Gauge, "Memory statistics for Timeout."},
	"Timeout blocks in use":                         {"memory", "timeout", false, "timeoutblocksinuse", metadata.Gauge, "Memory statistics for Timeout."},
	"Timeout memory":                                {"memory", "timeout", false, "timeoutmemory", metadata.Gauge, "Memory statistics for Timeout."},
	"Timeouts for forwarded updates":                {"query", "timeoutsforforwardedupdates", true, "", metadata.Counter, ""},
	"TKeyNego queries":                              {"query", "queries", false, "tkeynego", metadata.Counter, "The number of queries received by the DNS server. The 'Type' tag indicates the type of request that was received (A, MX, PTR, etc)."},
	"To forwarders":                                 {"query", "toforwarders", false, "", metadata.Counter, ""},
	"Tombstones read":                               {"domainservices", "dstombstonesread", false, "", metadata.Counter, ""},
	"Total failures":                                {"query", "totalfailures", false, "", metadata.Counter, ""},
	"Total LDAP search time":                        {"domainservices", "ldapsearchtime", false, "", metadata.Counter, ""},
	"Total memory":                                  {"memory", "totalmemory", false, "", metadata.Gauge, ""},
	"Total passes":                                  {"recursion", "lookuppasses", false, "", metadata.Counter, ""},
	"Total queries":                                 {"query", "totalqueries", false, "", metadata.Counter, ""},
	"Total questions recursed":                      {"recursion", "totalquestionsrecursed", false, "", metadata.Counter, ""},
	"Total records cached":                          {"record", "cachetotal", false, "", metadata.Counter, ""},
	"Total responses":                               {"recursion", "responses", false, "", metadata.Counter, ""},
	"Transfer failures":                             {"query", "transferfailures", false, "", metadata.Counter, ""},
	"Transfer successes":                            {"query", "transfersuccesses", false, "", metadata.Counter, ""},
	"UDP messages allocated":                        {"packet", "udpmessagesallocated", true, "", metadata.Counter, ""},
	"UDP messages freed":                            {"packet", "udpmessagesfreed", true, "", metadata.Counter, ""},
	"UDP messages in free list":                     {"packet", "udpinfreelist", false, "", metadata.Counter, ""},
	"UDP messages in use":                           {"packet", "udpinuse", false, "", metadata.Counter, ""},
	"UDP messages memory":                           {"packet", "udpmessagesmemory", true, "", metadata.Counter, ""},
	"UDP messages net allocations":                  {"packet", "udpmessagesnetallocations", true, "", metadata.Counter, ""},
	"UDP messages reallocated":                      {"memory", "udpmessagesreallocated", false, "", metadata.Gauge, ""},
	"UDP messages returned":                         {"packet", "udpreturn", false, "", metadata.Counter, ""},
	"UDP messages used":                             {"packet", "udpused", false, "", metadata.Counter, ""},
	"UDP Packet blocks allocated":                   {"packet", "udpalloc", false, "", metadata.Counter, ""},
	"UDP Packet blocks freed":                       {"packet", "udpfree", false, "", metadata.Counter, ""},
	"UDP Packet blocks in use":                      {"packet", "udpnetallocs", false, "", metadata.Counter, ""},
	"UDP Packet memory":                             {"packet", "udpmemory", false, "", metadata.Counter, ""},
	"UDP queries received":                          {"query", "queriesreceived", false, "udp", metadata.Counter, "The number of queries that the DNS server has received. The 'Type' tag contains whether or not the queries were TCP or UDP."},
	"UDP queries sent":                              {"query", "queriessent", false, "udp", metadata.Counter, "The number of queries that the DNS server has sent to other DNS servers (for example, recursive requests or zone transfers). The 'Type' tag contains whether or not the queries were TCP or UDP."},
	"UDP responses received":                        {"query", "responsesreceived", false, "udp", metadata.Counter, ""},
	"UDP responses sent":                            {"query", "respondessent", false, "udp", metadata.Counter, ""},
	"UnknownError":                                  {"error", "errors", false, "unknownerror", metadata.Counter, "The number of errors that the DNS server has processed. The 'Type' metric is the type of error that the DNS server has processed."},
	"Unsecure responses":                            {"recursion", "responseunsecure", false, "", metadata.Counter, ""},
	"Update aging off":                              {"domainservices", "updateagingoff", false, "", metadata.Counter, ""},
	"Update aging on":                               {"domainservices", "updateagingon", false, "", metadata.Counter, ""},
	"Update aging refresh":                          {"domainservices", "updateaginrefresh", false, "", metadata.Counter, ""},
	"Update blocks allocated":                       {"memory", "update", false, "updateblocksallocated", metadata.Gauge, "Memory statistics for Name updates."},
	"Update blocks freed":                           {"memory", "update", false, "updateblocksfreed", metadata.Gauge, "Memory statistics for Name updates."},
	"Update blocks in use":                          {"memory", "update", false, "updateblocksinuse", metadata.Gauge, "Memory statistics for Name updates."},
	"Update from admin":                             {"domainservices", "updateadmin", false, "", metadata.Counter, ""},
	"Update from auto config":                       {"domainservices", "updateautoconfig", false, "", metadata.Counter, ""},
	"Update from packet":                            {"domainservices", "updatepacket", false, "", metadata.Counter, ""},
	"Update from packet (precon)":                   {"domainservices", "updatepacketprecon", false, "", metadata.Counter, ""},
	"Update from scavenge":                          {"domainservices", "updatescavenge", false, "", metadata.Counter, ""},
	"Update List blocks allocated":                  {"memory", "updatelist", false, "updatelistblocksallocated", metadata.Gauge, "Memory statistics for Record update list."},
	"Update List blocks freed":                      {"memory", "updatelist", false, "updatelistblocksfreed", metadata.Gauge, "Memory statistics for Record update list."},
	"Update List blocks in use":                     {"memory", "updatelist", false, "updatelistblocksinuse", metadata.Gauge, "Memory statistics for Record update list."},
	"Update List memory":                            {"memory", "updatelist", false, "updatelistmemory", metadata.Gauge, "Memory statistics for Record update list."},
	"Update lists":                                  {"domainservices", "updatelists", false, "", metadata.Counter, ""},
	"Update memory":                                 {"memory", "updatememory", false, "", metadata.Gauge, ""},
	"Update nodes":                                  {"domainservices", "updatenodes", false, "", metadata.Counter, ""},
	"Update nodes read":                             {"domainservices", "dsupdatenodesread", false, "", metadata.Counter, ""},
	"Update queries":                                {"query", "queries", false, "update", metadata.Counter, "The number of queries received by the DNS server. The 'Type' tag indicates the type of request that was received (A, MX, PTR, etc)."},
	"Update record changes":                         {"domainservices", "updaterecordchange", false, "", metadata.Counter, ""},
	"Update records read":                           {"domainservices", "dsupdaterecordsread", false, "", metadata.Counter, ""},
	"Update searches":                               {"domainservices", "dsupdatesearches", false, "", metadata.Counter, ""},
	"Update tombstones":                             {"domainservices", "updatetombstones", false, "", metadata.Counter, ""},
	"Update writes":                                 {"domainservices", "updatewrites", false, "", metadata.Counter, ""},
	"Updates completed":                             {"update", "completed", true, "", metadata.Counter, ""},
	"Updates empty (precon only)":                   {"update", "empty", true, "", metadata.Counter, ""},
	"Updates for A type":                            {"update", "updates", true, "a", metadata.Counter, "The number of dynamic updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"Updates for A6 type":                           {"update", "updates", true, "a6", metadata.Counter, "The number of dynamic updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"Updates for AAAA type":                         {"update", "updates", true, "aaaa", metadata.Counter, "The number of dynamic updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"Updates for AFSDB type":                        {"update", "updates", true, "afsdb", metadata.Counter, "The number of dynamic updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"Updates for ATMA type":                         {"update", "updates", true, "atma", metadata.Counter, "The number of dynamic updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"Updates for CERT type":                         {"update", "updates", true, "cert", metadata.Counter, "The number of dynamic updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"Updates for CNAME type":                        {"update", "updates", true, "cname", metadata.Counter, "The number of dynamic updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"Updates for EID type":                          {"update", "updates", true, "eid", metadata.Counter, "The number of dynamic updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"Updates for GPOS type":                         {"update", "updates", true, "gpos", metadata.Counter, "The number of dynamic updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"Updates for HINFO type":                        {"update", "updates", true, "hinfo", metadata.Counter, "The number of dynamic updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"Updates for ISDN type":                         {"update", "updates", true, "isdn", metadata.Counter, "The number of dynamic updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"Updates for KEY type":                          {"update", "updates", true, "key", metadata.Counter, "The number of dynamic updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"Updates for KX type":                           {"update", "updates", true, "kx", metadata.Counter, "The number of dynamic updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"Updates for LOC type":                          {"update", "updates", true, "loc", metadata.Counter, "The number of dynamic updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"Updates for MB type":                           {"update", "updates", true, "mb", metadata.Counter, "The number of dynamic updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"Updates for MG type":                           {"update", "updates", true, "mg", metadata.Counter, "The number of dynamic updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"Updates for MINFO type":                        {"update", "updates", true, "minfo", metadata.Counter, "The number of dynamic updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"Updates for mixed type":                        {"update", "updates", true, "mixed", metadata.Counter, "The number of dynamic updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"Updates for MR type":                           {"update", "updates", true, "mr", metadata.Counter, "The number of dynamic updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"Updates for MX type":                           {"update", "updates", true, "mx", metadata.Counter, "The number of dynamic updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"Updates for NAPTR type":                        {"update", "updates", true, "naptr", metadata.Counter, "The number of dynamic updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"Updates for NIMLOC type":                       {"update", "updates", true, "nimloc", metadata.Counter, "The number of dynamic updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"Updates for NS type":                           {"update", "updates", true, "ns", metadata.Counter, "The number of dynamic updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"Updates for NSAP type":                         {"update", "updates", true, "nsap", metadata.Counter, "The number of dynamic updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"Updates for NSAPPTR type":                      {"update", "updates", true, "nsapptr", metadata.Counter, "The number of dynamic updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"Updates for NULL type":                         {"update", "updates", true, "null", metadata.Counter, "The number of dynamic updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"Updates for NXT type":                          {"update", "updates", true, "nxt", metadata.Counter, "The number of dynamic updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"Updates for PTR type":                          {"update", "updates", true, "ptr", metadata.Counter, "The number of dynamic updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"Updates for PX type":                           {"update", "updates", true, "px", metadata.Counter, "The number of dynamic updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"Updates for RP type":                           {"update", "updates", true, "rp", metadata.Counter, "The number of dynamic updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"Updates for RT type":                           {"update", "updates", true, "rt", metadata.Counter, "The number of dynamic updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"Updates for SIG type":                          {"update", "updates", true, "sig", metadata.Counter, "The number of dynamic updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"Updates for SOA type":                          {"update", "updates", true, "soa", metadata.Counter, "The number of dynamic updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"Updates for SRV type":                          {"update", "updates", true, "srv", metadata.Counter, "The number of dynamic updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"Updates for TXT type":                          {"update", "updates", true, "txt", metadata.Counter, "The number of dynamic updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"Updates for unknown type":                      {"update", "updates", true, "unknown", metadata.Counter, "The number of dynamic updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"Updates for WKS type":                          {"update", "updates", true, "wks", metadata.Counter, "The number of dynamic updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"Updates for X25 type":                          {"update", "updates", true, "x25", metadata.Counter, "The number of dynamic updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"Updates for ZERO type":                         {"update", "updates", true, "zero", metadata.Counter, "The number of dynamic updates that the DNS server has received. The 'Type' tag indicates the type of request that was received (e.g. A, MX, PTR, etc)"},
	"Updates forwarded":                             {"update", "forwards", true, "", metadata.Counter, ""},
	"Updates forwarded via TCP":                     {"update", "tcpforwards", true, "", metadata.Counter, ""},
	"Updates in queue":                              {"update", "inqueue", true, "", metadata.Counter, ""},
	"Updates NoOps (duplicates)":                    {"update", "noops", true, "", metadata.Counter, ""},
	"Updates received":                              {"update", "received", true, "", metadata.Counter, ""},
	"Updates rejected":                              {"update", "rejected", true, "", metadata.Counter, ""},
	"Updates rejected with FormError":               {"update", "rejections", true, "rejectedwithformerror", metadata.Counter, "The number of updates that the DNS server rejected. The 'Type' tag indicates the reason that the update was rejected."},
	"Updates rejected with NameError":               {"update", "rejections", true, "rejectedwithnameerror", metadata.Counter, "The number of updates that the DNS server rejected. The 'Type' tag indicates the reason that the update was rejected."},
	"Updates rejected with NotAuth":                 {"update", "rejections", true, "rejectedwithnotauth", metadata.Counter, "The number of updates that the DNS server rejected. The 'Type' tag indicates the reason that the update was rejected."},
	"Updates rejected with NotImpl":                 {"update", "rejections", true, "rejectedwithnotimpl", metadata.Counter, "The number of updates that the DNS server rejected. The 'Type' tag indicates the reason that the update was rejected."},
	"Updates rejected with NotZone":                 {"update", "rejections", true, "rejectedwithnotzone", metadata.Counter, "The number of updates that the DNS server rejected. The 'Type' tag indicates the reason that the update was rejected."},
	"Updates rejected with NxRRSet":                 {"update", "rejections", true, "rejectedwithnxrrset", metadata.Counter, "The number of updates that the DNS server rejected. The 'Type' tag indicates the reason that the update was rejected."},
	"Updates rejected with Refused":                 {"update", "rejections", true, "rejectedwithrefused", metadata.Counter, "The number of updates that the DNS server rejected. The 'Type' tag indicates the reason that the update was rejected."},
	"Updates rejected with Refused (access denied)": {"update", "rejections", true, "rejectedwithrefusedaccessdenied", metadata.Counter, "The number of updates that the DNS server rejected. The 'Type' tag indicates the reason that the update was rejected."},
	"Updates rejected with Refused (nonsecure)":     {"update", "rejections", true, "rejectedwithrefusednonsecure", metadata.Counter, "The number of updates that the DNS server rejected. The 'Type' tag indicates the reason that the update was rejected."},
	"Updates rejected with YxDomain":                {"update", "rejections", true, "rejectedwithyxdomain", metadata.Counter, "The number of updates that the DNS server rejected. The 'Type' tag indicates the reason that the update was rejected."},
	"Updates rejected with YxRRSet":                 {"update", "rejections", true, "rejectedwithyxrrset", metadata.Counter, "The number of updates that the DNS server rejected. The 'Type' tag indicates the reason that the update was rejected."},
	"Updates retried":                               {"update", "retried", true, "", metadata.Counter, ""},
	"Updates suppressed":                            {"domainservices", "updatesuppressed", false, "", metadata.Counter, ""},
	"Updates timed out":                             {"update", "timedout", true, "", metadata.Counter, ""},
	"Verify failed TSIGs":                           {"security", "verifyfailedtsigs", false, "", metadata.Counter, ""},
	"Verify success TSIGs":                          {"security", "verifytsigverifysuccess", false, "", metadata.Counter, ""},
	"WINS forward lookup responses":                 {"wins", "winsresponses", false, "", metadata.Counter, ""},
	"WINS forward lookups":                          {"wins", "winslookups", false, "", metadata.Counter, ""},
	"WINS reverse lookup responses":                 {"wins", "winsreverseresponses", false, "", metadata.Counter, ""},
	"WINS reverse lookups":                          {"wins", "winsreverselookups", false, "", metadata.Counter, ""},
	"YxDomain":                                      {"error", "errors", false, "yxdomain", metadata.Counter, "The number of updates that the DNS server rejected. The 'Type' tag indicates the reason that the update was rejected."},
	"YxRRSet":                                       {"error", "errors", false, "yxrrset", metadata.Counter, "The number of errors that the DNS server has processed. The 'Type' metric is the type of error that the DNS server has processed."},
	"Zone blocks allocated":                         {"memory", "zone", false, "zoneblocksallocated", metadata.Gauge, "Memory statistics for Zone operations."},
	"Zone blocks freed":                             {"memory", "zone", false, "zoneblocksfreed", metadata.Gauge, "Memory statistics for Zone operations."},
	"Zone blocks in use":                            {"memory", "zone", false, "zoneblocksinuse", metadata.Gauge, "Memory statistics for Zone operations."},
	"Zone memory":                                   {"memory", "zone", false, "zonememory", metadata.Gauge, "Memory statistics for Zone operations."},
}
