"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerMiddleware = exports.removeForBot = exports.enableForBot = void 0;

require("bluebird-global");

var _lodash = _interopRequireDefault(require("lodash"));

var _dialogflow = require("./engines/dialogflow");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const configPerBot = {};

const enableForBot = (botId, config) => {
  configPerBot[botId] = config;
};

exports.enableForBot = enableForBot;

const removeForBot = botId => delete configPerBot[botId];

exports.removeForBot = removeForBot;

const registerMiddleware = async bp => {
  bp.events.registerMiddleware({
    name: 'nlu-extras.incoming',
    direction: 'incoming',
    order: 11,
    description: 'Process natural language in the form of text. Structured data with an action and parameters for that action is injected in the incoming message event.',
    handler: async (event, next) => {
      const config = configPerBot[event.botId];
      const enabled = config && config.dialogflow && config.dialogflow.enabled;

      if (!enabled || !event.preview || event.hasFlag(bp.IO.WellKnownFlags.SKIP_NATIVE_NLU)) {
        return next();
      }

      try {
        const result = await (0, _dialogflow.predict)(config.dialogflow, event.preview, event.nlu.language, event.nlu.includedContexts);

        if (config.primary === 'dialogflow-nlu') {
          event.nlu['engine'] = 'dialogflow';
          event.nlu['botpress-nlu'] = _lodash.default.cloneDeep(event.nlu);
          event.nlu.intent.name = result.intent;
          event.nlu.intent.confidence = result.confidence;
          Object.assign(event.nlu, {
            intents: [event.nlu.intent]
          });
          Object.assign(event.nlu.slots, result.slots);
        } else {
          event.nlu['engine'] = 'botpress';
          event.nlu['dialogflow-nlu'] = { ...result,
            engine: 'dialogflow'
          };
        }
      } catch (err) {
        bp.logger.warn('Error extracting NLU from Dialogflow: ' + err.message);
      } finally {
        next();
      }
    }
  });
};

exports.registerMiddleware = registerMiddleware;
//# sourceMappingURL=middleware.js.map