import { lang } from 'botpress/shared'
import moment from 'moment'
import ms from 'ms'
import React, { Component } from 'react'
import { IoIosArchive } from 'react-icons/io'
import { connect } from 'react-redux'
import Select from 'react-select'
import { Col, Jumbotron, Label, Row } from 'reactstrap'
import PageContainer from '~/App/PageContainer'
import SplitPage from '~/App/SplitPage'
import CheckRequirements from '~/Pages/Components/CheckRequirements'
import LoadingSection from '~/Pages/Components/LoadingSection'
import IncidentsTable from '~/Pages/Components/Monitoring/IncidentsTable'
import { fetchIncidents } from '~/reducers/monitoring'

interface Props {
  incidents: any
  loadingIncidents: boolean
  fetchIncidents: (from, to) => void
}

interface State {
  intervalId: any
  timeFrame: any
  autoRefresh: boolean
  timeFrameOptions: any
  error?: string
}

class Alerts extends Component<Props, State> {
  timeFrameOptions = [
    { value: '1h', label: lang.tr('admin.alerting.timespanHour') },
    { value: '2h', label: lang.tr('admin.alerting.timespanHours', { nb: 2 }) },
    { value: '5h', label: lang.tr('admin.alerting.timespanHours', { nb: 5 }) },
    { value: '15h', label: lang.tr('admin.alerting.timespanHours', { nb: 15 }) },
    { value: '24h', label: lang.tr('admin.alerting.timespanHours', { nb: 24 }) }
  ]

  state: State = {
    intervalId: undefined,
    timeFrame: undefined,
    timeFrameOptions: this.timeFrameOptions,
    autoRefresh: false,
    error: undefined
  }

  componentDidMount() {
    this.setState({ timeFrame: this.timeFrameOptions[0] }, this.queryData)
  }

  componentWillUnmount() {
    if (this.state.intervalId) {
      clearInterval(this.state.intervalId)
    }
  }

  async queryData() {
    if (!this.state.timeFrame) {
      return
    }

    try {
      const fromTime = moment()
        .subtract(ms(this.state.timeFrame.value))
        .toDate()
        .getTime()

      const toTime = moment()
        .toDate()
        .getTime()

      this.props.fetchIncidents(fromTime, toTime)
    } catch (error) {
      this.setState({ error: error.message })
    }
  }

  handleTimeFrameChanged = timeFrame => this.setState({ timeFrame }, this.queryData)
  handleAutoRefreshChanged = event => {
    const autoRefresh = event.target.checked
    let intervalId

    if (autoRefresh && !this.state.intervalId) {
      intervalId = setInterval(() => this.queryData(), 10000)
    } else if (!autoRefresh && this.state.intervalId) {
      clearInterval(this.state.intervalId)
    }
    this.setState({ autoRefresh, intervalId })
  }

  renderTables() {
    if (!this.props.incidents) {
      return null
    }

    return (
      <div>
        <h4>{lang.tr('admin.alerting.activeIncidents')}</h4>
        <IncidentsTable data={this.props.incidents.active} />

        <h4>{lang.tr('admin.alerting.resolvedIncidents')}</h4>
        <IncidentsTable data={this.props.incidents.resolved} />
      </div>
    )
  }

  renderNoData() {
    return (
      <PageContainer title={lang.tr('admin.alerting.alertingAndIncidents')}>
        <Jumbotron>
          <Row>
            <Col style={{ textAlign: 'center' }} sm="12" md={{ size: 8, offset: 2 }}>
              <h1>
                <IoIosArchive />
                &nbsp; {lang.tr('admin.alerting.notEnabled')}
              </h1>
              <p>{lang.tr('admin.alerting.makeSureEnabled')}</p>
            </Col>
          </Row>
        </Jumbotron>
      </PageContainer>
    )
  }

  renderChild() {
    if (!this.props.incidents) {
      return this.renderNoData()
    }

    if (this.props.loadingIncidents) {
      return <LoadingSection />
    }

    return <SplitPage sideMenu={this.renderSideMenu()}>{this.renderTables()}</SplitPage>
  }

  renderSideMenu() {
    const reactSelectStyle = {
      control: base => ({ ...base, minHeight: 30 }),
      dropdownIndicator: base => ({ ...base, padding: 4 }),
      clearIndicator: base => ({ ...base, padding: 4 }),
      valueContainer: base => ({ ...base, padding: '0px 6px' }),
      input: base => ({ ...base, margin: 0, padding: 0 })
    }

    return (
      <div style={{ fontSize: '80%' }}>
        <strong>{lang.tr('admin.alerting.timeFrame')}</strong>
        <Select
          styles={reactSelectStyle}
          options={this.timeFrameOptions}
          value={this.state.timeFrame}
          onChange={this.handleTimeFrameChanged}
          isSearchable={false}
        />
        <strong>{lang.tr('admin.alerting.autoRefresh')}</strong>
        <br />
        <Label>
          <input
            style={{ marginTop: 8 }}
            name="autoRefresh"
            type="checkbox"
            checked={this.state.autoRefresh}
            onChange={this.handleAutoRefreshChanged}
          />{' '}
          <strong>{lang.tr('enabled')}</strong>
        </Label>
      </div>
    )
  }

  render() {
    return (
      <PageContainer title={lang.tr('admin.alerting.alertingAndIncidents')} fullWidth={true} superAdmin={true}>
        <CheckRequirements requirements={['redis', 'pro', 'monitoring']} feature="alerting">
          {this.renderChild()}
        </CheckRequirements>
      </PageContainer>
    )
  }
}

const mapStateToProps = state => ({
  incidents: state.monitoring.incidents
})

const mapDispatchToProps = { fetchIncidents }

export default connect(mapStateToProps, mapDispatchToProps)(Alerts)
