#!/bin/bash

if [[ ! -f bin/gren_token ]]; then
    echo "File bin/gren_token is needed; should contain a Github token with repo access"
    exit 1
fi

run_gren() {
    docker run -it \
        -v $(pwd):/data \
        -w /data \
        -u $(id -u):$(id -g) \
        -e GREN_GITHUB_TOKEN=$(cat bin/gren_token) \
        ignite-relnotes \
        /bin/bash -c "gren $@"
}

make_tidy_autogen() {
    make autogen tidy graph
    if [[ $(git status --short) != "" ]]; then
        git add -A
        git commit -m "Ran 'make autogen tidy graph'"
    fi
}

gen_changelog_md() {
    file="CHANGELOG.md"
    echo '<!-- Note: This file is autogenerated based on files in docs/releases. Run hack/release.sh to update -->' > ${file}
    echo "" >> ${file}
    echo "# Changelog" >> ${file}
    echo "" >> ${file}
    
    # Generate docs/releases/next.md based of GH release notes
    run_gren "changelog"
    # Add the new release and existing ones to the changelog
    cat docs/releases/${FULL_VERSION}.md docs/releases/next.md >> ${file}
    # Remove the temporary file
    rm docs/releases/next.md
}

write_changelog() {
    # Build the gren image
    docker build -t ignite-relnotes hack/relnotes

    # Generate the changelog draft
    if [[ ! -f docs/releases/${FULL_VERSION}.md ]]; then
        # Push the tag provisionally, we'll later update it
        echo "Tagging the current commit ${FULL_VERSION} temporarily in order to run gren..."
        git tag -f ${FULL_VERSION}
        git push upstream --tags -f

        echo "Creating a changelog for PRs between tags ${FULL_VERSION}..${PREVIOUS_TAG}"
        run_gren "changelog --generate --tags=${FULL_VERSION}..${PREVIOUS_TAG}"
        mv docs/releases/next.md docs/releases/${FULL_VERSION}.md
        # Add an extra newline in the end of the changelog
        echo "" >> docs/releases/${FULL_VERSION}.md
    fi

    read -p "Please manually fixup the changelog file now. Continue? [y/N] " confirm
    if [[ ! ${confirm} =~ ^[Yy]$ ]]; then
        exit 1
    fi

    # Generate the CHANGELOG.md file
    gen_changelog_md

    # Proceed with making the commit
    read -p "Are you sure you want to do a commit for the changelog? [y/N] " confirm
    if [[ ! ${confirm} =~ ^[Yy]$ ]]; then
        exit 1
    fi

    git add -A
    git commit -m "Release ${FULL_VERSION}"
}

build_push_release_artifacts() {
    make release
    # Do this at a later stage
    #make -C images push-all
}
