#' @include tabula.R utilities.R
NULL

# Class definitions ============================================================
## Permutation order -----------------------------------------------------------
#' Permutation order
#'
#' An S4 class to represent a permutation order.
#' @param object A \code{PermutationOrder} object.
#' @slot rows A \code{\link{integer}} vector giving the rows permutation.
#' @slot columns A \code{\link{integer}} vector giving the columns permutation.
#' @slot seriation A \code{\link{character}} vector indicating the seriation
#'  method used.
#' @author N. Frerebeau
#' @docType class
#' @aliases PermutationOrder-class
setClass(
  Class = "PermutationOrder",
  slots = c(rows = "integer",
            columns = "integer",
            method = "character")
)

## Numeric matrix --------------------------------------------------------------
setClass(
  Class = "NumericMatrix",
  contains = "matrix"
)

#' Count matrix
#'
#' An S4 class to represent a count matrix.
#' @inheritParams base::matrix
#' @note
#'  This class extends the \code{base} \link[base]{matrix}.
#' @seealso \link[base]{matrix}
#' @family abundance matrix
#' @example inst/examples/ex-abundance-class.R
#' @author N. Frerebeau
#' @docType class
#' @aliases CountMatrix-class
setClass(
  Class = "CountMatrix",
  contains = "NumericMatrix"
)

#' Frequency matrix
#'
#' An S4 class to represent a frequency matrix.
#' @param object A \code{FrequencyMatrix} object.
#' @slot total A \code{\link{numeric}} vector.
#' @details
#'  To ensure data integrity, a \code{FrequencyMatrix} can only be created by
#'  coercion from a \linkS4class{CountMatrix} (see examples).
#' @note This class extends the \code{base} \link[base]{matrix}.
#' @seealso \link[base]{matrix}
#' @family abundance matrix
#' @example inst/examples/ex-abundance-class.R
#' @author N. Frerebeau
#' @docType class
#' @aliases FrequencyMatrix-class
setClass(
  Class = "FrequencyMatrix",
  slots = c(totals = "numeric"),
  contains = "NumericMatrix"
)

## Logical matrix --------------------------------------------------------------
setClass(
  Class = "LogicalMatrix",
  contains = "matrix"
)

#' Incidence matrix
#'
#' An S4 class to represent an incidence (presence/absence) matrix.
#' @inheritParams base::matrix
#' @note This class extends the \code{base} \link[base]{matrix}.
#' @seealso \link[base]{matrix}
#' @family logical matrix
#' @example inst/examples/ex-logical-class.R
#' @author N. Frerebeau
#' @docType class
#' @aliases IncidenceMatrix-class
setClass(
  Class = "IncidenceMatrix",
  contains = "LogicalMatrix"
)

#' Co-occurrence matrix
#'
#' An S4 class to represent a co-occurrence matrix.
#' @details
#'  A co-occurrence matrix is a symetric matrix with zeros on its main diagonal,
#'  which works out which pairs of taxa occur together in at least one sample
#' @note This class extends the \code{base} \link[base]{matrix}.
#' @seealso \link[base]{matrix}
#' @family logical matrix
#' @example inst/examples/ex-logical-class.R
#' @author N. Frerebeau
#' @docType class
#' @aliases OccurrenceMatrix-class
setClass(
  Class = "OccurrenceMatrix",
  contains = "LogicalMatrix"
)

# Class validation =============================================================
## PermutationOrder ------------------------------------------------------------
setValidity(
  Class = "PermutationOrder",
  method = function(object) {
    errors <- c()
    # Get data
    rows <- object@rows
    columns <- object@columns
    method <- object@method

    if (length(rows) != 0) {
      if (!is.integer(rows))
        errors <- c(errors, "whole numbers are expected")
      if (any(is.na(rows)))
        errors <- c(errors, "NA values were detected")
      if (!any(is.nan(rows)))
        if (any(rows <= 0))
          errors <- c(errors, "strictly positive values are expected")
    }
    if (length(columns) != 0) {
      if (!is.integer(columns))
        errors <- c(errors, "whole numbers are expected")
      if (any(is.na(columns)))
        errors <- c(errors, "NA values were detected")
      if (!any(is.nan(columns)))
        if (any(columns <= 0))
          errors <- c(errors, "strictly positive values are expected")
    }
    if (length(rows) != 0 | length(columns) != 0) {
      if (length(method) == 1) {
        if (!is.character(method))
          errors <- c(errors, "a character string is expected")
      } else {
        errors <- c(errors, "should be of length 1")
      }
    }
    # Return errors if any
    if (length(errors) != 0) {
      stop(paste(errors, collapse = "\n"))
    } else {
      return(TRUE)
    }
  }
)

## NumericMatrix ---------------------------------------------------------------
setValidity(
  Class = "NumericMatrix",
  method = function(object) {
    errors <- c()
    # Get data
    data <- S3Part(object, strictS3 = TRUE, "matrix")
    if (length(data) != 0) {
      if (!is.numeric(data))
        errors <- c(errors, "numeric values are expected")
      if (any(is.na(data)))
        errors <- c(errors, "NA values were detected")
      if (any(is.infinite(data)))
        errors <- c(errors, "infinite numbers were detected")
      if (!any(is.nan(data)))
        if (any(data < 0))
          errors <- c(errors, "positive values are expected")
    }
    # Return errors if any
    if (length(errors) != 0) {
      stop(paste(errors, collapse = "\n"))
    } else {
      return(TRUE)
    }
  }
)

## CountMatrix -----------------------------------------------------------------
setValidity(
  Class = "CountMatrix",
  method = function(object) {
    errors <- c()
    # Get data
    data <- methods::S3Part(object, strictS3 = TRUE, "matrix")
    if (length(data) != 0) {
      if (sum(!isWholeNumber(data)) != 0)
        errors <- c(errors, "whole numbers are expected")
      if (isBinary(data))
        errors <- c(errors, "you should consider using an incidence matrix")
    }
    # Return errors, if any
    if (length(errors) != 0) {
      stop(paste(errors, collapse = "\n"))
    } else {
      return(TRUE)
    }
  }
)

## FrequencyMatrix -------------------------------------------------------------
setValidity(
  Class = "FrequencyMatrix",
  method = function(object) {
    errors <- c()
    # Get data
    data <- methods::S3Part(object, strictS3 = TRUE, "matrix")
    totals <- object@totals
    if (length(data) != 0) {
      if (!isEqual(rowSums(data, na.rm = TRUE)))
        errors <- c(errors, "frequencies are expected")
      if (isBinary(data))
        errors <- c(errors, "you should consider using an incidence matrix")
      if (length(totals) != nrow(data))
        errors <- c(errors, "wrong row sums")
    }
    # Return errors, if any
    if (length(errors) != 0) {
      stop(paste(errors, collapse = "\n"))
    } else {
      return(TRUE)
    }
  }
)

## LogicalMatrix ---------------------------------------------------------------
setValidity(
  Class = "LogicalMatrix",
  method = function(object) {
    errors <- c()
    # Get data
    data <- methods::S3Part(object, strictS3 = TRUE, "matrix")
    if (length(data) != 0) {
      if (!is.logical(data))
        errors <- c("logical values are expected")
      if (any(is.na(data)))
        errors <- c(errors, "NA values were detected")
    }
    # Return errors, if any
    if (length(errors) != 0) {
      stop(paste(errors, collapse = "\n"))
    } else {
      return(TRUE)
    }
  }
)

## OccurrenceMatrix ------------------------------------------------------------
setValidity(
  Class = "OccurrenceMatrix",
  method = function(object) {
    errors <- c()
    # Get data
    data <- methods::S3Part(object, strictS3 = TRUE, "matrix")

    if (length(data) != 0) {
      if (nrow(data) != ncol(data))
        errors <- c(errors, "a square matrix is expected")
      if (!identical(rownames(data), colnames(data)))
        errors <- c(errors, "rows and columns should have the same names")
    }
    # Return errors, if any
    if (length(errors) != 0) {
      stop(paste(errors, collapse = "\n"))
    } else {
      return(TRUE)
    }
  }
)

# Class constructors ===========================================================
## PermutationOrder ------------------------------------------------------------
setMethod(
  f = "initialize",
  signature = "PermutationOrder",
  definition = function(.Object, rows, columns, method) {
    if (!missing(rows)) .Object@rows <- rows
    if (!missing(columns)) .Object@columns <- columns
    if (!missing(method)) .Object@method <- method
    methods::validObject(.Object)
    if (getOption("verbose")) {
      message(paste(class(.Object), "instance initialized.", sep = " "))
    }
    return(.Object)
  }
)

## *Matrix ---------------------------------------------------------------------
initialize_matrix <- function(.Object, ...) {
  .Object <- methods::callNextMethod(.Object, ...)
  methods::validObject(.Object)
  if (getOption("verbose")) {
    message(paste(class(.Object), "instance initialized.", sep = " "))
  }
  return(.Object)
}
setMethod("initialize", "CountMatrix", initialize_matrix)
setMethod("initialize", "FrequencyMatrix", initialize_matrix)
setMethod("initialize", "IncidenceMatrix", initialize_matrix)
setMethod("initialize", "OccurrenceMatrix", initialize_matrix)

# Show =========================================================================
## PermutationOrder ------------------------------------------------------------
setMethod(
  f = "show",
  signature = "PermutationOrder",
  definition = function(object) {
    cat("Permutation order for matrix seriation:", "\n",
        "  Row order:", object@rows, "\n",
        "  Column order:", object@columns, "\n",
        "  Method:", object@method,
        sep = " "
    )
  }
)
## Numeric matrix --------------------------------------------------------------
setMethod(
  f = "show",
  signature = "CountMatrix",
  definition = function(object) {
    data <- methods::S3Part(object, strictS3 = TRUE, "matrix")
    m <- nrow(data)
    p <- ncol(data)
    cat(paste(m, "x", p, "count data matrix:", sep = " "), "\n", sep = " ")
    print(data)
  }
)
setMethod(
  f = "show",
  signature = "FrequencyMatrix",
  definition = function(object) {
    data <- methods::S3Part(object, strictS3 = TRUE, "matrix")
    m <- nrow(data)
    p <- ncol(data)
    cat(paste(m, "x", p, "frequency data matrix:", sep = " "), "\n", sep = " ")
    print(data)
  }
)
## Logical matrix --------------------------------------------------------------
setMethod(
  f = "show",
  signature = "IncidenceMatrix",
  definition = function(object) {
    data <- methods::S3Part(object, strictS3 = TRUE, "matrix")
    m <- nrow(data)
    p <- ncol(data)
    cat(paste(m, "x", p, "presence/absence data matrix:", sep = " "), "\n",
        sep = " ")
    print(data)
  }
)
setMethod(
  f = "show",
  signature = "OccurrenceMatrix",
  definition = function(object) {
    data <- methods::S3Part(object, strictS3 = TRUE, "matrix")
    m <- nrow(data)
    p <- ncol(data)
    cat(paste(m, "x", p, "co-occurrence matrix:", sep = " "), "\n",
        sep = " ")
    print(data)
  }
)

# Accessors ====================================================================
#' Accessors
#'
#' @param object An object.
#' @author N. Frerebeau
#' @docType methods
#' @name accessors
#' @rdname accessors
NULL

#' @rdname accessors
setGeneric("columns", function(object) standardGeneric("columns"))

#' @rdname accessors
setGeneric("method", function(object) standardGeneric("method"))

#' @rdname accessors
setGeneric("rows", function(object) standardGeneric("rows"))

#' @rdname accessors
setGeneric("totals", function(object) standardGeneric("totals"))

#' @export
#' @describeIn PermutationOrder Returns the rows permutation.
#' @aliases rows,PermutationOrder-method
setMethod("rows", "PermutationOrder", function(object) object@rows)

#' @export
#' @describeIn PermutationOrder Returns the columns permutation.
#' @aliases columns,PermutationOrder-method
setMethod("columns", "PermutationOrder", function(object) object@columns)

#' @export
#' @describeIn PermutationOrder Returns the method used for seriation.
#' @aliases method,PermutationOrder-method
setMethod("method", "PermutationOrder", function(object) object@method)

#' @export
#' @describeIn FrequencyMatrix Returns the row sums (counts).
#' @aliases totals,FrequencyMatrix-method
setMethod("totals", "FrequencyMatrix", function(object) object@totals)
