## Trial and Report

[`Trial`][amltk.optimization.trial.Trial] - typically the output of
[`Optimizer.ask()`][amltk.optimization.Optimizer.ask], indicating
what the optimizer would like to evaluate next. 
e provide a host of convenience methods attached to the `Trial` to make it easy to
save results, store artifacts, and more.

[`Trial.Report`][amltk.optimization.trial.Trial.Report] - 
the output of a [`trial.success(cost=...)`][amltk.optimization.trial.Trial.success] or
[`trial.fail(cost=...)`][amltk.optimization.trial.Trial.fail] call.
Provides an easy way to report back to the optimizer's
[`tell()`][amltk.optimization.Optimizer.tell].



### Trial
A [`Trial`][amltk.optimization.Trial] encapsulates some configuration
that needs to be evaluated. Typically, this is what is generated by an
[`Optimizer.ask()`][amltk.optimization.Optimizer.ask] call.

- [`trial.success()`][amltk.optimization.Trial.success] to generate a
success [`Report`][amltk.optimization.Trial.Report], typically
passing what your chosen optimizer expects, e.g., `"loss"` or `"cost"`.

- [`trial.fail()`][amltk.optimization.Trial.fail] to generate a
failure [`Report`][amltk.optimization.Trial.Report].
If an exception is passed to `fail()`, it will be attached to the report along with any traceback it can deduce.
Each [`Optimizer`][amltk.optimization.Optimizer] will take care of what to do from here.

```python exec="true" source="material-block" result="python"
from amltk.optimization import Trial, Metric
from amltk.store import PathBucket

cost = Metric("cost", minimize=True)

def target_function(trial: Trial) -> Trial.Report:
    x = trial.config["x"]
    y = trial.config["y"]

    with trial.profile("expensive-calculation"):
        cost = x**2 - y

    return trial.success(cost=cost)

# ... usually obtained from an optimizer
trial = Trial.create(
    name="some-unique-name",
    config={"x": 1, "y": 2},
    metrics=[cost]
)

report = target_function(trial)
print(report.df())
trial.bucket.rmdir()  # markdown-exec: hide
```


What you can return with [`trial.success()`][amltk.optimization.Trial.success]
or [`trial.fail()`][amltk.optimization.Trial.fail] depends on the
[`metrics`][amltk.optimization.Trial.metrics] of the trial. Typically,
an optimizer will provide the trial with the list of [metrics](../optimization/metrics.md)

Some important properties are that they have a unique
[`.name`][amltk.optimization.Trial.name] given the optimization run,
a candidate [`.config`][amltk.optimization.Trial.config] to evaluate,
a possible [`.seed`][amltk.optimization.Trial.seed] to use,
and an [`.info`][amltk.optimization.Trial.info] object, which is the optimizer
specific information, if required by you.

!!! tip "Reporting success (or failure)"

    When using the [`success()`][amltk.optimization.trial.Trial.success]
    method, make sure to provide values for all metrics specified in the
    [`.metrics`][amltk.optimization.Trial.metrics] attribute.
    Usually these are set by the optimizer generating the `Trial`.

    If you instead report using [`fail()`][amltk.optimization.trial.Trial.success],
    any metric not specified will be set to the
    [`.worst`][amltk.optimization.Metric.worst] value of the metric.

    Each metric has a unique name, and it's crucial to use the correct names when
    reporting success, otherwise an error will occur.

    ??? example "Reporting success for metrics"

        For example:

        ```python exec="true" result="python" source="material-block"
        from amltk.optimization import Trial, Metric

        # Gotten from some optimizer usually, i.e. via `optimizer.ask()`
        trial = Trial.create(
            name="example_trial",
            config={"param": 42},
            metrics=[Metric(name="accuracy", minimize=False)]
        )

        # Incorrect usage (will raise an error)
        try:
            report = trial.success(invalid_metric=0.95)
        except ValueError as error:
            print(error)

        # Correct usage
        report = trial.success(accuracy=0.95)
        trial.bucket.rmdir()  # markdown-exec: hide
        ```

If using [`Plugins`][amltk.scheduling.plugins.Plugin], they may insert
some extra objects in the [`.extra`][amltk.optimization.Trial.extras] dict.

To profile your trial, you can wrap the logic you'd like to check with
[`trial.profile()`][amltk.optimization.Trial.profile], which will automatically
profile the block of code for memory before and after as well as time taken.

If you've [`profile()`][amltk.optimization.Trial.profile]'ed any intervals,
you can access them by name through
[`trial.profiles`][amltk.optimization.Trial.profiles].
Please see the [`Profiler`][amltk.profiling.profiler.Profiler]
for more.

??? example "Profiling with a trial."

    ```python exec="true" source="material-block" result="python" title="profile"
    from amltk.optimization import Trial

    trial = Trial.create(name="some-unique-name", config={})

    # ... somewhere where you've begun your trial.
    with trial.profile("some_interval"):
        for work in range(100):
            pass

    print(trial.profiler.df())
    trial.bucket.rmdir()  # markdown-exec: hide
    ```

You can also record anything you'd like into the
[`.summary`][amltk.optimization.Trial.summary], a plain `#!python dict`
or use [`trial.store()`][amltk.optimization.Trial.store] to store artifacts
related to the trial.

??? tip "What to put in `.summary`?"

    For large items, e.g. predictions or models, these are highly advised to
    [`.store()`][amltk.optimization.Trial.store] to disk, especially if using
    a `Task` for multiprocessing.

    Further, if serializing the report using the
    [`report.df()`][amltk.optimization.Trial.Report.df],
    returning a single row,
    or a [`History`][amltk.optimization.History]
    with [`history.df()`][amltk.optimization.History.df] for a dataframe consisting
    of many of the reports, then you'd likely only want to store things
    that are scalar and can be serialised to disk by a pandas DataFrame.


### Report
The [`Trial.Report`][amltk.optimization.Trial.Report] encapsulates
a [`Trial`][amltk.optimization.Trial], its status and any metrics/exceptions
that may have occured.

Typically you will not create these yourself, but instead use
[`trial.success()`][amltk.optimization.Trial.success] or
[`trial.fail()`][amltk.optimization.Trial.fail] to generate them.

```python exec="true" source="material-block" result="python"
from amltk.optimization import Trial, Metric

loss = Metric("loss", minimize=True)

trial = Trial.create(name="trial", config={"x": 1}, metrics=[loss])

with trial.profile("fitting"):
    # Do some work
    # ...
    report = trial.success(loss=1)

print(report.df())
trial.bucket.rmdir()  # markdown-exec: hide
```

These reports are used to report back metrics to an
[`Optimizer`][amltk.optimization.Optimizer]
with [`Optimizer.tell()`][amltk.optimization.Optimizer.tell] but can also be
stored for your own uses.

You can access the original trial with the
[`.trial`][amltk.optimization.Trial.Report.trial] attribute, and the
[`Status`][amltk.optimization.Trial.Status] of the trial with the
[`.status`][amltk.optimization.Trial.Report.status] attribute.

You may also want to check out the [`History`][amltk.optimization.History] class
for storing a collection of `Report`s, allowing for an easier time to convert
them to a dataframe or perform some common Hyperparameter optimization parsing
of metrics.
