/* Copyright 2018 The TensorFlow Authors. All Rights Reserved.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
==============================================================================*/

#ifndef TENSORFLOW_GRAPPLER_GRAPPLER_TEST_H_
#define TENSORFLOW_GRAPPLER_GRAPPLER_TEST_H_

#include <vector>

#include "tensorflow/core/framework/attr_value.pb.h"
#include "tensorflow/core/framework/graph.pb.h"
#include "tensorflow/core/framework/types.h"
#include "tensorflow/core/grappler/grappler_item.h"
#include "tensorflow/core/grappler/utils.h"
#include "tensorflow/core/platform/test.h"
#include "tensorflow/core/public/session_options.h"

namespace tensorflow {
namespace grappler {

class GrapplerTest : public ::testing::Test {
 public:
  GrapplerTest();

 protected:
  std::vector<Tensor> EvaluateNodes(
      const GraphDef& graph, const std::vector<string>& node_names) const;

  std::vector<Tensor> EvaluateNodes(
      const GraphDef& graph, const std::vector<string>& node_names,
      const std::vector<std::pair<string, Tensor>>& inputs) const;

  std::vector<Tensor> EvaluateFetchNodes(const GrapplerItem& item) const;

  NodeDef* AddNode(const string& name, const string& op,
                   const std::vector<string>& inputs,
                   const std::vector<std::pair<string, AttrValue>>& attributes,
                   GraphDef* graph) const;

  void CompareGraphs(GraphDef want, GraphDef got) const;

  // Check if node 'src' is directly connected to the input($position) of 'dst'.
  bool IsNodesDirectlyConnected(const NodeMap& node_map, const string& src,
                                const string& dst, int position = 0);

  // Count nodes of the given op-type in a graph.
  int CountOpNodes(const GraphDef& graph, const string& op);

  // Get a random tansor with given shape.
  template <DataType DTYPE>
  Tensor GenerateRandomTensor(const TensorShape& shape) const {
    typedef typename EnumToDataType<DTYPE>::Type T;
    Tensor tensor(DTYPE, shape);
    tensor.flat<T>() = tensor.flat<T>().random();
    return tensor;
  }

 private:
  SessionOptions options_;
};

}  // end namespace grappler
}  // end namespace tensorflow

#endif  // TENSORFLOW_GRAPPLER_GRAPPLER_TEST_H_
