// Copyright (c) 2015-present, Qihoo, Inc.  All rights reserved.
// This source code is licensed under the BSD-style license found in the
// LICENSE file in the root directory of this source tree. An additional grant
// of patent rights can be found in the PATENTS file in the same directory.

#include "include/pika_conf.h"

#include <glog/logging.h>

#include <algorithm>
#include <strings.h>

#include "slash/include/env.h"

#include "include/pika_define.h"

PikaConf::PikaConf(const std::string& path)
    : slash::BaseConf(path), conf_path_(path) {
  pthread_rwlock_init(&rwlock_, NULL);
  local_meta_ = new PikaMeta();
}

PikaConf::~PikaConf() {
  pthread_rwlock_destroy(&rwlock_);
  delete local_meta_;
}

Status PikaConf::InternalGetTargetTable(const std::string& table_name, uint32_t* const target) {
  int32_t table_index = -1;
  for (size_t idx = 0; table_structs_.size(); ++idx) {
    if (table_structs_[idx].table_name == table_name) {
      table_index = idx;
      break;
    }
  }
  if (table_index == -1) {
    return Status::NotFound("table : " + table_name + " not found");
  }
  *target = table_index;
  return Status::OK();
}

Status PikaConf::TablePartitionsSanityCheck(const std::string& table_name,
                                            const std::set<uint32_t>& partition_ids,
                                            bool is_add) {
  RWLock l(&rwlock_, false);
  uint32_t table_index = 0;
  Status s = InternalGetTargetTable(table_name, &table_index);
  if (!s.ok()) {
    return s;
  }
  // Sanity Check
  for (const auto& id : partition_ids) {
    if (id >= table_structs_[table_index].partition_num) {
      return Status::Corruption("partition index out of range");
    } else if (is_add && table_structs_[table_index].partition_ids.count(id) != 0) {
      return Status::Corruption("partition : " + std::to_string(id) + " exist");
    } else if (!is_add && table_structs_[table_index].partition_ids.count(id) == 0) {
      return Status::Corruption("partition : " + std::to_string(id) + " not exist");
    }
  }
  return Status::OK();
}

Status PikaConf::AddTablePartitions(const std::string& table_name,
                                    const std::set<uint32_t>& partition_ids) {
  Status s = TablePartitionsSanityCheck(table_name, partition_ids, true);
  if (!s.ok()) {
    return s;
  }

  RWLock l(&rwlock_, true);
  uint32_t index = 0;
  s = InternalGetTargetTable(table_name, &index);
  if (s.ok()) {
    for (const auto& id : partition_ids) {
      table_structs_[index].partition_ids.insert(id);
    }
    s = local_meta_->StableSave(table_structs_);
  }
  return s;
}

Status PikaConf::RemoveTablePartitions(const std::string& table_name,
                                       const std::set<uint32_t>& partition_ids) {
  Status s = TablePartitionsSanityCheck(table_name, partition_ids, false);
  if (!s.ok()) {
    return s;
  }

  RWLock l(&rwlock_, true);
  uint32_t index = 0;
  s = InternalGetTargetTable(table_name, &index);
  if (s.ok()) {
    for (const auto& id : partition_ids) {
      table_structs_[index].partition_ids.erase(id);
    }
    s = local_meta_->StableSave(table_structs_);
  }
  return s;
}

int PikaConf::Load()
{
  int ret = LoadConf();
  if (ret != 0) {
    return ret;
  }

  GetConfInt("timeout", &timeout_);
  if (timeout_ < 0) {
      timeout_ = 60; // 60s
  }
  GetConfStr("server-id", &server_id_);
  if (server_id_.empty()) {
    server_id_ = "1";
  }
  GetConfStr("requirepass", &requirepass_);
  GetConfStr("masterauth", &masterauth_);
  GetConfStr("userpass", &userpass_);
  GetConfInt("maxclients", &maxclients_);
  if (maxclients_ <= 0) {
    maxclients_ = 20000;
  }
  GetConfInt("root-connection-num", &root_connection_num_);
  if (root_connection_num_ < 0) {
      root_connection_num_ = 2;
  }

  std::string swe;
  GetConfStr("slowlog-write-errorlog", &swe);
  slowlog_write_errorlog_.store(swe == "yes" ? true : false);

  int tmp_slowlog_log_slower_than;
  GetConfInt("slowlog-log-slower-than", &tmp_slowlog_log_slower_than);
  slowlog_log_slower_than_.store(tmp_slowlog_log_slower_than);
  GetConfInt("slowlog-max-len", &slowlog_max_len_);
  if (slowlog_max_len_ == 0) {
    slowlog_max_len_ = 128;
  }
  std::string user_blacklist;
  GetConfStr("userblacklist", &user_blacklist);
  slash::StringSplit(user_blacklist, COMMA, user_blacklist_);
  for (auto& item : user_blacklist_) {
    slash::StringToLower(item);
  }

  GetConfStr("dump-path", &bgsave_path_);
  bgsave_path_ = bgsave_path_.empty() ? "./dump/" : bgsave_path_;
  if (bgsave_path_[bgsave_path_.length() - 1] != '/') {
    bgsave_path_ += "/";
  }
  GetConfInt("dump-expire", &expire_dump_days_);
  if (expire_dump_days_ < 0 ) {
      expire_dump_days_ = 0;
  }
  GetConfStr("dump-prefix", &bgsave_prefix_);

  GetConfInt("expire-logs-nums", &expire_logs_nums_);
  if (expire_logs_nums_ <= 10 ) {
      expire_logs_nums_ = 10;
  }
  GetConfInt("expire-logs-days", &expire_logs_days_);
  if (expire_logs_days_ <= 0 ) {
      expire_logs_days_ = 1;
  }
  GetConfStr("compression", &compression_);
  // set slave read only true as default
  slave_read_only_ = true;
  GetConfInt("slave-priority", &slave_priority_);

  //
  // Immutable Sections
  //
  GetConfInt("port", &port_);
  GetConfStr("log-path", &log_path_);
  log_path_ = log_path_.empty() ? "./log/" : log_path_;
  if (log_path_[log_path_.length() - 1] != '/') {
    log_path_ += "/";
  }
  GetConfStr("db-path", &db_path_);
  db_path_ = db_path_.empty() ? "./db/" : db_path_;
  if (db_path_[db_path_.length() - 1] != '/') {
    db_path_ += "/";
  }
  local_meta_->SetPath(db_path_);

  GetConfInt("thread-num", &thread_num_);
  if (thread_num_ <= 0) {
    thread_num_ = 12;
  }
  if (thread_num_ > 24) {
    thread_num_ = 24;
  }
  GetConfInt("thread-pool-size", &thread_pool_size_);
  if (thread_pool_size_ <= 0) {
    thread_pool_size_ = 12;
  }
  if (thread_pool_size_ > 24) {
    thread_pool_size_ = 24;
  }
  GetConfInt("sync-thread-num", &sync_thread_num_);
  if (sync_thread_num_ <= 0) {
    sync_thread_num_ = 3;
  }
  if (sync_thread_num_ > 24) {
    sync_thread_num_ = 24;
  }

  std::string instance_mode;
  GetConfStr("instance-mode", &instance_mode);
  classic_mode_.store(instance_mode.empty()
          || !strcasecmp(instance_mode.data(), "classic"));

  if (classic_mode_.load()) {
    GetConfInt("databases", &databases_);
    if (databases_ < 1 || databases_ > 8) {
      LOG(FATAL) << "config databases error, limit [1 ~ 8], the actual is: "
          << databases_;
    }
    for (int idx = 0; idx < databases_; ++idx) {
      table_structs_.push_back({"db" + std::to_string(idx), 1, {0}});
    }
  } else {
    GetConfInt("default-slot-num", &default_slot_num_);
    if (default_slot_num_ <= 0) {
      LOG(FATAL) << "config default-slot-num error,"
          << " it should greater than zero, the actual is: "
          << default_slot_num_;
    }
    std::string pika_meta_path = db_path_ + kPikaMeta;
    if (!slash::FileExists(pika_meta_path)) {
      local_meta_->StableSave({{"db0", static_cast<uint32_t>(default_slot_num_), {}}});
    }
    Status s = local_meta_->ParseMeta(&table_structs_);
    if (!s.ok()) {
      LOG(FATAL) << "parse meta file error";
    }
  }
  default_table_ = table_structs_[0].table_name;

  compact_cron_ = "";
  GetConfStr("compact-cron", &compact_cron_);
  if (compact_cron_ != "") {
    bool have_week = false;
    std::string compact_cron, week_str;
    int slash_num = count(compact_cron_.begin(), compact_cron_.end(), '/');
    if (slash_num == 2) {
      have_week = true;
      std::string::size_type first_slash = compact_cron_.find("/");
      week_str = compact_cron_.substr(0, first_slash);
      compact_cron = compact_cron_.substr(first_slash + 1);
    } else {
      compact_cron = compact_cron_;
    }

    std::string::size_type len = compact_cron.length();
    std::string::size_type colon = compact_cron.find("-");
    std::string::size_type underline = compact_cron.find("/");
    if (colon == std::string::npos || underline == std::string::npos ||
        colon >= underline || colon + 1 >= len ||
        colon + 1 == underline || underline + 1 >= len) {
        compact_cron_ = "";
    } else {
      int week = std::atoi(week_str.c_str());
      int start = std::atoi(compact_cron.substr(0, colon).c_str());
      int end = std::atoi(compact_cron.substr(colon + 1, underline).c_str());
      int usage = std::atoi(compact_cron.substr(underline + 1).c_str());
      if ((have_week && (week < 1 || week > 7)) || start < 0 || start > 23 || end < 0 || end > 23 || usage < 0 || usage > 100) {
        compact_cron_ = "";
      }
    }
  }

  compact_interval_ = "";
  GetConfStr("compact-interval", &compact_interval_);
  if (compact_interval_ != "") {
    std::string::size_type len = compact_interval_.length();
    std::string::size_type slash = compact_interval_.find("/");
    if (slash == std::string::npos || slash + 1 >= len) {
      compact_interval_ = "";
    } else {
      int interval = std::atoi(compact_interval_.substr(0, slash).c_str());
      int usage = std::atoi(compact_interval_.substr(slash+1).c_str());
      if (interval <= 0 || usage < 0 || usage > 100) {
        compact_interval_ = "";
      }
    }
  }

  // write_buffer_size
  GetConfInt64("write-buffer-size", &write_buffer_size_);
  if (write_buffer_size_ <= 0 ) {
    write_buffer_size_ = 268435456;       // 256Mb
  }

  // max_write_buffer_size
  GetConfInt64("max-write-buffer-size", &max_write_buffer_size_);
  if (max_write_buffer_size_ <= 0) {
    max_write_buffer_size_ = 10737418240;  // 10Gb
  }

  // max_client_response_size
  GetConfInt64("max-client-response-size", &max_client_response_size_);
  if (max_client_response_size_ <= 0) {
    max_client_response_size_ = 1073741824; // 1Gb
  }

  // target_file_size_base
  GetConfInt("target-file-size-base", &target_file_size_base_);
  if (target_file_size_base_ <= 0) {
    target_file_size_base_ = 1048576;     // 10Mb
  }

  max_cache_statistic_keys_ = 0;
  GetConfInt("max-cache-statistic-keys", &max_cache_statistic_keys_);
  if (max_cache_statistic_keys_ <= 0) {
    max_cache_statistic_keys_ = 0;
  }

  small_compaction_threshold_ = 5000;
  GetConfInt("small-compaction-threshold", &small_compaction_threshold_);
  if (small_compaction_threshold_ <= 0
    || small_compaction_threshold_ >= 100000) {
    small_compaction_threshold_ = 5000;
  }

  max_background_flushes_ = 1;
  GetConfInt("max-background-flushes", &max_background_flushes_);
  if (max_background_flushes_ <= 0) {
    max_background_flushes_ = 1;
  }
  if (max_background_flushes_ >= 4) {
    max_background_flushes_ = 4;
  }

  max_background_compactions_ = 2;
  GetConfInt("max-background-compactions", &max_background_compactions_);
  if (max_background_compactions_ <= 0) {
    max_background_compactions_ = 2;
  }
  if (max_background_compactions_ >= 8) {
    max_background_compactions_ = 8;
  }

  max_cache_files_ = 5000;
  GetConfInt("max-cache-files", &max_cache_files_);
  if (max_cache_files_ < -1) {
    max_cache_files_ = 5000;
  }
  max_bytes_for_level_multiplier_ = 10;
  GetConfInt("max-bytes-for-level-multiplier", &max_bytes_for_level_multiplier_);
  if (max_bytes_for_level_multiplier_ < 10) {
    max_bytes_for_level_multiplier_ = 5;
  }

  block_size_ = 4 * 1024;
  GetConfInt64("block-size", &block_size_);
  if (block_size_ <= 0) {
    block_size_ = 4 * 1024;
  }

  block_cache_ = 8 * 1024 * 1024;
  GetConfInt64("block-cache", &block_cache_);
  if (block_cache_ < 0) {
    block_cache_ = 8 * 1024 * 1024;
  }

  std::string sbc;
  GetConfStr("share-block-cache", &sbc);
  share_block_cache_ = (sbc == "yes") ? true : false;

  std::string ciafb;
  GetConfStr("cache-index-and-filter-blocks", &ciafb);
  cache_index_and_filter_blocks_ = (ciafb == "yes") ? true : false;

  std::string offh;
  GetConfStr("optimize-filters-for-hits", &offh);
  optimize_filters_for_hits_ = (offh == "yes") ? true : false;

  std::string lcdlb;
  GetConfStr("level-compaction-dynamic-level-bytes", &lcdlb);
  level_compaction_dynamic_level_bytes_ = (lcdlb == "yes") ? true : false;

  // daemonize
  std::string dmz;
  GetConfStr("daemonize", &dmz);
  daemonize_ =  (dmz == "yes") ? true : false;

  // binlog
  std::string wb;
  GetConfStr("write-binlog", &wb);
  write_binlog_ = (wb == "no") ? false : true;
  GetConfInt("binlog-file-size", &binlog_file_size_);
  if (binlog_file_size_ < 1024
    || static_cast<int64_t>(binlog_file_size_) > (1024LL * 1024 * 1024)) {
    binlog_file_size_ = 100 * 1024 * 1024;    // 100M
  }
  GetConfStr("pidfile", &pidfile_);

  // db sync
  GetConfStr("db-sync-path", &db_sync_path_);
  db_sync_path_ = db_sync_path_.empty() ? "./dbsync/" : db_sync_path_;
  if (db_sync_path_[db_sync_path_.length() - 1] != '/') {
    db_sync_path_ += "/";
  }
  GetConfInt("db-sync-speed", &db_sync_speed_);
  if (db_sync_speed_ < 0 || db_sync_speed_ > 1024) {
    db_sync_speed_ = 1024;
  }
  // network interface
  network_interface_ = "";
  GetConfStr("network-interface", &network_interface_);

  // slaveof
  slaveof_ = "";
  GetConfStr("slaveof", &slaveof_);
  return ret;
}

void PikaConf::TryPushDiffCommands(const std::string& command, const std::string& value) {
  if (!CheckConfExist(command)) {
    diff_commands_[command] = value;
  }
}

int PikaConf::ConfigRewrite() {
  std::string userblacklist = suser_blacklist();

  RWLock l(&rwlock_, true);
  // Only set value for config item that can be config set.
  SetConfInt("timeout", timeout_);
  SetConfStr("requirepass", requirepass_);
  SetConfStr("masterauth", masterauth_);
  SetConfStr("userpass", userpass_);
  SetConfStr("userblacklist", userblacklist);
  SetConfStr("dump-prefix", bgsave_prefix_);
  SetConfInt("maxclients", maxclients_);
  SetConfInt("dump-expire", expire_dump_days_);
  SetConfInt("expire-logs-days", expire_logs_days_);
  SetConfInt("expire-logs-nums", expire_logs_nums_);
  SetConfInt("root-connection-num", root_connection_num_);
  SetConfStr("slowlog-write-errorlog", slowlog_write_errorlog_.load() ? "yes" : "no");
  SetConfInt("slowlog-log-slower-than", slowlog_log_slower_than_.load());
  SetConfInt("slowlog-max-len", slowlog_max_len_);
  SetConfStr("write-binlog", write_binlog_ ? "yes" : "no");
  SetConfInt("max-cache-statistic-keys", max_cache_statistic_keys_);
  SetConfInt("small-compaction-threshold", small_compaction_threshold_);
  SetConfInt("max-client-response-size", max_client_response_size_);
  SetConfInt("db-sync-speed", db_sync_speed_);
  SetConfStr("compact-cron", compact_cron_);
  SetConfStr("compact-interval", compact_interval_);
  SetConfInt("slave-priority", slave_priority_);
  // slaveof config item is special
  SetConfStr("slaveof", slaveof_);

  if (!diff_commands_.empty()) {
    std::vector<slash::BaseConf::Rep::ConfItem> filtered_items;
    for (const auto& diff_command : diff_commands_) {
      if (!diff_command.second.empty()) {
        slash::BaseConf::Rep::ConfItem item(slash::BaseConf::Rep::kConf, diff_command.first, diff_command.second);
        filtered_items.push_back(item);
      }
    }
    if (!filtered_items.empty()) {
      slash::BaseConf::Rep::ConfItem comment_item(slash::BaseConf::Rep::kComment, "# Generated by CONFIG REWRITE\n");
      PushConfItem(comment_item);
      for (const auto& item : filtered_items) {
        PushConfItem(item);
      }
    }
    diff_commands_.clear();
  }
  return WriteBack();
}
