//! Convert [`winit`] types into [`iced_native`] types, and viceversa.
//!
//! [`winit`]: https://github.com/rust-windowing/winit
//! [`iced_native`]: https://github.com/hecrj/iced/tree/master/native
use crate::keyboard;
use crate::mouse;
use crate::touch;
use crate::window;
use crate::{Event, Mode, Point};

/// Converts a winit window event into an iced event.
pub fn window_event(
    event: &winit::event::WindowEvent,
    scale_factor: f64,
    modifiers: winit::keyboard::ModifiersState,
) -> Option<Event> {
    use winit::event::WindowEvent;

    match event {
        WindowEvent::Resized(new_size) => {
            let logical_size = new_size.to_logical(scale_factor);

            Some(Event::Window(window::Event::Resized {
                width: logical_size.width,
                height: logical_size.height,
            }))
        }
        WindowEvent::CloseRequested => {
            Some(Event::Window(window::Event::CloseRequested))
        }
        WindowEvent::CursorMoved { position, .. } => {
            let position = position.to_logical::<f64>(scale_factor);

            Some(Event::Mouse(mouse::Event::CursorMoved {
                position: Point::new(position.x as f32, position.y as f32),
            }))
        }
        WindowEvent::CursorEntered { .. } => {
            Some(Event::Mouse(mouse::Event::CursorEntered))
        }
        WindowEvent::CursorLeft { .. } => {
            Some(Event::Mouse(mouse::Event::CursorLeft))
        }
        WindowEvent::MouseInput { button, state, .. } => {
            let button = mouse_button(*button)?;

            Some(Event::Mouse(match state {
                winit::event::ElementState::Pressed => {
                    mouse::Event::ButtonPressed(button)
                }
                winit::event::ElementState::Released => {
                    mouse::Event::ButtonReleased(button)
                }
            }))
        }
        WindowEvent::MouseWheel { delta, .. } => match delta {
            winit::event::MouseScrollDelta::LineDelta(delta_x, delta_y) => {
                Some(Event::Mouse(mouse::Event::WheelScrolled {
                    delta: mouse::ScrollDelta::Lines {
                        x: *delta_x,
                        y: *delta_y,
                    },
                }))
            }
            winit::event::MouseScrollDelta::PixelDelta(position) => {
                Some(Event::Mouse(mouse::Event::WheelScrolled {
                    delta: mouse::ScrollDelta::Pixels {
                        x: position.x as f32,
                        y: position.y as f32,
                    },
                }))
            }
        },
        WindowEvent::KeyboardInput { event, .. } => Some(Event::Keyboard({
            let key_code = key_code(&event.logical_key)?;
            let modifiers = self::modifiers(modifiers);

            match event.state {
                winit::event::ElementState::Pressed => {
                    keyboard::Event::KeyPressed {
                        key_code,
                        modifiers,
                    }
                }
                winit::event::ElementState::Released => {
                    keyboard::Event::KeyReleased {
                        key_code,
                        modifiers,
                    }
                }
            }
        })),
        WindowEvent::ModifiersChanged(new_modifiers) => {
            Some(Event::Keyboard(keyboard::Event::ModifiersChanged(
                self::modifiers(new_modifiers.state()),
            )))
        }
        WindowEvent::Focused(focused) => Some(Event::Window(if *focused {
            window::Event::Focused
        } else {
            window::Event::Unfocused
        })),
        WindowEvent::HoveredFile(path) => {
            Some(Event::Window(window::Event::FileHovered(path.clone())))
        }
        WindowEvent::DroppedFile(path) => {
            Some(Event::Window(window::Event::FileDropped(path.clone())))
        }
        WindowEvent::HoveredFileCancelled => {
            Some(Event::Window(window::Event::FilesHoveredLeft))
        }
        WindowEvent::Touch(touch) => {
            Some(Event::Touch(touch_event(*touch, scale_factor)))
        }
        _ => None,
    }
}

/// Converts a [`Mode`] to a [`winit`] fullscreen mode.
///
/// [`winit`]: https://github.com/rust-windowing/winit
pub fn fullscreen(
    monitor: Option<winit::monitor::MonitorHandle>,
    mode: Mode,
) -> Option<winit::window::Fullscreen> {
    match mode {
        Mode::Windowed | Mode::Hidden => None,
        Mode::Fullscreen => {
            Some(winit::window::Fullscreen::Borderless(monitor))
        }
    }
}

/// Converts a [`Mode`] to a visibility flag.
pub fn visible(mode: Mode) -> bool {
    match mode {
        Mode::Windowed | Mode::Fullscreen => true,
        Mode::Hidden => false,
    }
}

/// Converts a `MouseCursor` from [`iced_native`] to a [`winit`] cursor icon.
///
/// [`winit`]: https://github.com/rust-windowing/winit
/// [`iced_native`]: https://github.com/hecrj/iced/tree/master/native
pub fn mouse_interaction(
    interaction: mouse::Interaction,
) -> winit::window::CursorIcon {
    use mouse::Interaction;

    match interaction {
        Interaction::Idle => winit::window::CursorIcon::Default,
        Interaction::Pointer => winit::window::CursorIcon::Pointer,
        Interaction::Working => winit::window::CursorIcon::Progress,
        Interaction::Grab => winit::window::CursorIcon::Grab,
        Interaction::Grabbing => winit::window::CursorIcon::Grabbing,
        Interaction::Crosshair => winit::window::CursorIcon::Crosshair,
        Interaction::Text => winit::window::CursorIcon::Text,
        Interaction::ResizingHorizontally => {
            winit::window::CursorIcon::EwResize
        }
        Interaction::ResizingVertically => winit::window::CursorIcon::NsResize,
    }
}

/// Converts a `MouseButton` from [`winit`] to an [`iced_native`] mouse button.
///
/// [`winit`]: https://github.com/rust-windowing/winit
/// [`iced_native`]: https://github.com/hecrj/iced/tree/master/native
pub fn mouse_button(
    mouse_button: winit::event::MouseButton,
) -> Option<mouse::Button> {
    Some(match mouse_button {
        winit::event::MouseButton::Left => mouse::Button::Left,
        winit::event::MouseButton::Right => mouse::Button::Right,
        winit::event::MouseButton::Middle => mouse::Button::Middle,
        winit::event::MouseButton::Other(other) => {
            mouse::Button::Other(other as u8)
        }
        winit::event::MouseButton::Back
        | winit::event::MouseButton::Forward => return None,
    })
}

/// Converts some `ModifiersState` from [`winit`] to an [`iced_native`]
/// modifiers state.
///
/// [`winit`]: https://github.com/rust-windowing/winit
/// [`iced_native`]: https://github.com/hecrj/iced/tree/master/native
pub fn modifiers(
    modifiers: winit::keyboard::ModifiersState,
) -> keyboard::Modifiers {
    keyboard::Modifiers {
        shift: modifiers.shift_key(),
        control: modifiers.control_key(),
        alt: modifiers.alt_key(),
        logo: modifiers.super_key(),
    }
}

/// Converts a physical cursor position to a logical `Point`.
pub fn cursor_position(
    position: winit::dpi::PhysicalPosition<f64>,
    scale_factor: f64,
) -> Point {
    let logical_position = position.to_logical(scale_factor);

    Point::new(logical_position.x, logical_position.y)
}

/// Converts a `Touch` from [`winit`] to an [`iced_native`] touch event.
///
/// [`winit`]: https://github.com/rust-windowing/winit
/// [`iced_native`]: https://github.com/hecrj/iced/tree/master/native
pub fn touch_event(
    touch: winit::event::Touch,
    scale_factor: f64,
) -> touch::Event {
    let id = touch::Finger(touch.id);
    let position = {
        let location = touch.location.to_logical::<f64>(scale_factor);

        Point::new(location.x as f32, location.y as f32)
    };

    match touch.phase {
        winit::event::TouchPhase::Started => {
            touch::Event::FingerPressed { id, position }
        }
        winit::event::TouchPhase::Moved => {
            touch::Event::FingerMoved { id, position }
        }
        winit::event::TouchPhase::Ended => {
            touch::Event::FingerLifted { id, position }
        }
        winit::event::TouchPhase::Cancelled => {
            touch::Event::FingerLost { id, position }
        }
    }
}

/// Converts a `VirtualKeyCode` from [`winit`] to an [`iced_native`] key code.
///
/// [`winit`]: https://github.com/rust-windowing/winit
/// [`iced_native`]: https://github.com/hecrj/iced/tree/master/native
pub fn key_code(key: &winit::keyboard::Key) -> Option<keyboard::KeyCode> {
    use keyboard::KeyCode;

    Some(match key {
        winit::keyboard::Key::Named(key) => match key {
            winit::keyboard::NamedKey::Escape => KeyCode::Escape,
            winit::keyboard::NamedKey::F1 => KeyCode::F1,
            winit::keyboard::NamedKey::F2 => KeyCode::F2,
            winit::keyboard::NamedKey::F3 => KeyCode::F3,
            winit::keyboard::NamedKey::F4 => KeyCode::F4,
            winit::keyboard::NamedKey::F5 => KeyCode::F5,
            winit::keyboard::NamedKey::F6 => KeyCode::F6,
            winit::keyboard::NamedKey::F7 => KeyCode::F7,
            winit::keyboard::NamedKey::F8 => KeyCode::F8,
            winit::keyboard::NamedKey::F9 => KeyCode::F9,
            winit::keyboard::NamedKey::F10 => KeyCode::F10,
            winit::keyboard::NamedKey::F11 => KeyCode::F11,
            winit::keyboard::NamedKey::F12 => KeyCode::F12,
            winit::keyboard::NamedKey::F13 => KeyCode::F13,
            winit::keyboard::NamedKey::F14 => KeyCode::F14,
            winit::keyboard::NamedKey::F15 => KeyCode::F15,
            winit::keyboard::NamedKey::F16 => KeyCode::F16,
            winit::keyboard::NamedKey::F17 => KeyCode::F17,
            winit::keyboard::NamedKey::F18 => KeyCode::F18,
            winit::keyboard::NamedKey::F19 => KeyCode::F19,
            winit::keyboard::NamedKey::F20 => KeyCode::F20,
            winit::keyboard::NamedKey::F21 => KeyCode::F21,
            winit::keyboard::NamedKey::F22 => KeyCode::F22,
            winit::keyboard::NamedKey::F23 => KeyCode::F23,
            winit::keyboard::NamedKey::F24 => KeyCode::F24,
            winit::keyboard::NamedKey::ScrollLock => KeyCode::Scroll,
            winit::keyboard::NamedKey::Pause => KeyCode::Pause,
            winit::keyboard::NamedKey::Insert => KeyCode::Insert,
            winit::keyboard::NamedKey::Home => KeyCode::Home,
            winit::keyboard::NamedKey::Delete => KeyCode::Delete,
            winit::keyboard::NamedKey::End => KeyCode::End,
            winit::keyboard::NamedKey::PageDown => KeyCode::PageDown,
            winit::keyboard::NamedKey::PageUp => KeyCode::PageUp,
            winit::keyboard::NamedKey::ArrowLeft => KeyCode::Left,
            winit::keyboard::NamedKey::ArrowUp => KeyCode::Up,
            winit::keyboard::NamedKey::ArrowRight => KeyCode::Right,
            winit::keyboard::NamedKey::ArrowDown => KeyCode::Down,
            winit::keyboard::NamedKey::Backspace => KeyCode::Backspace,
            winit::keyboard::NamedKey::Enter => KeyCode::Enter,
            winit::keyboard::NamedKey::Space => KeyCode::Space,
            winit::keyboard::NamedKey::Compose => KeyCode::Compose,
            winit::keyboard::NamedKey::NumLock => KeyCode::Numlock,
            winit::keyboard::NamedKey::Convert => KeyCode::Convert,
            winit::keyboard::NamedKey::KanaMode => KeyCode::Kana,
            winit::keyboard::NamedKey::KanjiMode => KeyCode::Kanji,
            winit::keyboard::NamedKey::MediaStop => KeyCode::MediaStop,
            winit::keyboard::NamedKey::AudioVolumeMute => KeyCode::Mute,
            winit::keyboard::NamedKey::MediaTrackNext => KeyCode::NextTrack,
            winit::keyboard::NamedKey::NonConvert => KeyCode::NoConvert,
            winit::keyboard::NamedKey::MediaPlayPause => KeyCode::PlayPause,
            winit::keyboard::NamedKey::Power => KeyCode::Power,
            winit::keyboard::NamedKey::MediaTrackPrevious => KeyCode::PrevTrack,
            winit::keyboard::NamedKey::Tab => KeyCode::Tab,
            winit::keyboard::NamedKey::AudioVolumeDown => KeyCode::VolumeDown,
            winit::keyboard::NamedKey::AudioVolumeUp => KeyCode::VolumeUp,
            winit::keyboard::NamedKey::WakeUp => KeyCode::Wake,
            winit::keyboard::NamedKey::Copy => KeyCode::Copy,
            winit::keyboard::NamedKey::Paste => KeyCode::Paste,
            winit::keyboard::NamedKey::Cut => KeyCode::Cut,
            _ => return None,
        },
        winit::keyboard::Key::Character(c) => match c.as_str() {
            "a" | "A" => KeyCode::A,
            "b" | "B" => KeyCode::B,
            "c" | "C" => KeyCode::C,
            "d" | "D" => KeyCode::D,
            "e" | "E" => KeyCode::E,
            "f" | "F" => KeyCode::F,
            "g" | "G" => KeyCode::G,
            "h" | "H" => KeyCode::H,
            "i" | "I" => KeyCode::I,
            "j" | "J" => KeyCode::J,
            "k" | "K" => KeyCode::K,
            "l" | "L" => KeyCode::L,
            "m" | "M" => KeyCode::M,
            "n" | "N" => KeyCode::N,
            "o" | "O" => KeyCode::O,
            "p" | "P" => KeyCode::P,
            "q" | "Q" => KeyCode::Q,
            "r" | "R" => KeyCode::R,
            "s" | "S" => KeyCode::S,
            "t" | "T" => KeyCode::T,
            "u" | "U" => KeyCode::U,
            "v" | "V" => KeyCode::V,
            "w" | "W" => KeyCode::W,
            "x" | "X" => KeyCode::X,
            "y" | "Y" => KeyCode::Y,
            "z" | "Z" => KeyCode::Z,
            "'" => KeyCode::Apostrophe,
            "*" => KeyCode::Asterisk,
            "\\" => KeyCode::Backslash,
            "^" => KeyCode::Caret,
            ":" => KeyCode::Colon,
            "," => KeyCode::Comma,
            "=" => KeyCode::Equals,
            "-" => KeyCode::Minus,
            "+" => KeyCode::Plus,
            ";" => KeyCode::Semicolon,
            "/" => KeyCode::Slash,
            " " => KeyCode::Space,
            "_" => KeyCode::Underline,
            _ => return None,
        },
        winit::keyboard::Key::Unidentified(_)
        | winit::keyboard::Key::Dead(_) => return None,
    })
}
