<?php

namespace ccxt\async;

// PLEASE DO NOT EDIT THIS FILE, IT IS GENERATED AND WILL BE OVERWRITTEN:
// https://github.com/ccxt/ccxt/blob/master/CONTRIBUTING.md#how-to-contribute-code

use Exception; // a common import
use \ccxt\ExchangeError;
use \ccxt\InvalidOrder;

class novadax extends Exchange {

    public function describe() {
        return $this->deep_extend(parent::describe (), array(
            'id' => 'novadax',
            'name' => 'NovaDAX',
            'countries' => array( 'BR' ), // Brazil
            'rateLimit' => 50,
            'version' => 'v1',
            // new metainfo interface
            'has' => array(
                'CORS' => false,
                'cancelOrder' => true,
                'createOrder' => true,
                'fetchAccounts' => true,
                'fetchBalance' => true,
                'fetchClosedOrders' => true,
                'fetchDeposits' => true,
                'fetchMarkets' => true,
                'fetchMyTrades' => true,
                'fetchOHLCV' => true,
                'fetchOpenOrders' => true,
                'fetchOrder' => true,
                'fetchOrders' => true,
                'fetchOrderTrades' => true,
                'fetchOrderBook' => true,
                'fetchTicker' => true,
                'fetchTickers' => true,
                'fetchTime' => true,
                'fetchTrades' => true,
                'fetchTransactions' => true,
                'fetchWithdrawals' => true,
                'withdraw' => true,
            ),
            'timeframes' => array(
                '1m' => 'ONE_MIN',
                '5m' => 'FIVE_MIN',
                '15m' => 'FIFTEEN_MIN',
                '30m' => 'HALF_HOU',
                '1h' => 'ONE_HOU',
                '1d' => 'ONE_DAY',
                '1w' => 'ONE_WEE',
                '1M' => 'ONE_MON',
            ),
            'urls' => array(
                'logo' => 'https://user-images.githubusercontent.com/1294454/92337550-2b085500-f0b3-11ea-98e7-5794fb07dd3b.jpg',
                'api' => array(
                    'public' => 'https://api.novadax.com',
                    'private' => 'https://api.novadax.com',
                ),
                'www' => 'https://www.novadax.com.br',
                'doc' => array(
                    'https://doc.novadax.com/pt-BR/',
                ),
                'fees' => 'https://www.novadax.com.br/fees-and-limits',
                'referral' => 'https://www.novadax.com.br/?s=ccxt',
            ),
            'api' => array(
                'public' => array(
                    'get' => array(
                        'common/symbol',
                        'common/symbols',
                        'common/timestamp',
                        'market/tickers',
                        'market/ticker',
                        'market/depth',
                        'market/trades',
                        'market/kline/history',
                    ),
                ),
                'private' => array(
                    'get' => array(
                        'orders/get',
                        'orders/list',
                        'orders/fill',
                        'orders/fills',
                        'account/getBalance',
                        'account/subs',
                        'account/subs/balance',
                        'account/subs/transfer/record',
                        'wallet/query/deposit-withdraw',
                    ),
                    'post' => array(
                        'orders/create',
                        'orders/cancel',
                        'account/withdraw/coin',
                        'account/subs/transfer',
                    ),
                ),
            ),
            'fees' => array(
                'trading' => array(
                    'tierBased' => false,
                    'percentage' => true,
                    'taker' => 0.5 / 100,
                    'maker' => 0.3 / 100,
                ),
            ),
            'requiredCredentials' => array(
                'apiKey' => true,
                'secret' => true,
            ),
            'exceptions' => array(
                'exact' => array(
                    'A99999' => '\\ccxt\\ExchangeError', // 500 Failed Internal error
                    // 'A10000' => '\\ccxt\\ExchangeError', // 200 Success Successful request
                    'A10001' => '\\ccxt\\BadRequest', // 400 Params error Parameter is invalid
                    'A10002' => '\\ccxt\\ExchangeError', // 404 Api not found API used is irrelevant
                    'A10003' => '\\ccxt\\AuthenticationError', // 403 Authentication failed Authentication is failed
                    'A10004' => '\\ccxt\\RateLimitExceeded', // 429 Too many requests Too many requests are made
                    'A10005' => '\\ccxt\\PermissionDenied', // 403 Kyc required Need to complete KYC firstly
                    'A10006' => '\\ccxt\\AccountSuspended', // 403 Customer canceled Account is canceled
                    'A10007' => '\\ccxt\\BadRequest', // 400 Account not exist Sub account does not exist
                    'A10011' => '\\ccxt\\BadSymbol', // 400 Symbol not exist Trading symbol does not exist
                    'A10012' => '\\ccxt\\BadSymbol', // 400 Symbol not trading Trading symbol is temporarily not available
                    'A10013' => '\\ccxt\\OnMaintenance', // 503 Symbol maintain Trading symbol is in maintain
                    'A30001' => '\\ccxt\\OrderNotFound', // 400 Order not found Queried order is not found
                    'A30002' => '\\ccxt\\InvalidOrder', // 400 Order amount is too small Order amount is too small
                    'A30003' => '\\ccxt\\InvalidOrder', // 400 Order amount is invalid Order amount is invalid
                    'A30004' => '\\ccxt\\InvalidOrder', // 400 Order value is too small Order value is too small
                    'A30005' => '\\ccxt\\InvalidOrder', // 400 Order value is invalid Order value is invalid
                    'A30006' => '\\ccxt\\InvalidOrder', // 400 Order price is invalid Order price is invalid
                    'A30007' => '\\ccxt\\InsufficientFunds', // 400 Insufficient balance The balance is insufficient
                    'A30008' => '\\ccxt\\InvalidOrder', // 400 Order was closed The order has been executed
                    'A30009' => '\\ccxt\\InvalidOrder', // 400 Order canceled The order has been cancelled
                    'A30010' => '\\ccxt\\CancelPending', // 400 Order cancelling The order is being cancelled
                    'A30011' => '\\ccxt\\InvalidOrder', // 400 Order price too high The order price is too high
                    'A30012' => '\\ccxt\\InvalidOrder', // 400 Order price too low The order price is too low
                ),
                'broad' => array(
                ),
            ),
            'options' => array(
                'fetchOHLCV' => array(
                    'volume' => 'amount', // 'amount' for base volume or 'vol' for quote volume
                ),
            ),
        ));
    }

    public function fetch_time($params = array ()) {
        $response = yield $this->publicGetCommonTimestamp ($params);
        //
        //     {
        //         "code":"A10000",
        //         "data":1599090512080,
        //         "message":"Success"
        //     }
        //
        return $this->safe_integer($response, 'data');
    }

    public function fetch_markets($params = array ()) {
        $response = yield $this->publicGetCommonSymbols ($params);
        //
        //     {
        //         "code":"A10000",
        //         "$data":array(
        //             array(
        //                 "amountPrecision":8,
        //                 "baseCurrency":"BTC",
        //                 "minOrderAmount":"0.001",
        //                 "minOrderValue":"25",
        //                 "pricePrecision":2,
        //                 "quoteCurrency":"BRL",
        //                 "$status":"ONLINE",
        //                 "$symbol":"BTC_BRL",
        //                 "valuePrecision":2
        //             ),
        //         ),
        //         "message":"Success"
        //     }
        //
        $result = array();
        $data = $this->safe_value($response, 'data', array());
        for ($i = 0; $i < count($data); $i++) {
            $market = $data[$i];
            $baseId = $this->safe_string($market, 'baseCurrency');
            $quoteId = $this->safe_string($market, 'quoteCurrency');
            $id = $this->safe_string($market, 'symbol');
            $base = $this->safe_currency_code($baseId);
            $quote = $this->safe_currency_code($quoteId);
            $symbol = $base . '/' . $quote;
            $precision = array(
                'amount' => $this->safe_integer($market, 'amountPrecision'),
                'price' => $this->safe_integer($market, 'pricePrecision'),
                'cost' => $this->safe_integer($market, 'valuePrecision'),
            );
            $limits = array(
                'amount' => array(
                    'min' => $this->safe_float($market, 'minOrderAmount'),
                    'max' => null,
                ),
                'price' => array(
                    'min' => null,
                    'max' => null,
                ),
                'cost' => array(
                    'min' => $this->safe_float($market, 'minOrderValue'),
                    'max' => null,
                ),
            );
            $status = $this->safe_string($market, 'status');
            $active = ($status === 'ONLINE');
            $result[] = array(
                'id' => $id,
                'symbol' => $symbol,
                'base' => $base,
                'quote' => $quote,
                'baseId' => $baseId,
                'quoteId' => $quoteId,
                'precision' => $precision,
                'limits' => $limits,
                'info' => $market,
                'active' => $active,
            );
        }
        return $result;
    }

    public function parse_ticker($ticker, $market = null) {
        //
        // fetchTicker, fetchTickers
        //
        //     {
        //         "ask":"61946.1",
        //         "baseVolume24h":"164.41930186",
        //         "bid":"61815",
        //         "high24h":"64930.72",
        //         "lastPrice":"61928.41",
        //         "low24h":"61156.32",
        //         "open24h":"64512.46",
        //         "quoteVolume24h":"10308157.95",
        //         "$symbol":"BTC_BRL",
        //         "$timestamp":1599091115090
        //     }
        //
        $timestamp = $this->safe_integer($ticker, 'timestamp');
        $marketId = $this->safe_string($ticker, 'symbol');
        $symbol = $this->safe_symbol($marketId, $market, '_');
        $open = $this->safe_float($ticker, 'open24h');
        $last = $this->safe_float($ticker, 'lastPrice');
        $percentage = null;
        $change = null;
        $average = null;
        if (($last !== null) && ($open !== null)) {
            $change = $last - $open;
            $percentage = $change / $open * 100;
            $average = $this->sum($last, $open) / 2;
        }
        $baseVolume = $this->safe_float($ticker, 'baseVolume24h');
        $quoteVolume = $this->safe_float($ticker, 'quoteVolume24h');
        $vwap = $this->vwap($baseVolume, $quoteVolume);
        return array(
            'symbol' => $symbol,
            'timestamp' => $timestamp,
            'datetime' => $this->iso8601($timestamp),
            'high' => $this->safe_float($ticker, 'high24h'),
            'low' => $this->safe_float($ticker, 'low24h'),
            'bid' => $this->safe_float($ticker, 'bid'),
            'bidVolume' => null,
            'ask' => $this->safe_float($ticker, 'ask'),
            'askVolume' => null,
            'vwap' => $vwap,
            'open' => $open,
            'close' => $last,
            'last' => $last,
            'previousClose' => null,
            'change' => $change,
            'percentage' => $percentage,
            'average' => $average,
            'baseVolume' => $baseVolume,
            'quoteVolume' => $quoteVolume,
            'info' => $ticker,
        );
    }

    public function fetch_ticker($symbol, $params = array ()) {
        yield $this->load_markets();
        $market = $this->market($symbol);
        $request = array(
            'symbol' => $market['id'],
        );
        $response = yield $this->publicGetMarketTicker (array_merge($request, $params));
        //
        //     {
        //         "code":"A10000",
        //         "$data":array(
        //             "ask":"61946.1",
        //             "baseVolume24h":"164.41930186",
        //             "bid":"61815",
        //             "high24h":"64930.72",
        //             "lastPrice":"61928.41",
        //             "low24h":"61156.32",
        //             "open24h":"64512.46",
        //             "quoteVolume24h":"10308157.95",
        //             "$symbol":"BTC_BRL",
        //             "timestamp":1599091115090
        //         ),
        //         "message":"Success"
        //     }
        //
        $data = $this->safe_value($response, 'data', array());
        return $this->parse_ticker($data, $market);
    }

    public function fetch_tickers($symbols = null, $params = array ()) {
        yield $this->load_markets();
        $response = yield $this->publicGetMarketTickers ($params);
        //
        //     {
        //         "code":"A10000",
        //         "$data":array(
        //             array(
        //                 "ask":"61879.36",
        //                 "baseVolume24h":"164.40955092",
        //                 "bid":"61815",
        //                 "high24h":"64930.72",
        //                 "lastPrice":"61820.04",
        //                 "low24h":"61156.32",
        //                 "open24h":"64624.19",
        //                 "quoteVolume24h":"10307493.92",
        //                 "$symbol":"BTC_BRL",
        //                 "timestamp":1599091291083
        //             ),
        //         ),
        //         "message":"Success"
        //     }
        //
        $data = $this->safe_value($response, 'data', array());
        $result = array();
        for ($i = 0; $i < count($data); $i++) {
            $ticker = $this->parse_ticker($data[$i]);
            $symbol = $ticker['symbol'];
            $result[$symbol] = $ticker;
        }
        return $this->filter_by_array($result, 'symbol', $symbols);
    }

    public function fetch_order_book($symbol, $limit = null, $params = array ()) {
        yield $this->load_markets();
        $request = array(
            'symbol' => $this->market_id($symbol),
        );
        if ($limit !== null) {
            $request['limit'] = $limit; // default 10, max 20
        }
        $response = yield $this->publicGetMarketDepth (array_merge($request, $params));
        //
        //     {
        //         "code":"A10000",
        //         "$data":array(
        //             "asks":[
        //                 ["0.037159","0.3741"],
        //                 ["0.037215","0.2706"],
        //                 ["0.037222","1.8459"],
        //             ],
        //             "bids":[
        //                 ["0.037053","0.3857"],
        //                 ["0.036969","0.8101"],
        //                 ["0.036953","1.5226"],
        //             ],
        //             "$timestamp":1599280414448
        //         ),
        //         "message":"Success"
        //     }
        //
        $data = $this->safe_value($response, 'data', array());
        $timestamp = $this->safe_integer($data, 'timestamp');
        return $this->parse_order_book($data, $timestamp, 'bids', 'asks');
    }

    public function parse_trade($trade, $market = null) {
        //
        // public fetchTrades
        //
        //     {
        //         "$amount":"0.0632",
        //         "$price":"0.037288",
        //         "$side":"BUY",
        //         "$timestamp":1599279694576
        //     }
        //
        // private fetchOrderTrades
        //
        //     {
        //         "$id" => "608717046691139584",
        //         "$orderId" => "608716957545402368",
        //         "$symbol" => "BTC_BRL",
        //         "$side" => "BUY",
        //         "$amount" => "0.0988",
        //         "$price" => "45514.76",
        //         "$fee" => "0.0000988 BTC",
        //         "role" => "MAKER",
        //         "$timestamp" => 1565171053345
        //     }
        //
        // private fetchMyTrades
        //
        //     {
        //         "$id" => "608717046691139584",
        //         "$orderId" => "608716957545402368",
        //         "$symbol" => "BTC_BRL",
        //         "$side" => "BUY",
        //         "$amount" => "0.0988",
        //         "$price" => "45514.76",
        //         "$fee" => "0.0000988 BTC",
        //         "feeAmount" => "0.0000988",
        //         "feeCurrency" => "BTC",
        //         "role" => "MAKER",
        //         "$timestamp" => 1565171053345
        //     }
        //
        $id = $this->safe_string($trade, 'id');
        $orderId = $this->safe_string($trade, 'orderId');
        $timestamp = $this->safe_integer($trade, 'timestamp');
        $side = $this->safe_string_lower($trade, 'side');
        $price = $this->safe_float($trade, 'price');
        $amount = $this->safe_float($trade, 'amount');
        $cost = $this->safe_float($trade, 'volume');
        if (($cost === null) && ($amount !== null) && ($price !== null)) {
            $cost = $amount * $price;
        }
        $marketId = $this->safe_string($trade, 'symbol');
        $symbol = $this->safe_symbol($marketId, $market, '_');
        $takerOrMaker = $this->safe_string_lower($trade, 'role');
        $feeString = $this->safe_string($trade, 'fee');
        $fee = null;
        if ($feeString !== null) {
            $parts = explode(' ', $feeString);
            $feeCurrencyId = $this->safe_string($parts, 1);
            $feeCurrencyCode = $this->safe_currency_code($feeCurrencyId);
            $fee = array(
                'cost' => $this->safe_float($parts, 0),
                'currency' => $feeCurrencyCode,
            );
        }
        return array(
            'id' => $id,
            'order' => $orderId,
            'timestamp' => $timestamp,
            'datetime' => $this->iso8601($timestamp),
            'symbol' => $symbol,
            'type' => null,
            'side' => $side,
            'price' => $price,
            'amount' => $amount,
            'cost' => $cost,
            'takerOrMaker' => $takerOrMaker,
            'fee' => $fee,
            'info' => $trade,
        );
    }

    public function fetch_trades($symbol, $since = null, $limit = null, $params = array ()) {
        yield $this->load_markets();
        $market = $this->market($symbol);
        $request = array(
            'symbol' => $market['id'],
        );
        if ($limit !== null) {
            $request['limit'] = $limit; // default 100
        }
        $response = yield $this->publicGetMarketTrades (array_merge($request, $params));
        //
        //     {
        //         "code":"A10000",
        //         "$data":array(
        //             array("amount":"0.0632","price":"0.037288","side":"BUY","timestamp":1599279694576),
        //             array("amount":"0.0052","price":"0.03715","side":"SELL","timestamp":1599276606852),
        //             array("amount":"0.0058","price":"0.037188","side":"SELL","timestamp":1599275187812),
        //         ),
        //         "message":"Success"
        //     }
        //
        $data = $this->safe_value($response, 'data', array());
        return $this->parse_trades($data, $market, $since, $limit);
    }

    public function fetch_ohlcv($symbol, $timeframe = '1m', $since = null, $limit = null, $params = array ()) {
        yield $this->load_markets();
        $market = $this->market($symbol);
        $request = array(
            'symbol' => $market['id'],
            'unit' => $this->timeframes[$timeframe],
        );
        $duration = $this->parse_timeframe($timeframe);
        $now = $this->seconds();
        if ($limit === null) {
            $limit = 3000; // max
        }
        if ($since === null) {
            $request['from'] = $now - $limit * $duration;
            $request['to'] = $now;
        } else {
            $startFrom = intval($since / 1000);
            $request['from'] = $startFrom;
            $request['to'] = $this->sum($startFrom, $limit * $duration);
        }
        $response = yield $this->publicGetMarketKlineHistory (array_merge($request, $params));
        //
        //     {
        //         "code" => "A10000",
        //         "$data" => array(
        //             {
        //                 "amount" => 8.25709100,
        //                 "closePrice" => 62553.20,
        //                 "count" => 29,
        //                 "highPrice" => 62592.87,
        //                 "lowPrice" => 62553.20,
        //                 "openPrice" => 62554.23,
        //                 "score" => 1602501480,
        //                 "$symbol" => "BTC_BRL",
        //                 "vol" => 516784.2504067500
        //             }
        //         ),
        //         "message" => "Success"
        //     }
        //
        $data = $this->safe_value($response, 'data', array());
        return $this->parse_ohlcvs($data, $market, $timeframe, $since, $limit);
    }

    public function parse_ohlcv($ohlcv, $market = null) {
        //
        //     {
        //         "amount" => 8.25709100,
        //         "closePrice" => 62553.20,
        //         "count" => 29,
        //         "highPrice" => 62592.87,
        //         "lowPrice" => 62553.20,
        //         "openPrice" => 62554.23,
        //         "score" => 1602501480,
        //         "symbol" => "BTC_BRL",
        //         "vol" => 516784.2504067500
        //     }
        //
        $options = $this->safe_value($this->options, 'fetchOHLCV', array());
        $volumeField = $this->safe_string($options, 'volume', 'amount'); // or vol
        return array(
            $this->safe_timestamp($ohlcv, 'score'),
            $this->safe_float($ohlcv, 'openPrice'),
            $this->safe_float($ohlcv, 'highPrice'),
            $this->safe_float($ohlcv, 'lowPrice'),
            $this->safe_float($ohlcv, 'closePrice'),
            $this->safe_float($ohlcv, $volumeField),
        );
    }

    public function fetch_balance($params = array ()) {
        yield $this->load_markets();
        $response = yield $this->privateGetAccountGetBalance ($params);
        //
        //     {
        //         "$code" => "A10000",
        //         "$data" => array(
        //             {
        //                 "available" => "1.23",
        //                 "$balance" => "0.23",
        //                 "currency" => "BTC",
        //                 "hold" => "1"
        //             }
        //         ),
        //         "message" => "Success"
        //     }
        //
        $data = $this->safe_value($response, 'data', array());
        $result = array( 'info' => $response );
        for ($i = 0; $i < count($data); $i++) {
            $balance = $data[$i];
            $currencyId = $this->safe_string($balance, 'currency');
            $code = $this->safe_currency_code($currencyId);
            $account = $this->account();
            $account['total'] = $this->safe_float($balance, 'available');
            $account['free'] = $this->safe_float($balance, 'balance');
            $account['used'] = $this->safe_float($balance, 'hold');
            $result[$code] = $account;
        }
        return $this->parse_balance($result);
    }

    public function create_order($symbol, $type, $side, $amount, $price = null, $params = array ()) {
        yield $this->load_markets();
        $market = $this->market($symbol);
        $uppercaseType = strtoupper($type);
        $uppercaseSide = strtoupper($side);
        $request = array(
            'symbol' => $market['id'],
            'type' => $uppercaseType, // LIMIT, MARKET
            'side' => $uppercaseSide, // or SELL
            // 'accountId' => '...', // subaccount id, optional
            // 'amount' => $this->amount_to_precision($symbol, $amount),
            // "$price" => "1234.5678", // required for LIMIT and STOP orders
        );
        if ($uppercaseType === 'LIMIT') {
            $request['price'] = $this->price_to_precision($symbol, $price);
            $request['amount'] = $this->amount_to_precision($symbol, $amount);
        } else if ($uppercaseType === 'MARKET') {
            if ($uppercaseSide === 'SELL') {
                $request['amount'] = $this->amount_to_precision($symbol, $amount);
            } else if ($uppercaseSide === 'BUY') {
                $value = $this->safe_float($params, 'value');
                $createMarketBuyOrderRequiresPrice = $this->safe_value($this->options, 'createMarketBuyOrderRequiresPrice', true);
                if ($createMarketBuyOrderRequiresPrice) {
                    if ($price !== null) {
                        if ($value === null) {
                            $value = $amount * $price;
                        }
                    } else if ($value === null) {
                        throw new InvalidOrder($this->id . " createOrder() requires the $price argument with $market buy orders to calculate total order cost ($amount to spend), where cost = $amount * $price-> Supply a $price argument to createOrder() call if you want the cost to be calculated for you from $price and $amount, or, alternatively, add .options['createMarketBuyOrderRequiresPrice'] = false and supply the total cost $value in the 'amount' argument or in the 'value' extra parameter (the exchange-specific behaviour)");
                    }
                } else {
                    $value = ($value === null) ? $amount : $value;
                }
                $precision = $market['precision']['price'];
                $request['value'] = $this->decimal_to_precision($value, TRUNCATE, $precision, $this->precisionMode);
            }
        }
        $response = yield $this->privatePostOrdersCreate (array_merge($request, $params));
        //
        //     {
        //         "code" => "A10000",
        //         "$data" => array(
        //             "$amount" => "0.001",
        //             "averagePrice" => null,
        //             "filledAmount" => "0",
        //             "filledFee" => "0",
        //             "filledValue" => "0",
        //             "id" => "633679992971251712",
        //             "$price" => "35000",
        //             "$side" => "BUY",
        //             "status" => "PROCESSING",
        //             "$symbol" => "BTC_BRL",
        //             "timestamp" => 1571122683535,
        //             "$type" => "LIMIT",
        //             "$value" => "35"
        //         ),
        //         "message" => "Success"
        //     }
        //
        $data = $this->safe_value($response, 'data', array());
        return $this->parse_order($data, $market);
    }

    public function cancel_order($id, $symbol = null, $params = array ()) {
        yield $this->load_markets();
        $request = array(
            'id' => $id,
        );
        $response = yield $this->privatePostOrdersCancel (array_merge($request, $params));
        //
        //     {
        //         "code" => "A10000",
        //         "$data" => array(
        //             "result" => true
        //         ),
        //         "message" => "Success"
        //     }
        //
        $data = $this->safe_value($response, 'data', array());
        return $this->parse_order($data);
    }

    public function fetch_order($id, $symbol = null, $params = array ()) {
        yield $this->load_markets();
        $request = array(
            'id' => $id,
        );
        $response = yield $this->privateGetOrdersGet (array_merge($request, $params));
        //
        //     {
        //         "code" => "A10000",
        //         "$data" => array(
        //             "$id" => "608695623247466496",
        //             "$symbol" => "BTC_BRL",
        //             "type" => "MARKET",
        //             "side" => "SELL",
        //             "price" => null,
        //             "averagePrice" => "0",
        //             "amount" => "0.123",
        //             "filledAmount" => "0",
        //             "value" => null,
        //             "filledValue" => "0",
        //             "filledFee" => "0",
        //             "status" => "REJECTED",
        //             "timestamp" => 1565165945588
        //         ),
        //         "message" => "Success"
        //     }
        //
        $data = $this->safe_value($response, 'data', array());
        return $this->parse_order($data);
    }

    public function fetch_orders($symbol = null, $since = null, $limit = null, $params = array ()) {
        yield $this->load_markets();
        $request = array(
            // 'symbol' => $market['id'],
            // 'status' => 'SUBMITTED,PROCESSING', // SUBMITTED, PROCESSING, PARTIAL_FILLED, CANCELING, FILLED, CANCELED, REJECTED
            // 'fromId' => '...', // order id to begin with
            // 'toId' => '...', // order id to end up with
            // 'fromTimestamp' => $since,
            // 'toTimestamp' => $this->milliseconds(),
            // 'limit' => $limit, // default 100, max 100
        );
        $market = null;
        if ($symbol !== null) {
            $market = $this->market($symbol);
            $request['symbol'] = $market['id'];
        }
        if ($limit !== null) {
            $request['limit'] = $limit; // default 100, max 100
        }
        if ($since !== null) {
            $request['fromTimestamp'] = $since;
        }
        $response = yield $this->privateGetOrdersList (array_merge($request, $params));
        //
        //     {
        //         "code" => "A10000",
        //         "$data" => array(
        //             array(
        //                 "id" => "608695678650028032",
        //                 "$symbol" => "BTC_BRL",
        //                 "type" => "MARKET",
        //                 "side" => "SELL",
        //                 "price" => null,
        //                 "averagePrice" => "0",
        //                 "amount" => "0.123",
        //                 "filledAmount" => "0",
        //                 "value" => null,
        //                 "filledValue" => "0",
        //                 "filledFee" => "0",
        //                 "status" => "REJECTED",
        //                 "timestamp" => 1565165958796
        //             ),
        //         ),
        //         "message" => "Success"
        //     }
        //
        $data = $this->safe_value($response, 'data', array());
        return $this->parse_orders($data, $market, $since, $limit);
    }

    public function fetch_open_orders($symbol = null, $since = null, $limit = null, $params = array ()) {
        $request = array(
            'status' => 'SUBMITTED,PROCESSING,PARTIAL_FILLED,CANCELING',
        );
        return yield $this->fetch_orders($symbol, $since, $limit, array_merge($request, $params));
    }

    public function fetch_closed_orders($symbol = null, $since = null, $limit = null, $params = array ()) {
        $request = array(
            'status' => 'FILLED,CANCELED,REJECTED',
        );
        return yield $this->fetch_orders($symbol, $since, $limit, array_merge($request, $params));
    }

    public function fetch_order_trades($id, $symbol = null, $since = null, $limit = null, $params = array ()) {
        yield $this->load_markets();
        $request = array(
            'id' => $id,
        );
        $response = yield $this->privateGetOrdersFill (array_merge($request, $params));
        $market = null;
        if ($symbol !== null) {
            $market = $this->market($symbol);
        }
        $data = $this->safe_value($response, 'data', array());
        //
        //     {
        //         "code" => "A10000",
        //         "$data" => array(
        //             array(
        //                 "$id" => "608717046691139584",
        //                 "orderId" => "608716957545402368",
        //                 "$symbol" => "BTC_BRL",
        //                 "side" => "BUY",
        //                 "amount" => "0.0988",
        //                 "price" => "45514.76",
        //                 "fee" => "0.0000988 BTC",
        //                 "role" => "MAKER",
        //                 "timestamp" => 1565171053345
        //             ),
        //         ),
        //         "message" => "Success"
        //     }
        //
        return $this->parse_trades($data, $market, $since, $limit);
    }

    public function parse_order_status($status) {
        $statuses = array(
            'SUBMITTED' => 'open',
            'PROCESSING' => 'open',
            'PARTIAL_FILLED' => 'open',
            'CANCELING' => 'open',
            'FILLED' => 'closed',
            'CANCELED' => 'canceled',
            'REJECTED' => 'rejected',
        );
        return $this->safe_string($statuses, $status, $status);
    }

    public function parse_order($order, $market = null) {
        //
        // createOrder, fetchOrders, fetchOrder
        //
        //     {
        //         "$amount" => "0.001",
        //         "averagePrice" => null,
        //         "filledAmount" => "0",
        //         "filledFee" => "0",
        //         "filledValue" => "0",
        //         "$id" => "633679992971251712",
        //         "$price" => "35000",
        //         "$side" => "BUY",
        //         "$status" => "PROCESSING",
        //         "$symbol" => "BTC_BRL",
        //         "$timestamp" => 1571122683535,
        //         "$type" => "LIMIT",
        //         "value" => "35"
        //     }
        //
        // cancelOrder
        //
        //     {
        //         "result" => true
        //     }
        //
        $id = $this->safe_string($order, 'id');
        $amount = $this->safe_float($order, 'amount');
        $price = $this->safe_float($order, 'price');
        $cost = $this->safe_float($order, 'filledValue');
        $type = $this->safe_string_lower($order, 'type');
        $side = $this->safe_string_lower($order, 'side');
        $status = $this->parse_order_status($this->safe_string($order, 'status'));
        $timestamp = $this->safe_integer($order, 'timestamp');
        $average = $this->safe_float($order, 'averagePrice');
        $filled = $this->safe_float($order, 'filledAmount');
        $remaining = null;
        if (($amount !== null) && ($filled !== null)) {
            $remaining = max (0, $amount - $filled);
        }
        $fee = null;
        $feeCost = $this->safe_float($order, 'filledFee');
        if ($feeCost !== null) {
            $fee = array(
                'cost' => $feeCost,
                'currency' => null,
            );
        }
        $marketId = $this->safe_string($order, 'symbol');
        $symbol = $this->safe_symbol($marketId, $market, '_');
        return array(
            'id' => $id,
            'clientOrderId' => null,
            'info' => $order,
            'timestamp' => $timestamp,
            'datetime' => $this->iso8601($timestamp),
            'lastTradeTimestamp' => null,
            'symbol' => $symbol,
            'type' => $type,
            'timeInForce' => null,
            'postOnly' => null,
            'side' => $side,
            'price' => $price,
            'stopPrice' => null,
            'amount' => $amount,
            'cost' => $cost,
            'average' => $average,
            'filled' => $filled,
            'remaining' => $remaining,
            'status' => $status,
            'fee' => $fee,
            'trades' => null,
        );
    }

    public function withdraw($code, $amount, $address, $tag = null, $params = array ()) {
        yield $this->load_markets();
        $currency = $this->currency($code);
        $request = array(
            'code' => $currency['id'],
            'amount' => $this->currency_to_precision($code, $amount),
            'wallet' => $address,
        );
        if ($tag !== null) {
            $request['tag'] = $tag;
        }
        $response = yield $this->privatePostAccountWithdrawCoin (array_merge($request, $params));
        //
        //     {
        //         "$code":"A10000",
        //         "data" => "DR123",
        //         "message":"Success"
        //     }
        //
        return $this->parse_transaction($response, $currency);
    }

    public function fetch_accounts($params = array ()) {
        $response = yield $this->privateGetAccountSubs ($params);
        //
        //     {
        //         "code" => "A10000",
        //         "$data" => array(
        //             {
        //                 "subId" => "CA648856083527372800",
        //                 "state" => "Normal",
        //                 "subAccount" => "003",
        //                 "subIdentify" => "003"
        //             }
        //         ),
        //         "message" => "Success"
        //     }
        //
        $data = $this->safe_value($response, 'data', array());
        $result = array();
        for ($i = 0; $i < count($data); $i++) {
            $account = $data[$i];
            $accountId = $this->safe_string($account, 'subId');
            $type = $this->safe_string($account, 'subAccount');
            $result[] = array(
                'id' => $accountId,
                'type' => $type,
                'currency' => null,
                'info' => $account,
            );
        }
        return $result;
    }

    public function fetch_deposits($code = null, $since = null, $limit = null, $params = array ()) {
        $request = array(
            'type' => 'coin_in',
        );
        return yield $this->fetch_transactions($code, $since, $limit, array_merge($request, $params));
    }

    public function fetch_withdrawals($code = null, $since = null, $limit = null, $params = array ()) {
        $request = array(
            'type' => 'coin_out',
        );
        return yield $this->fetch_transactions($code, $since, $limit, array_merge($request, $params));
    }

    public function fetch_transactions($code = null, $since = null, $limit = null, $params = array ()) {
        yield $this->load_markets();
        $request = array(
            // 'currency' => $currency['id'],
            // 'type' => 'coin_in', // 'coin_out'
            // 'direct' => 'asc', // 'desc'
            // 'size' => $limit, // default 100
            // 'start' => id, // offset id
        );
        $currency = null;
        if ($code !== null) {
            $currency = $this->currency($code);
            $request['currency'] = $currency['id'];
        }
        if ($limit !== null) {
            $request['size'] = $limit;
        }
        $response = yield $this->privateGetWalletQueryDepositWithdraw (array_merge($request, $params));
        //
        //     {
        //         "$code" => "A10000",
        //         "$data" => array(
        //             {
        //                 "id" => "DR562339304588709888",
        //                 "type" => "COIN_IN",
        //                 "$currency" => "XLM",
        //                 "chain" => "XLM",
        //                 "address" => "GCUTK7KHPJC3ZQJ3OMWWFHAK2OXIBRD4LNZQRCCOVE7A2XOPP2K5PU5Q",
        //                 "addressTag" => "1000009",
        //                 "amount" => 1.0,
        //                 "state" => "SUCCESS",
        //                 "txHash" => "39210645748822f8d4ce673c7559aa6622e6e9cdd7073bc0fcae14b1edfda5f4",
        //                 "createdAt" => 1554113737000,
        //                 "updatedAt" => 1601371273000
        //             }
        //         ),
        //         "message" => "Success"
        //     }
        //
        $data = $this->safe_value($response, 'data', array());
        return $this->parse_transactions($data, $currency, $since, $limit);
    }

    public function parse_transaction_status($status) {
        // Pending the record is wait broadcast to chain
        // x/M confirming the comfirming state of tx, the M is total confirmings needed
        // SUCCESS the record is success full
        // FAIL the record failed
        $parts = explode(' ', $status);
        $status = $this->safe_string($parts, 1, $status);
        $statuses = array(
            'Pending' => 'pending',
            'confirming' => 'pending',
            'SUCCESS' => 'ok',
            'FAIL' => 'failed',
        );
        return $this->safe_string($statuses, $status, $status);
    }

    public function parse_transaction($transaction, $currency = null) {
        //
        // withdraw
        //
        //     {
        //         "$code":"A10000",
        //         "data" => "DR123",
        //         "message":"Success"
        //     }
        //
        // fetchTransactions
        //
        //     {
        //         "$id" => "DR562339304588709888",
        //         "$type" => "COIN_IN",
        //         "$currency" => "XLM",
        //         "chain" => "XLM",
        //         "$address" => "GCUTK7KHPJC3ZQJ3OMWWFHAK2OXIBRD4LNZQRCCOVE7A2XOPP2K5PU5Q",
        //         "addressTag" => "1000009",
        //         "$amount" => 1.0,
        //         "state" => "SUCCESS",
        //         "txHash" => "39210645748822f8d4ce673c7559aa6622e6e9cdd7073bc0fcae14b1edfda5f4",
        //         "createdAt" => 1554113737000,
        //         "updatedAt" => 1601371273000
        //     }
        //
        $id = $this->safe_string_2($transaction, 'id', 'data');
        $type = $this->safe_string($transaction, 'type');
        if ($type === 'COIN_IN') {
            $type = 'deposit';
        } else if ($type === 'COIN_OUT') {
            $type = 'withdraw';
        }
        $amount = $this->safe_float($transaction, 'amount');
        $address = $this->safe_string($transaction, 'address');
        $tag = $this->safe_string($transaction, 'addressTag');
        $txid = $this->safe_string($transaction, 'txHash');
        $timestamp = $this->safe_integer($transaction, 'createdAt');
        $updated = $this->safe_integer($transaction, 'updatedAt');
        $currencyId = $this->safe_string($transaction, 'currency');
        $code = $this->safe_currency_code($currencyId, $currency);
        $status = $this->parse_transaction_status($this->safe_string($transaction, 'state'));
        return array(
            'info' => $transaction,
            'id' => $id,
            'currency' => $code,
            'amount' => $amount,
            'address' => $address,
            'addressTo' => $address,
            'addressFrom' => null,
            'tag' => $tag,
            'tagTo' => $tag,
            'tagFrom' => null,
            'status' => $status,
            'type' => $type,
            'updated' => $updated,
            'txid' => $txid,
            'timestamp' => $timestamp,
            'datetime' => $this->iso8601($timestamp),
            'fee' => null,
        );
    }

    public function fetch_my_trades($symbol = null, $since = null, $limit = null, $params = array ()) {
        yield $this->load_markets();
        $request = array(
            //  'orderId' => id, // Order ID, string
            //  'symbol' => $market['id'], // The trading $symbol, like BTC_BRL, string
            //  'fromId' => fromId, // Search fill id to begin with, string
            //  'toId' => toId, // Search fill id to end up with, string
            //  'fromTimestamp' => $since, // Search order fill time to begin with, in milliseconds, string
            //  'toTimestamp' => $this->milliseconds(), // Search order fill time to end up with, in milliseconds, string
            //  'limit' => $limit, // The number of fills to return, default 100, max 100, string
            //  'accountId' => subaccountId, // Sub account ID, if not informed, the fills will be return under master account, string
        );
        $market = null;
        if ($symbol !== null) {
            $market = $this->market($symbol);
            $request['symbol'] = $market['id'];
        }
        if ($limit !== null) {
            $request['limit'] = $limit;
        }
        if ($since !== null) {
            $request['fromTimestamp'] = $since;
        }
        $response = yield $this->privateGetOrdersFills (array_merge($request, $params));
        //
        //     {
        //         "code" => "A10000",
        //         "$data" => array(
        //             array(
        //                 "id" => "608717046691139584",
        //                 "orderId" => "608716957545402368",
        //                 "$symbol" => "BTC_BRL",
        //                 "side" => "BUY",
        //                 "amount" => "0.0988",
        //                 "price" => "45514.76",
        //                 "fee" => "0.0000988 BTC",
        //                 "feeAmount" => "0.0000988",
        //                 "feeCurrency" => "BTC",
        //                 "role" => "MAKER",
        //                 "timestamp" => 1565171053345
        //             ),
        //             {
        //                 "id" => "608717065729085441",
        //                 "orderId" => "608716957545402368",
        //                 "$symbol" => "BTC_BRL",
        //                 "side" => "BUY",
        //                 "amount" => "0.0242",
        //                 "price" => "45514.76",
        //                 "fee" => "0.0000242 BTC",
        //                 "feeAmount" => "0.0000988",
        //                 "feeCurrency" => "BTC",
        //                 "role" => "MAKER",
        //                 "timestamp" => 1565171057882
        //             }
        //         ),
        //         "message" => "Success"
        //     }
        //
        $data = $this->safe_value($response, 'data', array());
        return $this->parse_trades($data, $market, $since, $limit);
    }

    public function sign($path, $api = 'public', $method = 'GET', $params = array (), $headers = null, $body = null) {
        $request = '/' . $this->version . '/' . $this->implode_params($path, $params);
        $url = $this->urls['api'][$api] . $request;
        $query = $this->omit($params, $this->extract_params($path));
        if ($api === 'public') {
            if ($query) {
                $url .= '?' . $this->urlencode($query);
            }
        } else if ($api === 'private') {
            $this->check_required_credentials();
            $timestamp = (string) $this->milliseconds();
            $headers = array(
                'X-Nova-Access-Key' => $this->apiKey,
                'X-Nova-Timestamp' => $timestamp,
            );
            $queryString = null;
            if ($method === 'POST') {
                $body = $this->json($query);
                $queryString = $this->hash($body, 'md5');
                $headers['Content-Type'] = 'application/json';
            } else {
                if ($query) {
                    $url .= '?' . $this->urlencode($query);
                }
                $queryString = $this->urlencode($this->keysort($query));
            }
            $auth = $method . "\n" . $request . "\n" . $queryString . "\n" . $timestamp; // eslint-disable-line quotes
            $headers['X-Nova-Signature'] = $this->hmac($this->encode($auth), $this->encode($this->secret));
        }
        return array( 'url' => $url, 'method' => $method, 'body' => $body, 'headers' => $headers );
    }

    public function handle_errors($code, $reason, $url, $method, $headers, $body, $response, $requestHeaders, $requestBody) {
        if ($response === null) {
            return;
        }
        //
        //     array("$code":"A10003","data":array(),"$message":"Authentication failed, Invalid accessKey.")
        //
        $errorCode = $this->safe_string($response, 'code');
        if ($errorCode !== 'A10000') {
            $message = $this->safe_string($response, 'message');
            $feedback = $this->id . ' ' . $body;
            $this->throw_exactly_matched_exception($this->exceptions['exact'], $errorCode, $feedback);
            $this->throw_broadly_matched_exception($this->exceptions['broad'], $message, $feedback);
            throw new ExchangeError($feedback); // unknown $message
        }
    }
}
