// Licensed to Cloudera, Inc. under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  Cloudera, Inc. licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

define(function (require, exports, module) {
  "use strict";

  var oop = require("../lib/oop");
  var TextHighlightRules = require("./text_highlight_rules").TextHighlightRules;

  var FlinkHighlightRules = function () {
    // regexp must not have capturing parentheses. Use (?:) instead.
    // regexps are ordered -> the first match is used

    var keywords = (
        "ADD|ADMIN|AFTER|ALL|ALTER|ANALYZE|AND|ARCHIVE|AS|ASC|AUTHORIZATION|BEFORE|BETWEEN|BOTH|BUCKET|BUCKETS|BY|CACHE|CASCADE|CASE|CAST|CHANGE|CLUSTER|CLUSTERED|CLUSTERSTATUS|COLLECTION|COLUMN|COLUMNS|COMMENT|COMMIT|COMPACT|COMPACTIONS|COMPUTE|CONCATENATE|CONF|CONSTRAINT|CONTINUE|CREATE|CROSS|CUBE|CURRENT|CURRENT_DATE|CURRENT_TIMESTAMP|CURSOR|DATA|DATABASE|DATABASES|DATETIME|DAY|DAYOFWEEK|DBPROPERTIES|DEFERRED|DEFINED|DELETE|DEPENDENCY|DESC|DESCRIBE|DIRECTORIES|DIRECTORY|DISABLE|DISTINCT|DISTRIBUTE|DROP|ELEM_TYPE|ELSE|ENABLE|END|ESCAPED|EXCHANGE|EXCLUSIVE|EXISTS|EXPLAIN|EXPORT|EXTENDED|EXTERNAL|EXTRACT|FETCH|FIELDS|FILE|FILEFORMAT|FIRST|FLOOR|FOLLOWING|FOR|FOREIGN|FORMAT|FORMATTED|FROM|FULL|FUNCTION|FUNCTIONS|GRANT|GROUP|GROUPING|HAVING|HOLD_DDLTIME|HOP|HOUR|IDXPROPERTIES|IF|IGNORE|IMPORT|IN|INDEX|INDEXES|INNER|INPATH|INPUTDRIVER|INPUTFORMAT|INSERT|INTERSECT|INTERVAL|INTO|IS|ITEMS|JAR|JOIN|KEY|KEYS|KEY_TYPE|LATERAL|LEFT|LESS|LIKE|LIMIT|LINES|LOAD|LOCAL|LOCATION|LOCK|LOCKS|LOGICAL|LONG|MACRO|MAPJOIN|MATERIALIZED|MINUS|MINUTE|MONTH|MORE|MSCK|NONE|NOSCAN|NOVALIDATE|NOT|NO_DROP|OF|OFFLINE|ON|ONLY|OPTION|OR|ORDER|OUT|OUTER|OUTPUTDRIVER|OUTPUTFORMAT|OVER|OVERWRITE|OWNER|PARTIALSCAN|PARTITION|PARTITIONED|PARTITIONS|PERCENT|PLUS|PRECEDING|PRECISION|PRESERVE|PRETTY|PRIMARY|PRINCIPALS|PROCEDURE|PROTECTION|PURGE|RANGE|READ|READONLY|READS|REBUILD|RECORDREADER|RECORDWRITER|REDUCE|REFERENCES|REGEXP|RELOAD|RENAME|REPAIR|REPLACE|RESTRICT|REVOKE|REWRITE|RIGHT|RLIKE|ROLE|ROLES|ROLLBACK|ROLLUP|ROW|ROWS|SCHEMA|SCHEMAS|SECOND|SELECT|SEMI|SERDEPROPERTIES|SERVER|SESSION|SET|SETS|SHARED|SHOW|SHOW_DATABASE|SKEWED|SORT|SORTED|SSL|START|STATISTICS|STORED|STREAMTABLE|SYNC|TABLE|TABLES|TABLESAMPLE|TBLPROPERTIES|TEMPORARY|TERMINATED|THEN|TO|TOUCH|TRANSACTIONAL|TRANSACTIONS|TRANSFORM|TRIGGER|TRUNCATE|TUMBLE|UNARCHIVE|UNBOUNDED|UNDO|UNION|UNIQUEJOIN|UNLOCK|UNSET|UNSIGNED|UTC_TIMESTAMP|UPDATE|URI|USE|USER|USING|UTC|UTCTIMESTAMP|VALUES|VALUE_TYPE|VIEW|VIEWS|WHEN|WHERE|WHILE|WINDOW|WITH|YEAR"
    );

    var builtinConstants = (
        "FALSE|NULL|TRUE"
    );

    var builtinFunctions = (
        "ABS|ACOS|ADD_MONTHS|AES_DECRYPT|AES_ENCRYPT|ARRAY|ARRAY_CONTAINS|ASCII|ASIN|ATAN|AVG|BASE64|BIN|BINARY|BROUND|CAST|CBRT|CEIL|CEILING|COALESCE|COLLECT_LIST|COLLECT_SET|CONCAT|CONCAT_WS|CONTEXT_NGRAMS|CONV|CORR|COS|COUNT|COVAR_POP|COVAR_SAMP|CRC32|CREATE_UNION|CUME_DIST|CURRENT_DATABASE|CURRENT_DATE|CURRENT_TIMESTAMP|CURRENT_USER|DATE_ADD|DATE_FORMAT|DATE_SUB|DATEDIFF|DAY|DAYOFMONTH|DECODE|DEGREES|DENSE_RANK|E|ENCODE|EXP|EXPLODE|FACTORIAL|FIND_IN_SET|FIRST_VALUE|FLOOR|FORMAT_NUMBER|FROM_UNIXTIME|FROM_UTC_TIMESTAMP|GET_JSON_OBJECT|GREATEST|HASH|HEX|HISTOGRAM_NUMERIC|HOP_END|HOP_PROCTIME|HOP_ROWTIME|HOP_START|HOUR|IF|IN_FILE|INITCAP|INLINE|INSTR|ISNOTNULL|ISNULL|JAVA_METHOD|JSON_TUPLE|LAG|LAST_DAY|LAST_VALUE|LCASE|LEAD|LEAST|LENGTH|LEVENSHTEIN|LISTAGG|LN|LOCATE|LOG|LOG2|LOG10|LOWER|LPAD|LTRIM|MAP|MAP_KEYS|MAP_VALUES|MAX|MD5|MIN|MINUTE|MONTH|MONTHS_BETWEEN|NAMED_STRUCT|NEGATIVE|NEXT_DAY|NGRAMS|NTILE|NVL|PARSE_URL|PARSE_URL_TUPLE|PERCENT_RANK|PERCENTILE|PERCENTILE_APPROX|PI|PMOD|POSEXPLODE|POSITIVE|POW|POWER|PRINTF|PROCTIME|QUARTER|RADIANS|RAND|RANK|REFLECT|REGEXP_EXTRACT|REGEXP_REPLACE|REPEAT|REVERSE|ROUND|ROW_NUMBER|RPAD|RTRIM|SECOND|SENTENCES|SESSION_END|SESSION_PROCTIME|SESSION_ROWTIME|SESSION_START|SHA|SHA1|SHA2|SHIFTLEFT|SHIFTRIGHT|SHIFTRIGHTUNSIGNED|SIGN|SIN|SIZE|SORT_ARRAY|SOUNDEX|SPACE|SPLIT|SQRT|STACK|STDDEV_POP|STDDEV_SAMP|STR_TO_MAP|STRUCT|SUBSTR|SUBSTRING|SUBSTRING_INDEX|SUM|TAN|TO_DATE|TO_UTC_TIMESTAMP|TRANSLATE|TRIM|TRUNC|TUMBLE_END|TUMBLE_PROCTIME|TUMBLE_ROWTIME|TUMBLE_START|UCASE|UNBASE64|UNHEX|UNIX_TIMESTAMP|UPPER|VAR_POP|VAR_SAMP|VARIANCE|WEEKOFYEAR|XPATH|XPATH_BOOLEAN|XPATH_DOUBLE|XPATH_FLOAT|XPATH_INT|XPATH_LONG|XPATH_NUMBER|XPATH_SHORT|XPATH_STRING|YEAR"
    );

    var dataTypes = (
        "ARRAY|BIGINT|BINARY|BOOLEAN|CHAR|DATE|DECIMAL|DELIMITED|DOUBLE|FLOAT|INT|INTEGER|JSONFILE|MAP|NUMERIC|PRECISION|RCFILE|SEQUENCEFILE|SERDE|SMALLINT|STRING|STRUCT|TEXTFILE|TIME|TIMESTAMP|TINYINT|UNIONTYPE|VARCHAR"
    );

    var keywordMapper = this.createKeywordMapper({
        "support.function": builtinFunctions,
        "keyword": keywords,
        "constant.language": builtinConstants,
        "storage.type": dataTypes
    }, "identifier", true);

    this.$rules = {
      start: [
        {
            token : "comment",
            regex : "--.*$"
        },  {
            token : "comment",
            start : "/\\*",
            end : "\\*/"
        }, {
            token : "string",           // " string
            regex : '".*?"'
        }, {
            token : "string",           // ' string
            regex : "'.*?'"
        }, {
            token : "constant.numeric", // float
            regex : "[+-]?\\d+(?:(?:\\.\\d*)?(?:[eE][+-]?\\d+)?)?\\b"
        }, {
            token : keywordMapper,
            regex : "[a-zA-Z_$][a-zA-Z0-9_$]*\\b"
        }, {
            token : "keyword.operator",
            regex : "\\+|\\-|\\/|\\/\\/|%|<@>|@>|<@|&|\\^|~|<|>|<=|=>|==|!=|<>|="
        }, {
            token : "paren.lparen",
            regex : "[\\(]"
        }, {
            token : "paren.rparen",
            regex : "[\\)]"
        }, {
            token : "text",
            regex : "\\s+"
        }
      ]
    };

    this.normalizeRules();
  };

  FlinkHighlightRules.metaData = {
    fileTypes: ["flink", "ql"],
    name: "Flink",
    scopeName: "source.flink"
  };

  oop.inherits(FlinkHighlightRules, TextHighlightRules);

  exports.FlinkHighlightRules = FlinkHighlightRules;
});
