// Licensed to Cloudera, Inc. under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  Cloudera, Inc. licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

const originalSetInterval = window.setInterval;
const originalClearInterval = window.clearInterval;
let hueIntervals = [];

/**
 * @param {string} [app] - context for the intervals
 */
window.getIntervals = function(app) {
  if (app) {
    return hueIntervals.filter(obj => {
      return obj.app === app;
    });
  }
  return hueIntervals;
};

/**
 * @param {Function} fn - the function to be called at intervals
 * @param {Number} timeout - timeout in milliseconds
 * @param {string} [app] - context for the interval
 */
window.setInterval = function(fn, timeout, app) {
  const id = originalSetInterval.call(window, fn, timeout);
  hueIntervals.push({
    fn: fn,
    timeout: timeout,
    app: app,
    id: id,
    originalId: id,
    status: 'running'
  });
  return id;
};

/**
 * @param {Number} id - the original interval id generated by window.setInterval
 */

window.clearInterval = function(id) {
  const foundIntervals = hueIntervals.filter(obj => {
    return obj.originalId === id;
  });
  if (foundIntervals && foundIntervals.length > 0) {
    originalClearInterval.call(window, foundIntervals[0].id);
  } else {
    originalClearInterval.call(window, id);
  }
  hueIntervals = hueIntervals.filter(obj => {
    return obj.originalId !== id;
  });
};

/**
 * @param {string} app - context for the intervals to be suspended
 */
window.pauseAppIntervals = function(app) {
  hueIntervals.forEach(interval => {
    if (
      typeof interval.app !== 'undefined' &&
      interval.app !== null &&
      (interval.app === app || (interval.app.split('-') && interval.app.split('-')[0] === app))
    ) {
      interval.status = 'paused';
      originalClearInterval.call(window, interval.id);
    }
  });
};

/**
 * @param {string} app - context for the intervals to be restarted
 */
window.resumeAppIntervals = function(app) {
  hueIntervals.forEach(interval => {
    if (
      typeof interval.app !== 'undefined' &&
      interval.app !== null &&
      (interval.app === app || interval.app.split('-')[0] === app) &&
      interval.status === 'paused'
    ) {
      interval.status = 'running';
      interval.id = originalSetInterval.call(window, interval.fn, interval.timeout);
    }
  });
};

/**
 * @param {string} app - context for the intervals to be cleared
 */
window.clearAppIntervals = function(app) {
  hueIntervals.forEach(interval => {
    if (
      interval.app === app ||
      (typeof interval.app !== 'undefined' &&
        interval.app !== null &&
        interval.app.split('-')[0] === app)
    ) {
      window.clearInterval(interval.originalId);
    }
  });
};
