/**
 * Internal dependencies
 */
import applyWithColors from './colors';
import { BackgroundPanel } from '../../../components/background';
import DimensionsControl from '../../../components/dimensions-control/';

/**
 * WordPress dependencies
 */
import { __ } from '@wordpress/i18n';
import { compose } from '@wordpress/compose';
import { InspectorControls, PanelColorSettings } from '@wordpress/block-editor';
import { PanelBody, withFallbackStyles } from '@wordpress/components';

/**
 * Fallback styles
 */
const { getComputedStyle } = window;
const applyFallbackStyles = withFallbackStyles( ( node, ownProps ) => {
	const { backgroundColor } = ownProps.attributes;
	const editableNode = node.querySelector( '[contenteditable="true"]' );
	//verify if editableNode is available, before using getComputedStyle.
	const computedStyles = editableNode ? getComputedStyle( editableNode ) : null;
	return {
		fallbackBackgroundColor: backgroundColor || ! computedStyles ? undefined : computedStyles.backgroundColor,
	};
} );

/**
 * Inspector controls
 *
 * @param {Object} props
 */
const Inspector = ( props ) => {
	const {
		attributes,
		backgroundColor,
		setAttributes,
		setBackgroundColor,
		textColor,
		setTextColor,
	} = props;

	const {
		paddingBottom,
		paddingLeft,
		paddingRight,
		paddingSize,
		paddingTop,
		paddingBottomTablet,
		paddingLeftTablet,
		paddingRightTablet,
		paddingTopTablet,
		paddingBottomMobile,
		paddingLeftMobile,
		paddingRightMobile,
		paddingTopMobile,
		paddingSyncUnits,
		paddingSyncUnitsTablet,
		paddingSyncUnitsMobile,
		paddingUnit,
	} = attributes;

	return (
		<>
			<InspectorControls>
				<PanelBody title={ __( 'Feature settings', 'coblocks' ) }>
					<DimensionsControl { ...props }
						type={ 'padding' }
						label={ __( 'Padding', 'coblocks' ) }
						valueTop={ paddingTop }
						valueRight={ paddingRight }
						valueBottom={ paddingBottom }
						valueLeft={ paddingLeft }
						valueTopTablet={ paddingTopTablet }
						valueRightTablet={ paddingRightTablet }
						valueBottomTablet={ paddingBottomTablet }
						valueLeftTablet={ paddingLeftTablet }
						valueTopMobile={ paddingTopMobile }
						valueRightMobile={ paddingRightMobile }
						valueBottomMobile={ paddingBottomMobile }
						valueLeftMobile={ paddingLeftMobile }
						unit={ paddingUnit }
						syncUnits={ paddingSyncUnits }
						syncUnitsTablet={ paddingSyncUnitsTablet }
						syncUnitsMobile={ paddingSyncUnitsMobile }
						dimensionSize={ paddingSize }
					/>
				</PanelBody>
				<PanelColorSettings
					title={ __( 'Color settings', 'coblocks' ) }
					initialOpen={ false }
					colorSettings={ [
						{
							value: backgroundColor.color,
							onChange: ( nextBackground ) => {
								setBackgroundColor( nextBackground );

								//add padding if there's none
								if ( ! paddingSize || paddingSize === 'no' ) {
									setAttributes( { paddingSize: 'medium' } );
								}

								//reset when cleared
								if ( ! nextBackground ) {
									setAttributes( { paddingSize: 'no' } );
								}
							},
							label: __( 'Background color', 'coblocks' ),
						}, {
							value: textColor.color,
							onChange: setTextColor,
							label: __( 'Text color', 'coblocks' ),
						},
					] }
				>
				</PanelColorSettings>
				<BackgroundPanel { ...props }
					hasOverlay={ true }
				/>
			</InspectorControls>
		</>
	);
};

export default compose( [
	applyWithColors,
	applyFallbackStyles,
] )( Inspector );
