
/**
 * Internal dependencies
 */
import applyWithColors from './colors';
import { BackgroundPanel } from '../../../components/background';
import DimensionsControl from '../../../components/dimensions-control';

/**
 * WordPress dependencies
 */
import { __ } from '@wordpress/i18n';
import { compose } from '@wordpress/compose';
import { InspectorControls, PanelColorSettings } from '@wordpress/block-editor';
import { PanelBody, RangeControl, withFallbackStyles } from '@wordpress/components';

/**
 * Fallback styles
 */
const { getComputedStyle } = window;

const FallbackStyles = withFallbackStyles( ( node, ownProps ) => {
	const { backgroundColor } = ownProps.attributes;

	const editableNode = node.querySelector( '[contenteditable="true"]' );

	//verify if editableNode is available, before using getComputedStyle.
	const computedStyles = editableNode ? getComputedStyle( editableNode ) : null;

	return {
		fallbackBackgroundColor: backgroundColor || ! computedStyles ? undefined : computedStyles.backgroundColor,
	};
} );

/**
 * Inspector controls
 *
 * @param {Object} props
 */
const Inspector = ( props ) => {
	const {
		clientId,
		attributes,
		setAttributes,
		backgroundColor,
		setBackgroundColor,
		setTextColor,
		textColor,
		nextBlockClient,
		nextBlockClientId,
		lastId,
		updateBlockAttributes,
	} = props;

	const {
		width,
		marginTop,
		marginRight,
		marginBottom,
		marginLeft,
		marginTopTablet,
		marginRightTablet,
		marginBottomTablet,
		marginLeftTablet,
		marginTopMobile,
		marginRightMobile,
		marginBottomMobile,
		marginLeftMobile,
		marginUnit,
		marginSyncUnits,
		marginSyncUnitsTablet,
		marginSyncUnitsMobile,
		marginSize,
		paddingTop,
		paddingRight,
		paddingBottom,
		paddingLeft,
		paddingTopTablet,
		paddingRightTablet,
		paddingBottomTablet,
		paddingLeftTablet,
		paddingTopMobile,
		paddingRightMobile,
		paddingBottomMobile,
		paddingLeftMobile,
		paddingUnit,
		paddingSyncUnits,
		paddingSyncUnitsTablet,
		paddingSyncUnitsMobile,
		paddingSize,
	} = attributes;

	const onChangeWidth = ( newWidth ) => {
		const diff = parseFloat( width ) - newWidth;
		const nextBlockWidth = parseFloat( nextBlockClient.attributes.width ) + diff;

		if ( nextBlockWidth > 9 ) {
			setAttributes( { width: parseFloat( newWidth ).toFixed( 2 ) } );
			updateBlockAttributes( nextBlockClientId, { width: parseFloat( nextBlockWidth ).toFixed( 2 ) } );
		}
	};

	return (
		<>
			<InspectorControls>
				<PanelBody title={ __( 'Column settings', 'coblocks' ) } className="components-panel__body--column-settings">
					<DimensionsControl { ...props }
						type={ 'padding' }
						label={ __( 'Padding', 'coblocks' ) }
						help={ __( 'Space inside of the container.', 'coblocks' ) }
						valueTop={ paddingTop }
						valueRight={ paddingRight }
						valueBottom={ paddingBottom }
						valueLeft={ paddingLeft }
						valueTopTablet={ paddingTopTablet }
						valueRightTablet={ paddingRightTablet }
						valueBottomTablet={ paddingBottomTablet }
						valueLeftTablet={ paddingLeftTablet }
						valueTopMobile={ paddingTopMobile }
						valueRightMobile={ paddingRightMobile }
						valueBottomMobile={ paddingBottomMobile }
						valueLeftMobile={ paddingLeftMobile }
						unit={ paddingUnit }
						syncUnits={ paddingSyncUnits }
						syncUnitsTablet={ paddingSyncUnitsTablet }
						syncUnitsMobile={ paddingSyncUnitsMobile }
						dimensionSize={ paddingSize }
					/>
					<DimensionsControl { ...props }
						type={ 'margin' }
						label={ __( 'Margin', 'coblocks' ) }
						help={ __( 'Space around the container.', 'coblocks' ) }
						valueTop={ marginTop }
						valueRight={ marginRight }
						valueBottom={ marginBottom }
						valueLeft={ marginLeft }
						valueTopTablet={ marginTopTablet }
						valueRightTablet={ marginRightTablet }
						valueBottomTablet={ marginBottomTablet }
						valueLeftTablet={ marginLeftTablet }
						valueTopMobile={ marginTopMobile }
						valueRightMobile={ marginRightMobile }
						valueBottomMobile={ marginBottomMobile }
						valueLeftMobile={ marginLeftMobile }
						unit={ marginUnit }
						syncUnits={ marginSyncUnits }
						syncUnitsTablet={ marginSyncUnitsTablet }
						syncUnitsMobile={ marginSyncUnitsMobile }
						dimensionSize={ marginSize }
					/>
					{ ( lastId !== clientId )
						? (
							<RangeControl
								label={ __( 'Width', 'coblocks' ) }
								value={ parseFloat( width ) }
								onChange={ ( newWidth ) => onChangeWidth( newWidth ) }
								min={ 10.00 }
								max={ 100.00 }
								step={ 0.01 }
							/>
						) : null }
				</PanelBody>
				<PanelColorSettings
					title={ __( 'Color settings', 'coblocks' ) }
					initialOpen={ false }
					colorSettings={ [
						{
							value: backgroundColor.color,
							onChange: setBackgroundColor,
							label: __( 'Background color', 'coblocks' ),
						},
						{
							value: textColor.color,
							onChange: setTextColor,
							label: __( 'Text color', 'coblocks' ),
						},
					] }
				>
				</PanelColorSettings>
				<BackgroundPanel { ...props }
					hasOverlay={ true }
				/>
			</InspectorControls>
		</>
	);
};

export default compose( [
	applyWithColors,
	FallbackStyles,
] )( Inspector );
