/*
 * Copyright (c) 2018 David Sargent
 *
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted (subject to the limitations in the disclaimer below) provided that
 * the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice, this list
 * of conditions and the following disclaimer.
 *
 * Redistributions in binary form must reproduce the above copyright notice, this
 * list of conditions and the following disclaimer in the documentation and/or
 * other materials provided with the distribution.
 * Neither the name of NAME nor the names of its contributors may be used to
 * endorse or promote products derived from this software without specific prior
 * written permission.
 *
 * NO EXPRESS OR IMPLIED LICENSES TO ANY PARTY'S PATENT RIGHTS ARE GRANTED BY THIS
 * LICENSE. THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESSFOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

package com.google.blocks.ftcrobotcontroller.onbotjava.handlers.file;

import com.google.blocks.ftcrobotcontroller.JavaModeWebHandlers;
import com.google.blocks.ftcrobotcontroller.onbotjava.JavaSourceFile;
import com.google.blocks.ftcrobotcontroller.onbotjava.OnBotJavaFileSystemUtils;
import com.google.blocks.ftcrobotcontroller.onbotjava.RegisterWebHandler;
import com.google.blocks.ftcrobotcontroller.onbotjava.StandardResponses;
import com.qualcomm.robotcore.util.ReadWriteFile;
import com.qualcomm.robotcore.util.RobotLog;

import org.firstinspires.ftc.robotcore.internal.opmode.OnBotJavaManager;
import org.firstinspires.ftc.robotcore.internal.system.AppUtil;
import org.firstinspires.ftc.robotcore.internal.webserver.WebHandler;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import fi.iki.elonen.NanoHTTPD;

import static com.google.blocks.ftcrobotcontroller.onbotjava.OnBotJavaFileSystemUtils.EXT_JAVA_FILE;
import static com.google.blocks.ftcrobotcontroller.onbotjava.OnBotJavaFileSystemUtils.PATH_SEPARATOR;
import static com.google.blocks.ftcrobotcontroller.onbotjava.StandardResponses.badRequest;
import static com.google.blocks.ftcrobotcontroller.onbotjava.StandardResponses.serverError;

@RegisterWebHandler(uri = JavaModeWebHandlers.URI_FILE_UPLOAD, usesParamGenerator = false)
public class UploadFiles implements WebHandler {
    private static final String TAG = UploadFiles.class.getName();

    @Override
    public NanoHTTPD.Response getResponse(NanoHTTPD.IHTTPSession session) throws IOException, NanoHTTPD.ResponseException {
        Map<String, String> files = new HashMap<>();
        session.parseBody(files);
        return uploadFile(session.getParameters(), files);
    }

    private NanoHTTPD.Response uploadFile(Map<String, List<String>> data, Map<String, String> files) {
        if (!data.containsKey("file")) return badRequest();
        final String newFileName = data.get("file").get(0).replaceAll("(?:\\.\\.|\\\\|/)", "");
        final File src = new File(files.get("file"));

        String newFolderName = null;
        if (data.containsKey("folder")) {
            newFolderName = data.get("folder").get(0);
            if (newFolderName.contains(".."))
                newFolderName = null; // keep things secure
        }

        File dest;
        if (newFileName.endsWith(EXT_JAVA_FILE)) {
            dest = OnBotJavaManager.srcDir;
            if (newFolderName == null) {
                final String code = ReadWriteFile.readFile(src);
                String group = JavaSourceFile.extractPackageNameFromContents(code);

                if (group != null) {
                    String folder = group.replaceAll("\\.", PATH_SEPARATOR);
                    dest = new File(dest, folder);
                } else {
                    dest = OnBotJavaManager.srcDir;
                }
            }
        } else if (newFileName.endsWith(OnBotJavaFileSystemUtils.EXT_ZIP_FILE) || newFileName.endsWith(".jar")) {
            dest = OnBotJavaManager.jarDir;
        } else {
            dest = OnBotJavaManager.srcDir;
        }

        dest = new File(dest, newFileName);
        dest.getParentFile().mkdirs();

        try {
            AppUtil.getInstance().copyFile(src, dest);
        } catch (IOException e) {
            RobotLog.ee(TAG, e, "Error copying file \"%s\"", newFileName);
            return serverError();
        }

        return StandardResponses.successfulJsonRequest(newFileName);
    }
}
