/*
 * Copyright (c) 2018 David Sargent
 *
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted (subject to the limitations in the disclaimer below) provided that
 * the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice, this list
 * of conditions and the following disclaimer.
 *
 * Redistributions in binary form must reproduce the above copyright notice, this
 * list of conditions and the following disclaimer in the documentation and/or
 * other materials provided with the distribution.
 * Neither the name of NAME nor the names of its contributors may be used to
 * endorse or promote products derived from this software without specific prior
 * written permission.
 *
 * NO EXPRESS OR IMPLIED LICENSES TO ANY PARTY'S PATENT RIGHTS ARE GRANTED BY THIS
 * LICENSE. THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESSFOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

package com.google.blocks.ftcrobotcontroller.onbotjava;

import android.support.annotation.NonNull;

import com.qualcomm.robotcore.util.ReadWriteFile;
import com.qualcomm.robotcore.util.RobotLog;

import org.firstinspires.ftc.robotcore.internal.files.FileModifyObserver;
import org.firstinspires.ftc.robotcore.internal.opmode.OnBotJavaManager;

import java.io.File;
import java.io.FileNotFoundException;

import fi.iki.elonen.NanoHTTPD;

import static com.google.blocks.ftcrobotcontroller.onbotjava.StandardResponses.serverError;

public class BuildMonitor {
    private final Object buildCompletionNotifier = new Object();
    private final Object buildInformationUpdateLock = new Object();

    // Status src
    private static final File buildStartedFile = OnBotJavaManager.buildStartedFile;
    private static final File buildCompleteFile = OnBotJavaManager.buildCompleteFile;

    // Status variables
    private final FileModifyObserver runningObserver;
    private final FileModifyObserver completedObserver;

    /*
     *      <li>status: As the build progresses, several src are written. These src may be
     *                monitored for changes (see {@link FileModifyObserver}) as triggers to take
     *                actions to process the output of the build.
     *                <ol>
     *                      <li>buildStarted.txt:    updated when the build starts</li>
     *                      <li>buildSuccessful.txt: updated when the build has been determined
     *                                               to be successful</li>
     *                      <li>buildComplete.txt:   updated when the build finishes, whether
     *                                               or not it was successful</li>
     *                      <li>buildLog.txt:        contains unstructured output from the compiler
     *                                               and other build tools. Note that after successful
     *                                               builds, this will likely be empty.</li>
     *                </ol>
     *                To this list it might be reasonable to add buildLog.xml that had a structured
     *                version of buildLog.txt, but that has not yet been implemented. Feedback is welcome.
     *                </li>
     */
    // Status variables
    private boolean isRunning = false;
    private boolean lastBuildSuccessful = false;
    private boolean lastBuildCompleted = false;
    private boolean closed = false;

    BuildMonitor() {
        //final String statusDirPath = OnBotJavaManager.statusDir;
        File statusDir = OnBotJavaManager.statusDir;
        if (!statusDir.isDirectory()) statusDir.mkdirs();

        runningObserver = new FileModifyObserver(buildStartedFile,
            new FileModifyObserver.Listener() {
                @Override
                public void onFileChanged(int event, File file) {
                    synchronized (buildInformationUpdateLock) {
                        isRunning = true;
                        lastBuildCompleted = false;
                        lastBuildSuccessful = false;
                    }
                }
            }
        );
        completedObserver = new FileModifyObserver(buildCompleteFile, new
            FileModifyObserver.Listener() {
                @Override
                public void onFileChanged(int event, File file) {
                    synchronized (buildInformationUpdateLock) {
                        isRunning = false;
                        lastBuildCompleted = true;
                        lastBuildSuccessful = OnBotJavaManager.getBuildStatus() == OnBotJavaManager.BuildStatus.SUCCESSFUL;
                        synchronized (buildCompletionNotifier) {
                            buildCompletionNotifier.notifyAll();
                        }
                    }
                }
            }
        );
    }

    public BuildStatus currentBuildStatus() {
        if (closed) throw new IllegalStateException("BuildMonitor has been closed!");
        synchronized (buildInformationUpdateLock) {
            return new BuildStatus(lastBuildCompleted, isRunning, lastBuildSuccessful);
        }
    }

    @NonNull
    public NanoHTTPD.Response currentBuildLog() {
        try {
            return OnBotJavaFileSystemUtils.serveFile(OnBotJavaManager.buildLogFile.getAbsolutePath());
        } catch (FileNotFoundException e) {
            return serverError();
        }
    }

    public boolean waitForRunningBuildCompletion() {
        synchronized (buildCompletionNotifier) {
            try {
                while (isRunning) {
                    buildCompletionNotifier.wait();
                }
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                return true;
            }
        }

        return false;
    }

    public void close() {
        closed = true;
        runningObserver.close();
        completedObserver.close();
    }

    @Override
    public void finalize() throws Throwable {
        super.finalize();
        if (closed) return;
        RobotLog.ww("FetchBuildStatus", "Did not call close(), running finalizer");
        close();
    }

    public long launchBuild() {
        final File buildStartFile = OnBotJavaManager.buildRequestFile;
        if (OnBotJavaWebInterfaceManager.instance().buildMonitor().waitForRunningBuildCompletion())
            return 0;
        final long startTime = System.currentTimeMillis();
        ReadWriteFile.writeFile(buildStartFile, Long.toString(startTime) + " - begin build");
        return startTime;
    }

    public static class BuildStatus {
        final boolean completed;
        final boolean running;
        final boolean successful;
        final long timestamp;

        private BuildStatus(boolean completed, boolean running, boolean successful) {
            this.completed = completed;
            this.running = running;
            this.successful = successful;
            this.timestamp = System.nanoTime();
        }
    }
}
