/*
Copyright (c) 2018 Craig MacFarlane

All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted (subject to the limitations in the disclaimer below) provided that
the following conditions are met:

Redistributions of source code must retain the above copyright notice, this list
of conditions and the following disclaimer.

Redistributions in binary form must reproduce the above copyright notice, this
list of conditions and the following disclaimer in the documentation and/or
other materials provided with the distribution.

Neither the name of Craig MacFarlane nor the names of his contributors may be used to
endorse or promote products derived from this software without specific prior
written permission.

NO EXPRESS OR IMPLIED LICENSES TO ANY PARTY'S PATENT RIGHTS ARE GRANTED BY THIS
LICENSE. THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESSFOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
package org.firstinspires.ftc.robotcore.internal.network;

import android.bluetooth.BluetoothAdapter;
import android.content.Context;
import android.net.NetworkInfo;
import android.net.wifi.SupplicantState;
import android.net.wifi.WifiConfiguration;
import android.net.wifi.WifiInfo;
import android.net.wifi.WifiManager;
import android.provider.Settings;

import com.qualcomm.robotcore.util.RobotLog;

import org.firstinspires.ftc.robotcore.internal.system.AppUtil;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

/*
 * A collection of useful wifi related utilities.
 */
public class WifiUtil {

    private static final String NO_AP = "None";

    public static boolean isAirplaneModeOn()
    {
        return Settings.Global.getInt(AppUtil.getDefContext().getContentResolver(), Settings.Global.AIRPLANE_MODE_ON, 0) != 0;
    }

    public static boolean isBluetoothOn()
    {
        return BluetoothAdapter.getDefaultAdapter() != null && BluetoothAdapter.getDefaultAdapter().isEnabled();
    }

    public static boolean isWifiEnabled()
    {
        WifiManager mgr = getWifiManager();
        int state = mgr.getWifiState();
        RobotLog.i("state = " + state);
        return getWifiManager().isWifiEnabled();
    }

    public static boolean isWifiApEnabled()
    {
        WifiManager wifiMgr = getWifiManager();
        try {
            Method isEnabled = wifiMgr.getClass().getMethod("isWifiApEnabled");
            return (Boolean)isEnabled.invoke(wifiMgr);
        } catch (NoSuchMethodException | IllegalAccessException | InvocationTargetException e) {
            RobotLog.e("Could not invoke isWifiApEnabled " + e.toString());
        }

        return false;
    }

    public static boolean isWifiConnected()
    {
        /*
         * The supplicant state may be in a state of obtaining an ip address even when wifi is not enabled!
         * Ergo, one can not rely upon the WifiManager alone to determine connection state.
         */
        if (!isWifiEnabled()) {
            return false;
        }

        WifiManager m = getWifiManager();
        SupplicantState s = m.getConnectionInfo().getSupplicantState();
        NetworkInfo.DetailedState state = WifiInfo.getDetailedStateOf(s);

        return (state == NetworkInfo.DetailedState.CONNECTED ||
                state == NetworkInfo.DetailedState.OBTAINING_IPADDR);
    }

    /*
     * getConnectedSsid
     *
     * This will return the name of an access point a device is connected.  Note however that
     * it will not return results if the device is acting as an access point.  e.g. Don't invoke
     * this on a control hub and expect it to return what you expect it to.
     */
    public static String getConnectedSsid()
    {
        if (!isWifiConnected()) {
            return NO_AP;
        } else {
            WifiInfo wifiInfo = getWifiManager().getConnectionInfo();
            return wifiInfo.getSSID().replace("\"", "");
        }
    }

    protected static WifiManager getWifiManager()
    {
        return (WifiManager)AppUtil.getDefContext().getApplicationContext().getSystemService(Context.WIFI_SERVICE);
    }
}
