/*
Copyright 2019 The Kubernetes Authors.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package coredns

import (
	"encoding/json"
	"fmt"
	"net"

	etcdc "github.com/coreos/etcd/client"
	dnsmsg "github.com/miekg/coredns/middleware/etcd/msg"
	"golang.org/x/net/context"
	"k8s.io/klog"
	"k8s.io/kops/dnsprovider/pkg/dnsprovider"
	"k8s.io/kops/dnsprovider/pkg/dnsprovider/rrstype"
)

// Compile time check for interface adherence
var _ dnsprovider.ResourceRecordSets = ResourceRecordSets{}

type ResourceRecordSets struct {
	zone *Zone
}

func (rrsets ResourceRecordSets) List() ([]dnsprovider.ResourceRecordSet, error) {
	var list []dnsprovider.ResourceRecordSet
	return list, fmt.Errorf("OperationNotSupported")
}

func (rrsets ResourceRecordSets) Get(name string) ([]dnsprovider.ResourceRecordSet, error) {
	getOpts := &etcdc.GetOptions{
		Recursive: true,
	}
	etcdPathPrefix := rrsets.zone.zones.intf.etcdPathPrefix
	response, err := rrsets.zone.zones.intf.etcdKeysAPI.Get(context.Background(), dnsmsg.Path(name, etcdPathPrefix), getOpts)
	if err != nil {
		if etcdc.IsKeyNotFound(err) {
			klog.V(2).Infof("Subdomain %q does not exist", name)
			return nil, nil
		}
		return nil, fmt.Errorf("Failed to get service from etcd, err: %v", err)
	}
	if emptyResponse(response) {
		klog.V(2).Infof("Subdomain %q does not exist in etcd", name)
		return nil, nil
	}

	var list []dnsprovider.ResourceRecordSet

	for _, node := range response.Node.Nodes {
		service := dnsmsg.Service{}
		err = json.Unmarshal([]byte(node.Value), &service)
		if err != nil {
			return nil, fmt.Errorf("Failed to unmarshall json data, err: %v", err)
		}

		rrset := ResourceRecordSet{name: name, rrdatas: []string{}, rrsets: &rrsets}
		ip := net.ParseIP(service.Host)
		switch {
		case ip == nil:
			rrset.rrsType = rrstype.CNAME
		case ip.To4() != nil:
			rrset.rrsType = rrstype.A
		case ip.To16() != nil:
			rrset.rrsType = rrstype.AAAA
		default:
			// Cannot occur
		}
		rrset.rrdatas = append(rrset.rrdatas, service.Host)
		rrset.ttl = int64(service.TTL)
		list = append(list, rrset)
	}

	return list, nil
}

func (rrsets ResourceRecordSets) StartChangeset() dnsprovider.ResourceRecordChangeset {
	return &ResourceRecordChangeset{
		zone:   rrsets.zone,
		rrsets: &rrsets,
	}
}

func (rrsets ResourceRecordSets) New(name string, rrdatas []string, ttl int64, rrsType rrstype.RrsType) dnsprovider.ResourceRecordSet {
	return ResourceRecordSet{
		name:    name,
		rrdatas: rrdatas,
		ttl:     ttl,
		rrsType: rrsType,
		rrsets:  &rrsets,
	}
}

// Zone returns the parent zone
func (rrset ResourceRecordSets) Zone() dnsprovider.Zone {
	return rrset.zone
}

func emptyResponse(resp *etcdc.Response) bool {
	return resp == nil || resp.Node == nil || (len(resp.Node.Value) == 0 && len(resp.Node.Nodes) == 0)
}
