/*
Copyright 2019 The Kubernetes Authors.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package awsup

import (
	"fmt"

	"k8s.io/klog"
)

// I believe one vCPU ~ 3 ECUS, and 60 CPU credits would be needed to use one vCPU for an hour
const BurstableCreditsToECUS float32 = 3.0 / 60.0

type AWSMachineTypeInfo struct {
	Name              string
	MemoryGB          float32
	ECU               float32
	Cores             int
	EphemeralDisks    []int
	Burstable         bool
	GPU               bool
	MaxPods           int
	InstanceENIs      int
	InstanceIPsPerENI int
}

type EphemeralDevice struct {
	DeviceName  string
	VirtualName string
	SizeGB      int
}

func (m *AWSMachineTypeInfo) EphemeralDevices() []*EphemeralDevice {
	var disks []*EphemeralDevice
	for i, sizeGB := range m.EphemeralDisks {
		d := &EphemeralDevice{
			SizeGB: sizeGB,
		}

		if i >= 20 {
			// TODO: What drive letters do we use?
			klog.Fatalf("ephemeral devices for > 20 not yet implemented")
		}
		d.DeviceName = "/dev/sd" + string('c'+i)
		d.VirtualName = fmt.Sprintf("ephemeral%d", i)

		disks = append(disks, d)
	}
	return disks
}

func GetMachineTypeInfo(machineType string) (*AWSMachineTypeInfo, error) {
	for i := range MachineTypes {
		m := &MachineTypes[i]
		if m.Name == machineType {
			return m, nil
		}
	}

	return nil, fmt.Errorf("instance type not handled: %q", machineType)
}

var MachineTypes []AWSMachineTypeInfo = []AWSMachineTypeInfo{
	// This is tedious, but seems simpler than trying to have some logic and then a lot of exceptions

	// NOTE: Content below is auto generated by `make update-machine-types`
	// BEGIN GENERATED CONTENT

	// WARNING: ENIs not known for m6g.medium
	// WARNING: IPs per ENI not known for m6g.medium
	// WARNING: ENIs not known for m6g.8xlarge
	// WARNING: IPs per ENI not known for m6g.8xlarge
	// WARNING: ENIs not known for m6g.16xlarge
	// WARNING: IPs per ENI not known for m6g.16xlarge
	// WARNING: ENIs not known for m6g.2xlarge
	// WARNING: IPs per ENI not known for m6g.2xlarge
	// WARNING: ENIs not known for m6g.xlarge
	// WARNING: IPs per ENI not known for m6g.xlarge
	// WARNING: ENIs not known for m6g.4xlarge
	// WARNING: IPs per ENI not known for m6g.4xlarge
	// WARNING: ENIs not known for m6g.12xlarge
	// WARNING: IPs per ENI not known for m6g.12xlarge
	// WARNING: ENIs not known for m6g.large
	// WARNING: IPs per ENI not known for m6g.large

	// a1 family
	{
		Name:              "a1.medium",
		MemoryGB:          2,
		ECU:               0,
		Cores:             1,
		InstanceENIs:      2,
		InstanceIPsPerENI: 4,
		EphemeralDisks:    nil,
	},

	{
		Name:              "a1.large",
		MemoryGB:          4,
		ECU:               0,
		Cores:             2,
		InstanceENIs:      3,
		InstanceIPsPerENI: 10,
		EphemeralDisks:    nil,
	},

	{
		Name:              "a1.xlarge",
		MemoryGB:          8,
		ECU:               0,
		Cores:             4,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    nil,
	},

	{
		Name:              "a1.2xlarge",
		MemoryGB:          16,
		ECU:               0,
		Cores:             8,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    nil,
	},

	{
		Name:              "a1.4xlarge",
		MemoryGB:          32,
		ECU:               0,
		Cores:             16,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    nil,
	},

	{
		Name:              "a1.metal",
		MemoryGB:          32,
		ECU:               0,
		Cores:             16,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    nil,
	},

	// c1 family
	{
		Name:              "c1.medium",
		MemoryGB:          1.7,
		ECU:               5,
		Cores:             2,
		InstanceENIs:      2,
		InstanceIPsPerENI: 6,
		EphemeralDisks:    []int{350},
	},

	{
		Name:              "c1.xlarge",
		MemoryGB:          7,
		ECU:               20,
		Cores:             8,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    []int{420, 420, 420, 420},
	},

	// c3 family
	{
		Name:              "c3.large",
		MemoryGB:          3.75,
		ECU:               7,
		Cores:             2,
		InstanceENIs:      3,
		InstanceIPsPerENI: 10,
		EphemeralDisks:    []int{16, 16},
	},

	{
		Name:              "c3.xlarge",
		MemoryGB:          7.5,
		ECU:               14,
		Cores:             4,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    []int{40, 40},
	},

	{
		Name:              "c3.2xlarge",
		MemoryGB:          15,
		ECU:               28,
		Cores:             8,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    []int{80, 80},
	},

	{
		Name:              "c3.4xlarge",
		MemoryGB:          30,
		ECU:               55,
		Cores:             16,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    []int{160, 160},
	},

	{
		Name:              "c3.8xlarge",
		MemoryGB:          60,
		ECU:               108,
		Cores:             36,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    []int{320, 320},
	},

	// c4 family
	{
		Name:              "c4.large",
		MemoryGB:          3.75,
		ECU:               8,
		Cores:             2,
		InstanceENIs:      3,
		InstanceIPsPerENI: 10,
		EphemeralDisks:    nil,
	},

	{
		Name:              "c4.xlarge",
		MemoryGB:          7.5,
		ECU:               16,
		Cores:             4,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    nil,
	},

	{
		Name:              "c4.2xlarge",
		MemoryGB:          15,
		ECU:               31,
		Cores:             8,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    nil,
	},

	{
		Name:              "c4.4xlarge",
		MemoryGB:          30,
		ECU:               62,
		Cores:             16,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    nil,
	},

	{
		Name:              "c4.8xlarge",
		MemoryGB:          60,
		ECU:               132,
		Cores:             36,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    nil,
	},

	// c5 family
	{
		Name:              "c5.large",
		MemoryGB:          4,
		ECU:               10,
		Cores:             2,
		InstanceENIs:      3,
		InstanceIPsPerENI: 10,
		EphemeralDisks:    nil,
	},

	{
		Name:              "c5.xlarge",
		MemoryGB:          8,
		ECU:               20,
		Cores:             4,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    nil,
	},

	{
		Name:              "c5.2xlarge",
		MemoryGB:          16,
		ECU:               39,
		Cores:             8,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    nil,
	},

	{
		Name:              "c5.4xlarge",
		MemoryGB:          32,
		ECU:               73,
		Cores:             16,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    nil,
	},

	{
		Name:              "c5.9xlarge",
		MemoryGB:          72,
		ECU:               139,
		Cores:             36,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    nil,
	},

	{
		Name:              "c5.12xlarge",
		MemoryGB:          96,
		ECU:               188,
		Cores:             48,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    nil,
	},

	{
		Name:              "c5.18xlarge",
		MemoryGB:          144,
		ECU:               281,
		Cores:             72,
		InstanceENIs:      15,
		InstanceIPsPerENI: 50,
		EphemeralDisks:    nil,
	},

	{
		Name:              "c5.24xlarge",
		MemoryGB:          192,
		ECU:               375,
		Cores:             96,
		InstanceENIs:      15,
		InstanceIPsPerENI: 50,
		EphemeralDisks:    nil,
	},

	{
		Name:              "c5.metal",
		MemoryGB:          192,
		ECU:               375,
		Cores:             96,
		InstanceENIs:      15,
		InstanceIPsPerENI: 50,
		EphemeralDisks:    nil,
	},

	// c5d family
	{
		Name:              "c5d.large",
		MemoryGB:          4,
		ECU:               10,
		Cores:             2,
		InstanceENIs:      3,
		InstanceIPsPerENI: 10,
		EphemeralDisks:    []int{50},
	},

	{
		Name:              "c5d.xlarge",
		MemoryGB:          8,
		ECU:               20,
		Cores:             4,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    []int{100},
	},

	{
		Name:              "c5d.2xlarge",
		MemoryGB:          16,
		ECU:               39,
		Cores:             8,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    []int{200},
	},

	{
		Name:              "c5d.4xlarge",
		MemoryGB:          32,
		ECU:               73,
		Cores:             16,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    []int{400},
	},

	{
		Name:              "c5d.9xlarge",
		MemoryGB:          72,
		ECU:               139,
		Cores:             36,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    []int{900},
	},

	{
		Name:              "c5d.12xlarge",
		MemoryGB:          96,
		ECU:               188,
		Cores:             48,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    []int{900, 900},
	},

	{
		Name:              "c5d.18xlarge",
		MemoryGB:          144,
		ECU:               281,
		Cores:             72,
		InstanceENIs:      15,
		InstanceIPsPerENI: 50,
		EphemeralDisks:    []int{900, 900},
	},

	{
		Name:              "c5d.24xlarge",
		MemoryGB:          192,
		ECU:               375,
		Cores:             96,
		InstanceENIs:      15,
		InstanceIPsPerENI: 50,
		EphemeralDisks:    []int{900, 900, 900, 900},
	},

	{
		Name:              "c5d.metal",
		MemoryGB:          192,
		ECU:               375,
		Cores:             96,
		InstanceENIs:      15,
		InstanceIPsPerENI: 50,
		EphemeralDisks:    []int{900, 900, 900, 900},
	},

	// c5n family
	{
		Name:              "c5n.large",
		MemoryGB:          5.25,
		ECU:               10,
		Cores:             2,
		InstanceENIs:      3,
		InstanceIPsPerENI: 10,
		EphemeralDisks:    nil,
	},

	{
		Name:              "c5n.xlarge",
		MemoryGB:          10.5,
		ECU:               20,
		Cores:             4,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    nil,
	},

	{
		Name:              "c5n.2xlarge",
		MemoryGB:          21,
		ECU:               39,
		Cores:             8,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    nil,
	},

	{
		Name:              "c5n.4xlarge",
		MemoryGB:          42,
		ECU:               73,
		Cores:             16,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    nil,
	},

	{
		Name:              "c5n.9xlarge",
		MemoryGB:          96,
		ECU:               139,
		Cores:             36,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    nil,
	},

	{
		Name:              "c5n.18xlarge",
		MemoryGB:          192,
		ECU:               281,
		Cores:             72,
		InstanceENIs:      15,
		InstanceIPsPerENI: 50,
		EphemeralDisks:    nil,
	},

	{
		Name:              "c5n.metal",
		MemoryGB:          192,
		ECU:               0,
		Cores:             72,
		InstanceENIs:      15,
		InstanceIPsPerENI: 50,
		EphemeralDisks:    nil,
	},

	// cc2 family
	{
		Name:              "cc2.8xlarge",
		MemoryGB:          60.5,
		ECU:               88,
		Cores:             32,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    []int{840, 840, 840, 840},
	},

	// cr1 family
	{
		Name:              "cr1.8xlarge",
		MemoryGB:          244,
		ECU:               88,
		Cores:             32,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    []int{120, 120},
	},

	// d2 family
	{
		Name:              "d2.xlarge",
		MemoryGB:          30.5,
		ECU:               14,
		Cores:             4,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    []int{2000, 2000, 2000},
	},

	{
		Name:              "d2.2xlarge",
		MemoryGB:          61,
		ECU:               28,
		Cores:             8,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    []int{2000, 2000, 2000, 2000, 2000, 2000},
	},

	{
		Name:              "d2.4xlarge",
		MemoryGB:          122,
		ECU:               56,
		Cores:             16,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    []int{2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000},
	},

	{
		Name:              "d2.8xlarge",
		MemoryGB:          244,
		ECU:               116,
		Cores:             36,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    []int{2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000},
	},

	// f1 family
	{
		Name:              "f1.2xlarge",
		MemoryGB:          122,
		ECU:               31,
		Cores:             8,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    []int{470},
	},

	{
		Name:              "f1.4xlarge",
		MemoryGB:          244,
		ECU:               58,
		Cores:             16,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    []int{940},
	},

	{
		Name:              "f1.16xlarge",
		MemoryGB:          976,
		ECU:               201,
		Cores:             64,
		InstanceENIs:      8,
		InstanceIPsPerENI: 50,
		EphemeralDisks:    []int{940, 940, 940, 940},
	},

	// g2 family
	{
		Name:              "g2.2xlarge",
		MemoryGB:          15,
		ECU:               26,
		Cores:             8,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    []int{60},
		GPU:               true,
	},

	{
		Name:              "g2.8xlarge",
		MemoryGB:          60,
		ECU:               104,
		Cores:             32,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    []int{120, 120},
		GPU:               true,
	},

	// g3 family
	{
		Name:              "g3.4xlarge",
		MemoryGB:          122,
		ECU:               58,
		Cores:             16,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    nil,
		GPU:               true,
	},

	{
		Name:              "g3.8xlarge",
		MemoryGB:          244,
		ECU:               97,
		Cores:             32,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    nil,
		GPU:               true,
	},

	{
		Name:              "g3.16xlarge",
		MemoryGB:          488,
		ECU:               201,
		Cores:             64,
		InstanceENIs:      15,
		InstanceIPsPerENI: 50,
		EphemeralDisks:    nil,
		GPU:               true,
	},

	// g3s family
	{
		Name:              "g3s.xlarge",
		MemoryGB:          30.5,
		ECU:               13,
		Cores:             4,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    nil,
		GPU:               true,
	},

	// g4dn family
	{
		Name:              "g4dn.xlarge",
		MemoryGB:          16,
		ECU:               0,
		Cores:             4,
		InstanceENIs:      3,
		InstanceIPsPerENI: 10,
		EphemeralDisks:    []int{125},
		GPU:               true,
	},

	{
		Name:              "g4dn.2xlarge",
		MemoryGB:          32,
		ECU:               0,
		Cores:             8,
		InstanceENIs:      3,
		InstanceIPsPerENI: 10,
		EphemeralDisks:    []int{225},
		GPU:               true,
	},

	{
		Name:              "g4dn.4xlarge",
		MemoryGB:          64,
		ECU:               0,
		Cores:             16,
		InstanceENIs:      3,
		InstanceIPsPerENI: 10,
		EphemeralDisks:    []int{225},
		GPU:               true,
	},

	{
		Name:              "g4dn.8xlarge",
		MemoryGB:          128,
		ECU:               0,
		Cores:             32,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    []int{900},
		GPU:               true,
	},

	{
		Name:              "g4dn.12xlarge",
		MemoryGB:          192,
		ECU:               0,
		Cores:             48,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    []int{900},
		GPU:               true,
	},

	{
		Name:              "g4dn.16xlarge",
		MemoryGB:          256,
		ECU:               0,
		Cores:             64,
		InstanceENIs:      15,
		InstanceIPsPerENI: 50,
		EphemeralDisks:    []int{900},
		GPU:               true,
	},

	// h1 family
	{
		Name:              "h1.2xlarge",
		MemoryGB:          32,
		ECU:               31,
		Cores:             8,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    []int{2000},
	},

	{
		Name:              "h1.4xlarge",
		MemoryGB:          64,
		ECU:               58,
		Cores:             16,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    []int{2000, 2000},
	},

	{
		Name:              "h1.8xlarge",
		MemoryGB:          128,
		ECU:               97,
		Cores:             32,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    []int{2000, 2000, 2000, 2000},
	},

	{
		Name:              "h1.16xlarge",
		MemoryGB:          256,
		ECU:               201,
		Cores:             64,
		InstanceENIs:      15,
		InstanceIPsPerENI: 50,
		EphemeralDisks:    []int{2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000},
	},

	// hs1 family
	{
		Name:              "hs1.8xlarge",
		MemoryGB:          117,
		ECU:               35,
		Cores:             17,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    []int{2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000},
	},

	// i2 family
	{
		Name:              "i2.xlarge",
		MemoryGB:          30.5,
		ECU:               14,
		Cores:             4,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    []int{800},
	},

	{
		Name:              "i2.2xlarge",
		MemoryGB:          61,
		ECU:               27,
		Cores:             8,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    []int{800, 800},
	},

	{
		Name:              "i2.4xlarge",
		MemoryGB:          122,
		ECU:               53,
		Cores:             16,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    []int{800, 800, 800, 800},
	},

	{
		Name:              "i2.8xlarge",
		MemoryGB:          244,
		ECU:               104,
		Cores:             32,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    []int{800, 800, 800, 800, 800, 800, 800, 800},
	},

	// i3 family
	{
		Name:              "i3.large",
		MemoryGB:          15.25,
		ECU:               8,
		Cores:             2,
		InstanceENIs:      3,
		InstanceIPsPerENI: 10,
		EphemeralDisks:    []int{475},
	},

	{
		Name:              "i3.xlarge",
		MemoryGB:          30.5,
		ECU:               16,
		Cores:             4,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    []int{950},
	},

	{
		Name:              "i3.2xlarge",
		MemoryGB:          61,
		ECU:               31,
		Cores:             8,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    []int{1900},
	},

	{
		Name:              "i3.4xlarge",
		MemoryGB:          122,
		ECU:               53,
		Cores:             16,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    []int{1900, 1900},
	},

	{
		Name:              "i3.8xlarge",
		MemoryGB:          244,
		ECU:               97,
		Cores:             32,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    []int{1900, 1900, 1900, 1900},
	},

	{
		Name:              "i3.16xlarge",
		MemoryGB:          488,
		ECU:               201,
		Cores:             64,
		InstanceENIs:      15,
		InstanceIPsPerENI: 50,
		EphemeralDisks:    []int{1900, 1900, 1900, 1900, 1900, 1900, 1900, 1900},
	},

	{
		Name:              "i3.metal",
		MemoryGB:          512,
		ECU:               208,
		Cores:             64,
		InstanceENIs:      15,
		InstanceIPsPerENI: 50,
		EphemeralDisks:    []int{1900, 1900, 1900, 1900, 1900, 1900, 1900, 1900},
	},

	// i3en family
	{
		Name:              "i3en.large",
		MemoryGB:          16,
		ECU:               10,
		Cores:             2,
		InstanceENIs:      3,
		InstanceIPsPerENI: 10,
		EphemeralDisks:    []int{1250},
	},

	{
		Name:              "i3en.xlarge",
		MemoryGB:          32,
		ECU:               0,
		Cores:             4,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    []int{2500},
	},

	{
		Name:              "i3en.2xlarge",
		MemoryGB:          64,
		ECU:               37,
		Cores:             8,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    []int{2500, 2500},
	},

	{
		Name:              "i3en.3xlarge",
		MemoryGB:          96,
		ECU:               0,
		Cores:             12,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    []int{7500},
	},

	{
		Name:              "i3en.6xlarge",
		MemoryGB:          192,
		ECU:               0,
		Cores:             24,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    []int{7500, 7500},
	},

	{
		Name:              "i3en.12xlarge",
		MemoryGB:          384,
		ECU:               168,
		Cores:             48,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    []int{7500, 7500, 7500, 7500},
	},

	{
		Name:              "i3en.24xlarge",
		MemoryGB:          768,
		ECU:               337,
		Cores:             96,
		InstanceENIs:      15,
		InstanceIPsPerENI: 50,
		EphemeralDisks:    []int{7500, 7500, 7500, 7500, 7500, 7500, 7500, 7500},
	},

	{
		Name:              "i3en.metal",
		MemoryGB:          768,
		ECU:               0,
		Cores:             96,
		InstanceENIs:      15,
		InstanceIPsPerENI: 50,
		EphemeralDisks:    []int{7500, 7500, 7500, 7500, 7500, 7500, 7500, 7500},
	},

	// inf1 family
	{
		Name:              "inf1.xlarge",
		MemoryGB:          8,
		ECU:               0,
		Cores:             4,
		InstanceENIs:      4,
		InstanceIPsPerENI: 10,
		EphemeralDisks:    nil,
	},

	{
		Name:              "inf1.2xlarge",
		MemoryGB:          16,
		ECU:               0,
		Cores:             8,
		InstanceENIs:      4,
		InstanceIPsPerENI: 10,
		EphemeralDisks:    nil,
	},

	{
		Name:              "inf1.6xlarge",
		MemoryGB:          48,
		ECU:               0,
		Cores:             24,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    nil,
	},

	{
		Name:              "inf1.24xlarge",
		MemoryGB:          192,
		ECU:               0,
		Cores:             96,
		InstanceENIs:      15,
		InstanceIPsPerENI: 50,
		EphemeralDisks:    nil,
	},

	// m1 family
	{
		Name:              "m1.small",
		MemoryGB:          1.7,
		ECU:               1,
		Cores:             1,
		InstanceENIs:      2,
		InstanceIPsPerENI: 4,
		EphemeralDisks:    []int{160},
	},

	{
		Name:              "m1.medium",
		MemoryGB:          3.75,
		ECU:               2,
		Cores:             1,
		InstanceENIs:      2,
		InstanceIPsPerENI: 6,
		EphemeralDisks:    []int{410},
	},

	{
		Name:              "m1.large",
		MemoryGB:          7.5,
		ECU:               4,
		Cores:             2,
		InstanceENIs:      3,
		InstanceIPsPerENI: 10,
		EphemeralDisks:    []int{420, 420},
	},

	{
		Name:              "m1.xlarge",
		MemoryGB:          15,
		ECU:               8,
		Cores:             4,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    []int{420, 420, 420, 420},
	},

	// m2 family
	{
		Name:              "m2.xlarge",
		MemoryGB:          17.1,
		ECU:               6.5,
		Cores:             2,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    []int{420},
	},

	{
		Name:              "m2.2xlarge",
		MemoryGB:          34.2,
		ECU:               13,
		Cores:             4,
		InstanceENIs:      4,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    []int{850},
	},

	{
		Name:              "m2.4xlarge",
		MemoryGB:          68.4,
		ECU:               26,
		Cores:             8,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    []int{840, 840},
	},

	// m3 family
	{
		Name:              "m3.medium",
		MemoryGB:          3.75,
		ECU:               3,
		Cores:             1,
		InstanceENIs:      2,
		InstanceIPsPerENI: 6,
		EphemeralDisks:    []int{4},
	},

	{
		Name:              "m3.large",
		MemoryGB:          7.5,
		ECU:               6.5,
		Cores:             2,
		InstanceENIs:      3,
		InstanceIPsPerENI: 10,
		EphemeralDisks:    []int{32},
	},

	{
		Name:              "m3.xlarge",
		MemoryGB:          15,
		ECU:               13,
		Cores:             4,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    []int{40, 40},
	},

	{
		Name:              "m3.2xlarge",
		MemoryGB:          30,
		ECU:               26,
		Cores:             8,
		InstanceENIs:      4,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    []int{80, 80},
	},

	// m4 family
	{
		Name:              "m4.large",
		MemoryGB:          8,
		ECU:               6.5,
		Cores:             2,
		InstanceENIs:      2,
		InstanceIPsPerENI: 10,
		EphemeralDisks:    nil,
	},

	{
		Name:              "m4.xlarge",
		MemoryGB:          16,
		ECU:               13,
		Cores:             4,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    nil,
	},

	{
		Name:              "m4.2xlarge",
		MemoryGB:          32,
		ECU:               26,
		Cores:             8,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    nil,
	},

	{
		Name:              "m4.4xlarge",
		MemoryGB:          64,
		ECU:               53.5,
		Cores:             16,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    nil,
	},

	{
		Name:              "m4.10xlarge",
		MemoryGB:          160,
		ECU:               124.5,
		Cores:             40,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    nil,
	},

	{
		Name:              "m4.16xlarge",
		MemoryGB:          256,
		ECU:               188,
		Cores:             64,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    nil,
	},

	// m5 family
	{
		Name:              "m5.large",
		MemoryGB:          8,
		ECU:               8,
		Cores:             2,
		InstanceENIs:      3,
		InstanceIPsPerENI: 10,
		EphemeralDisks:    nil,
	},

	{
		Name:              "m5.xlarge",
		MemoryGB:          16,
		ECU:               16,
		Cores:             4,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    nil,
	},

	{
		Name:              "m5.2xlarge",
		MemoryGB:          32,
		ECU:               37,
		Cores:             8,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    nil,
	},

	{
		Name:              "m5.4xlarge",
		MemoryGB:          64,
		ECU:               70,
		Cores:             16,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    nil,
	},

	{
		Name:              "m5.8xlarge",
		MemoryGB:          128,
		ECU:               128,
		Cores:             32,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    nil,
	},

	{
		Name:              "m5.12xlarge",
		MemoryGB:          192,
		ECU:               168,
		Cores:             48,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    nil,
	},

	{
		Name:              "m5.16xlarge",
		MemoryGB:          256,
		ECU:               256,
		Cores:             64,
		InstanceENIs:      15,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    nil,
	},

	{
		Name:              "m5.24xlarge",
		MemoryGB:          384,
		ECU:               337,
		Cores:             96,
		InstanceENIs:      15,
		InstanceIPsPerENI: 50,
		EphemeralDisks:    nil,
	},

	{
		Name:              "m5.metal",
		MemoryGB:          384,
		ECU:               345,
		Cores:             96,
		InstanceENIs:      15,
		InstanceIPsPerENI: 50,
		EphemeralDisks:    nil,
	},

	// m5a family
	{
		Name:              "m5a.large",
		MemoryGB:          8,
		ECU:               0,
		Cores:             2,
		InstanceENIs:      3,
		InstanceIPsPerENI: 10,
		EphemeralDisks:    nil,
	},

	{
		Name:              "m5a.xlarge",
		MemoryGB:          16,
		ECU:               0,
		Cores:             4,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    nil,
	},

	{
		Name:              "m5a.2xlarge",
		MemoryGB:          32,
		ECU:               0,
		Cores:             8,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    nil,
	},

	{
		Name:              "m5a.4xlarge",
		MemoryGB:          64,
		ECU:               0,
		Cores:             16,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    nil,
	},

	{
		Name:              "m5a.8xlarge",
		MemoryGB:          128,
		ECU:               0,
		Cores:             32,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    nil,
	},

	{
		Name:              "m5a.12xlarge",
		MemoryGB:          192,
		ECU:               0,
		Cores:             48,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    nil,
	},

	{
		Name:              "m5a.16xlarge",
		MemoryGB:          256,
		ECU:               0,
		Cores:             64,
		InstanceENIs:      15,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    nil,
	},

	{
		Name:              "m5a.24xlarge",
		MemoryGB:          384,
		ECU:               0,
		Cores:             96,
		InstanceENIs:      15,
		InstanceIPsPerENI: 50,
		EphemeralDisks:    nil,
	},

	// m5ad family
	{
		Name:              "m5ad.large",
		MemoryGB:          8,
		ECU:               0,
		Cores:             2,
		InstanceENIs:      3,
		InstanceIPsPerENI: 10,
		EphemeralDisks:    []int{75},
	},

	{
		Name:              "m5ad.xlarge",
		MemoryGB:          16,
		ECU:               0,
		Cores:             4,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    []int{150},
	},

	{
		Name:              "m5ad.2xlarge",
		MemoryGB:          32,
		ECU:               0,
		Cores:             8,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    []int{300},
	},

	{
		Name:              "m5ad.4xlarge",
		MemoryGB:          64,
		ECU:               0,
		Cores:             16,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    []int{300, 300},
	},

	{
		Name:              "m5ad.12xlarge",
		MemoryGB:          192,
		ECU:               0,
		Cores:             48,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    []int{900, 900},
	},

	{
		Name:              "m5ad.24xlarge",
		MemoryGB:          384,
		ECU:               0,
		Cores:             96,
		InstanceENIs:      15,
		InstanceIPsPerENI: 50,
		EphemeralDisks:    []int{900, 900, 900, 900},
	},

	// m5d family
	{
		Name:              "m5d.large",
		MemoryGB:          8,
		ECU:               10,
		Cores:             2,
		InstanceENIs:      3,
		InstanceIPsPerENI: 10,
		EphemeralDisks:    []int{75},
	},

	{
		Name:              "m5d.xlarge",
		MemoryGB:          16,
		ECU:               16,
		Cores:             4,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    []int{150},
	},

	{
		Name:              "m5d.2xlarge",
		MemoryGB:          32,
		ECU:               37,
		Cores:             8,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    []int{300},
	},

	{
		Name:              "m5d.4xlarge",
		MemoryGB:          64,
		ECU:               70,
		Cores:             16,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    []int{300, 300},
	},

	{
		Name:              "m5d.8xlarge",
		MemoryGB:          128,
		ECU:               128,
		Cores:             32,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    []int{600, 600},
	},

	{
		Name:              "m5d.12xlarge",
		MemoryGB:          192,
		ECU:               168,
		Cores:             48,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    []int{900, 900},
	},

	{
		Name:              "m5d.16xlarge",
		MemoryGB:          256,
		ECU:               256,
		Cores:             64,
		InstanceENIs:      15,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    []int{600, 600, 600, 600},
	},

	{
		Name:              "m5d.24xlarge",
		MemoryGB:          384,
		ECU:               337,
		Cores:             96,
		InstanceENIs:      15,
		InstanceIPsPerENI: 50,
		EphemeralDisks:    []int{900, 900, 900, 900},
	},

	{
		Name:              "m5d.metal",
		MemoryGB:          384,
		ECU:               345,
		Cores:             96,
		InstanceENIs:      15,
		InstanceIPsPerENI: 50,
		EphemeralDisks:    []int{900, 900, 900, 900},
	},

	// m5dn family
	{
		Name:              "m5dn.large",
		MemoryGB:          8,
		ECU:               0,
		Cores:             2,
		InstanceENIs:      3,
		InstanceIPsPerENI: 10,
		EphemeralDisks:    []int{75},
	},

	{
		Name:              "m5dn.xlarge",
		MemoryGB:          16,
		ECU:               0,
		Cores:             4,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    []int{150},
	},

	{
		Name:              "m5dn.2xlarge",
		MemoryGB:          32,
		ECU:               0,
		Cores:             8,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    []int{300},
	},

	{
		Name:              "m5dn.4xlarge",
		MemoryGB:          64,
		ECU:               0,
		Cores:             16,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    []int{300, 300},
	},

	{
		Name:              "m5dn.8xlarge",
		MemoryGB:          128,
		ECU:               0,
		Cores:             32,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    []int{600, 600},
	},

	{
		Name:              "m5dn.12xlarge",
		MemoryGB:          192,
		ECU:               0,
		Cores:             48,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    []int{900, 900},
	},

	{
		Name:              "m5dn.16xlarge",
		MemoryGB:          256,
		ECU:               0,
		Cores:             64,
		InstanceENIs:      15,
		InstanceIPsPerENI: 50,
		EphemeralDisks:    []int{600, 600, 600, 600},
	},

	{
		Name:              "m5dn.24xlarge",
		MemoryGB:          384,
		ECU:               0,
		Cores:             96,
		InstanceENIs:      15,
		InstanceIPsPerENI: 50,
		EphemeralDisks:    []int{900, 900, 900, 900},
	},

	// m5n family
	{
		Name:              "m5n.large",
		MemoryGB:          8,
		ECU:               0,
		Cores:             2,
		InstanceENIs:      3,
		InstanceIPsPerENI: 10,
		EphemeralDisks:    nil,
	},

	{
		Name:              "m5n.xlarge",
		MemoryGB:          16,
		ECU:               0,
		Cores:             4,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    nil,
	},

	{
		Name:              "m5n.2xlarge",
		MemoryGB:          32,
		ECU:               0,
		Cores:             8,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    nil,
	},

	{
		Name:              "m5n.4xlarge",
		MemoryGB:          64,
		ECU:               0,
		Cores:             16,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    nil,
	},

	{
		Name:              "m5n.8xlarge",
		MemoryGB:          128,
		ECU:               0,
		Cores:             32,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    nil,
	},

	{
		Name:              "m5n.12xlarge",
		MemoryGB:          192,
		ECU:               0,
		Cores:             48,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    nil,
	},

	{
		Name:              "m5n.16xlarge",
		MemoryGB:          256,
		ECU:               0,
		Cores:             64,
		InstanceENIs:      15,
		InstanceIPsPerENI: 50,
		EphemeralDisks:    nil,
	},

	{
		Name:              "m5n.24xlarge",
		MemoryGB:          384,
		ECU:               0,
		Cores:             96,
		InstanceENIs:      15,
		InstanceIPsPerENI: 50,
		EphemeralDisks:    nil,
	},

	// m6g family
	{
		Name:              "m6g.medium",
		MemoryGB:          4,
		ECU:               0,
		Cores:             1,
		InstanceENIs:      0,
		InstanceIPsPerENI: 0,
		EphemeralDisks:    nil,
	},

	{
		Name:              "m6g.large",
		MemoryGB:          8,
		ECU:               0,
		Cores:             2,
		InstanceENIs:      0,
		InstanceIPsPerENI: 0,
		EphemeralDisks:    nil,
	},

	{
		Name:              "m6g.xlarge",
		MemoryGB:          16,
		ECU:               0,
		Cores:             4,
		InstanceENIs:      0,
		InstanceIPsPerENI: 0,
		EphemeralDisks:    nil,
	},

	{
		Name:              "m6g.2xlarge",
		MemoryGB:          32,
		ECU:               0,
		Cores:             8,
		InstanceENIs:      0,
		InstanceIPsPerENI: 0,
		EphemeralDisks:    nil,
	},

	{
		Name:              "m6g.4xlarge",
		MemoryGB:          64,
		ECU:               0,
		Cores:             16,
		InstanceENIs:      0,
		InstanceIPsPerENI: 0,
		EphemeralDisks:    nil,
	},

	{
		Name:              "m6g.8xlarge",
		MemoryGB:          128,
		ECU:               0,
		Cores:             32,
		InstanceENIs:      0,
		InstanceIPsPerENI: 0,
		EphemeralDisks:    nil,
	},

	{
		Name:              "m6g.12xlarge",
		MemoryGB:          192,
		ECU:               0,
		Cores:             48,
		InstanceENIs:      0,
		InstanceIPsPerENI: 0,
		EphemeralDisks:    nil,
	},

	{
		Name:              "m6g.16xlarge",
		MemoryGB:          256,
		ECU:               0,
		Cores:             64,
		InstanceENIs:      0,
		InstanceIPsPerENI: 0,
		EphemeralDisks:    nil,
	},

	// p2 family
	{
		Name:              "p2.xlarge",
		MemoryGB:          61,
		ECU:               16,
		Cores:             4,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    nil,
		GPU:               true,
	},

	{
		Name:              "p2.8xlarge",
		MemoryGB:          488,
		ECU:               97,
		Cores:             32,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    nil,
		GPU:               true,
	},

	{
		Name:              "p2.16xlarge",
		MemoryGB:          732,
		ECU:               201,
		Cores:             64,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    nil,
		GPU:               true,
	},

	// p3 family
	{
		Name:              "p3.2xlarge",
		MemoryGB:          61,
		ECU:               31,
		Cores:             8,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    nil,
		GPU:               true,
	},

	{
		Name:              "p3.8xlarge",
		MemoryGB:          244,
		ECU:               97,
		Cores:             32,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    nil,
		GPU:               true,
	},

	{
		Name:              "p3.16xlarge",
		MemoryGB:          488,
		ECU:               201,
		Cores:             64,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    nil,
		GPU:               true,
	},

	// p3dn family
	{
		Name:              "p3dn.24xlarge",
		MemoryGB:          768,
		ECU:               337,
		Cores:             96,
		InstanceENIs:      15,
		InstanceIPsPerENI: 50,
		EphemeralDisks:    []int{900, 900},
		GPU:               true,
	},

	// r3 family
	{
		Name:              "r3.large",
		MemoryGB:          15.25,
		ECU:               6.5,
		Cores:             2,
		InstanceENIs:      3,
		InstanceIPsPerENI: 10,
		EphemeralDisks:    []int{32},
	},

	{
		Name:              "r3.xlarge",
		MemoryGB:          30.5,
		ECU:               13,
		Cores:             4,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    []int{80},
	},

	{
		Name:              "r3.2xlarge",
		MemoryGB:          61,
		ECU:               26,
		Cores:             8,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    []int{160},
	},

	{
		Name:              "r3.4xlarge",
		MemoryGB:          122,
		ECU:               52,
		Cores:             16,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    []int{320},
	},

	{
		Name:              "r3.8xlarge",
		MemoryGB:          244,
		ECU:               104,
		Cores:             32,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    []int{320, 320},
	},

	// r4 family
	{
		Name:              "r4.large",
		MemoryGB:          15.25,
		ECU:               7,
		Cores:             2,
		InstanceENIs:      3,
		InstanceIPsPerENI: 10,
		EphemeralDisks:    nil,
	},

	{
		Name:              "r4.xlarge",
		MemoryGB:          30.5,
		ECU:               16,
		Cores:             4,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    nil,
	},

	{
		Name:              "r4.2xlarge",
		MemoryGB:          61,
		ECU:               31,
		Cores:             8,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    nil,
	},

	{
		Name:              "r4.4xlarge",
		MemoryGB:          122,
		ECU:               58,
		Cores:             16,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    nil,
	},

	{
		Name:              "r4.8xlarge",
		MemoryGB:          244,
		ECU:               97,
		Cores:             32,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    nil,
	},

	{
		Name:              "r4.16xlarge",
		MemoryGB:          488,
		ECU:               195,
		Cores:             64,
		InstanceENIs:      15,
		InstanceIPsPerENI: 50,
		EphemeralDisks:    nil,
	},

	// r5 family
	{
		Name:              "r5.large",
		MemoryGB:          16,
		ECU:               10,
		Cores:             2,
		InstanceENIs:      3,
		InstanceIPsPerENI: 10,
		EphemeralDisks:    nil,
	},

	{
		Name:              "r5.xlarge",
		MemoryGB:          32,
		ECU:               19,
		Cores:             4,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    nil,
	},

	{
		Name:              "r5.2xlarge",
		MemoryGB:          64,
		ECU:               37,
		Cores:             8,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    nil,
	},

	{
		Name:              "r5.4xlarge",
		MemoryGB:          128,
		ECU:               70,
		Cores:             16,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    nil,
	},

	{
		Name:              "r5.8xlarge",
		MemoryGB:          256,
		ECU:               128,
		Cores:             32,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    nil,
	},

	{
		Name:              "r5.12xlarge",
		MemoryGB:          384,
		ECU:               168,
		Cores:             48,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    nil,
	},

	{
		Name:              "r5.16xlarge",
		MemoryGB:          512,
		ECU:               256,
		Cores:             64,
		InstanceENIs:      15,
		InstanceIPsPerENI: 50,
		EphemeralDisks:    nil,
	},

	{
		Name:              "r5.24xlarge",
		MemoryGB:          768,
		ECU:               337,
		Cores:             96,
		InstanceENIs:      15,
		InstanceIPsPerENI: 50,
		EphemeralDisks:    nil,
	},

	{
		Name:              "r5.metal",
		MemoryGB:          768,
		ECU:               347,
		Cores:             96,
		InstanceENIs:      15,
		InstanceIPsPerENI: 50,
		EphemeralDisks:    nil,
	},

	// r5a family
	{
		Name:              "r5a.large",
		MemoryGB:          16,
		ECU:               0,
		Cores:             2,
		InstanceENIs:      3,
		InstanceIPsPerENI: 10,
		EphemeralDisks:    nil,
	},

	{
		Name:              "r5a.xlarge",
		MemoryGB:          32,
		ECU:               0,
		Cores:             4,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    nil,
	},

	{
		Name:              "r5a.2xlarge",
		MemoryGB:          64,
		ECU:               0,
		Cores:             8,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    nil,
	},

	{
		Name:              "r5a.4xlarge",
		MemoryGB:          128,
		ECU:               0,
		Cores:             16,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    nil,
	},

	{
		Name:              "r5a.8xlarge",
		MemoryGB:          256,
		ECU:               0,
		Cores:             32,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    nil,
	},

	{
		Name:              "r5a.12xlarge",
		MemoryGB:          384,
		ECU:               0,
		Cores:             48,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    nil,
	},

	{
		Name:              "r5a.16xlarge",
		MemoryGB:          512,
		ECU:               0,
		Cores:             64,
		InstanceENIs:      15,
		InstanceIPsPerENI: 50,
		EphemeralDisks:    nil,
	},

	{
		Name:              "r5a.24xlarge",
		MemoryGB:          768,
		ECU:               0,
		Cores:             96,
		InstanceENIs:      15,
		InstanceIPsPerENI: 50,
		EphemeralDisks:    nil,
	},

	// r5ad family
	{
		Name:              "r5ad.large",
		MemoryGB:          16,
		ECU:               0,
		Cores:             2,
		InstanceENIs:      3,
		InstanceIPsPerENI: 10,
		EphemeralDisks:    []int{75},
	},

	{
		Name:              "r5ad.xlarge",
		MemoryGB:          32,
		ECU:               0,
		Cores:             4,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    []int{150},
	},

	{
		Name:              "r5ad.2xlarge",
		MemoryGB:          64,
		ECU:               0,
		Cores:             8,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    []int{300},
	},

	{
		Name:              "r5ad.4xlarge",
		MemoryGB:          128,
		ECU:               0,
		Cores:             16,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    []int{300, 300},
	},

	{
		Name:              "r5ad.12xlarge",
		MemoryGB:          384,
		ECU:               0,
		Cores:             48,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    []int{900, 900},
	},

	{
		Name:              "r5ad.24xlarge",
		MemoryGB:          768,
		ECU:               0,
		Cores:             96,
		InstanceENIs:      15,
		InstanceIPsPerENI: 50,
		EphemeralDisks:    []int{900, 900, 900, 900},
	},

	// r5d family
	{
		Name:              "r5d.large",
		MemoryGB:          16,
		ECU:               10,
		Cores:             2,
		InstanceENIs:      3,
		InstanceIPsPerENI: 10,
		EphemeralDisks:    []int{75},
	},

	{
		Name:              "r5d.xlarge",
		MemoryGB:          32,
		ECU:               19,
		Cores:             4,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    []int{150},
	},

	{
		Name:              "r5d.2xlarge",
		MemoryGB:          64,
		ECU:               37,
		Cores:             8,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    []int{300},
	},

	{
		Name:              "r5d.4xlarge",
		MemoryGB:          128,
		ECU:               70,
		Cores:             16,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    []int{300, 300},
	},

	{
		Name:              "r5d.8xlarge",
		MemoryGB:          256,
		ECU:               128,
		Cores:             32,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    []int{600, 600},
	},

	{
		Name:              "r5d.12xlarge",
		MemoryGB:          384,
		ECU:               168,
		Cores:             48,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    []int{900, 900},
	},

	{
		Name:              "r5d.16xlarge",
		MemoryGB:          512,
		ECU:               256,
		Cores:             64,
		InstanceENIs:      15,
		InstanceIPsPerENI: 50,
		EphemeralDisks:    []int{600, 600, 600, 600},
	},

	{
		Name:              "r5d.24xlarge",
		MemoryGB:          768,
		ECU:               337,
		Cores:             96,
		InstanceENIs:      15,
		InstanceIPsPerENI: 50,
		EphemeralDisks:    []int{900, 900, 900, 900},
	},

	{
		Name:              "r5d.metal",
		MemoryGB:          768,
		ECU:               347,
		Cores:             96,
		InstanceENIs:      15,
		InstanceIPsPerENI: 50,
		EphemeralDisks:    []int{900, 900, 900, 900},
	},

	// r5dn family
	{
		Name:              "r5dn.large",
		MemoryGB:          16,
		ECU:               0,
		Cores:             2,
		InstanceENIs:      3,
		InstanceIPsPerENI: 10,
		EphemeralDisks:    []int{75},
	},

	{
		Name:              "r5dn.xlarge",
		MemoryGB:          32,
		ECU:               0,
		Cores:             4,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    []int{150},
	},

	{
		Name:              "r5dn.2xlarge",
		MemoryGB:          64,
		ECU:               0,
		Cores:             8,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    []int{300},
	},

	{
		Name:              "r5dn.4xlarge",
		MemoryGB:          128,
		ECU:               0,
		Cores:             16,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    []int{300, 300},
	},

	{
		Name:              "r5dn.8xlarge",
		MemoryGB:          256,
		ECU:               0,
		Cores:             32,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    []int{600, 600},
	},

	{
		Name:              "r5dn.12xlarge",
		MemoryGB:          384,
		ECU:               0,
		Cores:             48,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    []int{900, 900},
	},

	{
		Name:              "r5dn.16xlarge",
		MemoryGB:          512,
		ECU:               0,
		Cores:             64,
		InstanceENIs:      15,
		InstanceIPsPerENI: 50,
		EphemeralDisks:    []int{600, 600, 600, 600},
	},

	{
		Name:              "r5dn.24xlarge",
		MemoryGB:          768,
		ECU:               0,
		Cores:             96,
		InstanceENIs:      15,
		InstanceIPsPerENI: 50,
		EphemeralDisks:    []int{900, 900, 900, 900},
	},

	// r5n family
	{
		Name:              "r5n.large",
		MemoryGB:          16,
		ECU:               0,
		Cores:             2,
		InstanceENIs:      3,
		InstanceIPsPerENI: 10,
		EphemeralDisks:    nil,
	},

	{
		Name:              "r5n.xlarge",
		MemoryGB:          32,
		ECU:               0,
		Cores:             4,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    nil,
	},

	{
		Name:              "r5n.2xlarge",
		MemoryGB:          64,
		ECU:               0,
		Cores:             8,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    nil,
	},

	{
		Name:              "r5n.4xlarge",
		MemoryGB:          128,
		ECU:               0,
		Cores:             16,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    nil,
	},

	{
		Name:              "r5n.8xlarge",
		MemoryGB:          256,
		ECU:               0,
		Cores:             32,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    nil,
	},

	{
		Name:              "r5n.12xlarge",
		MemoryGB:          384,
		ECU:               0,
		Cores:             48,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    nil,
	},

	{
		Name:              "r5n.16xlarge",
		MemoryGB:          512,
		ECU:               0,
		Cores:             64,
		InstanceENIs:      15,
		InstanceIPsPerENI: 50,
		EphemeralDisks:    nil,
	},

	{
		Name:              "r5n.24xlarge",
		MemoryGB:          768,
		ECU:               0,
		Cores:             96,
		InstanceENIs:      15,
		InstanceIPsPerENI: 50,
		EphemeralDisks:    nil,
	},

	// t1 family
	{
		Name:              "t1.micro",
		MemoryGB:          0.613,
		ECU:               1 * BurstableCreditsToECUS,
		Cores:             1,
		InstanceENIs:      2,
		InstanceIPsPerENI: 2,
		EphemeralDisks:    nil,
		Burstable:         true,
	},

	// t2 family
	{
		Name:              "t2.nano",
		MemoryGB:          0.5,
		ECU:               3 * BurstableCreditsToECUS,
		Cores:             1,
		InstanceENIs:      2,
		InstanceIPsPerENI: 2,
		EphemeralDisks:    nil,
		Burstable:         true,
	},

	{
		Name:              "t2.micro",
		MemoryGB:          1,
		ECU:               6 * BurstableCreditsToECUS,
		Cores:             1,
		InstanceENIs:      2,
		InstanceIPsPerENI: 2,
		EphemeralDisks:    nil,
		Burstable:         true,
	},

	{
		Name:              "t2.small",
		MemoryGB:          2,
		ECU:               12 * BurstableCreditsToECUS,
		Cores:             1,
		InstanceENIs:      3,
		InstanceIPsPerENI: 4,
		EphemeralDisks:    nil,
		Burstable:         true,
	},

	{
		Name:              "t2.medium",
		MemoryGB:          4,
		ECU:               24 * BurstableCreditsToECUS,
		Cores:             2,
		InstanceENIs:      3,
		InstanceIPsPerENI: 6,
		EphemeralDisks:    nil,
		Burstable:         true,
	},

	{
		Name:              "t2.large",
		MemoryGB:          8,
		ECU:               36 * BurstableCreditsToECUS,
		Cores:             2,
		InstanceENIs:      3,
		InstanceIPsPerENI: 12,
		EphemeralDisks:    nil,
		Burstable:         true,
	},

	{
		Name:              "t2.xlarge",
		MemoryGB:          16,
		ECU:               54 * BurstableCreditsToECUS,
		Cores:             4,
		InstanceENIs:      3,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    nil,
		Burstable:         true,
	},

	{
		Name:              "t2.2xlarge",
		MemoryGB:          32,
		ECU:               81.6 * BurstableCreditsToECUS,
		Cores:             8,
		InstanceENIs:      3,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    nil,
		Burstable:         true,
	},

	// t3 family
	{
		Name:              "t3.nano",
		MemoryGB:          0.5,
		ECU:               6 * BurstableCreditsToECUS,
		Cores:             2,
		InstanceENIs:      2,
		InstanceIPsPerENI: 2,
		EphemeralDisks:    nil,
		Burstable:         true,
	},

	{
		Name:              "t3.micro",
		MemoryGB:          1,
		ECU:               12 * BurstableCreditsToECUS,
		Cores:             2,
		InstanceENIs:      2,
		InstanceIPsPerENI: 2,
		EphemeralDisks:    nil,
		Burstable:         true,
	},

	{
		Name:              "t3.small",
		MemoryGB:          2,
		ECU:               24 * BurstableCreditsToECUS,
		Cores:             2,
		InstanceENIs:      3,
		InstanceIPsPerENI: 4,
		EphemeralDisks:    nil,
		Burstable:         true,
	},

	{
		Name:              "t3.medium",
		MemoryGB:          4,
		ECU:               24 * BurstableCreditsToECUS,
		Cores:             2,
		InstanceENIs:      3,
		InstanceIPsPerENI: 6,
		EphemeralDisks:    nil,
		Burstable:         true,
	},

	{
		Name:              "t3.large",
		MemoryGB:          8,
		ECU:               36 * BurstableCreditsToECUS,
		Cores:             2,
		InstanceENIs:      3,
		InstanceIPsPerENI: 12,
		EphemeralDisks:    nil,
		Burstable:         true,
	},

	{
		Name:              "t3.xlarge",
		MemoryGB:          16,
		ECU:               96 * BurstableCreditsToECUS,
		Cores:             4,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    nil,
		Burstable:         true,
	},

	{
		Name:              "t3.2xlarge",
		MemoryGB:          32,
		ECU:               192 * BurstableCreditsToECUS,
		Cores:             8,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    nil,
		Burstable:         true,
	},

	// t3a family
	{
		Name:              "t3a.nano",
		MemoryGB:          0.5,
		ECU:               0 * BurstableCreditsToECUS,
		Cores:             2,
		InstanceENIs:      2,
		InstanceIPsPerENI: 2,
		EphemeralDisks:    nil,
		Burstable:         true,
	},

	{
		Name:              "t3a.micro",
		MemoryGB:          1,
		ECU:               0 * BurstableCreditsToECUS,
		Cores:             2,
		InstanceENIs:      2,
		InstanceIPsPerENI: 2,
		EphemeralDisks:    nil,
		Burstable:         true,
	},

	{
		Name:              "t3a.small",
		MemoryGB:          2,
		ECU:               0 * BurstableCreditsToECUS,
		Cores:             2,
		InstanceENIs:      3,
		InstanceIPsPerENI: 4,
		EphemeralDisks:    nil,
		Burstable:         true,
	},

	{
		Name:              "t3a.medium",
		MemoryGB:          4,
		ECU:               0 * BurstableCreditsToECUS,
		Cores:             2,
		InstanceENIs:      3,
		InstanceIPsPerENI: 6,
		EphemeralDisks:    nil,
		Burstable:         true,
	},

	{
		Name:              "t3a.large",
		MemoryGB:          8,
		ECU:               0 * BurstableCreditsToECUS,
		Cores:             2,
		InstanceENIs:      3,
		InstanceIPsPerENI: 12,
		EphemeralDisks:    nil,
		Burstable:         true,
	},

	{
		Name:              "t3a.xlarge",
		MemoryGB:          16,
		ECU:               0 * BurstableCreditsToECUS,
		Cores:             4,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    nil,
		Burstable:         true,
	},

	{
		Name:              "t3a.2xlarge",
		MemoryGB:          32,
		ECU:               0 * BurstableCreditsToECUS,
		Cores:             8,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    nil,
		Burstable:         true,
	},

	// x1 family
	{
		Name:              "x1.16xlarge",
		MemoryGB:          976,
		ECU:               174.5,
		Cores:             64,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    []int{1920},
	},

	{
		Name:              "x1.32xlarge",
		MemoryGB:          1952,
		ECU:               349,
		Cores:             128,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    []int{1920, 1920},
	},

	// x1e family
	{
		Name:              "x1e.xlarge",
		MemoryGB:          122,
		ECU:               12,
		Cores:             4,
		InstanceENIs:      3,
		InstanceIPsPerENI: 10,
		EphemeralDisks:    []int{120},
	},

	{
		Name:              "x1e.2xlarge",
		MemoryGB:          244,
		ECU:               23,
		Cores:             8,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    []int{240},
	},

	{
		Name:              "x1e.4xlarge",
		MemoryGB:          488,
		ECU:               47,
		Cores:             16,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    []int{480},
	},

	{
		Name:              "x1e.8xlarge",
		MemoryGB:          976,
		ECU:               91,
		Cores:             32,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    []int{960},
	},

	{
		Name:              "x1e.16xlarge",
		MemoryGB:          1952,
		ECU:               179,
		Cores:             64,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    []int{1920},
	},

	{
		Name:              "x1e.32xlarge",
		MemoryGB:          3904,
		ECU:               340,
		Cores:             128,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    []int{1920, 1920},
	},

	// z1d family
	{
		Name:              "z1d.large",
		MemoryGB:          16,
		ECU:               12,
		Cores:             2,
		InstanceENIs:      3,
		InstanceIPsPerENI: 10,
		EphemeralDisks:    []int{75},
	},

	{
		Name:              "z1d.xlarge",
		MemoryGB:          32,
		ECU:               23,
		Cores:             4,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    []int{150},
	},

	{
		Name:              "z1d.2xlarge",
		MemoryGB:          64,
		ECU:               45,
		Cores:             8,
		InstanceENIs:      4,
		InstanceIPsPerENI: 15,
		EphemeralDisks:    []int{300},
	},

	{
		Name:              "z1d.3xlarge",
		MemoryGB:          96,
		ECU:               75,
		Cores:             12,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    []int{450},
	},

	{
		Name:              "z1d.6xlarge",
		MemoryGB:          192,
		ECU:               116,
		Cores:             24,
		InstanceENIs:      8,
		InstanceIPsPerENI: 30,
		EphemeralDisks:    []int{900},
	},

	{
		Name:              "z1d.12xlarge",
		MemoryGB:          384,
		ECU:               235,
		Cores:             48,
		InstanceENIs:      15,
		InstanceIPsPerENI: 50,
		EphemeralDisks:    []int{900, 900},
	},

	{
		Name:              "z1d.metal",
		MemoryGB:          384,
		ECU:               271,
		Cores:             48,
		InstanceENIs:      15,
		InstanceIPsPerENI: 50,
		EphemeralDisks:    []int{900, 900},
	},

	// END GENERATED CONTENT
}
