#include "common/buffer/buffer_impl.h"
#include "common/mongo/bson_impl.h"
#include "common/mongo/codec_impl.h"

using testing::Eq;
using testing::NiceMock;
using testing::Pointee;

namespace Mongo {

class TestDecoderCallbacks : public DecoderCallbacks {
public:
  void decodeBase64(std::string&& message) override { decodeBase64_(message); }
  void decodeGetMore(GetMoreMessagePtr&& message) override { decodeGetMore_(message); }
  void decodeInsert(InsertMessagePtr&& message) override { decodeInsert_(message); }
  void decodeKillCursors(KillCursorsMessagePtr&& message) override { decodeKillCursors_(message); }
  void decodeQuery(QueryMessagePtr&& message) override { decodeQuery_(message); }
  void decodeReply(ReplyMessagePtr&& message) override { decodeReply_(message); }

  MOCK_METHOD1(decodeBase64_, void(std::string& message));
  MOCK_METHOD1(decodeGetMore_, void(GetMoreMessagePtr& message));
  MOCK_METHOD1(decodeInsert_, void(InsertMessagePtr& message));
  MOCK_METHOD1(decodeKillCursors_, void(KillCursorsMessagePtr& message));
  MOCK_METHOD1(decodeQuery_, void(QueryMessagePtr& message));
  MOCK_METHOD1(decodeReply_, void(ReplyMessagePtr& message));
};

class MongoCodecImplTest : public testing::Test {
public:
  Buffer::OwnedImpl output_;
  EncoderImpl encoder_{output_};
  NiceMock<TestDecoderCallbacks> callbacks_;
  DecoderImpl decoder_{callbacks_};
};

TEST_F(MongoCodecImplTest, QueryEqual) {
  {
    QueryMessageImpl q1(0, 0);
    QueryMessageImpl q2(1, 1);
    EXPECT_FALSE(q1 == q2);
  }

  {
    QueryMessageImpl q1(0, 0);
    q1.fullCollectionName("hello");
    QueryMessageImpl q2(0, 0);
    q2.fullCollectionName("world");
    EXPECT_FALSE(q1 == q2);
  }

  {
    QueryMessageImpl q1(0, 0);
    q1.query(Bson::DocumentImpl::create()->addString("hello", "world"));
    QueryMessageImpl q2(0, 0);
    q2.query(Bson::DocumentImpl::create()->addString("world", "hello"));
    EXPECT_FALSE(q1 == q2);
  }

  {
    QueryMessageImpl q1(0, 0);
    q1.returnFieldsSelector(Bson::DocumentImpl::create()->addString("hello", "world"));
    QueryMessageImpl q2(0, 0);
    q2.returnFieldsSelector(Bson::DocumentImpl::create()->addString("world", "hello"));
    EXPECT_FALSE(q1 == q2);
  }
}

TEST_F(MongoCodecImplTest, Query) {
  QueryMessageImpl query(1, 1);
  query.flags(0x4);
  query.fullCollectionName("test");
  query.numberToSkip(20);
  query.numberToReturn(-1);
  query.query(Bson::DocumentImpl::create()
                  ->addDouble("double", 2.1)
                  ->addString("string", "string_value")
                  ->addDocument("document", Bson::DocumentImpl::create())
                  ->addArray("array", Bson::DocumentImpl::create())
                  ->addBinary("binary", "binary_value")
                  ->addObjectId("object_id", Bson::Field::ObjectId())
                  ->addBoolean("true", true)
                  ->addBoolean("false", false)
                  ->addDatetime("datetime", 1)
                  ->addNull("null")
                  ->addRegex("regex", {"hello", ""})
                  ->addInt32("int32", 1)
                  ->addTimestamp("timestamp", 1000)
                  ->addInt64("int64", 2));
  encoder_.encodeQuery(query);

  QueryMessageImpl query2(2, 2);
  query2.fullCollectionName("test2");
  query2.query(Bson::DocumentImpl::create()->addString("string2", "string2_value"));
  query2.returnFieldsSelector(Bson::DocumentImpl::create()->addDouble("double2", -2.3));
  encoder_.encodeQuery(query2);

  EXPECT_CALL(callbacks_, decodeQuery_(Pointee(Eq(query))));
  EXPECT_CALL(callbacks_, decodeQuery_(Pointee(Eq(query2))));
  decoder_.onData(output_);
}

TEST_F(MongoCodecImplTest, ReplyEqual) {
  {
    ReplyMessageImpl r1(0, 0);
    ReplyMessageImpl r2(1, 1);
    EXPECT_FALSE(r1 == r2);
  }

  {
    ReplyMessageImpl r1(0, 0);
    r1.cursorId(1);
    ReplyMessageImpl r2(0, 0);
    r2.cursorId(2);
    EXPECT_FALSE(r1 == r2);
  }

  {
    ReplyMessageImpl r1(0, 0);
    r1.numberReturned(1);
    r1.documents().push_back(Bson::DocumentImpl::create()->addString("hello", "world"));
    ReplyMessageImpl r2(0, 0);
    r2.numberReturned(1);
    r2.documents().push_back(Bson::DocumentImpl::create()->addString("world", "hello"));
    EXPECT_FALSE(r1 == r2);
  }
}

TEST_F(MongoCodecImplTest, Reply) {
  ReplyMessageImpl reply(2, 2);
  reply.flags(0x8);
  reply.cursorId(20000);
  reply.startingFrom(20);
  reply.numberReturned(2);
  reply.documents().push_back(Bson::DocumentImpl::create());
  reply.documents().push_back(Bson::DocumentImpl::create());
  encoder_.encodeReply(reply);

  EXPECT_CALL(callbacks_, decodeReply_(Pointee(Eq(reply))));
  decoder_.onData(output_);
}

TEST_F(MongoCodecImplTest, GetMoreEqual) {
  {
    GetMoreMessageImpl g1(0, 0);
    GetMoreMessageImpl g2(1, 1);
    EXPECT_FALSE(g1 == g2);
  }

  {
    GetMoreMessageImpl g1(0, 0);
    g1.cursorId(1);
    GetMoreMessageImpl g2(0, 0);
    g1.cursorId(2);
    EXPECT_FALSE(g1 == g2);
  }
}

TEST_F(MongoCodecImplTest, GetMore) {
  GetMoreMessageImpl get_more(3, 3);
  get_more.fullCollectionName("test");
  get_more.numberToReturn(20);
  get_more.cursorId(20000);
  encoder_.encodeGetMore(get_more);

  EXPECT_CALL(callbacks_, decodeGetMore_(Pointee(Eq(get_more))));
  decoder_.onData(output_);
}

TEST_F(MongoCodecImplTest, InsertEqual) {
  {
    InsertMessageImpl i1(0, 0);
    InsertMessageImpl i2(1, 1);
    EXPECT_FALSE(i1 == i2);
  }

  {
    InsertMessageImpl i1(0, 0);
    i1.fullCollectionName("hello");
    InsertMessageImpl i2(0, 0);
    i2.fullCollectionName("world");
    EXPECT_FALSE(i1 == i2);
  }

  {
    InsertMessageImpl i1(0, 0);
    i1.fullCollectionName("hello");
    i1.documents().push_back(Bson::DocumentImpl::create()->addString("hello", "world"));
    InsertMessageImpl i2(0, 0);
    i2.fullCollectionName("hello");
    i2.documents().push_back(Bson::DocumentImpl::create()->addString("world", "hello"));
    EXPECT_FALSE(i1 == i2);
  }
}

TEST_F(MongoCodecImplTest, Insert) {
  InsertMessageImpl insert(4, 4);
  insert.flags(0x8);
  insert.fullCollectionName("test");
  insert.documents().push_back(Bson::DocumentImpl::create());
  insert.documents().push_back(Bson::DocumentImpl::create());
  encoder_.encodeInsert(insert);

  EXPECT_CALL(callbacks_, decodeInsert_(Pointee(Eq(insert))));
  decoder_.onData(output_);
}

TEST_F(MongoCodecImplTest, KillCursorsEqual) {
  {
    KillCursorsMessageImpl k1(0, 0);
    KillCursorsMessageImpl k2(1, 1);
    EXPECT_FALSE(k1 == k2);
  }

  {
    KillCursorsMessageImpl k1(0, 0);
    k1.numberOfCursorIds(1);
    KillCursorsMessageImpl k2(0, 0);
    k2.numberOfCursorIds(2);
    EXPECT_FALSE(k1 == k2);
  }

  {
    KillCursorsMessageImpl k1(0, 0);
    k1.numberOfCursorIds(1);
    k1.cursorIds({1});
    KillCursorsMessageImpl k2(0, 0);
    k2.numberOfCursorIds(1);
    k2.cursorIds({2});
    EXPECT_FALSE(k1 == k2);
  }
}

TEST_F(MongoCodecImplTest, KillCursors) {
  KillCursorsMessageImpl kill(5, 5);
  kill.numberOfCursorIds(2);
  kill.cursorIds({20000, 40000});
  encoder_.encodeKillCursors(kill);

  EXPECT_CALL(callbacks_, decodeKillCursors_(Pointee(Eq(kill))));
  decoder_.onData(output_);
}

TEST_F(MongoCodecImplTest, EncodeExceptions) {
  QueryMessageImpl q(0, 0);
  EXPECT_THROW(encoder_.encodeQuery(q), EnvoyException);
  q.fullCollectionName("hello");
  EXPECT_THROW(encoder_.encodeQuery(q), EnvoyException);
  q.query(Bson::DocumentImpl::create());
  encoder_.encodeQuery(q);

  KillCursorsMessageImpl k(0, 0);
  EXPECT_THROW(encoder_.encodeKillCursors(k), EnvoyException);
  k.numberOfCursorIds(1);
  EXPECT_THROW(encoder_.encodeKillCursors(k), EnvoyException);
  k.cursorIds({1});
  encoder_.encodeKillCursors(k);

  InsertMessageImpl i(0, 0);
  EXPECT_THROW(encoder_.encodeInsert(i), EnvoyException);
  i.fullCollectionName("hello");
  EXPECT_THROW(encoder_.encodeInsert(i), EnvoyException);
  i.documents().push_back(Bson::DocumentImpl::create());
  encoder_.encodeInsert(i);

  GetMoreMessageImpl g(0, 0);
  EXPECT_THROW(encoder_.encodeGetMore(g), EnvoyException);
  g.fullCollectionName("hello");
  EXPECT_THROW(encoder_.encodeGetMore(g), EnvoyException);
  g.cursorId(1);
  encoder_.encodeGetMore(g);
}

TEST_F(MongoCodecImplTest, PartialMessages) {
  output_.add("2");
  decoder_.onData(output_);
  output_.drain(output_.length());

  Bson::BufferHelper::writeInt32(output_, 100);
  decoder_.onData(output_);
  EXPECT_EQ(4U, output_.length());
}

TEST_F(MongoCodecImplTest, InvalidMessage) {
  Bson::BufferHelper::writeInt32(output_, 16); // Size
  Bson::BufferHelper::writeInt32(output_, 0);  // Request ID
  Bson::BufferHelper::writeInt32(output_, 1);  // Response to
  Bson::BufferHelper::writeInt32(output_, 2);  // Invalid op
  EXPECT_THROW(decoder_.onData(output_), EnvoyException);
}

} // Mongo
