var EventEmitter = require('events').EventEmitter;

var CHANGE_EVENT = 'change';
var _events = new EventEmitter;

_events.setMaxListeners(0);

function notifyChange() {
  _events.emit(CHANGE_EVENT);
}

var _activeRoutes = [];
var _activeParams = {};
var _activeQuery = {};

function routeIsActive(routeName) {
  return _activeRoutes.some(function (route) {
    return route.props.name === routeName;
  });
}

function paramsAreActive(params) {
  for (var property in params) {
    if (_activeParams[property] !== String(params[property]))
      return false;
  }

  return true;
}

function queryIsActive(query) {
  for (var property in query) {
    if (_activeQuery[property] !== String(query[property]))
      return false;
  }

  return true;
}

/**
 * The ActiveStore keeps track of which routes, URL and query parameters are
 * currently active on a page. <Link>s subscribe to the ActiveStore to know
 * whether or not they are active.
 */
var ActiveStore = {

  addChangeListener: function (listener) {
    _events.on(CHANGE_EVENT, listener);
  },

  removeChangeListener: function (listener) {
    _events.removeListener(CHANGE_EVENT, listener);
  },

  /**
   * Updates the currently active state and notifies all listeners.
   * This is automatically called by routes as they become active.
   */
  updateState: function (state) {
    state = state || {};

    _activeRoutes = state.activeRoutes || [];
    _activeParams = state.activeParams || {};
    _activeQuery = state.activeQuery || {};

    notifyChange();
  },

  /**
   * Returns true if the route with the given name, URL parameters, and query
   * are all currently active.
   */
  isActive: function (routeName, params, query) {
    var isActive = routeIsActive(routeName) && paramsAreActive(params);

    if (query)
      return isActive && queryIsActive(query);

    return isActive;
  }

};

module.exports = ActiveStore;
