var ActiveStore = require('../stores/ActiveStore');

/**
 * A mixin for components that need to know about the routes, params,
 * and query that are currently active. Components that use it get two
 * things:
 *
 *   1. An `isActive` static method they can use to check if a route,
 *      params, and query are active.
 *   2. An `updateActiveState` instance method that is called when the
 *      active state changes.
 *
 * Example:
 *
 *   var Tab = React.createClass({
 *     
 *     mixins: [ Router.ActiveState ],
 *
 *     getInitialState: function () {
 *       return {
 *         isActive: false
 *       };
 *     },
 *   
 *     updateActiveState: function () {
 *       this.setState({
 *         isActive: Tab.isActive(routeName, params, query)
 *       })
 *     }
 *   
 *   });
 */
var ActiveState = {

  statics: {

    /**
     * Returns true if the route with the given name, URL parameters, and query
     * are all currently active.
     */
    isActive: ActiveStore.isActive

  },

  componentWillMount: function () {
    ActiveStore.addChangeListener(this.handleActiveStateChange);
  },

  componentDidMount: function () {
    if (this.updateActiveState)
      this.updateActiveState();
  },

  componentWillUnmount: function () {
    ActiveStore.removeChangeListener(this.handleActiveStateChange);
  },

  handleActiveStateChange: function () {
    if (this.isMounted() && typeof this.updateActiveState === 'function')
      this.updateActiveState();
  }

};

module.exports = ActiveState;
