var LocationActions = require('../actions/LocationActions');
var History = require('../utils/History');
var Path = require('../utils/Path');

/**
 * Returns the current URL path from `window.location`, including query string.
 */
function getWindowPath() {
  return Path.decode(
    window.location.pathname + window.location.search
  );
}

var _changeListeners = [];

function notifyChange(type) {
  var change = {
    path: getWindowPath(),
    type: type
  };

  _changeListeners.forEach(function (listener) {
    listener(change);
  });
}

var _isListening = false;

function onPopState() {
  notifyChange(LocationActions.POP);
}

/**
 * A Location that uses HTML5 history.
 */
var HistoryLocation = {

  addChangeListener: function (listener) {
    _changeListeners.push(listener);

    if (_isListening)
      return;

    if (window.addEventListener) {
      window.addEventListener('popstate', onPopState, false);
    } else {
      window.attachEvent('popstate', onPopState);
    }

    _isListening = true;
  },

  removeChangeListener: function(listener) {
    for (var i = 0, l = _changeListeners.length; i < l; i ++) {
      if (_changeListeners[i] === listener) {
        _changeListeners.splice(i, 1);
        break;
      }
    }

    if (window.addEventListener) {
      window.removeEventListener('popstate', onPopState);
    } else {
      window.removeEvent('popstate', onPopState);
    }

    if (_changeListeners.length === 0)
      _isListening = false;
  },



  push: function (path) {
    window.history.pushState({ path: path }, '', Path.encode(path));
    History.length += 1;
    notifyChange(LocationActions.PUSH);
  },

  replace: function (path) {
    window.history.replaceState({ path: path }, '', Path.encode(path));
    notifyChange(LocationActions.REPLACE);
  },

  pop: function () {
    History.back();
  },

  getCurrentPath: getWindowPath,

  toString: function () {
    return '<HistoryLocation>';
  }

};

module.exports = HistoryLocation;
