// ==========================================================================
// Project:  Ember Metal
// Copyright: ©2011 Strobe Inc. and contributors.
// License:   Licensed under MIT license (see license.js)
// ==========================================================================
/*globals ember_assert */

require('ember-metal/core');
require('ember-metal/platform');
require('ember-metal/utils');
require('ember-metal/accessors');

var USE_ACCESSORS = Ember.USE_ACCESSORS;
var GUID_KEY = Ember.GUID_KEY;
var META_KEY = Ember.META_KEY;
var meta = Ember.meta;
var o_create = Ember.platform.create;
var o_defineProperty = Ember.platform.defineProperty;
var SIMPLE_PROPERTY, WATCHED_PROPERTY;

// ..........................................................
// DESCRIPTOR
//

var SIMPLE_DESC = {
  writable: true,
  configurable: true,
  enumerable: true,
  value: null
};

/**
  @private
  @constructor

  Objects of this type can implement an interface to responds requests to
  get and set.  The default implementation handles simple properties.

  You generally won't need to create or subclass this directly.
*/
var Dc = Ember.Descriptor = function() {};

var setup = Dc.setup = function(obj, keyName, value) {
  SIMPLE_DESC.value = value;
  o_defineProperty(obj, keyName, SIMPLE_DESC);
  SIMPLE_DESC.value = null;
};

var Dp = Ember.Descriptor.prototype;

/**
  Called whenever we want to set the property value.  Should set the value
  and return the actual set value (which is usually the same but may be
  different in the case of computed properties.)

  @param {Object} obj
    The object to set the value on.

  @param {String} keyName
    The key to set.

  @param {Object} value
    The new value

  @returns {Object} value actual set value
*/
Dp.set = function(obj, keyName, value) {
  obj[keyName] = value;
  return value;
};

/**
  Called whenever we want to get the property value.  Should retrieve the
  current value.

  @param {Object} obj
    The object to get the value on.

  @param {String} keyName
    The key to retrieve

  @returns {Object} the current value
*/
Dp.get = function(obj, keyName) {
  return w_get(obj, keyName, obj);
};

/**
  This is called on the descriptor to set it up on the object.  The
  descriptor is responsible for actually defining the property on the object
  here.

  The passed `value` is the transferValue returned from any previous
  descriptor.

  @param {Object} obj
    The object to set the value on.

  @param {String} keyName
    The key to set.

  @param {Object} value
    The transfer value from any previous descriptor.

  @returns {void}
*/
Dp.setup = setup;

/**
  This is called on the descriptor just before another descriptor takes its
  place.  This method should at least return the 'transfer value' of the
  property - which is the value you want to passed as the input to the new
  descriptor's setup() method.

  It is not generally necessary to actually 'undefine' the property as a new
  property descriptor will redefine it immediately after this method returns.

  @param {Object} obj
    The object to set the value on.

  @param {String} keyName
    The key to set.

  @returns {Object} transfer value
*/
Dp.teardown = function(obj, keyName) {
  return obj[keyName];
};

Dp.val = function(obj, keyName) {
  return obj[keyName];
};

// ..........................................................
// SIMPLE AND WATCHED PROPERTIES
//

// if accessors are disabled for the app then this will act as a guard when
// testing on browsers that do support accessors.  It will throw an exception
// if you do foo.bar instead of Ember.get(foo, 'bar')

// The exception to this is that any objects managed by Ember but not a descendant
// of Ember.Object will not throw an exception, instead failing silently. This
// prevent errors with other libraries that may attempt to access special
// properties on standard objects like Array. Usually this happens when copying
// an object by looping over all properties.

if (!USE_ACCESSORS) {
  Ember.Descriptor.MUST_USE_GETTER = function() {
    if (this instanceof Ember.Object) {
      ember_assert('Must use Ember.get() to access this property', false);
    }
  };

  Ember.Descriptor.MUST_USE_SETTER = function() {
    if (this instanceof Ember.Object) {
      if (this.isDestroyed) {
        ember_assert('You cannot set observed properties on destroyed objects', false);
      } else {
        ember_assert('Must use Ember.set() to access this property', false);
      }
    }
  };
}

var WATCHED_DESC = {
  configurable: true,
  enumerable:   true,
  set: Ember.Descriptor.MUST_USE_SETTER
};

/** @private */
function w_get(obj, keyName, values) {
  values = values || meta(obj, false).values;

  if (values) {
    var ret = values[keyName];
    if (ret !== undefined) { return ret; }
    if (obj.unknownProperty) { return obj.unknownProperty(keyName); }
  }

}

/** @private */
function w_set(obj, keyName, value) {
  var m = meta(obj), watching;

  watching = m.watching[keyName]>0 && value!==m.values[keyName];
  if (watching) Ember.propertyWillChange(obj, keyName);
  m.values[keyName] = value;
  if (watching) Ember.propertyDidChange(obj, keyName);
  return value;
}

var WATCHED_GETTERS = {};
/** @private */
function mkWatchedGetter(keyName) {
  var ret = WATCHED_GETTERS[keyName];
  if (!ret) {
    ret = WATCHED_GETTERS[keyName] = function() {
      return w_get(this, keyName);
    };
  }
  return ret;
}

var WATCHED_SETTERS = {};
/** @private */
function mkWatchedSetter(keyName) {
  var ret = WATCHED_SETTERS[keyName];
  if (!ret) {
    ret = WATCHED_SETTERS[keyName] = function(value) {
      return w_set(this, keyName, value);
    };
  }
  return ret;
}

/**
  @private

  Private version of simple property that invokes property change callbacks.
*/
WATCHED_PROPERTY = new Ember.Descriptor();

if (Ember.platform.hasPropertyAccessors) {
  WATCHED_PROPERTY.get = w_get ;
  WATCHED_PROPERTY.set = w_set ;

  if (USE_ACCESSORS) {
    WATCHED_PROPERTY.setup = function(obj, keyName, value) {
      WATCHED_DESC.get = mkWatchedGetter(keyName);
      WATCHED_DESC.set = mkWatchedSetter(keyName);
      o_defineProperty(obj, keyName, WATCHED_DESC);
      WATCHED_DESC.get = WATCHED_DESC.set = null;
      if (value !== undefined) meta(obj).values[keyName] = value;
    };

  } else {
    WATCHED_PROPERTY.setup = function(obj, keyName, value) {
      WATCHED_DESC.get = mkWatchedGetter(keyName);
      o_defineProperty(obj, keyName, WATCHED_DESC);
      WATCHED_DESC.get = null;
      if (value !== undefined) meta(obj).values[keyName] = value;
    };
  }

  WATCHED_PROPERTY.teardown = function(obj, keyName) {
    var ret = meta(obj).values[keyName];
    delete meta(obj).values[keyName];
    return ret;
  };

// NOTE: if platform does not have property accessors then we just have to
// set values and hope for the best.  You just won't get any warnings...
} else {

  WATCHED_PROPERTY.set = function(obj, keyName, value) {
    var m = meta(obj), watching;

    watching = m.watching[keyName]>0 && value!==obj[keyName];
    if (watching) Ember.propertyWillChange(obj, keyName);
    obj[keyName] = value;
    if (watching) Ember.propertyDidChange(obj, keyName);
    return value;
  };

}

/**
  The default descriptor for simple properties.  Pass as the third argument
  to Ember.defineProperty() along with a value to set a simple value.

  @static
  @default Ember.Descriptor
*/
Ember.SIMPLE_PROPERTY = new Ember.Descriptor();
SIMPLE_PROPERTY = Ember.SIMPLE_PROPERTY;

SIMPLE_PROPERTY.unwatched = WATCHED_PROPERTY.unwatched = SIMPLE_PROPERTY;
SIMPLE_PROPERTY.watched   = WATCHED_PROPERTY.watched   = WATCHED_PROPERTY;


// ..........................................................
// DEFINING PROPERTIES API
//

/** @private */
function hasDesc(descs, keyName) {
  if (keyName === 'toString') return 'function' !== typeof descs.toString;
  else return !!descs[keyName];
}

/**
  @private

  NOTE: This is a low-level method used by other parts of the API.  You almost
  never want to call this method directly.  Instead you should use Ember.mixin()
  to define new properties.

  Defines a property on an object.  This method works much like the ES5
  Object.defineProperty() method except that it can also accept computed
  properties and other special descriptors.

  Normally this method takes only three parameters.  However if you pass an
  instance of Ember.Descriptor as the third param then you can pass an optional
  value as the fourth parameter.  This is often more efficient than creating
  new descriptor hashes for each property.

  ## Examples

      // ES5 compatible mode
      Ember.defineProperty(contact, 'firstName', {
        writable: true,
        configurable: false,
        enumerable: true,
        value: 'Charles'
      });

      // define a simple property
      Ember.defineProperty(contact, 'lastName', Ember.SIMPLE_PROPERTY, 'Jolley');

      // define a computed property
      Ember.defineProperty(contact, 'fullName', Ember.computed(function() {
        return this.firstName+' '+this.lastName;
      }).property('firstName', 'lastName').cacheable());
*/
Ember.defineProperty = function(obj, keyName, desc, val) {
  var m = meta(obj, false), descs = m.descs, watching = m.watching[keyName]>0, override = true;

  if (val === undefined) {
    override = false;
    val = hasDesc(descs, keyName) ? descs[keyName].teardown(obj, keyName) : obj[keyName];
  } else if (hasDesc(descs, keyName)) {
    descs[keyName].teardown(obj, keyName);
  }

  if (!desc) desc = SIMPLE_PROPERTY;

  if (desc instanceof Ember.Descriptor) {
    m = meta(obj, true);
    descs = m.descs;

    desc = (watching ? desc.watched : desc.unwatched) || desc;
    descs[keyName] = desc;
    desc.setup(obj, keyName, val, watching);

  // compatibility with ES5
  } else {
    if (descs[keyName]) meta(obj).descs[keyName] = null;
    o_defineProperty(obj, keyName, desc);
  }

  // if key is being watched, override chains that
  // were initialized with the prototype
  if (override && watching) Ember.overrideChains(obj, keyName, m);

  return this;
};

/**
  Creates a new object using the passed object as its prototype.  On browsers
  that support it, this uses the built in Object.create method.  Else one is
  simulated for you.

  This method is a better choice than Object.create() because it will make
  sure that any observers, event listeners, and computed properties are
  inherited from the parent as well.

  @param {Object} obj
    The object you want to have as the prototype.

  @returns {Object} the newly created object
*/
Ember.create = function(obj, props) {
  var ret = o_create(obj, props);
  if (GUID_KEY in ret) Ember.generateGuid(ret, 'ember');
  if (META_KEY in ret) Ember.rewatch(ret); // setup watch chains if needed.
  return ret;
};

/**
  @private

  Creates a new object using the passed object as its prototype.  This method
  acts like `Ember.create()` in every way except that bindings, observers, and
  computed properties will be activated on the object.

  The purpose of this method is to build an object for use in a prototype
  chain. (i.e. to be set as the `prototype` property on a constructor
  function).  Prototype objects need to inherit bindings, observers and
  other configuration so they pass it on to their children.  However since
  they are never 'live' objects themselves, they should not fire or make
  other changes when various properties around them change.

  You should use this method anytime you want to create a new object for use
  in a prototype chain.

  @param {Object} obj
    The base object.

  @param {Object} hash
    Optional hash of properties to define on the object.

  @returns {Object} new object
*/
Ember.createPrototype = function(obj, props) {
  var ret = o_create(obj, props);
  meta(ret, true).proto = ret;
  if (GUID_KEY in ret) Ember.generateGuid(ret, 'ember');
  if (META_KEY in ret) Ember.rewatch(ret); // setup watch chains if needed.
  return ret;
};
