package fs

import (
	"regexp"

	"github.com/photoprism/photoprism/pkg/rnd"
)

var DscNameRegexp = regexp.MustCompile("\\D{3}[\\d_]\\d{4,8}_?\\d{0,6}_?\\d{0,6}[\\.jpgJPGXx]{0,4}")
var UniqueNameRegexp = regexp.MustCompile("[a-f0-9]{8,16}_[a-f0-9]{6,16}_[A-Za-z0-9]{1,20}_?[A-Za-z0-9]{0,4}") // Example: 8263987746_d0a6055c58_o
var UUIDNameRegexp = regexp.MustCompile("[A-Fa-f0-9\\-]{16,36}_?[A-Za-z0-9_]{0,20}")                           // Example: 8263987746_d0a6055c58_o

// IsInt tests if the file base is an integer number.
func IsInt(s string) bool {
	if s == "" {
		return false
	}

	for _, r := range s {
		if r < 48 || r > 57 {
			return false
		}
	}

	return true
}

// IsAsciiID tests if the string is a file name that only contains uppercase ascii letters and numbers like "IQVG4929".
func IsAsciiID(s string) bool {
	if s == "" {
		return false
	}

	for _, r := range s {
		if (r < 65 || r > 90) && (r < 48 || r > 57) && r != 45 && r != 95 {
			return false
		}
	}

	return true
}

// IsUniqueName tests if the string looks like a unique file name.
func IsUniqueName(s string) bool {
	if s == "" {
		return false
	}

	if m := UniqueNameRegexp.FindString(s); s == m {
		return true
	} else if m := UUIDNameRegexp.FindString(s); s == m {
		return true
	}

	return false
}

// IsDscName tests if the string looks like a file name generated by a camera.
func IsDscName(s string) bool {
	if s == "" {
		return false
	}

	if m := DscNameRegexp.FindString(s); s == m {
		return true
	}

	return false
}

// IsGenerated tests if the file name looks like an automatically generated identifier.
func IsGenerated(fileName string) bool {
	if fileName == "" {
		return false
	}

	base := BasePrefix(fileName, false)

	if IsAsciiID(base) {
		return true
	} else if IsHash(base) {
		return true
	} else if IsInt(base) {
		return true
	} else if IsDscName(base) {
		return true
	} else if IsUniqueName(base) {
		return true
	} else if rnd.IsUID(base, 0) {
		return true
	} else if IsCanonical(base) {
		return true
	}

	return false
}
