// Copyright (c) 2018 Intel Corporation
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

///////////////////////////////////////////////////////////////////////////////////////////////////
#include <gtest/gtest.h>

#include <api/CPP/engine.hpp>
#include <api/CPP/input_layout.hpp>
#include <api/CPP/pyramid_roi_align.hpp>
#include <api/CPP/memory.hpp>
#include <api/CPP/topology.hpp>
#include <api/CPP/network.hpp>
#include <api/CPP/mutable_data.hpp>

#include "test_utils/test_utils.h"


using namespace cldnn;
using namespace tests;

enum Test_index { //order the same as test_data table
    BOXES = 0,
    IMAGE_META,
    P5,
    P4,
    P3,
    P2,
    POOL
};

struct Test_data
{
    const char *parameter_name;
    int32_t b, f, x, y;
    std::vector<float> dataTMP;
};

Test_data test_data [] = //order the same as enum Test_index
{
    {
        "boxes",
        1, 1, 1, 4,
        { 0.274695277f, 0.39985016f, 0.751607299f, 0.649529517f }
    },
    {   
        "image_meta",
        1, 1, 1, 93,
        { 0, 415, 640, 3, 1024, 1024, 3, 180, 0, 844, 1024, 1.6f }
    },
    {
        "P5",
        1, 1, 32, 32,
        {
            -2.33415818f, -1.46765602f,  -0.998123348f, -0.945146739f, -0.721071541f, -1.19279253f, -1.37023795f,   -1.61545324f,  -2.03868198f,  -1.72659981f,  -1.5598495f,   -1.55309856f,  -1.53211606f, -1.86645496f, -1.84540808f,   -1.68674099f, -1.60733783f, -1.43271363f, -1.37408626f,  -1.35044777f, -1.25868618f, -0.965965867f, -0.881696165f, -0.709434509f, -0.494760394f, -0.482933104f, -1.26238084f, -1.45486391f, -1.00801146f, -0.840218246f, -0.420806766f, 0.635412455f,
            -5.15252113f, -4.81609535f,  -4.33736563f,  -4.5069356f,   -4.69305611f,  -5.35324192f, -5.4090085f,    -5.18345022f,  -5.57966137f,  -6.08182287f,  -6.4237361f,   -6.63587379f,  -6.60395145f, -6.99704218f, -7.26061678f,   -7.13621283f, -6.92309761f, -6.54043388f, -6.0931859f,   -5.95154953f, -5.92886162f, -5.60794735f,  -5.39521217f,  -5.24937916f,  -4.93126583f,  -5.03314447f,  -6.35518694f, -5.97401428f, -4.61507177f, -3.88595009f,  -3.10539627f, -1.12507141f,
            -4.58263206f, -4.23551846f,  -3.71995449f,  -3.9303925f,   -4.22284889f,  -4.90389252f, -4.90515423f,   -4.35046101f,  -4.93061686f,  -5.62805653f,  -7.13111687f,  -8.04961014f,  -8.61973f,    -8.91128826f, -9.59987259f,   -9.77626991f, -9.34930134f, -8.41235256f, -7.99330997f,  -7.56377172f, -7.41074753f, -7.68792772f, -7.52159262f, -7.23604727f, -6.43461895f, -6.30558538f, -7.42862511f, -6.57217264f, -4.36673212f, -3.42791319f, -2.78279519f, -1.13899291f,
            -4.05928659f, -3.63066411f,  -3.08045626f,  -3.49022269f,  -3.33089471f,  -3.98461342f, -3.60927105f,   -3.47735429f,  -4.22189903f,  -5.61483288f,  -6.73310328f,  -7.82119894f,  -7.76680946f, -7.81351185f, -8.53846359f,   -8.85490894f, -8.87630653f, -8.05196667f, -7.37027693f,  -6.48965073f, -6.0011878f, -6.49297428f, -6.87221718f, -6.6889801f, -5.67975998f, -5.48370981f, -6.48479271f, -5.99923038f, -4.15075731f, -3.24771428f, -2.38959575f, -0.802779257f,
            -3.8221159f,  -3.2125051f,   -2.67735672f,  -3.35456967f,  -2.42953777f,  -1.97508657f, -0.0455740131f, 0.200172856f,  -1.73673642f,  -4.14228773f,  -6.05798674f,  -6.92922974f,  -6.31088972f, -5.24032164f, -5.8104291f,    -6.21769142f, -6.71948385f, -6.34254694f, -5.40050459f,  -3.83635306f, -2.84016895f, -3.47709227f, -4.53029394f, -4.79398346f, -4.15029287f, -4.34026718f, -5.05020094f, -4.96476984f, -3.85935163f, -3.06635952f, -2.21780515f, -0.550920606f,
            -3.38425207f, -2.47040701f,  -1.75725257f,  -2.67789435f,  -1.93510687f,  -0.023562137f, 3.12235284f,   3.195858f,     -0.502758205f, -3.64130497f,  -4.92483091f,  -5.37235212f,  -4.44142771f, -3.01087427f, -2.56460142f,   -3.36131048f, -4.67883253f, -4.97649288f, -4.15489054f,  -3.05888772f, -2.53061557f, -2.89280939f, -3.89569187f, -3.85883617f, -3.85448074f, -3.72637963f, -4.17853975f, -3.72458243f, -3.2028439f, -2.26282644f, -1.57095635f, -0.0362351872f,
            -2.86179805f, -1.77212584f,  -1.01908028f,  -2.22856259f,  -2.04378486f,  -0.389851034f, 2.5954473f,    3.546386f,     -0.572356939f, -3.22942686f,  -4.71709538f,  -5.06511068f,  -4.19580078f, -2.62281418f, -1.84743559f,   -1.72474909f, -2.85398459f, -3.05193329f, -2.1715126f,   -1.87324941f, -2.42470956f, -3.27851868f, -4.05942631f, -3.64058971f, -3.65105247f, -3.37935495f, -3.88859773f, -3.24483466f, -2.69226313f, -1.51380038f, -0.803811312f, 0.575846195f,
            -2.44617772f, -1.21146309f,  -0.406607807f, -1.79564178f,  -2.15529561f,  -1.86219978f, -0.642769337f,  -0.119694829f, -3.55873179f,  -6.07527542f,  -7.34461832f,  -7.5732069f,   -5.2650032f,  -2.78443551f, -2.01951551f,   -2.20919466f, -3.48502755f, -3.39159703f, -2.84414029f,  -3.01556158f, -4.17538118f, -4.6715436f, -4.51803017f, -3.98833418f, -4.03647232f, -3.56217432f, -4.35153055f, -3.35357046f, -2.34758973f, -0.991552889f, -0.410246134f, 0.853578329f,
            -2.32879257f, -0.983750522f, -0.21862191f,  -1.63332736f,  -2.70467782f,  -3.79070854f, -3.12105083f,   -3.37172794f,  -5.87286377f,  -7.56662798f,  -8.18826008f,  -7.51929522f,  -5.9531951f,  -4.06868601f, -2.65765858f,   -2.80148482f, -4.28907013f, -4.32930136f, -4.3640132f,   -4.59029436f, -5.4193697f, -5.89368916f, -5.6321454f, -5.52998543f, -5.09114599f, -3.59506583f, -3.95068288f, -3.30025363f, -2.04802871f, -0.637728035f, -0.245602071f, 0.879402578f,
            -2.35637832f, -0.938572884f, -0.137476623f, -1.41782618f,  -2.65590358f,  -4.25014019f, -4.0826478f,    -4.17878771f,  -5.6027894f,   -7.31306791f,  -7.89162493f,  -7.03756762f,  -6.09949017f, -5.60607052f, -4.94666481f,   -4.39400244f, -4.67201567f, -4.2205472f,  -4.38528776f,  -4.6779213f, -4.83282471f, -4.84141684f, -4.65654802f, -4.24497604f, -3.85145688f, -2.74431086f, -3.78755236f, -3.00524449f, -1.81372464f, -0.552992642f, -0.150228053f, 0.944489419f,
            -2.39807153f, -0.961493254f, -0.207601368f, -1.41579533f,  -2.26456952f,  -3.31752872f, -2.37754416f,   -2.27816534f,  -3.3359437f,   -4.83316755f,  -4.82455635f,  -5.1267004f,   -4.75627851f, -6.18640566f, -7.98392439f,   -9.12876225f, -8.12104893f, -7.43801117f, -5.90858698f,  -3.8132503f, -2.49779272f, -2.64403725f, -2.50610948f, -2.27564049f, -2.08231401f, -2.0385685f, -3.72143364f, -3.04797244f, -1.76300609f, -0.521960258f, -0.0881003886f, 0.961502016f,
            -2.44038081f, -1.01705039f,  -0.289608359f, -1.37090492f,  -1.93311131f,  -2.47754407f, -1.31518912f,   -0.804416537f, -0.930097163f, -0.780354142f, -0.834263086f, -1.50460267f,  -3.63839531f, -4.60880566f, -6.8964262f,    -8.66131878f, -9.60757637f, -8.79116344f, -6.86388493f,  -4.30527639f, -1.8283174f, -1.4908253f, -1.37629032f, -1.22827542f, -1.60703599f, -2.33176303f, -3.86254454f, -2.99731207f, -1.65976918f, -0.461797535f, 0.0194968097f, 0.998998225f,
            -2.46240711f, -1.03391945f,  -0.35371244f,  -1.40552509f,  -1.92847848f,  -2.80441093f, -1.44593406f,   -0.652132452f, -0.4637236f,   -0.377687186f, -0.223660469f, -1.29031694f,  -2.68966746f, -3.15799189f, -3.18843555f,   -4.4910984f, -6.69606543f, -8.33802032f,  -8.19927311f,  -6.32680511f, -3.98862648f, -2.22264123f, -1.55090904f, -1.1854068f, -1.3106786f, -1.90384912f, -3.67234707f, -2.88272882f, -1.53641987f, -0.362456888f, 0.0893754214f, 1.02051163f,
            -2.48206067f, -1.02961993f,  -0.368244141f, -1.42910719f,  -1.93446803f,  -2.968822f,   -1.83339584f,   -1.077631f,    -1.20465982f,  -1.57803464f,  -1.41360343f,  -1.76699162f,  -2.31551576f, -2.05016136f, -0.0285568349f, 1.02111804f, -1.09839404f, -3.57055283f,  -6.42463684f,  -6.38169003f, -6.04913425f, -3.92720795f, -2.87601185f, -2.27725315f, -1.91104662f, -1.94828415f, -3.19035602f, -2.59298229f, -1.44278193f, -0.386298746f, 0.0836858153f, 0.999346912f,
            -2.48712945f, -1.01729345f,  -0.474304944f, -1.67669559f,  -2.10705042f,  -3.42592764f, -2.34152699f,   -1.83562672f,  -1.90750253f,  -2.23259664f,  -1.80318487f,  -2.05461431f,  -2.2218473f,  -1.68138134f,  1.89481843f,   4.749331f,    4.48664188f,  1.76011801f,  -2.80741739f,  -5.01609373f, -6.86733389f, -4.95238161f, -3.11620855f, -2.35959673f, -2.14903998f, -2.22679043f, -3.25020576f, -2.55579758f, -1.45884585f, -0.450649738f, 0.0580532737f, 0.980433941f,
            -2.5185082f,  -1.06924045f,  -0.577468932f, -1.7359041f,   -2.2522819f,   -3.44346404f, -2.27338934f,   -1.50737846f,  -1.4048748f,   -1.7626915f,   -1.77618313f,  -2.55145335f,  -2.72144723f, -1.09168231f,  3.47705436f,   7.27473307f,  7.77128983f,  4.76851988f,  -0.231550142f, -4.59473372f, -7.91270256f, -5.9186945f, -3.17887211f, -1.95729899f, -2.12510371f, -2.66853952f, -3.79930806f, -2.93926597f, -1.47657454f, -0.51107496f, 0.0374269597f, 0.9673509f,
            -2.57245374f, -1.16771162f,  -0.721676588f, -1.80981266f,  -2.38730669f,  -3.6522367f,  -2.01576495f,   -0.8515746f,   -0.121799529f, -1.13752592f,  -1.98465598f,  -3.21510339f,  -3.90218043f, -1.90408611f,  3.62870288f,   9.53359127f,  12.2969809f,  9.25624657f,  3.08819818f,   -3.57391787f, -8.53378582f, -6.41586733f, -3.14953685f, -1.97396016f, -2.7328465f, -3.78186893f, -4.93579912f, -3.55470729f, -1.54245102f, -0.482002735f, 0.0237279348f, 0.970623732f,
            -2.6402328f,  -1.25508213f,  -0.813264728f, -1.85111022f,  -2.31478047f,  -3.37323236f, -1.72119153f,   -0.622631073f,  0.275214434f, -1.74099112f,  -3.82077885f,  -5.72362041f,  -7.07592487f, -5.2477479f,   1.65343058f,   9.84803104f,  13.9755783f,  12.027339f,   6.53266191f,   0.243630022f, -4.9232049f, -4.36105299f, -1.71283042f, -1.22028506f, -2.47615337f, -3.96648002f, -4.9211669f, -3.52139068f, -1.58175361f, -0.453389883f, 0.0172070079f, 0.974586606f,
            -2.69985747f, -1.30426204f,  -0.813042939f, -1.84938121f,  -2.33455706f,  -3.75564861f, -2.54689479f,   -2.26757884f,  -1.79824364f,  -2.93493605f,  -4.15734148f,  -4.67264462f,  -5.97829533f, -6.07628202f, -0.634435117f,  7.86048698f,  13.385828f,   13.8827438f,  9.38942051f,   3.89634967f, -1.39140749f, -2.39509726f, -1.62092125f, -1.5939455f, -2.25631547f, -3.52288079f, -4.53593159f, -3.25450349f, -1.60031211f, -0.435814232f, 0.0219062977f, 0.986854315f,
            -2.74063468f, -1.31302822f,  -0.820956767f, -1.81994605f,  -2.28283525f,  -3.5440836f,  -2.51103139f,   -2.81304479f,  -3.26139283f,  -3.37517047f,  -3.98655128f,  -4.15412378f,  -4.92545223f, -5.78675413f, -3.06408238f,   3.01499391f,  8.77478504f,  10.6144304f,  8.11615849f,   4.45580721f, 0.623039126f, -1.10865057f, -1.95774138f, -2.36074567f, -2.57845926f, -3.33297563f, -3.97079587f, -2.93356919f, -1.50071633f, -0.443875313f, 0.0236797072f, 0.991317093f,
            -2.77299833f, -1.32691216f,  -0.831916511f, -1.82886219f,  -2.0734787f,   -3.13335371f, -1.50032151f,   -1.46733963f,  -2.72959828f,  -3.5253818f,   -4.29566097f,  -5.57419872f,  -6.24431992f, -6.32591867f, -5.26826477f,   -3.04502487f, 0.449693143f, 3.47979259f,  3.50362659f,   2.58046269f, 0.579684913f, -0.919588447f, -2.08200479f, -2.6678884f, -2.59757757f, -3.0013814f, -3.42182064f, -2.75994992f, -1.48684669f, -0.477065891f, 0.0327885784f, 0.994787097f,
            -2.7904563f,  -1.33298481f,  -0.825692832f, -1.78411806f,  -1.98032236f,  -2.94529605f, -1.540254f,     -1.03917682f,  -1.87087965f,  -2.15394163f,  -2.24386406f,  -1.56417131f,  -1.79924405f, -2.09344101f, -3.65430427f,   -4.66693974f, -4.27157164f, -1.08878291f, -0.221785039f, -0.0799107477f, -0.684955359f, -1.22172666f, -1.90416121f, -2.04627061f, -2.09932423f, -2.7114203f, -3.33123398f, -2.65206981f, -1.4748162f, -0.431342453f, 0.0863730982f, 1.03362691f,
            -2.80970526f, -1.32318377f,  -0.788406253f, -1.62803352f,  -1.83336627f,  -2.71299958f, -1.29830825f,   -0.898415565f, -1.27306414f,  -1.4642626f,   -1.53942132f,  -0.524312437f, -1.13679814f, -2.15964532f, -3.81581545f,   -6.19301414f, -6.9342289f,  -4.5518117f,  -4.05187798f,  -3.89661026f, -2.73003149f, -1.90081847f, -1.18712986f, -1.05476069f, -1.45352709f, -2.40461349f, -3.57806826f, -2.67894101f, -1.34701252f, -0.292546421f, 0.223820776f, 1.15115368f,
            -2.83941913f, -1.31946158f,  -0.752137005f, -1.59541857f,  -1.98224044f,  -3.13006711f, -2.87664342f,   -2.74078941f,  -2.44921613f,  -1.53203559f,  -1.11937928f,  -0.268255889f, -1.06444466f, -2.87781739f, -4.91630268f,   -8.23729324f, -10.6890593f, -10.1742487f, -8.88589478f,  -7.06334209f, -4.42162704f, -2.8048737f, -0.9670524f, -0.169980749f, -0.62598449f, -1.46366549f, -3.44733119f, -2.70727062f, -1.12550855f, 0.0431886837f, 0.491125584f, 1.39527845f,
            -2.88625073f, -1.36332977f,  -0.782323718f, -1.70872879f,  -2.29862785f,  -3.65832949f, -3.41763759f,   -2.27270484f,  -1.15727568f,  -0.485867918f, -0.534794629f, -0.99851644f,  -1.86469233f, -3.56163645f, -6.06065321f,   -8.93986511f, -11.1936483f, -11.16537f,   -9.42015839f,  -7.1612606f, -4.54605007f, -3.13340139f, -1.05612564f, -0.218226328f, -0.347539067f, -0.917124569f, -3.23879743f, -2.66016054f, -1.1019274f, 0.280594468f, 0.802835882f, 1.70916617f,
            -2.95432734f, -1.55732679f,  -0.9671579f,   -1.87740719f,  -2.52375722f,  -3.9269383f,  -3.63090515f,   -2.16633034f,  -1.57592404f,  -1.65385628f,  -2.63003421f,  -3.4876802f,   -4.29189682f, -4.7487464f,  -5.76429272f,   -6.65200949f, -7.45039988f, -7.22736359f, -6.15258741f,  -5.31453133f, -3.85754275f, -3.2067554f, -1.73008275f, -1.35701323f, -1.16924942f, -1.25322843f, -3.28507686f, -2.95321226f, -1.38456213f, 0.187379554f, 0.978641272f, 1.96348953f,
            -3.11177945f, -1.80547488f,  -1.13023674f,  -1.9582721f,   -2.37351155f,  -3.67039227f, -3.1937058f,    -2.27774191f,  -2.11655211f,  -2.92763114f,  -3.51109672f,  -4.43897057f,  -4.60774946f, -5.22836876f, -5.26246691f,   -5.41725492f, -5.64507723f, -5.44532156f, -5.25552511f,  -5.40288162f, -4.75492859f, -4.50234127f, -3.85268068f, -3.71338868f, -3.31360817f, -3.09147811f, -4.54734945f, -3.58751845f, -1.86106849f, -0.0580402128f, 0.987123847f, 2.12943125f,
            -3.36467028f, -2.15916252f,  -1.39851403f,  -2.21555972f,  -2.6277256f,   -3.89018989f, -3.28536391f,   -2.4179709f,   -2.31355095f,  -3.14865518f,  -3.84860849f,  -4.44453287f,  -4.50857449f, -4.88197565f, -4.95770359f,   -5.04250717f, -4.74955845f, -4.8034606f,  -4.87089396f,  -5.45653677f, -5.71883726f, -5.90324974f, -5.92616558f, -5.50277519f, -5.18182898f, -5.07875252f, -6.3301487f, -4.71556807f, -2.65147376f, -0.510522306f, 0.768599629f, 2.15899801f,
            -3.76996517f, -2.77193499f,  -2.04029584f,  -2.67725992f,  -3.11456323f,  -4.35716057f, -3.96405196f,   -3.11866283f,  -2.89303422f,  -3.84127808f,  -4.63507318f,  -5.34559536f,  -5.6741724f,  -5.9913516f,  -5.89291143f,   -6.14835787f, -5.75908613f, -5.48700523f, -5.17146826f,  -5.74538183f, -6.23743486f, -6.26235199f, -6.18846273f, -5.73266459f, -5.36256504f, -5.36837292f, -6.48104477f, -4.97722006f, -3.1608839f, -1.36612868f, -0.0857250318f, 1.66240442f,
            -4.26793671f, -3.59083676f,  -2.84308076f,  -3.14333463f,  -3.37969398f,  -4.55007124f, -4.40458679f,   -3.53423572f,  -2.78584123f,  -3.32700229f,  -3.87822628f,  -5.09642506f,  -6.15807199f, -6.88138437f, -7.01429605f,   -7.22634697f, -7.04120684f, -6.64636993f, -5.79211712f,  -5.76786995f, -5.5597887f, -5.01553154f, -4.96951723f, -4.92054939f, -4.69466639f, -4.54826736f, -5.57798719f, -4.50945187f, -3.42488861f, -2.2323885f, -1.17007399f, 0.706006825f,
            -4.58093643f, -4.09917927f,  -3.6026299f,   -3.76272631f,  -4.10116673f,  -5.68298769f, -5.67115974f,   -5.20354462f,  -4.87026978f,  -5.25120115f,  -5.51101351f,  -6.41377878f,  -7.30511761f, -8.20695019f, -8.15464306f,   -8.10768127f, -7.72227478f, -7.57483578f, -6.83547497f,  -6.92473555f, -6.26031017f, -5.44693089f, -4.98586988f, -4.71777868f, -4.84076738f, -4.88040304f, -5.76190281f, -4.94208717f, -3.97660065f, -3.09410763f, -2.34518123f, -0.401388973f,
            -4.02035284f, -4.02879238f,  -4.01832962f,  -4.46334934f,  -5.42945766f,  -7.13510704f, -7.44949913f,   -7.30862284f,  -7.11234093f,  -7.16781139f,  -7.24586773f,  -7.57177401f,  -7.80264711f, -7.91191673f, -7.63455296f,   -7.31139612f, -7.24533272f, -7.21524429f, -7.19505501f,  -7.53508186f, -7.19776154f, -6.55349255f, -6.06127691f, -5.76581764f, -6.03102398f, -6.42573166f, -7.26578999f, -6.23923731f, -5.2162056f, -4.30992317f, -3.86889744f, -2.24626088f
        }
    },
    {
        "P4",
        1, 1, 64, 64,
        {/*Intentionaly 0 elements. Test for this elemnt does not used*/}
    },
    {
        "P3",
        1, 1, 128, 128,
        {/*Intentionaly 0 elements. Test for this elemnt does not used*/}
    },
    {
        "P2",
        1, 1, 128, 128,
        {/*Intentionaly 0 elements. Test for this elemnt does not used*/}
    },
    {
        "pool",
        1, 1, 7, 7,
        {/*Intentionaly 0 elements. Values not important - only layout*/}
    }
};

memory allocate_memory(Test_index key, const engine &engine) 
{
    auto ret = memory::allocate(engine, { data_types::f32, format::bfyx, { test_data[key].b, test_data[key].f, test_data[key].y, test_data[key].x } });
    set_values(ret, test_data[key].dataTMP);
    return ret;
}

TEST(pyramidROIAlign_gpu, basic_functionality)
{
    const auto& engine = get_test_engine();

    std::vector<float> answer =
    {
            -5.56710863f, -4.15980053f,  -3.62781334f, -4.4299016f,   -4.32974339f,  -4.59520054f, -5.14869022f,
            -4.04856586f, -6.20199442f,  -8.62770653f, -9.3613081f,   -7.69766426f,  -4.6893239f,  -1.79761052f,
            -2.1207974f,  -0.0283275843f, 2.62955427f,  0.693355441f, -3.21296549f,  -5.62806273f, -6.13721943f,
            -3.01667213f,  1.90189886f,   9.18445969f,  11.0731812f,   5.476161f,    -2.67103052f, -8.19120693f,
            -5.73783922f, -2.93177485f,   5.87217808f,  11.9360819f,   10.5841255f,   4.8481946f,  -0.81512779f,
            -2.63171887f, -3.56354189f,  -4.38874054f, -2.65824175f,   0.0660879612f, 0.36207819f, -0.571367621f,
            -2.00750613f, -4.5745883f,   -8.36942673f, -10.7424393f,  -9.67979145f,  -7.39468241f, -4.24828815f
    };

    auto boxes = allocate_memory(BOXES, engine);
    auto image_meta = allocate_memory(IMAGE_META, engine);
    auto P5_tensor = allocate_memory(P5, engine);
    auto P4_tensor = allocate_memory(P4, engine);
    auto P3_tensor = allocate_memory(P3, engine);
    auto P2_tensor = allocate_memory(P2, engine);
    auto pool_size = allocate_memory(POOL, engine);

    topology topo;
    topo.add(input_layout(test_data[BOXES].parameter_name, boxes.get_layout()));
    topo.add(input_layout(test_data[IMAGE_META].parameter_name, image_meta.get_layout()));
    topo.add(input_layout(test_data[P2].parameter_name, P2_tensor.get_layout()));
    topo.add(input_layout(test_data[P3].parameter_name, P3_tensor.get_layout()));
    topo.add(input_layout(test_data[P4].parameter_name, P4_tensor.get_layout()));
    topo.add(input_layout(test_data[P5].parameter_name, P5_tensor.get_layout()));
    topo.add(input_layout(test_data[POOL].parameter_name, pool_size.get_layout()));

    topo.add(pyramid_roi_align("pyramidROIAlign", 
            test_data[BOXES].parameter_name, 
            test_data[IMAGE_META].parameter_name,
            test_data[P2].parameter_name,
            test_data[P3].parameter_name,
            test_data[P4].parameter_name,
            test_data[P5].parameter_name,
            test_data[POOL].parameter_name));

    network net(engine, topo);
    net.set_input_data(test_data[BOXES].parameter_name, boxes);
    net.set_input_data(test_data[IMAGE_META].parameter_name, image_meta);
    net.set_input_data(test_data[P2].parameter_name, P2_tensor);
    net.set_input_data(test_data[P3].parameter_name, P3_tensor);
    net.set_input_data(test_data[P4].parameter_name, P4_tensor);
    net.set_input_data(test_data[P5].parameter_name, P5_tensor);
    net.set_input_data(test_data[POOL].parameter_name, pool_size);

    auto outputs = net.execute();

    EXPECT_EQ(outputs.size(), size_t(1));
    EXPECT_EQ(outputs.begin()->first, "pyramidROIAlign");

    auto output_mem = outputs.at("pyramidROIAlign").get_memory();
    auto output_ptr = output_mem.pointer<float>();

    int k = 0;
    for (float val1 : output_ptr)
    {
        EXPECT_NEAR(val1, answer[k++], 1e-5);
    }
}
