"""
 Copyright (c) 2017-2018 Intel Corporation

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

      http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
"""

import logging as log

import networkx as nx
import numpy as np

from mo.graph.graph import Node
from mo.ops.op import Op
from mo.utils.utils import refer_to_faq_msg


class Take(Op):
    ''' Implements regular numpy.take function

        It is assumed that there is no equivalent of this op in IE,
        so it is usually relevant to constant folding.
    '''
    op = 'Take'

    def __init__(self, graph: nx.MultiDiGraph, attrs: dict):
        mandatory_props = {
            'type': None, # do not set type as there is no IE equivalent
            'op': __class__.op,
            'axis': 0,
            'infer': __class__.infer
        }
        super().__init__(graph, mandatory_props, attrs)

    def supported_attrs(self):
        return [
            'axis',
        ]

    @staticmethod
    def infer(node: Node):
        assert len(node.in_nodes()) == 2
        data = node.in_node(0).value
        indices = node.in_node(1).value
        assert data is not None
        assert indices is not None
        assert node.axis is not None

        node.out_node(0).value = np.take(data, indices, node.axis)
        node.out_node(0).shape = np.array(node.out_node(0).value.shape, dtype=np.int64)
