// Copyright (C) 2018 Intel Corporation
//
// SPDX-License-Identifier: Apache-2.0
//

/**
 * \brief BMP reader
 * \file bmp.h
 */
#pragma once

#include <memory>
#include <string>
#include <format_reader.h>

#include "register.h"

namespace FormatReader {
/**
 * \class BitMap
 * \brief Reader for bmp files
 */
class BitMap : public Reader {
private:
    static Register<BitMap> reg;

    typedef struct {
        unsigned short type;                 /* Magic identifier            */
        unsigned int size;                   /* File size in bytes          */
        unsigned int reserved;
        unsigned int offset;                /* Offset to image data, bytes */
    } BmpHeader;

    typedef struct {
        unsigned int size;                  /* Header size in bytes      */
        int width, height;                  /* Width and height of image */
        unsigned short planes;              /* Number of colour planes   */
        unsigned short bits;                /* Bits per pixel            */
        unsigned int compression;           /* Compression type          */
        unsigned int imagesize;             /* Image size in bytes       */
        int xresolution, yresolution;       /* Pixels per meter          */
        unsigned int ncolours;              /* Number of colours         */
        unsigned int importantcolours;      /* Important colours         */
    } BmpInfoHeader;

public:
    /**
     * \brief Constructor of BMP reader
     * @param filename - path to input data
     * @return BitMap reader object
     */
    explicit BitMap(const std::string &filename);
    virtual ~BitMap() {
    }

    /**
     * \brief Get size
     * @return size
     */
    size_t size() const override {
        return _width * _height * 3;
    }

    void Release() noexcept override {
        delete this;
    }

    std::shared_ptr<unsigned char> getData(int width, int height) override {
        if ((width * height != 0) && (_width * _height != width * height)) {
            std::cout << "[ WARNING ] Image won't be resized! Please use OpenCV.\n";
            return nullptr;
        }
        return _data;
    }
};
}  // namespace FormatReader
