/*
 * SPDX-FileCopyrightText: 2023 The HedgeDoc developers (see AUTHORS file)
 *
 * SPDX-License-Identifier: AGPL-3.0-only
 */
import { AliasesSidebarEntry } from './specific-sidebar-entries/aliases-sidebar-entry/aliases-sidebar-entry'
import { DeleteNoteSidebarEntry } from './specific-sidebar-entries/delete-note-sidebar-entry/delete-note-sidebar-entry'
import { ExportMenuSidebarMenu } from './specific-sidebar-entries/export-menu-sidebar-menu'
import { ImportMenuSidebarMenu } from './specific-sidebar-entries/import-menu-sidebar-menu'
import { NoteInfoSidebarMenu } from './specific-sidebar-entries/note-info-sidebar-menu/note-info-sidebar-menu'
import { PermissionsSidebarEntry } from './specific-sidebar-entries/permissions-sidebar-entry/permissions-sidebar-entry'
import { PinNoteSidebarEntry } from './specific-sidebar-entries/pin-note-sidebar-entry/pin-note-sidebar-entry'
import { RevisionSidebarEntry } from './specific-sidebar-entries/revisions-sidebar-entry/revision-sidebar-entry'
import { ShareNoteSidebarEntry } from './specific-sidebar-entries/share-note-sidebar-entry/share-note-sidebar-entry'
import { UsersOnlineSidebarMenu } from './specific-sidebar-entries/users-online-sidebar-menu/users-online-sidebar-menu'
import styles from './style/sidebar.module.scss'
import { DocumentSidebarMenuSelection } from './types'
import React, { useCallback, useRef, useState } from 'react'
import { useClickAway } from 'react-use'

/**
 * Renders the sidebar for the editor.
 */
export const Sidebar: React.FC = () => {
  const sideBarRef = useRef<HTMLDivElement>(null)
  const [selectedMenu, setSelectedMenu] = useState<DocumentSidebarMenuSelection>(DocumentSidebarMenuSelection.NONE)

  useClickAway(sideBarRef, () => {
    setSelectedMenu(DocumentSidebarMenuSelection.NONE)
  })

  const toggleValue = useCallback(
    (toggleValue: DocumentSidebarMenuSelection): void => {
      const newValue = selectedMenu === toggleValue ? DocumentSidebarMenuSelection.NONE : toggleValue
      setSelectedMenu(newValue)
    },
    [selectedMenu]
  )

  const selectionIsNotNone = selectedMenu !== DocumentSidebarMenuSelection.NONE

  return (
    <div className={styles['slide-sidebar']}>
      <div ref={sideBarRef} className={`${styles['sidebar-inner']} ${selectionIsNotNone ? styles['show'] : ''}`}>
        <UsersOnlineSidebarMenu
          menuId={DocumentSidebarMenuSelection.USERS_ONLINE}
          selectedMenuId={selectedMenu}
          onClick={toggleValue}
        />
        <NoteInfoSidebarMenu
          menuId={DocumentSidebarMenuSelection.NOTE_INFO}
          selectedMenuId={selectedMenu}
          onClick={toggleValue}
        />
        <RevisionSidebarEntry hide={selectionIsNotNone} />
        <PermissionsSidebarEntry hide={selectionIsNotNone} />
        <AliasesSidebarEntry hide={selectionIsNotNone} />
        <ImportMenuSidebarMenu
          menuId={DocumentSidebarMenuSelection.IMPORT}
          selectedMenuId={selectedMenu}
          onClick={toggleValue}
        />
        <ExportMenuSidebarMenu
          menuId={DocumentSidebarMenuSelection.EXPORT}
          selectedMenuId={selectedMenu}
          onClick={toggleValue}
        />
        <ShareNoteSidebarEntry hide={selectionIsNotNone} />
        {/* TODO only show if user has permissions (Owner) (https://github.com/hedgedoc/hedgedoc/issues/5036) */}
        <DeleteNoteSidebarEntry hide={selectionIsNotNone} />
        <PinNoteSidebarEntry hide={selectionIsNotNone} />
      </div>
    </div>
  )
}
